package com.xforceplus.business.resource.controller;

import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.global.resource.ResourceApi;
import com.xforceplus.api.model.ResourceModel.Request.BindApis;
import com.xforceplus.api.model.ResourceModel.Request.Query;
import com.xforceplus.api.model.ResourceModel.Request.Save;
import com.xforceplus.bo.ResourceQueryBo;
import com.xforceplus.business.excel.ExcelFile;
import com.xforceplus.business.excel.QueryParam;
import com.xforceplus.business.file.controller.vo.ImportFileRespVo;
import com.xforceplus.business.file.service.ExportFileService;
import com.xforceplus.business.file.service.ImportFileService;
import com.xforceplus.business.resource.service.ResourcePortService;
import com.xforceplus.business.resource.service.ResourceService;
import com.xforceplus.domain.resource.ResourceDto;
import com.xforceplus.domain.resource.ResourceExtendDto;
import com.xforceplus.domain.resource.ServiceApiExtendDto;
import com.xforceplus.domain.validation.ValidationGroup;
import com.xforceplus.dto.resource.ResourceDTO;
import com.xforceplus.entity.ExcelFileStore;
import com.xforceplus.entity.Resource;
import com.xforceplus.tenant.security.autoscan.annotation.AuthorizedDefinition;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.utils.BasePathUtils;
import com.xforceplus.utils.DownloadUtils;
import com.xforceplus.utils.FileUtils;
import com.xforceplus.utils.excel.ResourceExcelUtils;
import com.xforceplus.utils.excel.ResourceWrapper;
import com.xforceplus.utils.filetransfer.FileTransferUtilsService;
import com.xforceplus.utils.yaml.ResourceYamlUtils;
import io.geewit.core.utils.reflection.BeanUtils;
import io.geewit.data.jpa.essential.domain.PageableFactory;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.validator.constraints.Range;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Controller;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import springfox.documentation.annotations.ApiIgnore;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import javax.validation.constraints.Min;
import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.xforceplus.api.common.Uri.PATH_GLOBAL_PREFIX;
import static com.xforceplus.business.excel.writer.ExcelConfigBusinessType.RESOURCE_EXCEL_FILE;
import static com.xforceplus.utils.FileUtils.DEFAULT_UPLOAD_MAX_FILE_SIZE;
import static com.xforceplus.utils.TemplateUtils.TEMPLATE;

@SuppressWarnings("all")
@Api(value = "资源码相关接口", tags = {"resources", "resource"})
@Validated
@Controller
public class ResourceController implements ResourceApi, com.xforceplus.api.tenant.resource.ResourceApi {
    private final static Logger logger = LoggerFactory.getLogger(ResourceController.class);

    private final FileTransferUtilsService fileTransferUtilsService;

    private final ResourceService resourceService;

    private final ResourcePortService resourcePortService;
    /**
     * 文件导出服务
     */

    private final ExportFileService exportFileService;

    private final ImportFileService importFileService;

    public ResourceController(ResourceService resourceService, ResourcePortService resourcePortService,
                              FileTransferUtilsService fileTransferUtilsService,
                              ExportFileService exportFileService, ImportFileService importFileService) {
        this.resourceService = resourceService;
        this.resourcePortService = resourcePortService;
        this.fileTransferUtilsService = fileTransferUtilsService;
        this.exportFileService = exportFileService;
        this.importFileService = importFileService;
    }

    /**
     * 导出资源码列表
     *
     * @return ResponseEntity<Long>
     */
    @AuthorizedDefinition(resources = {"xforce:operation:resource:read"})
    @ApiOperation(value = "异步导出资源码列表", notes = "异步导出资源码列表")
    @RequestMapping(name = "异步导出资源码列表", value = {PATH_GLOBAL_PREFIX + "/resources/async/export"}, method = RequestMethod.POST)
    @ResponseBody
    public ResponseEntity<ImportFileRespVo> asyncExport(@ApiParam(value = "request") @RequestBody Query query, @ApiParam(value = "sort") Sort sort) {
        //校验是选择列表
        if (ArrayUtils.isEmpty(query.getSheets())) {
            throw new IllegalArgumentException("请选择SheetName");
        }
        //判断是否登录
        if (UserInfoHolder.get() == null) {
            throw new IllegalArgumentException("请登录");
        }
        //异步执行
        QueryParam queryParam = QueryParam.builder()
                .param(ExcelFile.PARAMS_QUERY, query)
                .param(ExcelFile.PARAMS_SORT, sort)
                .build();

        ResourceQueryBo queryBo = new ResourceQueryBo();
        BeanUtils.copyProperties(query, queryBo);
        ExcelFileStore excelFileStore = this.exportFileService.asyncExcelExport(queryParam.params(), RESOURCE_EXCEL_FILE);
        ImportFileRespVo respVo = new ImportFileRespVo();
        respVo.setId(excelFileStore.getId());
        respVo.setBusinessType(RESOURCE_EXCEL_FILE.getBusinessName());
        //检查ExcelFile文件是否正确
        return ResponseEntity.ok(respVo);
    }

    /**
     * 导出资源码列表
     *
     * @return ResponseEntity<Long>
     */
    @AuthorizedDefinition(resources = {"xforce:operation:resource:read"})
    @ApiOperation(value = "导出资源码列表", notes = "导出资源码列表")
    @RequestMapping(name = "导出资源码列表", value = {"/api/global/resources/export"}, method = RequestMethod.POST)
    @ResponseBody
    public ResponseEntity<Long> export(@RequestBody List<String> resourceIds, HttpServletRequest request, HttpServletResponse response) {
        if (CollectionUtils.isEmpty(resourceIds)) {
            throw new IllegalArgumentException("id记录不能为空。");
        }
        List<Long> list = resourceIds.stream().map(Long::valueOf).collect(Collectors.toList());
        List<ResourceExtendDto> extendList = resourcePortService.getExportData(list);
        List<ServiceApiExtendDto> serviceApiList = resourcePortService.getServiceApiData(extendList);
        ResourceWrapper wrapper = new ResourceWrapper();
        wrapper.setResources(extendList);
        wrapper.setServiceApis(serviceApiList);
        String filePath = BasePathUtils.ensureFilePath(request);
        ResourceExcelUtils.writeV2007(filePath, wrapper);
        try {
            Long fileId = fileTransferUtilsService.upload(filePath);
            ResponseEntity<Long> responseEntity = ResponseEntity.ok(fileId);
            deleteFile(filePath);
            return responseEntity;
        } catch (Exception e) {
            deleteFile(filePath);
            logger.error("上传文件失败", e);
            throw new IllegalArgumentException("上传文件失败。");
        }
    }

    /**
     * 导出资源码列表
     */
    @AuthorizedDefinition(resources = {"xforce:operation:resource:read"})
    @ResponseBody
    @ApiOperation(value = "导出资源码yaml文件", notes = "导出资源码yaml文件")
    @RequestMapping(name = "导出资源码yaml文件", value = {PATH_GLOBAL_PREFIX + "/resources/yaml-export"}, method = RequestMethod.GET)
    public void exportYaml(@RequestParam("appId") Long appId, HttpServletRequest request, HttpServletResponse response) {
        if (appId == null || appId <= 0L) {
            throw new IllegalArgumentException("appId不能为空或者负值。");
        }
        List<ResourceExtendDto> extendList = resourcePortService.getExportDataByAppId(appId);
        if (CollectionUtils.isEmpty(extendList)) {
            throw new IllegalArgumentException("没有查询到资源码记录。");
        }
        Map<String, Object> yamlMap = resourcePortService.getYamlMapByResources(extendList);
        String filePath = BasePathUtils.ensureFilePath(request);
        filePath = filePath.replace(".xlsx", ".yml");
        ResourceYamlUtils.writeYmlResource(filePath, yamlMap);
        try {
            DownloadUtils.fileToDownload(response, filePath, "resourcecode.yml");
            deleteFile(filePath);
        } catch (Exception e) {
            deleteFile(filePath);
            logger.error("下载文件失败", e);
            throw new IllegalArgumentException("下载文件失败。");
        }
    }

    /**
     * 下载文件
     *
     * @return
     */
    @Deprecated
    @AuthorizedDefinition(resources = {"xforce:operation:resource:read"})
    @ApiOperation(value = "下载文件", notes = "下载文件")
    @RequestMapping(name = "下载文件", value = {"/api/global/resources/file/{fileId}"}, method = RequestMethod.GET)
    public void download(@PathVariable String fileId, HttpServletRequest request,
                         HttpServletResponse response) {
        String filePath;
        if (StringUtils.equals(TEMPLATE, fileId)) {
            filePath = BasePathUtils.getTemplatePath(request, "resourceImport.xlsx");
        } else {
            filePath = BasePathUtils.ensureFilePath(request);
            try {
                fileTransferUtilsService.download(filePath, Long.valueOf(fileId));
            } catch (Exception e) {
                this.deleteFile(filePath);
                logger.error("下载文件失败", e);
                throw new IllegalArgumentException("下载文件失败。");
            }
        }
        DownloadUtils.fileToDownload(response, filePath, "resourceImport.xlsx");
        if (!StringUtils.equals(TEMPLATE, fileId) && StringUtils.isNotBlank(filePath)) {
            this.deleteFile(filePath);
        }
    }

    /**
     * 异步导入资源码列表
     *
     * @return ResponseEntity<ImportFileRespVo> 导入对象
     */
    @AuthorizedDefinition(resources = {"xforce:operation:resource:save"})
    @ApiOperation(value = "异步导入资源码列表", notes = "异步导入资源码列表")
    @RequestMapping(name = "异步导入资源码列表", value = {PATH_GLOBAL_PREFIX + "/resources/async/import"}, method = RequestMethod.POST)
    @ResponseBody
    public ResponseEntity<ImportFileRespVo> asyncImport(@RequestParam(name = "file", required = false) MultipartFile file) {
        Long tenantId = 0L;
        Long userId = 0L;
        if (null != UserInfoHolder.get()) {
            tenantId = UserInfoHolder.get().getTenantId();
            userId = UserInfoHolder.get().getId();
        }

        //保存数据和上传文件，并转为异步处理
        ExcelFileStore excelFileStore = this.importFileService.create(file, RESOURCE_EXCEL_FILE, userId, tenantId);
        ImportFileRespVo respVo = new ImportFileRespVo();
        respVo.setBusinessType(RESOURCE_EXCEL_FILE.name());
        respVo.setFileId(excelFileStore.getSourceFileId());
        respVo.setId(excelFileStore.getId());
        //检查ExcelFile文件是否正确
        return ResponseEntity.ok(respVo);
    }


    /**
     * 导入资源码列表
     *
     * @return ResponseEntity<Long> fileId
     */
    @Deprecated
    @AuthorizedDefinition(resources = {"xforce:operation:resource:save"})
    @ApiOperation(value = "导入资源码列表", notes = "导入资源码列表")
    @RequestMapping(name = "导入资源码列表", value = {PATH_GLOBAL_PREFIX + "/resources/import"}, method = RequestMethod.POST)
    @ResponseBody
    public ResponseEntity<Long> importData(HttpServletRequest request, HttpServletResponse response, @RequestParam(name = "file", required = false) MultipartFile file) {
        if (file == null) {
            throw new IllegalArgumentException("请选择导入文件。");
        }
        //检查文件是否小于15M，大于等于15M，则报错
        FileUtils.checkFileSize(file, DEFAULT_UPLOAD_MAX_FILE_SIZE);
        String filePath = BasePathUtils.ensureFilePath(request);
        saveFileToLocal(file, filePath);
        ResourceWrapper resourceWrapper = ResourceExcelUtils.extract(filePath);
        ResourceWrapper result = resourcePortService.saveResourceData(resourceWrapper);
        Long fileId = null;
        if (CollectionUtils.isNotEmpty(result.getResources())) {
            String filePath2 = BasePathUtils.ensureFilePath(request);
            result.setSkipReason(false);
            ResourceExcelUtils.writeV2007(filePath2, result);
            try {
                fileId = fileTransferUtilsService.upload(filePath2);
                deleteFile(filePath2);
            } catch (Exception e) {
                deleteFile(filePath);
                deleteFile(filePath2);
                logger.error("上传文件失败", e);
                throw new IllegalArgumentException("上传文件失败。");
            }
        }
        ResponseEntity<Long> responseEntity = ResponseEntity.ok(fileId);
        deleteFile(filePath);
        return responseEntity;
    }


    /**
     * 保存文件到本地
     *
     * @param file     文件
     * @param basePath 路径
     */
    private String saveFileToLocal(MultipartFile file, String basePath) {
        File newFile = new File(basePath);
        try {
            file.transferTo(newFile);
        } catch (IOException e) {
            logger.error("导入文件失败", e);
            throw new IllegalArgumentException("导入文件失败");
        }
        return basePath;
    }

    @AuthorizedDefinition(resources = {"xforce:operation:resource:read", "xforce:operation:user-resource:read"})
    @ApiOperation(value = "获取资源码分页", notes = "通过分页参数，获取资源码分页")
    @Override
    public ResponseEntity<Page<Resource>> page(@ApiParam(value = "request") Query query,
                                               @ApiParam(value = "pageable") Pageable pageable) {
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        Set<String> withExtendParams = Stream.of("appName").collect(Collectors.toSet());
        if (StringUtils.isNotBlank(query.getWithExtendParams())) {
            withExtendParams.addAll(Arrays.stream(StringUtils.split(query.getWithExtendParams(), ",")).collect(Collectors.toSet()));
        }
        query.setWithExtendParams(withExtendParams.stream().collect(Collectors.joining(",")));
        Page<Resource> page = resourceService.page(query, currentPageable);
        return ResponseEntity.ok(page);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:resource:read"})
    @ApiIgnore
    @ApiOperation(value = "获取资源码列表", notes = "通过参数，获取资源码列表")
    @ResponseBody
    @RequestMapping(name = "资源码列表", value = ResourceApi.Path.LIST, method = RequestMethod.GET)
    public ResponseEntity<List<Resource>> list(Query query, Sort sort) {
        if (null == query.getResourceId()) {
            throw new IllegalArgumentException("功能集id不能为空");
        }
        Set<String> withExtendParams = Stream.of("appName").collect(Collectors.toSet());
        if (StringUtils.isNotBlank(query.getWithExtendParams())) {
            withExtendParams.addAll(Arrays.stream(StringUtils.split(query.getWithExtendParams(), ",")).collect(Collectors.toSet()));
        }
        query.setWithExtendParams(withExtendParams.stream().collect(Collectors.joining(",")));
        if (sort.stream().noneMatch(order -> "createTime".equals(order.getProperty()))) {
            sort = sort.and(Sort.by(Sort.Direction.DESC, "createTime"));
        }
        List<Resource> result = resourceService.list(query, sort);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:resource:save"})
    @ApiOperation(value = "新增资源码")
    @Override
    public ResponseEntity<Resource> create(@ApiParam(value = "model", required = true) @Validated(ValidationGroup.OnCreate.class) Save model) {
        Resource result = resourceService.save(model);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:resource:save"})
    @ApiOperation(value = "修改资源码详情")
    @Override
    public ResponseEntity<Resource> update(@ApiParam(value = "id", required = true) long id,
                                           @ApiParam(value = "model", required = true) Save model) {
        Resource result = resourceService.update(id, model);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:resource:read"})
    @ApiOperation(value = "获取资源码详情")
    @Override
    public ResponseEntity<Resource> info(@ApiParam(value = "id", required = true) long id) {
        Resource result = resourceService.findById(id);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:resource:save"})
    @ApiOperation(value = "删除资源码")
    @Override
    public ResponseEntity<String> delete(@ApiParam(value = "id", required = true) long id) {
        resourceService.deleteById(id);
        return ResponseEntity.ok("删除成功");
    }

    @AuthorizedDefinition(resources = {"xforce:operation:resource:save"})
    @ApiOperation("更新资源码状态")
    @Override
    public ResponseEntity<String> updateStatus(long id, int status) {
        resourceService.updateStatus(id, status);
        return ResponseEntity.ok("更新成功");
    }

    @AuthorizedDefinition(resources = {"xforce:operation:resource:save"})
    @ApiOperation("在父资源码下添加资源码")
    @Override
    public ResponseEntity<Resource> append(@ApiParam(value = "id", required = true) long id,
                                           @ApiParam(value = "model", required = true) Save model) {
        Resource result = resourceService.append(id, model);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:resource:save"})
    @ApiOperation("资源码批量绑定接口")
    @Override
    public ResponseEntity<String> bindApis(long id, BindApis bindApis) {
        resourceService.bindApis(id, bindApis);
        return ResponseEntity.ok("绑定成功");
    }

    private void deleteFile(String filePath) {
        File file = new File(filePath);
        if (file.exists()) {
            boolean deleteResult = file.delete();
            logger.info("删除文件结果:{}", deleteResult);
        }
    }

    @AuthorizedDefinition(resources = {"xforce:operation:resource:read"})
    @ApiOperation(value = "获取租户关联的资源码分页", notes = "通过分页参数，获取租户关联的资源码分页")
    @Override
    public ResponseEntity<Page<Resource>> page(long tenantId, Query query, Pageable pageable) {
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        if (tenantId > 0) {
            query.setTenantId(tenantId);
        }
        Set<String> withExtendParams = Stream.of("appName").collect(Collectors.toSet());
        if (StringUtils.isNotBlank(query.getWithExtendParams())) {
            withExtendParams.addAll(Arrays.stream(StringUtils.split(query.getWithExtendParams(), ",")).collect(Collectors.toSet()));
        }
        query.setWithExtendParams(withExtendParams.stream().collect(Collectors.joining(",")));
        Page<Resource> page = resourceService.page(query, currentPageable);
        return ResponseEntity.ok(page);
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:operation:resource:read"})
    @ApiOperation(value = "根据用户ID 获取用户role 关联的资源码", notes = "根据用户ID 获取用户role 关联的资源码")
    public ResponseEntity<List<ResourceDto>> getUserRoleResource(Long userId) {
        List<ResourceDTO> resourceDTOList = resourceService.getUserRoleResourceCode(userId);
        List<ResourceDto> resultList = resourceDTOList.stream().map(l -> {
            ResourceDto resourceDto = new ResourceDto();
            BeanUtils.copyProperties(l, resourceDto);
            return resourceDto;
        }).collect(Collectors.toList());

        return ResponseEntity.ok(resultList);
    }
}
