package com.xforceplus.business.resource.controller;

import com.fasterxml.jackson.annotation.JsonView;
import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.global.resource.ResourcesetApi;
import com.xforceplus.api.model.ResourcesetModel.Request.Create;
import com.xforceplus.api.model.ResourcesetModel.Request.Query;
import com.xforceplus.api.model.ResourcesetModel.Request.Save;
import com.xforceplus.bo.ResourceQueryBo;
import com.xforceplus.business.excel.ExcelFile;
import com.xforceplus.business.excel.QueryParam;
import com.xforceplus.business.file.controller.vo.ImportFileRespVo;
import com.xforceplus.business.file.service.ExportFileService;
import com.xforceplus.business.file.service.ImportFileService;
import com.xforceplus.business.resource.service.ResourcesetPortService;
import com.xforceplus.business.resource.service.ResourcesetService;
import com.xforceplus.domain.validation.ValidationGroup;
import com.xforceplus.entity.ExcelFileStore;
import com.xforceplus.entity.Resourceset;
import com.xforceplus.tenant.core.exception.UnknownException;
import com.xforceplus.tenant.security.autoscan.annotation.AuthorizedDefinition;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.utils.BasePathUtils;
import com.xforceplus.utils.DownloadUtils;
import com.xforceplus.utils.FileUtils;
import com.xforceplus.utils.excel.ResourcesetExcelUtils;
import com.xforceplus.utils.excel.ResourcesetWrapper;
import com.xforceplus.utils.filetransfer.FileTransferUtilsService;
import io.geewit.core.jackson.view.View;
import io.geewit.core.utils.reflection.BeanUtils;
import io.geewit.data.jpa.essential.domain.PageableFactory;
import io.geewit.data.jpa.essential.search.DynamicSpecifications;
import io.geewit.data.jpa.essential.search.SearchFilter;
import io.geewit.data.jpa.essential.web.servlet.Servlets;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.validator.constraints.Range;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Controller;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.context.request.WebRequest;
import org.springframework.web.multipart.MultipartFile;
import springfox.documentation.annotations.ApiIgnore;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import javax.validation.constraints.Min;
import java.io.*;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static com.xforceplus.api.common.Uri.PATH_GLOBAL_PREFIX;
import static com.xforceplus.business.excel.writer.ExcelConfigBusinessType.RESOURCE_SET_EXCEL_FILE;
import static com.xforceplus.utils.FileUtils.DEFAULT_UPLOAD_MAX_FILE_SIZE;

@SuppressWarnings("all")
@Api(value = "功能集相关接口", tags = {"resources", "resourceSet"})
@Validated
@Controller
public class ResourceSetController implements ResourcesetApi, com.xforceplus.api.tenant.resource.ResourceSetApi {

    /**
     * template 常量
     */
    public static final String TEMPLATE = "template";

    private final static Logger logger = LoggerFactory.getLogger(ResourceController.class);

    private final ResourcesetService resourcesetService;

    private final ResourcesetPortService resourcesetPortService;
    /**
     * 文件服务
     */
    private final FileTransferUtilsService fileTransferUtilsService;
    /**
     * 文件导出服务
     */
    private final ExportFileService exportFileService;
    /**
     * 文件导入服务
     */
    private final ImportFileService importFileService;


    public ResourceSetController(ResourcesetService resourcesetService, ResourcesetPortService resourcesetPortService,
                                 ExportFileService exportFileService,
                                 FileTransferUtilsService fileTransferUtilsService, ImportFileService importFileService) {
        this.resourcesetService = resourcesetService;
        this.resourcesetPortService = resourcesetPortService;
        this.exportFileService = exportFileService;
        this.fileTransferUtilsService = fileTransferUtilsService;
        this.importFileService = importFileService;
    }

    /**
     * 获取功能集分页列表
     *
     * @param request
     * @param pageable
     * @return
     */
    @AuthorizedDefinition(resources = {"xforce:operation:resourceset:read"})
    @ApiIgnore
    @ApiOperation(value = "获取功能集分页", notes = "通过分页参数，获取功能集分页")
    @ResponseBody
    @RequestMapping(name = "功能集分页列表查询", value = {"/api/global/resourcesset/query"}, method = RequestMethod.GET)
    public Page<Resourceset> page(WebRequest request, @ApiParam(value = "pageable") Pageable pageable) {
        Map<String, Object> searchParams = Servlets.getParametersStartingWith(request, "search_");
        Collection<SearchFilter> filters = SearchFilter.parse(searchParams);
        Specification<Resourceset> specification = DynamicSpecifications.bySearchFilter(filters);
        Page<Resourceset> page = resourcesetService.page(specification, pageable);
        return page;
    }

    @AuthorizedDefinition(resources = {"xforce:operation:resourceset:read"})
    @JsonView(View.Page.class)
    @ApiOperation(value = "获取功能集分页", notes = "通过分页参数，获取功能集分页")
    @Override
    public ResponseEntity<Page<Resourceset>> page(@ApiParam(value = "request") Query query,
                                                  @ApiParam(value = "pageable") Pageable pageable) {
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        query.setContainCustomFlag(false);
        Page<Resourceset> result = resourcesetService.page(query, currentPageable);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:resourceset:read"})
    @JsonView(View.Info.class)
    @ApiOperation(value = "获取功能集列表", notes = "通过参数，获取功能集列表")
    @ResponseBody
    @RequestMapping(name = "功能集列表", value = ResourcesetApi.Path.LIST, method = RequestMethod.GET)
    public ResponseEntity<List<Resourceset>> list(@ApiParam(value = "request") Query query,
                                                  @ApiParam(value = "sort") Sort sort) {
        if (sort.stream().noneMatch(order -> "createTime".equals(order.getProperty()))) {
            sort = sort.and(Sort.by(Sort.Direction.DESC, "createTime"));
        }
        List<Resourceset> result = resourcesetService.list(query, sort);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:resourceset:save"})
    @JsonView(View.Page.class)
    @ApiOperation(value = "新增功能集")
    @Override
    public ResponseEntity<Resourceset> create(@ApiParam(value = "model", required = true) Create model) {
        Resourceset result = resourcesetService.create(model);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:resourceset:save"})
    @JsonView(View.Page.class)
    @ApiOperation(value = "更新功能集")
    @Override
    public ResponseEntity<Resourceset> update(@ApiParam(value = "id", required = true) long id,
                                              @ApiParam(value = "model", required = true) Save model) {
        Resourceset result = resourcesetService.update(id, model);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:resourceset:save"})
    @ApiOperation(value = "修改功能集状态")
    @Override
    public ResponseEntity<String> updateStatus(long id, int status) {
        resourcesetService.updateStatus(id, status);
        return ResponseEntity.ok("修改成功");
    }

    @AuthorizedDefinition(resources = {"xforce:operation:resourceset:read"})
    @ApiOperation(value = "获取功能集详情")
    @Override
    public ResponseEntity<Resourceset> info(@ApiParam(value = "id", required = true) long id) {
        Resourceset result = resourcesetService.findById(id);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:resourceset:save"})
    @ApiOperation(value = "删除功能集")
    @Override
    public ResponseEntity<String> delete(@ApiParam(value = "id", required = true) long id) {
        resourcesetService.deleteById(id);
        return ResponseEntity.ok("删除成功");
    }

    /**
     * 导出功能集列表
     *
     * @return
     */
    @AuthorizedDefinition(resources = {"xforce:operation:resourceset:read"})
    @ApiOperation(value = "导出功能集列表", notes = "导出功能集列表")
    @RequestMapping(name = "导出功能集列表", value = {PATH_GLOBAL_PREFIX + "/resourcesets/export"}, method = RequestMethod.POST)
    @ResponseBody
    public ResponseEntity<Long> export(@RequestBody List<String> resourceSetIds, HttpServletRequest request, HttpServletResponse response) {
        if (CollectionUtils.isEmpty(resourceSetIds)) {
            throw new UnknownException("id记录不能为空。");
        }
        List<Long> list = resourceSetIds.stream().map(Long::valueOf).collect(Collectors.toList());
        ResourcesetWrapper resourcesetWrapper = resourcesetPortService.getExportData(list);
        String filePath = BasePathUtils.ensureFilePath(request);
        ResourcesetExcelUtils.writeV2007(filePath, resourcesetWrapper);
        try {
            Long fileId = fileTransferUtilsService.upload(filePath);
            ResponseEntity<Long> responseEntity = ResponseEntity.ok(fileId);
            deleteFile(filePath);
            return responseEntity;
        } catch (Exception e) {
            deleteFile(filePath);
            logger.error("上传文件失败", e);
            throw new UnknownException("上传文件失败。");
        }
    }

    /**
     * 导出功能集列表
     *
     * @return ResponseEntity<ImportFileRespVo>
     */
    @AuthorizedDefinition(resources = {"xforce:operation:resourceset:read"})
    @ApiOperation(value = "异步导出功能集列表", notes = "异步导出功能集列表")
    @RequestMapping(name = "异步导出功能集列表", value = {PATH_GLOBAL_PREFIX + "/resourcesets/async/export"}, method = RequestMethod.POST)
    @ResponseBody
    public ResponseEntity<ImportFileRespVo> asyncExport(@ApiParam(value = "request") @RequestBody Query query,
                                                        @ApiParam(value = "sort") Sort sort) {
        //校验是选择列表
        if (ArrayUtils.isEmpty(query.getSheets())) {
            throw new UnknownException("请选择SheetName");
        }
        //异步执行
        QueryParam queryParam = QueryParam.builder()
                .param(ExcelFile.PARAMS_QUERY, query)
                .param(ExcelFile.PARAMS_SORT, sort)
                .build();

        ResourceQueryBo queryBo = new ResourceQueryBo();
        BeanUtils.copyProperties(query, queryBo);
        ExcelFileStore excelFileStore = this.exportFileService.asyncExcelExport(queryParam.params(), RESOURCE_SET_EXCEL_FILE);
        ImportFileRespVo respVo = new ImportFileRespVo();
        respVo.setId(excelFileStore.getId());
        respVo.setBusinessType(RESOURCE_SET_EXCEL_FILE.getBusinessName());
        //检查ExcelFile文件是否正确
        return ResponseEntity.ok(respVo);
    }


    /**
     * 异步导入功能集列表
     *
     * @return ResponseEntity<ImportFileRespVo> 导入对象
     */
    @AuthorizedDefinition(resources = {"xforce:operation:resourceset:save"})
    @ApiOperation(value = "异步导入功能集列表", notes = "异步导入功能集列表")
    @RequestMapping(name = "异步导入功能集列表", value = {PATH_GLOBAL_PREFIX + "/resourcesets/async/import"}, method = RequestMethod.POST)
    @ResponseBody
    public ResponseEntity<ImportFileRespVo> asyncImport(@RequestParam(name = "file", required = false) MultipartFile file) {
        Long tenantId = 0L;
        Long userId = 0L;
        if (null != UserInfoHolder.get()) {
            tenantId = UserInfoHolder.get().getTenantId();
            userId = UserInfoHolder.get().getId();
        }
        //保存数据和上传文件，并转为异步处理
        ExcelFileStore excelFileStore = this.importFileService.create(file, RESOURCE_SET_EXCEL_FILE, userId, tenantId);
        ImportFileRespVo respVo = new ImportFileRespVo();
        respVo.setBusinessType(RESOURCE_SET_EXCEL_FILE.name());
        respVo.setFileId(excelFileStore.getSourceFileId());
        respVo.setId(excelFileStore.getId());
        //检查ExcelFile文件是否正确
        return ResponseEntity.ok(respVo);
    }

    /**
     * 下载文件
     *
     * @return
     */
    @AuthorizedDefinition(resources = {"xforce:operation:resourceset:read"})
    @ApiOperation(value = "下载文件", notes = "下载文件")
    @RequestMapping(name = "下载文件", value = {"/api/global/resourcesets/file/{fileId}"}, method = RequestMethod.GET)
    public void download(@PathVariable String fileId, HttpServletRequest request,
                         HttpServletResponse response) {
        String filePath;
        if (StringUtils.equals(TEMPLATE, fileId)) {
            filePath = BasePathUtils.getTemplatePath(request, "resourceset.xlsx");
        } else {
            filePath = BasePathUtils.ensureFilePath(request);
            try {
                fileTransferUtilsService.download(filePath, Long.valueOf(fileId));
            } catch (Exception e) {
                deleteFile(filePath);
                logger.error("下载文件失败", e);
                throw new UnknownException("下载文件失败。");
            }
        }
        DownloadUtils.fileToDownload(response, filePath, "resourceset.xlsx");
        deleteFile(filePath);
    }

    /**
     * 导入功能集列表
     *
     * @return
     */
    @AuthorizedDefinition(resources = {"xforce:operation:resourceset:save"})
    @ApiOperation(value = "导入功能集列表", notes = "导入功能集列表")
    @RequestMapping(name = "导入功能集列表", value = {PATH_GLOBAL_PREFIX + "/resourcesets/import"}, method = RequestMethod.POST)
    @ResponseBody
    public ResponseEntity<Long> importData(HttpServletRequest request, HttpServletResponse response, @RequestParam(name = "file", required = false) MultipartFile file) {
        if (file == null) {
            throw new UnknownException("请选择导入文件。");
        }
        FileUtils.checkFileSize(file, DEFAULT_UPLOAD_MAX_FILE_SIZE);
        String filePath = BasePathUtils.ensureFilePath(request);
        saveFileToLocal(file, filePath);
        ResourcesetWrapper resourcesetWrapper = ResourcesetExcelUtils.extract(filePath);
        ResourcesetWrapper result = resourcesetPortService.saveResourcesetData(resourcesetWrapper);
        Long fileId = null;
        if (CollectionUtils.isNotEmpty(result.getResourcesets())) {
            String filePath2 = BasePathUtils.ensureFilePath(request);
            result.setSkipReason(false);
            ResourcesetExcelUtils.writeV2007(filePath2, result);
            try {
                fileId = fileTransferUtilsService.upload(filePath2);
                deleteFile(filePath2);
            } catch (Exception e) {
                deleteFile(filePath);
                deleteFile(filePath2);
                logger.error("上传文件失败", e);
                throw new UnknownException("上传文件失败。");
            }
        }
        ResponseEntity<Long> responseEntity = ResponseEntity.ok(fileId);
        deleteFile(filePath);
        return responseEntity;
    }

    /**
     * 保存文件到本地
     */
    private String saveFileToLocal(MultipartFile file, String basePath) {
        File newFile = new File(basePath);
        try {
            file.transferTo(newFile);
        } catch (IOException e) {
            logger.error("导入文件失败", e);
            throw new UnknownException("导入文件失败");
        }
        return basePath;
    }

    private InputStream getInputStreamByFile(String newFilePath) {
        InputStream in;
        try {
            in = new FileInputStream(new File(newFilePath));
        } catch (FileNotFoundException e) {
            logger.error("读取文件失败,文件没有找到", e);
            throw new UnknownException("读取文件失败，文件没有找到。");
        }
        return in;
    }

    private void deleteFile(String filePath) {
        File file = new File(filePath);
        if (file.exists()) {
            boolean deleteResult = file.delete();
            logger.info("删除文件结果:{}", deleteResult);
        }
    }

    @AuthorizedDefinition(authorization = false)
    @JsonView(View.Page.class)
    @ApiOperation(value = "获取租户关联功能集分页", notes = "通过分页参数，获取租户关联功能集分页")
    @Override
    public ResponseEntity<Page<Resourceset>> page(Long tenantId, Query query, Pageable pageable) {
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        if (tenantId > 0) {
            query.setTenantId(tenantId);
        }
        Page<Resourceset> result = resourcesetService.page(query, currentPageable);
        return ResponseEntity.ok(result);
    }
}
