package com.xforceplus.business.resource.controller;

import com.fasterxml.jackson.annotation.JsonView;
import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.global.resource.ServicePackageApi;
import com.xforceplus.api.model.ServicePackageModel.Request.Query;
import com.xforceplus.api.model.ServicePackageModel.Request.Save;
import com.xforceplus.bo.ResourceQueryBo;
import com.xforceplus.business.excel.ExcelFile;
import com.xforceplus.business.excel.QueryParam;
import com.xforceplus.business.file.controller.vo.ImportFileRespVo;
import com.xforceplus.business.file.service.ExportFileService;
import com.xforceplus.business.file.service.ImportFileService;
import com.xforceplus.business.resource.service.ServicePackagePortService;
import com.xforceplus.business.resource.service.ServicePackageService;
import com.xforceplus.domain.TreeNode;
import com.xforceplus.domain.resource.view.ServicePackageView;
import com.xforceplus.domain.validation.ValidationGroup;
import com.xforceplus.entity.ExcelFileStore;
import com.xforceplus.entity.ServicePackage;
import com.xforceplus.tenant.core.exception.UnknownException;
import com.xforceplus.tenant.security.autoscan.annotation.AuthorizedDefinition;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.utils.BasePathUtils;
import com.xforceplus.utils.DownloadUtils;
import com.xforceplus.utils.excel.ServicePackageExcelUtils;
import com.xforceplus.utils.excel.ServicePackageWrapper;
import com.xforceplus.utils.filetransfer.FileTransferUtilsService;
import io.geewit.core.jackson.view.View;
import io.geewit.core.utils.reflection.BeanUtils;
import io.geewit.data.jpa.essential.domain.PageableFactory;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.validator.constraints.Range;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Controller;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import springfox.documentation.annotations.ApiIgnore;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import javax.validation.constraints.Min;
import java.io.File;
import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.xforceplus.api.common.Uri.PATH_GLOBAL_PREFIX;
import static com.xforceplus.business.excel.writer.ExcelConfigBusinessType.SERVICE_PACKAGE_EXCEL_FILE;
import static com.xforceplus.utils.TemplateUtils.TEMPLATE;


/**
 * 服务包相关接口
 *
 * @author geewit
 */
@SuppressWarnings("all")
@Api(value = "服务包相关接口", tags = {"resources", "servicePackage"})
@Validated
@Controller
public class ServicePackageController implements ServicePackageApi, com.xforceplus.api.tenant.resource.ServicePackageApi {
    private final static Logger logger = LoggerFactory.getLogger(ServicePackageController.class);
    private final ServicePackageService servicePackageService;
    private final ServicePackagePortService servicePackagePortService;
    private final FileTransferUtilsService fileTransferUtilsService;


    private final ExportFileService exportFileService;

    private final ImportFileService importFileService;

    public ServicePackageController(ServicePackageService servicePackageService,
                                    ServicePackagePortService servicePackagePortService,
                                    ExportFileService exportFileService,
                                    FileTransferUtilsService fileTransferUtilsService,
                                    ImportFileService importFileService) {
        this.servicePackageService = servicePackageService;
        this.servicePackagePortService = servicePackagePortService;
        this.fileTransferUtilsService = fileTransferUtilsService;
        this.exportFileService = exportFileService;
        this.importFileService = importFileService;
    }

    @AuthorizedDefinition(resources = {"xforce:operation:service-package:read"})
    @JsonView(ServicePackageView.ServicePackageInfo.class)
    @ApiOperation(value = "获取服务包分页", notes = "通过分页参数，获取服务包分页")
    @Override
    public ResponseEntity<Page<ServicePackage>> page(@ApiParam(value = "request") Query query,
                                                     @ApiParam(value = "pageable") Pageable pageable) {
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        Set<String> withExtendParams = Stream.of("appName").collect(Collectors.toSet());
        if (StringUtils.isNotBlank(query.getWithExtendParams())) {
            withExtendParams.addAll(Arrays.stream(StringUtils.split(query.getWithExtendParams(), ",")).collect(Collectors.toSet()));
        }
        query.setWithExtendParams(withExtendParams.stream().collect(Collectors.joining(",")));
        Page<ServicePackage> result = servicePackageService.page(query, currentPageable);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:service-package:read"})
    @JsonView(View.Page.class)
    @ApiIgnore
    @ApiOperation(value = "获取服务包列表", notes = "通过参数，获取服务包列表")
    @ResponseBody
    @RequestMapping(name = "服务包分页列表", value = ServicePackageApi.Path.LIST, method = RequestMethod.GET)
    public ResponseEntity<List<ServicePackage>> list(@ApiParam(value = "request") Query query,
                                                     @ApiParam(value = "sort") Sort sort) {
        if (sort.stream().noneMatch(order -> "createTime".equals(order.getProperty()))) {
            sort = Sort.by(Sort.Direction.DESC, "createTime").and(sort);
        }
        Set<String> withExtendParams = Stream.of("appName").collect(Collectors.toSet());
        if (StringUtils.isNotBlank(query.getWithExtendParams())) {
            withExtendParams.addAll(Arrays.stream(StringUtils.split(query.getWithExtendParams(), ",")).collect(Collectors.toSet()));
        }
        query.setWithExtendParams(withExtendParams.stream().collect(Collectors.joining(",")));
        List<ServicePackage> result = servicePackageService.list(query, sort);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:service-package:save"})
    @ApiOperation(value = "新建服务包")
    @Override
    public ResponseEntity<ServicePackage> create(@ApiParam(value = "model", required = true) Save model) {
        ServicePackage result = servicePackageService.create(model);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:service-package:save"})
    @JsonView(ServicePackageView.ServicePackageInfo.class)
    @ApiOperation(value = "根据id更新服务包")
    @Override
    public ResponseEntity<ServicePackage> update(@ApiParam(value = "id", required = true) long id,
                                                 @ApiParam(value = "model", required = true) @Validated(ValidationGroup.OnUpdate.class) Save model) {
        ServicePackage result = servicePackageService.update(id, model);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:service-package:read"})
    @JsonView(ServicePackageView.ServicePackageInfo.class)
    @ApiOperation(value = "获取服务包详情")
    @Override
    public ResponseEntity<ServicePackage> info(@ApiParam(value = "id", required = true) long id) {
        ServicePackage result = servicePackageService.findById(id);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:service-package:save"})
    @ApiOperation(value = "根据id更新服务包状态")
    @Override
    public ResponseEntity<String> updateStatus(@ApiParam(value = "id", required = true) long id,
                                               @ApiParam(value = "status", required = true) int status) {
        servicePackageService.updateStatus(id, status);
        return ResponseEntity.ok("更新成功");
    }

    @Override
    public ResponseEntity<Map<Long, Boolean>> validResourcesetsBound(long id, String resourcesetIds) {
        Set<Long> resourcesetIdSet = new HashSet<>();
        for (String resourcesetIdStr : StringUtils.split(resourcesetIds, ",")) {
            try {
                Long resourcesetId = Long.parseLong(resourcesetIdStr);
                resourcesetIdSet.add(resourcesetId);
            } catch (NumberFormatException ignored) {
            }
        }
        Map<Long, Boolean> result = servicePackageService.validResourcesetsBound(id, resourcesetIdSet);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:service-package:save"})
    @ApiOperation(value = "根据id删除服务包")
    @Override
    public ResponseEntity<String> delete(@ApiParam(value = "id", required = true) long id) {
        servicePackageService.deleteById(id);
        return ResponseEntity.ok("删除成功");
    }

    /**
     * 导出服务包列表
     *
     * @return
     */
    @Deprecated
    @AuthorizedDefinition(resources = {"xforce:operation:service-package:save"})
    @ApiOperation(value = "导出服务包列表", notes = "导出服务包列表")
    @RequestMapping(name = "导出服务包列表", value = {PATH_GLOBAL_PREFIX + "/packages/export"}, method = RequestMethod.POST)
    @ResponseBody
    public ResponseEntity<Long> export(@RequestBody List<String> servicePackageIds, HttpServletRequest request) {
        if (servicePackageIds == null || servicePackageIds.size() <= 0) {
            throw new IllegalArgumentException("id记录不能为空。");
        }
        List<Long> list = servicePackageIds.stream().map(Long::valueOf).collect(Collectors.toList());
        ServicePackageWrapper servicePackageWrapper = this.servicePackagePortService.getExportData(list);
        String filePath = BasePathUtils.ensureFilePath(request);
        ServicePackageExcelUtils.writeV2007(filePath, servicePackageWrapper);
        try {
            Long fileId = fileTransferUtilsService.upload(filePath);
            ResponseEntity<Long> responseEntity = ResponseEntity.ok(fileId);
            deleteFile(filePath);
            return responseEntity;
        } catch (Exception e) {
            deleteFile(filePath);
            logger.error("上传文件失败", e);
            throw new UnknownException("上传文件失败。");
        }
    }

    /**
     * 异步导出服务包列表
     *
     * @return
     */
    @AuthorizedDefinition(resources = {"xforce:operation:service-package:save"})
    @ApiOperation(value = "异步导出服务包列表", notes = "异步导出服务包列表")
    @RequestMapping(name = "异步导出服务包列表", value = {PATH_GLOBAL_PREFIX + "/packages/async/export"}, method = RequestMethod.POST)
    @ResponseBody
    public ResponseEntity<ImportFileRespVo> asyncExport(@ApiParam(value = "request") @RequestBody Query query,
                                                        @ApiParam(value = "sort") Sort sort) {
        //校验是选择列表
        if (ArrayUtils.isEmpty(query.getSheets())) {
            throw new UnknownException("请选择SheetName");
        }
        //异步执行
        QueryParam queryParam = QueryParam.builder()
                .param(ExcelFile.PARAMS_QUERY, query)
                .param(ExcelFile.PARAMS_SORT, sort)
                .build();

        ResourceQueryBo queryBo = new ResourceQueryBo();
        BeanUtils.copyProperties(query, queryBo);
        ExcelFileStore excelFileStore = this.exportFileService.asyncExcelExport(queryParam.params(), SERVICE_PACKAGE_EXCEL_FILE);
        ImportFileRespVo respVo = new ImportFileRespVo();
        respVo.setId(excelFileStore.getId());
        respVo.setBusinessType(SERVICE_PACKAGE_EXCEL_FILE.getBusinessName());
        //检查ExcelFile文件是否正确
        return ResponseEntity.ok(respVo);
    }


    /**
     * 异步导入服务包列表
     *
     * @return ResponseEntity<ImportFileRespVo> 导入对象
     */
    @AuthorizedDefinition(resources = {"xforce:operation:service-package:save"})
    @ApiOperation(value = "异步导入服务包列表", notes = "异步导入服务包列表")
    @RequestMapping(name = "异步导入服务包列表", value = {PATH_GLOBAL_PREFIX + "/packages/async/import"}, method = RequestMethod.POST)
    @ResponseBody
    public ResponseEntity<ImportFileRespVo> asyncImport(@RequestParam(name = "file", required = false) MultipartFile file) {
        Long tenantId = 0L;
        Long userId = 0L;
        if (null != UserInfoHolder.get()) {
            tenantId = UserInfoHolder.get().getTenantId();
            userId = UserInfoHolder.get().getId();
        }
        //保存数据和上传文件，并转为异步处理
        ExcelFileStore excelFileStore = this.importFileService.create(file, SERVICE_PACKAGE_EXCEL_FILE, userId, tenantId);
        ImportFileRespVo respVo = new ImportFileRespVo();
        respVo.setBusinessType(SERVICE_PACKAGE_EXCEL_FILE.name());
        respVo.setFileId(excelFileStore.getSourceFileId());
        respVo.setId(excelFileStore.getId());
        //检查ExcelFile文件是否正确
        return ResponseEntity.ok(respVo);
    }

    /**
     * 导入服务包列表
     *
     * @return
     */
    @AuthorizedDefinition(resources = {"xforce:operation:service-package:save"})
    @ApiOperation(value = "导入服务包列表", notes = "导入服务包列表")
    @RequestMapping(name = "导入服务包列表", value = {PATH_GLOBAL_PREFIX + "/packages/import"}, method = RequestMethod.POST)
    @ResponseBody
    public ResponseEntity<Long> importData(@RequestParam(name = "file", required = false) MultipartFile file, HttpServletRequest request) {
        if (file == null) {
            throw new UnknownException("请选择导入文件。");
        }
        //校验文件大小
        long size = file.getSize() / (1024 * 1024);  //byte转换为M
        if (size >= 15) {
            throw new UnknownException("导入文件不能超过15M");
        }
        String filePath = BasePathUtils.ensureFilePath(request);
        saveFileToLocal(file, filePath);
        ServicePackageWrapper servicePackageWrapper = ServicePackageExcelUtils.extract(filePath);
        ServicePackageWrapper result = servicePackagePortService.saveData(servicePackageWrapper);
        Long fileId = null;
        if (CollectionUtils.isNotEmpty(result.getServicePackages()) || CollectionUtils.isNotEmpty(result.getResourcesets())) {
            String filePath2 = BasePathUtils.ensureFilePath(request);
            result.setSkipReason(false);
            ServicePackageExcelUtils.writeV2007(filePath2, result);
            try {
                fileId = fileTransferUtilsService.upload(filePath2);
                deleteFile(filePath2);
            } catch (Exception e) {
                deleteFile(filePath);
                deleteFile(filePath2);
                logger.error("上传文件失败", e);
                throw new UnknownException("上传文件失败。");
            }
        }
        ResponseEntity<Long> responseEntity = ResponseEntity.ok(fileId);
        deleteFile(filePath);
        return responseEntity;
    }

    /**
     * 下载文件
     *
     * @return
     */
    @AuthorizedDefinition(resources = {"xforce:operation:service-package:read"})
    @ApiOperation(value = "下载文件", notes = "下载文件")
    @RequestMapping(name = "下载文件", value = {"/api/global/packages/file/{fileId}"}, method = RequestMethod.GET)
    public void download(@PathVariable String fileId,
                         HttpServletRequest request,
                         HttpServletResponse response) {
        String filePath;
        if (StringUtils.equals(TEMPLATE, fileId)) {
            filePath = BasePathUtils.getTemplatePath(request, "servicepackage.xlsx");
        } else {
            filePath = BasePathUtils.ensureFilePath(request);
            try {
                fileTransferUtilsService.download(filePath, Long.valueOf(fileId));
            } catch (Exception e) {
                deleteFile(filePath);
                logger.error("下载文件失败", e);
                throw new UnknownException("下载文件失败。");
            }
        }
        DownloadUtils.fileToDownload(response, filePath, "servicepackage.xlsx");
        deleteFile(filePath);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:service-package:read"})
    @JsonView(ServicePackageView.ServicePackageInfo.class)
    @ApiOperation(value = "获取租户相关的服务包分页", notes = "通过分页参数，获取租户相关的服务包分页")
    @Override
    public ResponseEntity<Page<ServicePackage>> page(long tenantId, Query query, Pageable pageable) {
        Set<String> withExtendParams = Stream.of("appName").collect(Collectors.toSet());
        if (StringUtils.isNotBlank(query.getWithExtendParams())) {
            withExtendParams.addAll(Arrays.stream(StringUtils.split(query.getWithExtendParams(), ",")).collect(Collectors.toSet()));
        }
        query.setWithExtendParams(withExtendParams.stream().collect(Collectors.joining(",")));
        Page<ServicePackage> result = servicePackageService.page(tenantId, query, pageable);
        return ResponseEntity.ok(result);
    }

    @JsonView(ServicePackageView.ServicePackageInfo.class)
    @ApiOperation(value = "获取当前租户相关的服务包分页", notes = "通过分页参数，获取租户相关的服务包分页")
    @ResponseBody
    @RequestMapping(name = "获取当前租户相关的服务包分页", value = "/api/current/packages", method = RequestMethod.GET)
    public ResponseEntity<Page<ServicePackage>> currentPage(Query query, Pageable pageable) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Page<ServicePackage> result = servicePackageService.page(authorizedUser.getTenantId(), query, pageable);
        return ResponseEntity.ok(result);
    }

    private void deleteFile(String filePath) {
        File file = new File(filePath);
        if (file.exists()) {
            boolean deleteResult = file.delete();
            logger.info("删除文件结果:{}", deleteResult);
        }
    }

    /**
     * 保存文件到本地
     */
    private void saveFileToLocal(MultipartFile file, String basePath) {

        File newFile = new File(basePath);
        try {
            file.transferTo(newFile);
        } catch (IOException e) {
            logger.error("导入文件失败", e);
            throw new UnknownException("导入文件失败");
        }
    }

    @ApiOperation(value = "获取当前租户相关的服务包-功能集-资源码-tree", notes = "获取当前租户相关的服务包-功能集-资源码-tree")
    @ResponseBody
    @RequestMapping(name = "获取当前租户相关的服务包-功能集-资源码-tree", value = "/api/current/packages/tree", method = RequestMethod.GET)
    public ResponseEntity<List<TreeNode>> currentTree(Query query) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        List<TreeNode> result = servicePackageService.tree(authorizedUser.getTenantId(), query);
        return ResponseEntity.ok(result);
    }
}
