package com.xforceplus.business.resource.service;

import com.alibaba.excel.event.AnalysisEventListener;
import com.xforceplus.api.model.ResourceModel;
import com.xforceplus.api.model.ResourceModel.Request.Query;
import com.xforceplus.api.model.ServiceApiModel;
import com.xforceplus.bo.ResourceQueryBo;
import com.xforceplus.business.app.service.AppService;
import com.xforceplus.business.excel.BusinessType;
import com.xforceplus.business.excel.ExcelSheet;
import com.xforceplus.business.excel.SimpleExcelWriter;
import com.xforceplus.business.excel.reader.Context;
import com.xforceplus.business.excel.reader.MessageRow;
import com.xforceplus.business.excel.reader.SimpleDataReadListener;
import com.xforceplus.business.excel.writer.ExcelConfigBusinessType;
import com.xforceplus.business.resource.dto.ResourceExcelDTO;
import com.xforceplus.business.resource.dto.ServiceApiExcelDTO;
import com.xforceplus.business.service.ExcelReaderService;
import com.xforceplus.business.service.ExcelWriteService;
import com.xforceplus.dao.ResourceApiRelDao;
import com.xforceplus.dao.ResourceDao;
import com.xforceplus.dao.RouteDao;
import com.xforceplus.dao.ServiceApiDao;
import com.xforceplus.dto.resource.ResourceDTO;
import com.xforceplus.dto.resource.ResourceServiceApiDTO;
import com.xforceplus.dto.resource.ServiceApiRouterDTO;
import com.xforceplus.entity.*;
import com.xforceplus.utils.ApiUtils;
import io.geewit.core.utils.reflection.BeanUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.NumberUtils;
import org.springframework.web.bind.annotation.RequestMethod;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.xforceplus.business.excel.ExcelFile.*;
import static com.xforceplus.business.resource.service.ResourceExcelConfig.ResourceServiceApiConfig.*;
import static com.xforceplus.business.resource.service.ResourceExcelServiceImpl.Message.RESOURCE_NOT_FIND;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.springframework.data.domain.Sort.Direction.DESC;

/**
 * <p>
 * Title: 资源导出导入Excel服务
 * </p>
 * <p>
 * Description: 资源导出导入Excel服务
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-10-19 19-08-39
 * @since V1.0
 */
@Service
public class ResourceExcelServiceImpl implements ExcelWriteService, ExcelReaderService {

    /**
     * 日志
     */
    private static final Logger log = LoggerFactory.getLogger(ResourceExcelServiceImpl.class);

    /**
     * 资源服务
     */
    private final ResourceService resourceService;

    @Autowired
    private ResourceDao resourceDao;
    /**
     * ServiceApiService
     */
    @Autowired
    private ServiceApiDao serviceApiDao;
    /**
     * 路由表
     */
    @Autowired
    private RouteDao routeDao;

    @Autowired
    private ResourceApiRelDao resourceApiRelDao;
    /**
     * AppService
     */
    private final AppService appService;

    private final ServiceApiService serviceApiService;

    /**
     * @param resourceService
     * @param serviceApiService
     */
    public ResourceExcelServiceImpl(ResourceService resourceService,
                                    AppService appService, ServiceApiService serviceApiService) {
        this.resourceService = resourceService;
        this.appService = appService;
        this.serviceApiService = serviceApiService;
    }

    /**
     * 获取导入类型，用于Event事件调整导入方法
     * @return ImportBusinessType
     */
    @Override
    public BusinessType getBusinessType() {
        return ExcelConfigBusinessType.RESOURCE_EXCEL_FILE;
    }

    /**
     * 导入定义
     *
     * @param context 上下文
     * @return Context
     */
    @Override
    public Context importExcel(Context context) {
        List<ExcelSheet> sheets = context.getExcelBook().getExcelSheets();
        List<Long> appIds = this.findAllByApp();
        for (ExcelSheet sheet : sheets) {
            String sheetName = sheet.getSheetName();
            //等于公司服务配置信息，则分页读取公司信息
            if (RESOURCE_SHEET.equals(sheetName)) {
                MessageRow messageRows = new MessageRow(sheetName);
                //构建监听器
                AnalysisEventListener<ResourceExcelDTO> excelDataListener = SimpleDataReadListener.listener(context,
                        (rows) -> this.saveResources(messageRows, context, rows, appIds));
                //开始处理
                context.getSimpleExcelReader().read(ResourceExcelDTO.class, excelDataListener, sheetName);
            } else if (SERVICE_API_SHEET.equals(sheetName)) {
                MessageRow messageRows = new MessageRow(sheetName);
                //构建监听器
                AnalysisEventListener<ServiceApiExcelDTO> excelDataListener = SimpleDataReadListener.listener(context,
                        (rows) -> this.saveServiceApi(messageRows, context, rows, appIds));
                //开始处理
                context.getSimpleExcelReader().read(ServiceApiExcelDTO.class, excelDataListener, sheetName);
            }
        }
        return context;
    }

    /**
     * 批量保存资源列表
     *
     * @param messageRows MessageRow
     * @param context     Context
     * @param rows        rows
     * @param appIds      appIds
     */
    private void saveServiceApi(MessageRow messageRows, Context context, List<ServiceApiExcelDTO> rows, List<Long> appIds) {
        //为空则不处理
        if (CollectionUtils.isEmpty(rows)) {
            return;
        }
        for (ServiceApiExcelDTO row : rows) {
            try {
                if (!row.getValidatedStatus()) {
                    messageRows.fail(row.getRowIndex(), row.getValidatedMessage());
                    continue;
                }

                if (SERVICE_API_ACTION_ADD_BIND.equals(row.getAction())) {
                    //新增&绑定
                    this.saveServiceApi(messageRows, row, appIds);
                    continue;
                } else if (SERVICE_API_ACTION_UN_BIND.equals(row.getAction())) {
                    //解绑
                    this.unBindServiceApi(messageRows, row, appIds);
                    continue;
                }
            } catch (Exception ex) {
                log.error(ex.getMessage(), ex);
                messageRows.fail(row.getRowIndex(), row.getAction() + "数据失败");
            }
        }
        context.messageRow(messageRows.getSheetName(), messageRows);
    }

    /**
     * 解绑ServiceApi&资源码
     *
     * @param messageRows MessageRow
     * @param row         ServiceApiExcelDTO
     */
    private void unBindServiceApi(MessageRow messageRows, ServiceApiExcelDTO row, List<Long> appIds) {
        Long appId = NumberUtils.parseNumber(row.getAppId(), Long.class);
        if (!appIds.contains(appId)) {
            messageRows.fail(row.getRowIndex(), "【" + appId + "】APP的数据不存在");
            return;
        }
        RequestMethod requestMethod = RequestMethod.valueOf(row.getRequestMethod());
        //转换大写的
        String resourceCode = row.getResourceCode();
        Optional<ServiceApi> serviceApi = Optional.empty();
        Optional<Route> routeOptional = this.findByPath(row.getPath());
        if (!routeOptional.isPresent()) {
            messageRows.fail(row.getRowIndex(), "【" + row.getPath() + "】不存在");
            return;
        }
        Long routeId = routeOptional.get().getRouteId();
        //判断ServiceApi是否已存在
        try {
            serviceApi = this.findByRouteIdApiUrlMethod(routeId, row.getServiceApiUrl(), requestMethod);
        } catch (IllegalArgumentException e) {
            messageRows.fail(row.getRowIndex(), e.getMessage());
            return;
        }
        if (!serviceApi.isPresent()) {
            messageRows.fail(row.getRowIndex(), "路径ID:【" + routeId + "】服务URL:【" + row.getServiceApiUrl()
                    + "】RequestMethod:【" + requestMethod.name() + "】的数据不存在");
            return;
        }
        //判断资源码是否存
        Optional<Resource> resourceOptional = this.findResourceByCodeAppId(appId, resourceCode);
        if (!resourceOptional.isPresent()) {
            messageRows.fail(row.getRowIndex(), "该【" + resourceCode + "】" + RESOURCE_NOT_FIND);
            return;
        }
        try {
            resourceApiRelDao.deleteByServiceApiIdAndResourceId(serviceApi.get().getServiceApiId(), resourceOptional.get().getResourceId());
            messageRows.success(row.getRowIndex());
        } catch (Exception e) {
            messageRows.fail(row.getRowIndex(), e.getMessage());
        }
    }

    /**
     * 已存在的数据绑定API
     *
     * @param messageRows  MessageRow
     * @param row          ServiceApiExcelDTO
     * @param resourceId   resourceId
     * @param resourceCode resourceCode
     * @param serviceApi   serviceApi
     */
    private void bindServiceApi(MessageRow messageRows, ServiceApiExcelDTO row, Long resourceId, String resourceCode, ServiceApi serviceApi) {

        //判断API是否已经绑定
        List<ResourceApiRel> resourceApiRelList = this.findResourceApiRelByServiceApiId(serviceApi.getServiceApiId());
        if (!CollectionUtils.isEmpty(resourceApiRelList)) {
            messageRows.fail(row.getRowIndex(), "该[" + serviceApi.getServiceApiName() + "]服务API已绑定资源码");
            return;
        }
        ServiceApiModel.Request.BindResources bindResources = new ServiceApiModel.Request.BindResources();
        bindResources.setOverwrite(false);
        bindResources.setResourceIds(Stream.of(resourceId).collect(Collectors.toList()));
        try {
            this.bindResources(serviceApi, bindResources);
            messageRows.success(row.getRowIndex());
        } catch (Exception e) {
            messageRows.fail(row.getRowIndex(), "绑定[" + resourceCode + "]资源码失败");
        }
    }

    /**
     * 查询所有APP
     *
     * @return Optional<App>
     */
    private List<Long> findAllByApp() {
        List<App> appList = appService.list();
        if (CollectionUtils.isEmpty(appList)) {
            return Collections.emptyList();
        }
        return appList.stream().map(App::getAppId).collect(Collectors.toList());
    }

    /**
     * 按Path查询路由
     *
     * @param path Path
     * @return Optional<Route>
     */
    public Optional<Route> findByPath(String path) {
        List<Route> routeList = routeDao.findByPath(path);
        if (CollectionUtils.isEmpty(routeList)) {
            return Optional.empty();
        }
        if (routeList.size() > 1) {
            throw new IllegalArgumentException("【" + path + "】路由存在不唯一记录");
        }
        return Optional.of(routeList.get(0));
    }

    /**
     * 保存并绑定ServiceApi&资源码
     *
     * @param messageRows MessageRow
     * @param row         ServiceApiExcelDTO
     */
    private void saveServiceApi(MessageRow messageRows, ServiceApiExcelDTO row, List<Long> appIds) {
        Long appId = NumberUtils.parseNumber(row.getAppId(), Long.class);
        if (!appIds.contains(appId)) {
            messageRows.fail(row.getRowIndex(), "【" + appId + "】APP的数据不存在");
            return;
        }
        //转换大写的
        RequestMethod requestMethod = RequestMethod.valueOf(StringUtils.upperCase(row.getRequestMethod()));
        String resourceCode = row.getResourceCode();
        String path = row.getPath();
        Optional<ServiceApi> serviceApi;
        Optional<Route> routeOptional;
        try {
            routeOptional = this.findByPath(path);
            if (!routeOptional.isPresent()) {
                messageRows.fail(row.getRowIndex(), "【" + path + "】路由不存在");
                return;
            }
        } catch (IllegalArgumentException e) {
            messageRows.fail(row.getRowIndex(), e.getMessage());
            return;
        }
        Long routeId = routeOptional.get().getRouteId();

        //判断资源码是否存
        Optional<Resource> resourceOptional = this.findResourceByCodeAppId(appId, resourceCode);
        if (!resourceOptional.isPresent()) {
            messageRows.fail(row.getRowIndex(), "该【" + resourceCode + "】" + RESOURCE_NOT_FIND);
            return;
        }
        //为一级资源码不能绑定服务API
        if (resourceOptional.get().getParentId() == null) {
            messageRows.fail(row.getRowIndex(), "【" + resourceCode + "】为一级资源码不能绑定服务API");
            return;
        }
        //判断ServiceApi是否已存在
        String hash = ApiUtils.hash(row.getServiceApiUrl(), requestMethod.name(), routeId);
        try {
            serviceApi = this.findByHash(hash);
        } catch (IllegalArgumentException e) {
            messageRows.fail(row.getRowIndex(), e.getMessage());
            return;
        }
        if (serviceApi.isPresent()) {
            log.info("rowIndex:{},路径Path:{},服务URL:{},RequestMethod:{},的数据已存在,进绑定数据", row.getRowIndex(), path, row.getServiceApiUrl(), requestMethod.name());
            this.bindServiceApi(messageRows, row, resourceOptional.get().getResourceId(), resourceCode, serviceApi.get());
            return;
        }
        ServiceApiModel.Request.Save save = new ServiceApiModel.Request.Save();
        save.setAppId(appId);
        save.setRouteId(routeId);
        save.setRequestMethod(requestMethod);
        save.setServiceApiName(row.getServiceApiName());
        save.setServiceApiPath(row.getServiceApiPath());
        save.setServiceApiUrl(row.getServiceApiUrl());
        //跳过
        save.setSkipAuth(true);
        //跳过权限
        save.setSkipCheck(true);
        //启用
        save.setStatus(1);
        ServiceApiModel.Request.BindResources bindResources = new ServiceApiModel.Request.BindResources();
        bindResources.setResourceIds(Stream.of(resourceOptional.get().getResourceId()).collect(Collectors.toList()));
        bindResources.setOverwrite(false);
        save.setBindResources(bindResources);
        try {
            serviceApiService.save(save);
            messageRows.success(row.getRowIndex());
        } catch (Exception e) {
            messageRows.fail(row.getRowIndex(), "保存数据失败");
        }
    }

    public Optional<ServiceApi> findByHash(String hash) {
        List<ServiceApi> serviceApis = serviceApiDao.findByHash(hash);
        if (CollectionUtils.isEmpty(serviceApis)) {
            return Optional.empty();
        }
        if (serviceApis.size() > 1) {
            throw new IllegalArgumentException("存在相同的hash值[" + hash + "]，请先修正数据");
        }
        return Optional.of(serviceApis.get(0));
    }

    /**
     * 批量保存资源数据
     *
     * @param messageRows 列表
     * @param context     上下文
     * @param rows        行数据
     */
    private void saveResources(MessageRow messageRows, Context context, List<ResourceExcelDTO> rows, List<Long> appIds) {
        //为空则不处理
        if (CollectionUtils.isEmpty(rows)) {
            return;
        }
        Collections.sort(rows);

        for (ResourceExcelDTO row : rows) {
            try {
                if (!row.getValidatedStatus()) {
                    messageRows.fail(row.getRowIndex(), row.getValidatedMessage());
                    continue;
                }
                if (RESOURCE_ACTION_ADD.equals(row.getAction())) {
                    //资源新增
                    this.saveResource(messageRows, row, appIds);
                    continue;
                } else if (RESOURCE_ACTION_UPDATE.equals(row.getAction())) {
                    //修改操作
                    this.updateResource(messageRows, row, appIds);
                    continue;
                }
            } catch (Exception ex) {
                log.error(ex.getMessage(), ex);
                messageRows.fail(row.getRowIndex(), row.getAction() + "数据失败");
            }
        }
        context.messageRow(messageRows.getSheetName(), messageRows);
    }

    /**
     * 资源码更新
     *
     * @param messageRows MessageRow
     * @param row         ResourceExcelDTO
     */
    private void updateResource(MessageRow messageRows, ResourceExcelDTO row, List<Long> appIds) {
        Long appId = NumberUtils.parseNumber(row.getAppId(), Long.class);
        String parentResourceCode = StringUtils.trimToEmpty(row.getParentResourceCode());
        String resourceCode = StringUtils.trimToEmpty(row.getResourceCode());
        //判断资源码是否存在并获取父级资源码ID
        Optional<Resource> resourceOptional = this.findResourceByCodeAppId(appId, resourceCode);
        if (!resourceOptional.isPresent()) {
            messageRows.fail(row.getRowIndex(), "该【" + resourceCode + "】" + RESOURCE_NOT_FIND);
            return;
        }

        //父级资源码是否存在并获取父级资源码ID
        Optional<Resource> parentResourceOptional = this.findResourceByCodeAppId(appId, parentResourceCode);
        if (StringUtils.isNotBlank(parentResourceCode) && !parentResourceOptional.isPresent()) {
            messageRows.fail(row.getRowIndex(), "该【" + parentResourceCode + "】上级资源码未找到");
            return;
        }

        Resource resource = resourceOptional.get();
        //判断父级资源码的父级资源必须为空，限制资源码层级为两级， 为了确保修改前可用，则判断是修改再再判断父组Id
        if (this.isParentResource(resourceCode, parentResourceOptional, resource.getParentId())) {
            messageRows.fail(row.getRowIndex(), "该【" + parentResourceCode + "】父级资源码层级过多");
            return;
        }

        //更新
        resource.setResourceName(row.getResourceName());
        if (parentResourceOptional.isPresent()) {
            resource.setParentId(resource.getParentId());
        }
        try {
            resourceService.update(resource);
            messageRows.success(row.getRowIndex());
        } catch (RuntimeException e) {
            messageRows.fail(row.getRowIndex(), "更新数据失败");
        }
    }

    /**
     * 判断父级资源码的父级资源必须为空，限制资源码层级为两级， 为了确保修改前可用，则判断是修改再再判断父组Id
     *
     * @param resourceCode     resourceCode
     * @param optionalResource Optional<Resource>
     * @param parentId         parentId
     * @return boolean
     */
    private boolean isParentResource(String resourceCode, Optional<Resource> optionalResource, Long parentId) {
        if (StringUtils.isNotBlank(resourceCode) && optionalResource.isPresent()) {
            Resource resource = optionalResource.get();
            if (resource.getResourceId().equals(parentId) && (resource.getParentId() != null && resource.getParentId() != 0)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 根据资源码和AppId查询Resource
     *
     * @param appId        AppId
     * @param resourceCode ResourceCode
     * @return
     */
    private Optional<Resource> findResourceByCodeAppId(Long appId, String resourceCode) {
        return this.resourceService.findByAppIdResourceCode(appId, resourceCode);
    }

    /**
     * 资源码新增
     * @param messageRows MessageRow
     * @param row 对象
     */
    private void saveResource(MessageRow messageRows, ResourceExcelDTO row, List<Long> appIds) {
        Long appId = NumberUtils.parseNumber(row.getAppId(), Long.class);
        if (!appIds.contains(appId)) {
            messageRows.fail(row.getRowIndex(), "【" + appId + "】APP的数据不存在");
            return;
        }

        String resourceCode = StringUtils.trimToEmpty(row.getResourceCode());
        //判断资源码是否存在并获取父级资源码ID
        Optional<Resource> resourceOptional = this.findResourceByCodeAppId(appId, resourceCode);
        if (resourceOptional.isPresent()) {
            messageRows.fail(row.getRowIndex(), "该【" + resourceCode + "】资源码已存在");
            return;
        }
        String parentResourceCode = StringUtils.trimToEmpty(row.getParentResourceCode());
        //父级资源码是否存在并获取父级资源码ID
        Optional<Resource> parentResourceOptional = this.findResourceByCodeAppId(appId, parentResourceCode);
        if (StringUtils.isNotBlank(parentResourceCode) && !parentResourceOptional.isPresent()) {
            messageRows.fail(row.getRowIndex(), "该【" + parentResourceCode + "】上级资源码未找到");
            return;
        }
        //判断父级资源码的父级资源必须为空，限制资源码层级为两级
        if (StringUtils.isNotBlank(parentResourceCode) && parentResourceOptional.isPresent() && (parentResourceOptional.get().getParentId() != null && parentResourceOptional.get().getParentId() != 0)) {
            messageRows.fail(row.getRowIndex(), "该【" + parentResourceCode + "】父级资源码层级过多");
            return;
        }
        ResourceModel.Request.Save save = new ResourceModel.Request.Save();
        save.setAppId(appId);
        save.setResourceCode(resourceCode);
        save.setResourceName(row.getResourceName());
        if (parentResourceOptional.isPresent()) {
            save.setParentId(parentResourceOptional.get().getResourceId());
        }
        //启用-1
        save.setStatus(1);
        //是否属于公司服务包
        try {
            this.resourceService.save(save);
            messageRows.success(row.getRowIndex());
        } catch (IllegalArgumentException e) {
            messageRows.fail(row.getRowIndex(), e.getMessage());
        }
    }

    /**
     * @param routeId
     * @param serviceApiPath
     * @param requestMethod
     * @return
     */
    public Optional<ServiceApi> findByRouteIdApiUrlMethod(Long routeId, String serviceApiUrl, RequestMethod requestMethod) {
        List<ServiceApi> serviceApis = this.serviceApiDao.findByRouteIdAndServiceApiUrlAndRequestMethod(routeId, serviceApiUrl, requestMethod);
        if (CollectionUtils.isEmpty(serviceApis)) {
            return Optional.empty();
        }
        if (serviceApis.size() > 1) {
            throw new IllegalArgumentException("路径ID:[" + routeId + "], Path:[" + serviceApiUrl + "], RequestMethod:[" + requestMethod.name() + "]不唯一数据，请先修正数据");
        }
        return Optional.of(serviceApis.get(0));
    }

    /**
     * 志出
     *
     * @param context 上下文
     * @return Context
     */
    @Override
    public void write(Context context) {
        Query query = context.getParam(PARAMS_QUERY, Query.class);
        Sort sort = context.getParam(PARAMS_SORT, Sort.class);
        List<String> sheetNames = Arrays.asList(query.getSheets());
        SimpleExcelWriter simpleExcelWriter = context.getSimpleExcelWriter();
        Integer maxPageSize = context.getMaxPageSize();
        ResourceQueryBo resourceQueryBo = new ResourceQueryBo();
        //复制查询条件
        BeanUtils.copyProperties(query, resourceQueryBo);
        //写入数据
        for (String sheetName : sheetNames) {
            //等于公司服务配置信息，则分页读取公司信息
            if (RESOURCE_SHEET.equals(sheetName)) {

                //资源码
                Page<ResourceDTO> page;
                Pageable pageable = PageRequest.of(START_PAGE, PAGE_SIZE, sort.and(Sort.by(DESC, SORT_FIELD)));
                do {
                    page = this.resourceService.pagingBy(pageable, resourceQueryBo);
                    //对象转换
                    Page<ResourceExcelDTO> excelPage = page.map(e -> {
                        ResourceExcelDTO dto = new ResourceExcelDTO();
                        dto.setAppId(e.getAppId() == null ? EMPTY : String.valueOf(e.getAppId()));
                        dto.setResourceName(StringUtils.trimToEmpty(e.getResourceName()));
                        dto.setResourceCode(StringUtils.trimToEmpty(e.getResourceCode()));
                        dto.setParentResourceCode(StringUtils.trimToEmpty(e.getParentCode()));
                        return dto;
                    });
                    simpleExcelWriter.fill(sheetName, excelPage.getContent());
                } while (page.hasNext() && (pageable = pageable.next()) != null && page.getNumber() < maxPageSize);
                //计算当前导出数据  this.calcSuccessSize(page,)
                simpleExcelWriter.param(SUCCESS_SIZE, this.calcSuccessSize(page));
            } else if (SERVICE_API_SHEET.equals(sheetName)) {
                Map<Long, String> routeMap = this.findRouteList();
                //资源码
                Page<ResourceServiceApiDTO> page;
                Pageable pageable = PageRequest.of(START_PAGE, PAGE_SIZE, sort.and(Sort.by(DESC, SORT_FIELD)));
                do {
                    page = this.resourceService.pagingByServiceApi(pageable, resourceQueryBo);
                    //对象转换
                    Page<ServiceApiExcelDTO> configPageDTO = page.map(e -> {
                        ServiceApiExcelDTO dto = new ServiceApiExcelDTO();
                        dto.setServiceApiName(StringUtils.trimToEmpty(e.getServiceApiName()));
                        dto.setPath(this.getRoutePath(routeMap, e.getRouteId()));
                        dto.setServiceApiUrl(StringUtils.trimToEmpty(e.getServiceApiUrl()));
                        dto.setAppId(e.getAppId() == null ? EMPTY : String.valueOf(e.getAppId()));
                        dto.setServiceApiPath(StringUtils.defaultIfEmpty(e.getServiceApiPath(), EMPTY));
                        dto.setRequestMethod(StringUtils.trimToEmpty(e.getRequestMethod()));
                        dto.setResourceCode(StringUtils.trimToEmpty(e.getResourceCode()));
                        return dto;
                    });
                    simpleExcelWriter.fill(sheetName, configPageDTO.getContent());
                } while (page.hasNext() && (pageable = pageable.next()) != null && page.getNumber() < maxPageSize);
                //计算当前导出数据  this.calcSuccessSize(page,)
                simpleExcelWriter.param(SUCCESS_SIZE, this.calcSuccessSize(page));
            }
        }
    }

    /**
     * 查到路由Path
     *
     * @param routeMap routeMap
     * @param routeId  routeId
     * @return String Path
     */
    private String getRoutePath(Map<Long, String> routeMap, Long routeId) {
        if (routeMap.containsKey(routeId)) {
            return StringUtils.trimToEmpty(routeMap.get(routeId));
        }
        return EMPTY;
    }

    /**
     * 路由列表
     *
     * @return Map<Long, String>
     */
    private Map<Long, String> findRouteList() {
        List<ServiceApiRouterDTO> routerDTOList = this.resourceService.findRouteList();
        if (CollectionUtils.isEmpty(routerDTOList)) {
            return Collections.emptyMap();
        }
        return routerDTOList.stream().collect(Collectors.toMap(ServiceApiRouterDTO::getRouteId, ServiceApiRouterDTO::getPath));
    }

    /**
     * 根据serviceApiId查询数据
     *
     * @param serviceApiId serviceApiId
     * @return List<ResourceApiRel>
     */
    public List<ResourceApiRel> findResourceApiRelByServiceApiId(Long serviceApiId) {
        return resourceApiRelDao.findByServiceApiId(serviceApiId);
    }

    public ServiceApi findServiceApiById(Long id) {
        return serviceApiDao.findById(id).orElseThrow(() -> new IllegalArgumentException("未找到实体"));
    }

    /**
     * @param id            接口id
     * @param bindResources
     */
    public void bindResources(long id, ServiceApiModel.Request.BindResources bindResources) {
        ServiceApi serviceApi = this.findServiceApiById(id);
        this.bindResources(serviceApi, bindResources);
    }

    public void bindResources(ServiceApi serviceApi, ServiceApiModel.Request.BindResources bindResources) {
        if (bindResources == null) {
            return;
        }
        List<Long> resourceIds = bindResources.getResourceIds();
        if (resourceIds == null) {
            return;
        }
        log.info("resourceIds = " + resourceIds.stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.joining(",")));
        List<ResourceApiRel> existRels = resourceApiRelDao.findByServiceApiId(serviceApi.getServiceApiId());
        log.info("exist api-resource-rels.size = " + existRels.size());
        //region 往绑定不在数据库中但是报文 resourcesetIds 中有的服务包-功能集关系
        Set<ResourceApiRel> insertingRels = resourceIds.stream().filter(resourceId -> existRels.stream().map(ResourceApiRel::getResourceId).noneMatch(existId -> existId.equals(resourceId))).map(resourceId -> {
            Optional<Resource> resourceOptional = resourceDao.findById(resourceId);
            if (resourceOptional.isPresent()) {
                ResourceApiRel rel = new ResourceApiRel();
                rel.setResourceId(resourceId);
                rel.setServiceApiId(serviceApi.getServiceApiId());
                return rel;
            } else {
                return null;
            }
        }).filter(Objects::nonNull).collect(Collectors.toSet());
        serviceApiService.bindResources(insertingRels);
        //endregion
        if (bindResources.isOverwrite()) {
            //region 从数据库中解绑不在报文 resourcesetIds 中的服务包-功能集关系
            existRels.stream().filter(rel -> resourceIds.stream().noneMatch(resourceId -> resourceId.equals(rel.getResourceId()))).forEach(rel -> {
                log.info("deleting Resource-Api-Rel record, {}", rel);
                resourceApiRelDao.deleteById(rel.getId());
            });
            //endregion
        }
    }

    /**
     * 错误信息
     */
    protected static final class Message {

        /**
         * 资源码不存在提示信息
         */
        public static final String RESOURCE_NOT_FIND = "资源码数据未找到";
    }
}
