package com.xforceplus.business.resource.service;

import com.xforceplus.dao.AppDao;
import com.xforceplus.dao.ResourceApiRelDao;
import com.xforceplus.dao.ResourceDao;
import com.xforceplus.dao.ServiceApiDao;
import com.xforceplus.domain.resource.ResourceDto;
import com.xforceplus.domain.resource.ResourceExtendDto;
import com.xforceplus.domain.resource.ServiceApiExtendDto;
import com.xforceplus.entity.Resource;
import com.xforceplus.entity.ResourceApiRel;
import com.xforceplus.entity.ServiceApi;
import com.xforceplus.utils.ApiUtils;
import com.xforceplus.utils.excel.QueryUtils;
import com.xforceplus.utils.excel.ResourceWrapper;
import io.geewit.core.utils.reflection.BeanUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 导入导出资源码服务类
 *
 * @author lengmz
 */
@Service
@Slf4j
public class ResourcePortService {

    @Autowired
    private ResourceService resourceService;
    @Autowired
    private ResourceApiRelDao resourceApiRelDao;
    @Autowired
    private ServiceApiDao serviceApiDao;
    @Autowired
    private ServiceApiService serviceApiService;
    @Autowired
    private ResourceDao resourceDao;
    @Autowired
    private AppDao appDao;


    private String operateUserName = "手动导入";


    /**
     * 根据资源码id导出资源码列表
     *
     * @param resourceIds
     * @return
     */
    public List<ResourceExtendDto> getExportData(List<Long> resourceIds) {
        Specification<Resource> specification = QueryUtils
                .getSpecification("resourceId", resourceIds);
        List<Resource> list = resourceService.list(specification, Sort.unsorted());
        if (list.isEmpty()) {
            throw new IllegalArgumentException("所选id记录不存在。");
        }
        List<Long> parentIds = list.stream().filter(p -> isSubResource(p.getParentId()))
                .map(p -> p.getParentId()).collect(Collectors.toList());
        List<Resource> parentList = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(parentIds)) {
            Specification<Resource> specification2 = QueryUtils
                    .getSpecification("resourceId", parentIds);
            parentList = resourceService.list(specification2, Sort.unsorted());
        }
        List<ResourceExtendDto> extendList =
                new ArrayList<>(list.size() + parentList.size());
        Map<Long, Integer> map = this.getParentMap(parentList);
        for (ResourceDto resourceDto : list) {
            //一级资源码
            if (!this.isSubResource(resourceDto.getParentId())) {
                Integer resourceId = map.get(resourceDto.getResourceId());
                //选中的一级资源码，但是所属下级没有选中
                if (resourceId == null) {
                    ResourceExtendDto resourceExtendDto = new ResourceExtendDto();
                    BeanUtils.copyProperties(resourceDto, resourceExtendDto);
                    extendList.add(resourceExtendDto);
                }
            }
        }
        for (ResourceDto resourceDto : parentList) {
            ResourceExtendDto resourceExtendDto = new ResourceExtendDto();
            BeanUtils.copyProperties(resourceDto, resourceExtendDto);
            extendList.add(resourceExtendDto);
        }
        map = this.getParentMap(extendList);
        for (Resource resource : list) {
            //一级资源码
            if (!isSubResource(resource.getParentId())) {
                continue;
            }
            ResourceExtendDto resourceExtendDto = new ResourceExtendDto();
            BeanUtils.copyProperties(resource, resourceExtendDto);
            Integer parentResourceIndex = map.get(resource.getParentId());
            if (parentResourceIndex != null) {
                resourceExtendDto.setParentResourceIndex(parentResourceIndex + 1);
            }
            extendList.add(resourceExtendDto);
        }
        return extendList;
    }

    /**
     * 根据appId导出资源码列表
     *
     * @param appId
     * @return
     */
    public List<ResourceExtendDto> getExportDataByAppId(Long appId) {
        List<Resource> list = resourceDao.findAll(new Specification<Resource>() {
            @Override
            public Predicate toPredicate(Root<Resource> root, CriteriaQuery<?> query, CriteriaBuilder builder) {
                List<Predicate> predicates = new ArrayList<>();
                predicates.add(builder.equal(root.<Long>get("appId"), appId));
                if (!predicates.isEmpty()) {
                    query.where(predicates.stream().toArray(Predicate[]::new));
                }
                return query.getRestriction();
            }
        }, Sort.unsorted());
        if (list.isEmpty()) {
            throw new IllegalArgumentException("该appId下没有资源码。");
        }
        List<ResourceExtendDto> extendList = new ArrayList<>(list.size());

        for (Resource resource : list) {
            ResourceExtendDto resourceExtendDto = new ResourceExtendDto();
            BeanUtils.copyProperties(resource, resourceExtendDto);
            extendList.add(resourceExtendDto);
        }
        return extendList;
    }

    /**
     * 是否为二级资源码
     * @param parentId
     * @return
     */
    private boolean isSubResource(Long parentId) {
        return parentId != null && parentId > 0L;
    }

    private <T extends ResourceDto> Map<Long, Integer> getParentMap(List<T> list) {
        Map<Long, Integer> map = new HashMap<>(list.size());
        if (list.isEmpty()) {
            return map;
        }
        for (int i = 0; i < list.size(); i++) {
            T resource = list.get(i);
            map.put(resource.getResourceId(), i);
        }
        return map;
    }

    public ResourceWrapper saveResourceData(ResourceWrapper wrapper) {
        ResourceWrapper result = new ResourceWrapper();
        List<ResourceExtendDto> resource = this.saveResource(wrapper.getResources());
        List<ServiceApiExtendDto> serviceApi = this.saveServiceApi(wrapper.getResources(), wrapper.getServiceApis());
        result.setResources(resource);
        result.setServiceApis(serviceApi);
        return result;
    }

    private List<ServiceApiExtendDto> saveServiceApi(List<ResourceExtendDto> resources, List<ServiceApiExtendDto> serviceApis) {
        List<ServiceApiExtendDto> exists = new ArrayList<>();
        for (ServiceApiExtendDto item : serviceApis) {
            String hash = ApiUtils.hash(item.getServiceApiUrl(), item.getRequestMethod().name(), item.getRouteId());
            List<ServiceApi> list = serviceApiDao.findByHash(hash);
            ServiceApi serviceApi;
            if (list.isEmpty()) {
                serviceApi = saveServiceApi(item);
            } else {
                serviceApi = list.get(0);
            }
            boolean result = serviceApiService.saveResourceApiRel(resources.get(item.getRelResourceIndex() - 1).getResourceId(), serviceApi.getServiceApiId(), operateUserName);
            if (!result) {
                item.setReason("该api已经存在并且跟资源码关联。");
                exists.add(item);
            }
        }
        return exists;
    }

    public List<ResourceExtendDto> saveResource(List<ResourceExtendDto> resources) {
        List<ResourceExtendDto> outResources = new ArrayList<>();
        List<String> resourceCodes = resources.stream().map(ResourceDto::getResourceCode).collect(Collectors.toList());
        List<Resource> existsResources = this.getResourceByCodes(resourceCodes);
        Map<String, Resource> existsMap = new HashMap<>(existsResources.size());
        for (Resource resource : existsResources) {
            existsMap.put(this.getResourceCodeKey(resource.getResourceCode(), resource.getAppId()), resource);
        }
        for (ResourceExtendDto item : resources) {
            if (null == item.getAppId() || !appDao.existsById(item.getAppId())) {
                item.setReason("无效的appId。");
                outResources.add(item);
                continue;
            }

            Resource resource = existsMap.get(this.getResourceCodeKey(item.getResourceCode(), item.getAppId()));
            if (item.getParentResourceIndex() != null) {
                item.setParentId(resources.get(item.getParentResourceIndex() - 1).getResourceId());
            } else {
                item.setParentId(0L);
            }
            if (resource == null) {
                resource = this.saveResource(item);
            } else {    //导入时已经存在
                if (this.isSubResource(item.getParentId())) {
                    resource.setParentId(item.getParentId());
                    this.saveResource(resource);
                }
                item.setReason("该资源码已经存在。");
                outResources.add(item);
            }
            item.setResourceId(resource.getResourceId());
        }
        return outResources;
    }

    private String getResourceCodeKey(String resourceCode, Long appId) {
        String appIdStr = (appId == null ? "" : String.valueOf(appId));
        return resourceCode + "_" + appIdStr;
    }

    public List<Resource> getResourceByCodes(List<String> resourceCodes) {
        Specification<Resource> specification = QueryUtils.getSpecification("resourceCode", resourceCodes);
        return resourceService.list(specification, Sort.unsorted());
    }

    public List<ServiceApiExtendDto> getServiceApiData(List<ResourceExtendDto> extendList) {
        if (CollectionUtils.isEmpty(extendList)) {
            return new ArrayList<>();
        }
        List<Long> resourceIds = extendList.stream().map(ResourceDto::getResourceId).collect(Collectors.toList());
        Specification<ResourceApiRel> specification = QueryUtils.getSpecification("resourceId", resourceIds);
        List<ResourceApiRel> resourceApiRels = this.resourceApiRelDao.findAll(specification, Sort.unsorted());
        if (CollectionUtils.isEmpty(resourceApiRels)) {
            log.info("CollectionUtils.isEmpty(resourceApiRels)");
            return new ArrayList<>();
        }
        Map<Long, Integer> idAndIndexMap = this.getResourceIdAndIndexMap(extendList);
        Map<Long, Integer> serviceApiIdAndIndexId = this.getServiceApiIdAndIndexId(resourceApiRels, idAndIndexMap);
        List<Long> serviceApiId = resourceApiRels.stream().map(ResourceApiRel::getServiceApiId).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(serviceApiId)) {
            log.info("CollectionUtils.isEmpty(serviceApiId)");
            return new ArrayList<>();
        }
        Specification<ServiceApi> specification2 = QueryUtils.getSpecification("serviceApiId", serviceApiId);
        List<ServiceApi> serviceApiList = this.serviceApiDao.findAll(specification2, Sort.unsorted());
        List<ServiceApiExtendDto> result = new ArrayList<>(serviceApiList.size());
        Set<String> keySets = new HashSet<>(serviceApiList.size());
        for (ServiceApi serviceApi : serviceApiList) {
            String uniqueKey = serviceApi.getServiceApiUrl() + serviceApi.getRequestMethod();
            if (serviceApi.getRouteId() != null && serviceApi.getRouteId() > 0L) {
                uniqueKey = uniqueKey + String.valueOf(serviceApi.getRouteId());
            }
            if (keySets.contains(uniqueKey)) {
                log.info("continue uniqueKey=={}", uniqueKey);
                continue;
            }
            keySets.add(uniqueKey);
            ServiceApiExtendDto serviceApiExtendDto = new ServiceApiExtendDto();
            BeanUtils.copyProperties(serviceApi, serviceApiExtendDto);
            Integer resourceIndex = serviceApiIdAndIndexId.get(serviceApi.getServiceApiId());
            serviceApiExtendDto.setRelResourceIndex(resourceIndex);
            result.add(serviceApiExtendDto);
        }
        return result;
    }

    private Map<Long, Integer> getResourceIdAndIndexMap(List<ResourceExtendDto> list) {
        Map<Long, Integer> map = new HashMap<>(list.size());
        if (list.isEmpty()) {
            return map;
        }
        for (int i = 0; i < list.size(); i++) {
            ResourceExtendDto resourceDto = list.get(i);
            map.put(resourceDto.getResourceId(), i + 1);
        }
        return map;
    }

    private Map<Long, Integer> getServiceApiIdAndIndexId(List<ResourceApiRel> list, Map<Long, Integer> resourceAndIndexMap) {
        Map<Long, Integer> map = new HashMap<>(list.size());
        if (list.isEmpty()) {
            return map;
        }
        for (ResourceApiRel item : list) {
            Integer index = resourceAndIndexMap.get(item.getResourceId());
            map.put(item.getServiceApiId(), index);
        }
        return map;
    }

    public <T extends ResourceDto> Resource saveResource(T model) {
        Resource entity = new Resource();
        BeanUtils.copyProperties(model, entity, Stream.of("children").toArray(String[]::new));
        Date now = Calendar.getInstance().getTime();
        if (entity.getResourceId() == null) {
            entity.setCreaterName(operateUserName);
            entity.setCreateTime(now);
        }
        entity.setUpdaterName(operateUserName);
        entity.setUpdateTime(now);
        return resourceDao.saveAndFlush(entity);
    }

    private ServiceApi saveServiceApi(ServiceApiExtendDto item) {
        ServiceApi entity = new ServiceApi();
        BeanUtils.copyProperties(item, entity);
        Date now = Calendar.getInstance().getTime();
        if (entity.getServiceApiId() == null) {
            entity.setCreaterName(operateUserName);
            entity.setCreateTime(now);
        }
        entity.setUpdaterName(operateUserName);
        entity.setUpdateTime(now);
        return serviceApiDao.saveAndFlush(entity);
    }


    public Map<String, Object> getYamlMapByResources(List<ResourceExtendDto> resourceList) {
        Map<Long, ResourceExtendDto> resourceIdAndItem = new HashMap<>();
        Map<Long, List<ResourceExtendDto>> parentIdAndItem = new HashMap<>();
        if (CollectionUtils.isEmpty(resourceList)) {
            return null;
        }
        for (ResourceExtendDto item : resourceList) {
            resourceIdAndItem.put(item.getResourceId(), item);
            if (item.getParentId() != null && item.getParentId() > 0) {
                List<ResourceExtendDto> children = parentIdAndItem.computeIfAbsent(item.getParentId(), k -> new ArrayList<>());
                children.add(item);
            }
        }
        Map<String, Object> rootMap = new HashMap<>(parentIdAndItem.size());
        for (ResourceExtendDto resourceExtendDto : resourceList) {
            Map<String, Object> resourceMap = new HashMap<>(parentIdAndItem.size());
            resourceMap.put("name", resourceExtendDto.getResourceName());
            List<ResourceExtendDto> children = parentIdAndItem.get(resourceExtendDto.getResourceId());
            if (CollectionUtils.isNotEmpty(children)) {
                for (Object item : children) {
                    ResourceExtendDto child = (ResourceExtendDto) item;
                    Map<String, Object> childNameMap = new HashMap<>();
                    childNameMap.put("name", child.getResourceName());
                    resourceMap.put(child.getResourceCode(), childNameMap);
                }
            }
            if (resourceExtendDto.getParentId() == null || resourceExtendDto.getParentId() <= 0) {
                rootMap.put(resourceExtendDto.getResourceCode(), resourceMap);
            }
        }
        return rootMap;
    }
}
