package com.xforceplus.business.resource.service;

import com.xforceplus.api.model.ResourceModel.Request;
import com.xforceplus.api.model.ResourcesetModel;
import com.xforceplus.api.utils.Separator;
import com.xforceplus.bo.ResourceQueryBo;
import com.xforceplus.dao.*;
import com.xforceplus.dto.resource.ResourceDTO;
import com.xforceplus.dto.resource.ResourceServiceApiDTO;
import com.xforceplus.dto.resource.ServiceApiRouterDTO;
import com.xforceplus.dto.resource.ServicePackageDTO;
import com.xforceplus.entity.*;
import com.xforceplus.query.ResourceQueryHelper;
import com.xforceplus.query.ResourcesetQueryHelper;
import io.geewit.core.utils.reflection.BeanUtils;
import io.geewit.data.jpa.essential.domain.EntityGraphs;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

@Service
public class ResourceService {
    private final static Logger logger = LoggerFactory.getLogger(ResourceService.class);

    private final ResourceDao resourceDao;

    private final ResourcesetDao resourcesetDao;

    private final ResourcesetResourceRelDao resourcesetResourceRelDao;

    private final ResourceApiRelDao resourceApiRelDao;

    private final ServiceApiDao serviceApiDao;

    private final RedisTemplate redisTemplate;

    @Autowired
    private ResourceExtendDao resourceExtendDao;

    @Autowired
    private ResourceCustomizedDao resourceCustomizedDao;

    @Autowired
    private AppDao appDao;

    public ResourceService(ResourceDao resourceDao, ResourcesetDao resourcesetDao, ResourcesetResourceRelDao resourcesetResourceRelDao, ResourceApiRelDao resourceApiRelDao, ServiceApiDao serviceApiDao, RedisTemplate redisTemplate) {
        this.resourceDao = resourceDao;
        this.resourcesetDao = resourcesetDao;
        this.resourcesetResourceRelDao = resourcesetResourceRelDao;
        this.resourceApiRelDao = resourceApiRelDao;
        this.serviceApiDao = serviceApiDao;
        this.redisTemplate = redisTemplate;
    }


    public Page<Resource> page(Request.Query query, Pageable pageable) {
        Page<Resource> page;
        if (query.isTupleSelection()) {
            page = resourceCustomizedDao.findAttributes(query, pageable);
        } else {
            Specification<Resource> specification = ResourceQueryHelper.querySpecification(query);
            page = resourceDao.findAll(specification, pageable, EntityGraphs.named(Resource.NAMED_ENTITY_GRAPH_DEFAULT));
        }
        this.fillResourceExtensions(query, page);
        return page;
    }

    public List<Resource> list(Request.Query query, Sort sort) {
        List<Resource> list;
        if (query.isTupleSelection()) {
            list = resourceCustomizedDao.findAttributes(query, sort);
        } else {
            Specification<Resource> specification = ResourceQueryHelper.querySpecification(query);
            list = resourceDao.findAll(specification, sort, EntityGraphs.named(Resource.NAMED_ENTITY_GRAPH_DEFAULT));
        }
        this.fillResourceExtensions(query, list);
        return list;
    }

    private void fillResourceExtensions(Request.Query query, Iterable<Resource> resources) {
        if (resources == null || resources.iterator() == null || !resources.iterator().hasNext()) {
            return;
        }
        boolean isFlat = query.isFlat();
        boolean needParentName = false;
        boolean needAppName = false;
        if (StringUtils.isNotBlank(query.getWithExtendParams())) {
            String[] withExtendParams = StringUtils.split(query.getWithExtendParams(), ",");
            for (String withExtendParam : withExtendParams) {
                if ("parentName".equalsIgnoreCase(withExtendParam)) {
                    needParentName = true;
                    break;
                } else if ("appName".equalsIgnoreCase(withExtendParam)) {
                    needAppName = true;
                    break;
                }
            }
        }
        if (!isFlat || needParentName || needAppName) {
            Map<Long, String> resourceNameMap = null;
            if (needParentName) {
                resourceNameMap = new HashMap<>();
            }
            Map<Long, String> appNameMap = null;
            Set<Long> appIds = new HashSet<>();
            for (Resource resource : resources) {
                if (resourceNameMap != null) {
                    resourceNameMap.put(resource.getResourceId(), resource.getResourceName());
                }
                if (!isFlat) {
                    List<Resource> children = resourceDao.findByParentId(resource.getResourceId());
                    if (CollectionUtils.isEmpty(children)) {
                        resource.setChildren(null);
                    } else {
                        resource.setChildren(children);
                    }
                }
                appIds.add(resource.getAppId());
            }
            if (needAppName) {
                List<App> apps = appDao.findAppsByAppIds(appIds);
                appNameMap = apps.stream().collect(Collectors.toMap(App::getAppId, App::getAppName));
            }
            if (resourceNameMap != null || appNameMap != null) {
                Set<Long> selectingResourceIds = null;
                if (resourceNameMap != null) {
                    selectingResourceIds = new HashSet<>();
                }
                for (Resource resource : resources) {
                    if (appNameMap != null) {
                        resource.setAppName(appNameMap.get(resource.getAppId()));
                    }
                    if (resourceNameMap != null) {
                        if (resource.getParentName() == null && resource.getParentId() != null && resource.getParentId() > 0) {
                            if (resourceNameMap.containsKey(resource.getParentId())) {
                                resource.setParentName(resourceNameMap.get(resource.getParentId()));
                            } else {
                                selectingResourceIds.add(resource.getParentId());
                            }
                        }
                    }
                }
                if (resourceNameMap != null) {
                    if (!selectingResourceIds.isEmpty()) {
                        List<Pair<Long, String>> resourceNamePairs = resourceDao.findResourceNamePairsByResourceIds(selectingResourceIds);
                        for (Pair<Long, String> pair : resourceNamePairs) {
                            resourceNameMap.put(pair.getLeft(), pair.getRight());
                        }
                    }
                    for (Resource resource : resources) {
                        if (resource.getParentName() == null && resource.getParentId() != null && resource.getParentId() > 0) {
                            if (resourceNameMap.containsKey(resource.getParentId())) {
                                resource.setParentName(resourceNameMap.get(resource.getParentId()));
                            }
                        }
                    }
                }
            }
        }
    }

    public List<Resource> list(Specification<Resource> specification, Sort sort) {
        return resourceDao.findAll(specification, sort, EntityGraphs.named(Resource.NAMED_ENTITY_GRAPH_DEFAULT));
    }

    public Optional<Resource> findOne(Request.Query query) {
        Specification<Resource> specification = ResourceQueryHelper.querySpecification(query);
        return resourceDao.findOne(specification);
    }

    public long count(Request.Query query) {
        Specification<Resource> specification = ResourceQueryHelper.querySpecification(query);
        return resourceDao.count(specification);
    }

    @Transactional(rollbackFor = Exception.class)
    public Resource save(Request.Save model) {
        if (null == model.getIsServicePackage()) {
            model.setIsServicePackage(true);
        }
        Resource entity = new Resource();
        BeanUtils.copyProperties(model, entity);
        entity.setResourceType(1);
        return resourceDao.saveAndFlush(entity);
    }

    /**
     * 更新数据
     *
     * @param resource 更新数据
     * @return Resource
     */
    @Transactional(rollbackFor = Exception.class)
    public Resource update(Resource resource) {
        resourceDao.saveAndFlush(resource);
        return resource;
    }

    @Transactional(rollbackFor = Exception.class)
    public Resource update(long resourceId, Request.Save model) {
        Resource existEntity = this.findById(resourceId);

        if (StringUtils.isBlank(model.getResourceName())) {
            model.setResourceName(null);
        }
        if (StringUtils.isBlank(model.getResourceCode())) {
            model.setResourceCode(null);
        }
        BeanUtils.copyProperties(model, existEntity);
        existEntity = resourceDao.saveAndFlush(existEntity);

        return existEntity;
    }

    public Resource findById(long resourceId) {
        Request.Query query = new Request.Query();
        query.setResourceId(resourceId);
        Optional<Resource> resourceOptional = this.findOne(query);
        return resourceOptional.orElseThrow(() -> new IllegalArgumentException("未找到资源码(" + resourceId + ")"));
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteById(long resourceId) {
        Resource resource = this.findById(resourceId);
        ResourcesetModel.Request.Query query = new ResourcesetModel.Request.Query();
        query.setResourceId(resourceId);
        Specification<Resourceset> specification = ResourcesetQueryHelper.querySpecification(query);
        long resourcesets = resourcesetDao.count(specification);
        if (resourcesets > 0) {
            throw new IllegalArgumentException("该资源码(" + resourceId + ")已经关联功能集合不能删除");
        }
        Request.Query resourceQuery = new Request.Query();
        resourceQuery.setParentId(resourceId);
        Specification<Resource> resourceSspecification = ResourceQueryHelper.querySpecification(resourceQuery);
        long resourceChildrenCount = resourceDao.count(resourceSspecification);
        if (resourceChildrenCount > 0) {
            throw new IllegalArgumentException("已经创建子资源码");
        }
        resourceDao.deleteById(resource.getResourceId());
        resourcesetResourceRelDao.deleteByResourceId(resourceId);
    }

    @Transactional(rollbackFor = Exception.class)
    public Resource append(long parentId, Request.Save model) {
        Resource parent = this.findById(parentId);
        if (null == model.getIsServicePackage()) {
            model.setIsServicePackage(true);
        }
        model.setAppId(parent.getAppId());
        Resource entity = new Resource();
        BeanUtils.copyProperties(model, entity);
        entity.setParentId(parentId);
        entity = resourceDao.saveAndFlush(entity);

        return entity;
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateStatus(long id, int status) {
        int result = resourceDao.updateStatus(id, status);
        if (result <= 0) {
            throw new IllegalArgumentException("找不到资源码(" + id + ")实体");
        }
    }

    public List<Resource> listByServiceApiId(long serviceApiId) {
        Request.Query query = new Request.Query();
        query.setServiceApiId(serviceApiId);
        query.setStatus(1);
        return this.list(query, Sort.unsorted());
    }

    @Transactional(rollbackFor = Exception.class)
    public void bindApis(long resourceId, Request.BindApis bindApis) {
        Resource resource = this.findById(resourceId);
        this.bindApis(resource, bindApis);
    }

    public void bindApis(Resource resource, Request.BindApis bindApis) {
        if (bindApis == null) {
            return;
        }
        List<Long> apiIds = bindApis.getApiIds();
        if (CollectionUtils.isEmpty(apiIds)) {
            return;
        }
        logger.info("apiIds :{}", apiIds.stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.joining(Separator.COMMA)));
        List<ResourceApiRel> existRels = resourceApiRelDao.findByResourceId(resource.getResourceId());
        logger.info("exist api-resource-rels.size:{} ", existRels.size());
        //region 往绑定不在数据库中但是报文 resourcesetIds 中有的服务包-功能集关系
        Set<ResourceApiRel> insertingRels = apiIds.stream().filter(
                        serviceApiId -> existRels.stream()
                                .map(ResourceApiRel::getServiceApiId)
                                .noneMatch(existId -> existId.equals(serviceApiId)))
                .map(serviceApiId -> {
                    Optional<ServiceApi> serviceApiOptional = serviceApiDao.findById(serviceApiId);
                    if (serviceApiOptional.isPresent()) {
                        ResourceApiRel rel = new ResourceApiRel();
                        rel.setResourceId(resource.getResourceId());
                        rel.setServiceApiId(serviceApiId);
                        return rel;
                    } else {
                        return null;
                    }
                }).filter(Objects::nonNull).collect(Collectors.toSet());
        this.bindApis(insertingRels);
        //endregion
        if (bindApis.isOverwrite()) {
            //region 从数据库中解绑不在报文 resourcesetIds 中的服务包-功能集关系
            existRels.stream().filter(rel -> apiIds.stream().noneMatch(apiId -> apiId.equals(rel.getServiceApiId()))).forEach(rel -> {
                logger.info("deleting Resource-Api-Rel record, {}", rel);
                try {
                    resourceApiRelDao.deleteById(rel.getId());
                } catch (Exception e) {
                    logger.warn(e.getMessage(), e);
                }
            });
            //endregion
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void bindApis(Collection<ResourceApiRel> rels) {
        if (!rels.isEmpty()) {
            for (ResourceApiRel rel : rels) {
                resourceApiRelDao.saveAndFlush(rel);
            }
        }
    }

    public List<Resource> listByResourcesetId(long resourcesetId, Integer status) {
        Request.Query query = new Request.Query();
        query.setResourcesetId(resourcesetId);
        query.setStatus(status);
        return this.list(query, Sort.unsorted());
    }

    /**
     * 分页查询 (需要添加@{@link Transactional readOnly=true})
     *
     * @param pageable 分页对象
     * @param queryBo  分页Bo
     * @return Page<ResourceDTO>
     */
    public Page<ResourceDTO> pagingBy(Pageable pageable, ResourceQueryBo queryBo) {
        return resourceExtendDao.pagingBy(pageable, queryBo);
    }

    /**
     * 服务Api分页查询 (需要添加@{@link Transactional readOnly=true})
     *
     * @param pageable 分页对象
     * @param queryBo  分页Bo
     * @return Page<ResourceDTO>
     */
    public Page<ResourceServiceApiDTO> pagingByServiceApi(Pageable pageable, ResourceQueryBo queryBo) {
        return resourceExtendDao.pagingByServiceApi(pageable, queryBo);
    }

    /**
     * 查询路由列表
     *
     * @return List<ServiceApiRouterDTO>
     */
    public List<ServiceApiRouterDTO> findRouteList() {
        return this.resourceExtendDao.findRouteAll();
    }

    /**
     * @param appId
     * @param resourceCode
     * @return Optional<Resource>
     * @throws IllegalArgumentException "【"+appId+"】,【"+resourceCode+"】资源码数据存在不唯一记录"
     */
    public Optional<Resource> findByAppIdResourceCode(Long appId, String resourceCode) {
        List<Resource> resourceList = resourceDao.findByAppIdAndResourceCode(appId, resourceCode);
        if (resourceList.isEmpty()) {
            return Optional.empty();
        }
        //如果
        if (resourceList.size() > 1) {
            throw new IllegalArgumentException("【" + appId + "】,【" + resourceCode + "】资源码数据存在不唯一记录");
        }
        return Optional.of(resourceList.get(0));
    }

    /**
     * 查询roleId关联的资源码列表
     *
     * @param roleId
     * @return
     */
    public List<Resource> listByRoleId(long roleId, Set<String> attributes) {
        Request.Query query = new Request.Query();
        query.setRoleId(roleId);
        query.setStatus(1);
        query.setAttributes(attributes);
        return this.list(query, Sort.unsorted());
    }

    /**
     * 根据功能集id集合查询关联的资源码列表
     *
     * @param resourcesetIds
     * @return
     */
    public List<Resource> listByResourcesetIds(Collection<Long> resourcesetIds) {
        if (resourcesetIds == null || resourcesetIds.isEmpty()) {
            return Collections.emptyList();
        }
        Request.Query query = new Request.Query();
        query.setResourcesetIds(resourcesetIds);
        query.setStatus(1);
        return this.list(query, Sort.unsorted());
    }

    /**
     * 根据功能集id集合查询关联的资源码列表
     *
     * @param resourcesetIds
     * @return
     */
    public List<Resource> listByResourcesetIds(Collection<Long> resourcesetIds, Set<String> attributes) {
        if (resourcesetIds == null || resourcesetIds.isEmpty()) {
            return Collections.emptyList();
        }
        Request.Query query = new Request.Query();
        query.setResourcesetIds(resourcesetIds);
        query.setAttributes(attributes);
        query.setStatus(1);
        return this.list(query, Sort.unsorted());
    }


    public List<ResourceDTO> getUserRoleResourceCode(Long userId) {
        //1. get from redis
        //2. if not find, get from msyql
        //3. save to redis

        return resourceExtendDao.getUserRoleResourceCode(userId);
    }


    public List<ResourceDTO> getUserCompanyResourceCode(Long userId, Long appId) {
        List<ServicePackageDTO> servicePackageDTOList = resourceExtendDao.getServicePackIdByUserIdAndAppId(userId, appId);
        if (CollectionUtils.isNotEmpty(servicePackageDTOList)) {
            List<Long> servicePackageIdList = servicePackageDTOList.stream().map(ServicePackageDTO::getServicePackageId).collect(Collectors.toList());
            return resourceExtendDao.getUserCompanyResourceByPackageIdList(servicePackageIdList);
        }

        return Collections.emptyList();
    }

    /**
     * 查询roleId集合关联的资源码列表
     *
     * @param roleId
     * @return
     */
    public List<Resource> listByRoleIds(Collection<Long> roleIds) {
        if (roleIds == null || roleIds.isEmpty()) {
            return Collections.emptyList();
        }
        List<Long> resourcesetIds = resourcesetDao.findResourcesetIdsByRoleIds(roleIds);
        return this.listByResourcesetIds(resourcesetIds);
    }

    /**
     * 查询roleId集合关联的资源码列表
     *
     * @param roleId
     * @return
     */
    public List<Resource> listByRoleIds(Collection<Long> roleIds, Set<String> attributes) {
        if (roleIds == null || roleIds.isEmpty()) {
            return Collections.emptyList();
        }
        List<Long> resourcesetIds = resourcesetDao.findResourcesetIdsByRoleIds(roleIds);
        return this.listByResourcesetIds(resourcesetIds, attributes);
    }


    public List<Resource> listByPackageIds(Collection<Long> packageIds) {
        if (packageIds == null || packageIds.isEmpty()) {
            return Collections.emptyList();
        }
        List<Long> resourcesetIds = resourcesetDao.findResourcesetIdsByPackageIds(packageIds);
        List<Resource> resources = this.listByResourcesetIds(resourcesetIds);
        return resources;
    }


    public List<Resource> listByPackageIds(Collection<Long> packageIds, Set<String> attributes) {
        if (packageIds == null || packageIds.isEmpty()) {
            return Collections.emptyList();
        }
        List<Long> resourcesetIds = resourcesetDao.findResourcesetIdsByPackageIds(packageIds);
        List<Resource> resources = this.listByResourcesetIds(resourcesetIds, attributes);
        return resources;
    }
}
