/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: ResourceSetExcelExportServiceImpl.java   2020-10-20 16-18-56
 * Author: Evan
 */
package com.xforceplus.business.resource.service;

import com.alibaba.excel.event.AnalysisEventListener;
import com.xforceplus.api.model.ResourceModel;
import com.xforceplus.api.model.ResourcesetModel;
import com.xforceplus.api.model.ResourcesetModel.Request.Query;
import com.xforceplus.bo.ResourceSetQueryBo;
import com.xforceplus.business.app.service.AppService;
import com.xforceplus.business.excel.BusinessType;
import com.xforceplus.business.excel.ExcelSheet;
import com.xforceplus.business.excel.SimpleExcelWriter;
import com.xforceplus.business.excel.reader.Context;
import com.xforceplus.business.excel.reader.MessageRow;
import com.xforceplus.business.excel.reader.SimpleDataReadListener;
import com.xforceplus.business.excel.writer.ExcelConfigBusinessType;
import com.xforceplus.business.resource.dto.ResourceSetExcelDTO;
import com.xforceplus.business.resource.dto.ResourceSetResourceExcelDTO;
import com.xforceplus.business.service.ExcelReaderService;
import com.xforceplus.business.service.ExcelWriteService;
import com.xforceplus.dto.resource.ResourceSetDTO;
import com.xforceplus.dto.resource.ResourceSetResourceDTO;
import com.xforceplus.entity.App;
import com.xforceplus.entity.Resource;
import com.xforceplus.entity.Resourceset;
import io.geewit.core.utils.reflection.BeanUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.NumberUtils;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import static com.xforceplus.business.excel.ExcelFile.*;
import static com.xforceplus.business.resource.service.ResourceExcelConfig.ResourceSetConfig.*;
import static com.xforceplus.business.resource.service.ResourceSetExcelInExportServiceImpl.Message.*;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.springframework.data.domain.Sort.Direction.DESC;

/**
 * <p>
 * Title: 功能集导出导入服务类
 * </p>
 * <p>
 * Description: 功能集导出导入服务类（功能集）
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-10-20 16-18-56
 * @since V1.0
 */
@Service
public class ResourceSetExcelInExportServiceImpl implements ExcelWriteService, ExcelReaderService {

    /**
     * 日志
     */
    private static final Logger log = LoggerFactory.getLogger(ResourceExcelServiceImpl.class);

    /**
     * 资源集服务
     */
    private final ResourcesetService resourcesetService;
    /**
     * 资源码服务
     */
    private final ResourceService resourceService;

    /**
     * AppService
     */
    private final AppService appService;


    public ResourceSetExcelInExportServiceImpl(ResourcesetService resourcesetService, ResourceService resourceService, AppService appService) {
        this.resourcesetService = resourcesetService;
        this.resourceService = resourceService;
        this.appService = appService;
    }

    /**
     * 获取导入类型，用于Event事件调整导入方法
     * @return ImportBusinessType
     */
    @Override
    public BusinessType getBusinessType() {
        return ExcelConfigBusinessType.RESOURCE_SET_EXCEL_FILE;
    }

    /**
     * 导入定义
     *
     * @param context 上下文
     * @return Context
     */
    @Override
    public Context importExcel(Context context) {
        List<ExcelSheet> sheets = context.getExcelBook().getExcelSheets();
        for (ExcelSheet sheet : sheets) {
            String sheetName = sheet.getSheetName();
            //等于公司服务配置信息，则分页读取公司信息
            if (RESOURCE_SET_SHEET.equals(sheetName)) {
                MessageRow messageRows = new MessageRow(sheetName);
                List<Long> appIds = this.findAllByApp();
                //构建监听器
                AnalysisEventListener<ResourceSetExcelDTO> excelDataListener = SimpleDataReadListener.listener(context, (rows) -> this.saveResourceSets(messageRows, context, rows, appIds));
                //开始处理
                context.getSimpleExcelReader().read(ResourceSetExcelDTO.class, excelDataListener, sheetName);
            } else if (RESOURCE_SHEET.equals(sheetName)) {
                MessageRow messageRows = new MessageRow(sheetName);
                //构建监听器
                AnalysisEventListener<ResourceSetResourceExcelDTO> excelDataListener = SimpleDataReadListener.listener(context, (rows) -> this.saveResourceSetResource(messageRows, context, rows));
                //开始处理
                context.getSimpleExcelReader().read(ResourceSetResourceExcelDTO.class, excelDataListener, sheetName);
            }
        }
        return context;
    }

    /**
     * @param messageRows
     * @param context
     * @param rows
     */
    private void saveResourceSetResource(MessageRow messageRows, Context context, List<ResourceSetResourceExcelDTO> rows) {
        //为空则不处理
        if (CollectionUtils.isEmpty(rows)) {
            return;
        }
        for (ResourceSetResourceExcelDTO row : rows) {
            try {
                if (!row.getValidatedStatus()) {
                    messageRows.fail(row.getRowIndex(), row.getValidatedMessage());
                } else if (RESOURCE_SET_RESOURCE_ACTION_BIND.equals(row.getAction())) {
                    //绑定
                    this.bindResources(messageRows, row);
                } else if (RESOURCE_SET_RESOURCE_ACTION_UN_BIND.equals(row.getAction())) {
                    //解绑
                    this.unBindResources(messageRows, row);
                }
            } catch (Exception ex) {
                log.error(ex.getMessage(), ex);
                messageRows.fail(row.getRowIndex(), row.getAction() + "数据失败");
            }
        }
        context.messageRow(messageRows.getSheetName(), messageRows);
    }

    /**
     * 解邦资源码
     *
     * @param messageRows
     * @param row
     */
    private void unBindResources(MessageRow messageRows, ResourceSetResourceExcelDTO row) {
        String resourceSetCode = StringUtils.trimToEmpty(row.getResourceSetCode());
        String resourceCode = StringUtils.trimToEmpty(row.getResourceCode());
        Optional<Resourceset> resourceSetOptional;
        try {
            resourceSetOptional = this.findByResourceSetCode(resourceSetCode);
            if (!resourceSetOptional.isPresent()) {
                messageRows.fail(row.getRowIndex(), "【" + resourceSetCode + "】" + RESOURCE_SET_NOT_NULL);
                return;
            }
        } catch (IllegalArgumentException e) {
            messageRows.fail(row.getRowIndex(), "解绑" + e.getMessage());
            return;
        }
        Resourceset resourceSet = resourceSetOptional.get();
        Long appId = resourceSet.getAppId();
        Optional<Resource> resource = this.findResourceByCodeAppId(appId, resourceCode);
        if (!resource.isPresent()) {
            messageRows.fail(row.getRowIndex(), "【" + resourceCode + "】" + RESOURCE_NOT_NULL);
            return;
        }
        //解邦资源码
        try {
            resourcesetService.unbindResources(resourceSet.getResourcesetId(), resource.get().getResourceId());
            messageRows.success(row.getRowIndex());
        } catch (IllegalArgumentException e) {
            messageRows.fail(row.getRowIndex(), "解绑" + e.getMessage());
        }
    }

    /**
     * 绑定资源码
     *
     * @param messageRows
     * @param row
     */
    private void bindResources(MessageRow messageRows, ResourceSetResourceExcelDTO row) {
        String resourceSetCode = StringUtils.trimToEmpty(row.getResourceSetCode());
        String resourceCode = StringUtils.trimToEmpty(row.getResourceCode());
        Optional<Resourceset> resourceSetOptional;
        try {
            resourceSetOptional = this.findByResourceSetCode(resourceSetCode);
            if (!resourceSetOptional.isPresent()) {
                messageRows.fail(row.getRowIndex(), "【" + resourceSetCode + "】" + RESOURCE_SET_NOT_NULL);
                return;
            }
        } catch (IllegalArgumentException e) {
            messageRows.fail(row.getRowIndex(), "绑定" + e.getMessage());
            return;
        }
        Resourceset resourceSet = resourceSetOptional.get();
        Long appId = resourceSet.getAppId();
        Optional<Resource> resource = this.findResourceByCodeAppId(appId, resourceCode);
        if (!resource.isPresent()) {
            messageRows.fail(row.getRowIndex(), "【" + resourceCode + "】" + RESOURCE_NOT_NULL);
            return;
        }
        ResourcesetModel.Request.BindResources bindResources = new ResourcesetModel.Request.BindResources();
        bindResources.setOverwrite(false);
        bindResources.setResourceIds(Arrays.asList(resource.get().getResourceId()));
        try {
            //绑定数据
            this.resourcesetService.bindResources(resourceSet, bindResources);
            messageRows.success(row.getRowIndex());
        } catch (Exception e) {
            messageRows.fail(row.getRowIndex(), "绑定" + e.getMessage());
        }
    }

    /**
     * 按功能集Code查询数据
     *
     * @param resourceSetCode
     * @return Optional<Resourceset>
     */
    private Optional<Resourceset> findByResourceSetCode(String resourceSetCode) {
        return this.resourcesetService.findByResourceSetCode(resourceSetCode);
    }

    /**
     * 根据资源码和AppId查询Resource
     *
     * @param appId        AppId
     * @param resourceCode ResourceCode
     * @return
     */
    private Optional<Resource> findResourceByCodeAppId(Long appId, String resourceCode) {
        //查询父级资源ID
        ResourceModel.Request.Query query = new ResourceModel.Request.Query();
        query.setAppId(appId);
        query.setResourceCode(resourceCode);
        return this.resourceService.findOne(query);
    }

    /**
     * @param messageRows
     * @param context
     * @param rows
     */
    private void saveResourceSets(MessageRow messageRows, Context context, List<ResourceSetExcelDTO> rows, List<Long> appIds) {
        //为空则不处理
        if (CollectionUtils.isEmpty(rows)) {
            return;
        }
        for (ResourceSetExcelDTO row : rows) {
            try {
                if (!row.getValidatedStatus()) {
                    messageRows.fail(row.getRowIndex(), row.getValidatedMessage());
                    continue;
                }
                if (RESOURCE_SET_ACTION_ADD.equals(row.getAction())) {
                    //保存
                    this.saveResourceSet(messageRows, row, appIds);
                } else if (RESOURCE_SET_ACTION_UPDATE.equals(row.getAction())) {
                    //更新
                    this.updateResourceSet(messageRows, row, appIds);
                }
            } catch (Exception ex) {
                log.error(ex.getMessage(), ex);
                messageRows.fail(row.getRowIndex(), row.getAction() + "数据失败");
            }
        }
        context.messageRow(messageRows.getSheetName(), messageRows);
    }

    /**
     * 更新资源码
     *
     * @param messageRows
     * @param row
     */
    private void updateResourceSet(MessageRow messageRows, ResourceSetExcelDTO row, List<Long> appIds) {
        String resourceSetCode = StringUtils.trimToEmpty(row.getResourceSetCode());
        Integer status = NumberUtils.parseNumber(row.getStatus(), Integer.class);
        Optional<Resourceset> resourceSetOptional;
        try {
            resourceSetOptional = this.findByResourceSetCode(resourceSetCode);
            if (!resourceSetOptional.isPresent()) {
                messageRows.fail(row.getRowIndex(), "【" + resourceSetCode + "】" + RESOURCE_SET_NOT_NULL);
                return;
            }
        } catch (IllegalArgumentException e) {
            messageRows.fail(row.getRowIndex(), "更新" + e.getMessage());
            return;
        }
        Long appId = NumberUtils.parseNumber(row.getAppId(), Long.class);
        if (!appIds.contains(appId)) {
            messageRows.fail(row.getRowIndex(), "【" + appId + "】APP的数据不存在");
            return;
        }
        ResourcesetModel.Request.Save resourceSet = new ResourcesetModel.Request.Save();
        resourceSet.setAppId(appId);
        resourceSet.setResourcesetCode(resourceSetCode);
        resourceSet.setResourcesetName(row.getResourceSetName());
        resourceSet.setResourcesetDesc(resourceSetOptional.get().getResourcesetDesc());
        resourceSet.setStatus(status);
        try {
            this.resourcesetService.update(resourceSetOptional.get().getResourcesetId(), resourceSet);
            messageRows.success(row.getRowIndex());
        } catch (Exception e) {
            messageRows.fail(row.getRowIndex(), e.getMessage());
        }
    }

    /**
     * 保存资源码
     *
     * @param messageRows
     * @param row
     * @param appIds
     */
    private void saveResourceSet(MessageRow messageRows, ResourceSetExcelDTO row, List<Long> appIds) {
        String resourceSetCode = StringUtils.trimToEmpty(row.getResourceSetCode());
        Integer status = NumberUtils.parseNumber(row.getStatus(), Integer.class);
        Optional<Resourceset> resourceSetOptional;
        try {
            resourceSetOptional = this.findByResourceSetCode(resourceSetCode);
            if (resourceSetOptional.isPresent()) {
                messageRows.fail(row.getRowIndex(), "【" + resourceSetCode + "】" + RESOURCE_SET_NULL);
                return;
            }
        } catch (IllegalArgumentException e) {
            messageRows.fail(row.getRowIndex(), "保存" + e.getMessage());
            return;
        }
        Long appId = NumberUtils.parseNumber(row.getAppId(), Long.class);
        if (!appIds.contains(appId)) {
            messageRows.fail(row.getRowIndex(), "【" + appId + "】APP的数据不存在");
            return;
        }
        ResourcesetModel.Request.Create resourceSet = new ResourcesetModel.Request.Create();
        resourceSet.setAppId(appId);
        resourceSet.setResourcesetCode(resourceSetCode);
        resourceSet.setResourcesetName(row.getResourceSetName());
        resourceSet.setResourcesetDesc(row.getResourceSetName());
        resourceSet.setStatus(status);
        try {
            this.resourcesetService.create(resourceSet);
            messageRows.success(row.getRowIndex());
        } catch (Exception e) {
            messageRows.fail(row.getRowIndex(), e.getMessage());
        }

    }

    /**
     * 查询所有APP
     *
     * @return Optional<App>
     */
    private List<Long> findAllByApp() {
        List<App> appList = this.appService.list();
        if (CollectionUtils.isEmpty(appList)) {
            return Collections.emptyList();
        }
        return appList.stream().map(App::getAppId).collect(Collectors.toList());
    }

    /**
     * 导出公共方法
     *
     * @param context 上下文
     * @return Context
     */
    @Override
    @SuppressWarnings("all")
    public void write(Context context) {
        Query query = context.getParam(PARAMS_QUERY, Query.class);
        Sort sort = context.getParam(PARAMS_SORT, Sort.class);
        List<String> sheetNames = Arrays.asList(query.getSheets());
        SimpleExcelWriter simpleExcelWriter = context.getSimpleExcelWriter();
        Integer maxPageSize = context.getMaxPageSize();
        ResourceSetQueryBo queryBo = new ResourceSetQueryBo();
        BeanUtils.copyProperties(query, queryBo);
        queryBo.setResourceSetIds(query.getResourcesetIds());
        for (String sheetName : sheetNames) {
            //功能集
            if (RESOURCE_SET_SHEET.equals(sheetName)) {
                Page<ResourceSetDTO> page;
                Pageable pageable = PageRequest.of(START_PAGE, PAGE_SIZE, sort.and(Sort.by(DESC, SORT_FIELD)));
                do {
                    page = this.resourcesetService.pagingBy(pageable, queryBo);
                    //对象转换
                    Page<ResourceSetExcelDTO> excelDTOPage = page.map(e -> {
                        ResourceSetExcelDTO dto = new ResourceSetExcelDTO();
                        dto.setAppId(e.getAppId() == null ? EMPTY : String.valueOf(e.getAppId()));
                        dto.setResourceSetName(StringUtils.trimToEmpty(e.getResourceSetName()));
                        dto.setResourceSetCode(StringUtils.trimToEmpty(e.getResourceSetCode()));
                        dto.setStatus(e.getStatus() == null ? EMPTY : String.valueOf(e.getStatus()));
                        return dto;
                    });
                    simpleExcelWriter.fill(sheetName, excelDTOPage.getContent());
                } while (page.hasNext() && (pageable = pageable.next()) != null && page.getNumber() < maxPageSize);
                //计算当前导出数据  this.calcSuccessSize(page,)
                simpleExcelWriter.param(SUCCESS_SIZE, this.calcSuccessSize(page));
            }
            //功能集与资源码导入
            if (RESOURCE_SHEET.equals(sheetName)) {
                Page<ResourceSetResourceDTO> page;
                Pageable pageable = PageRequest.of(START_PAGE, PAGE_SIZE, sort.and(Sort.by(DESC, SORT_FIELD)));
                do {
                    page = this.resourcesetService.pagingByResource(pageable, queryBo);
                    //对象转换
                    Page<ResourceSetResourceExcelDTO> excelDTOPage = page.map(e -> {
                        ResourceSetResourceExcelDTO dto = new ResourceSetResourceExcelDTO();
                        dto.setResourceSetCode(StringUtils.trimToEmpty(e.getResourceSetCode()));
                        dto.setResourceCode(StringUtils.trimToEmpty(e.getResourceCode()));
                        return dto;
                    });
                    simpleExcelWriter.fill(sheetName, excelDTOPage.getContent());
                } while (page.hasNext() && (pageable = pageable.next()) != null && page.getNumber() < maxPageSize);
                //计算当前导出数据  this.calcSuccessSize(page,)
                simpleExcelWriter.param(SUCCESS_SIZE, this.calcSuccessSize(page));
            }
        }
    }

    /**
     * 错误信息
     */
    protected static final class Message {
        /**
         * 功能集数据不存在提示信息
         */
        public static final String RESOURCE_SET_NOT_NULL = "功能集数据不存在";
        /**
         * 资源码不存在提示信息
         */
        public static final String RESOURCE_NOT_NULL = "资源码数据不存在";
        /**
         * 功能数据已经存在
         */
        public static final String RESOURCE_SET_NULL = "功能集数据已存在";
    }
}
