package com.xforceplus.business.resource.service;

import com.google.common.collect.Lists;
import com.xforceplus.dao.AppDao;
import com.xforceplus.dao.ResourcesetResourceRelDao;
import com.xforceplus.domain.resource.ResourceDto;
import com.xforceplus.domain.resource.ResourceExtendDto;
import com.xforceplus.domain.resource.ResourcesetDto;
import com.xforceplus.domain.resource.ResourcesetExtendDto;
import com.xforceplus.entity.Resource;
import com.xforceplus.entity.Resourceset;
import com.xforceplus.entity.ResourcesetResourceRel;
import com.xforceplus.utils.excel.QueryUtils;
import com.xforceplus.utils.excel.ResourcesetWrapper;
import io.geewit.core.utils.reflection.BeanUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 导入导出资源码服务类
 *
 * @author lengmz
 */
@Service
@Slf4j
public class ResourcesetPortService {

    @Autowired
    private ResourceService resourceService;
    @Autowired
    private ResourcesetResourceRelDao resourcesetResourceRelDao;
    @Autowired
    private ResourcesetService resourcesetService;
    @Autowired
    private ResourcePortService resourcePortService;
    @Autowired
    private AppDao appDao;

    private String operateUserName = "手动导入";


    public ResourcesetWrapper getExportData(List<Long> resourcesetIds) {
        Specification<Resourceset> specification = QueryUtils.getSpecification("resourcesetId", resourcesetIds);
        List<Resourceset> list = resourcesetService.list(specification, Sort.unsorted());
        if (CollectionUtils.isEmpty(list)) {
            throw new IllegalArgumentException("所选id记录不存在。");
        }
        Set<Long> resourceIds = list.stream().filter(p -> p.getResourceId() != null && p.getResourceId() > 0).map(ResourcesetDto::getResourceId).collect(Collectors.toSet());
        List<ResourceExtendDto> resourceDtos = new ArrayList<>(list.size());
        List<ResourceExtendDto> moduleResources = getModuleResource(resourceIds);
        if (CollectionUtils.isNotEmpty(moduleResources)) {
            resourceDtos.addAll(moduleResources);
        }
        List<ResourceExtendDto> childResources = getChildrenResource(list);
        if (CollectionUtils.isNotEmpty(childResources)) {
            resourceDtos.addAll(childResources);
        }
        List<ResourcesetExtendDto> extendList = new ArrayList<>(list.size());
        Map<Long, Integer> moduleResourceIdAndIndex = getModuleResourceIdAndIndex(moduleResources);
        for (ResourcesetDto resourcesetDto : list) {
            ResourcesetExtendDto resourcesetExtendDto = new ResourcesetExtendDto();
            BeanUtils.copyProperties(resourcesetDto, resourcesetExtendDto);
            Integer relResourceIndex = moduleResourceIdAndIndex.get(resourcesetDto.getResourceId());
            if (relResourceIndex != null) {
                resourcesetExtendDto.setRelResourceIndex(relResourceIndex + 1);
            }
            extendList.add(resourcesetExtendDto);
        }
        ResourcesetWrapper result = new ResourcesetWrapper();
        result.setResourcesets(extendList);
        result.setResources(resourceDtos);
        return result;
    }


    private Map<Long, Integer> getModuleResourceIdAndIndex(List<ResourceExtendDto> list) {
        Map<Long, Integer> map = new HashMap<>(list.size());
        if (list.isEmpty()) {
            return map;
        }
        for (int i = 0; i < list.size(); i++) {
            ResourceDto resourceDto = list.get(i);
            if (resourceDto.getResourceId() != null && resourceDto.getResourceId() > 0L) {
                map.put(resourceDto.getResourceId(), i);
            }
        }
        return map;
    }

    private List<ResourceExtendDto> getResource(Set<Long> resourceIds, List<Resourceset> list) {
        List<ResourceExtendDto> resourceList = new ArrayList<>(list.size());
        List<ResourceExtendDto> moduleResources = getModuleResource(resourceIds);
        if (CollectionUtils.isNotEmpty(moduleResources)) {
            resourceList.addAll(moduleResources);
        }
        List<ResourceExtendDto> childResources = getChildrenResource(list);
        if (CollectionUtils.isNotEmpty(childResources)) {
            resourceList.addAll(childResources);
        }
        return resourceList;
    }

    private List<ResourceExtendDto> getChildrenResource(List<Resourceset> list) {
        List<ResourceExtendDto> resourceList = new ArrayList<>();
        List<Long> resourcesetIds = list.stream().map(ResourcesetDto::getResourcesetId)
                .collect(Collectors.toList());
        Specification<ResourcesetResourceRel> specification = QueryUtils.getSpecification("resourcesetId", resourcesetIds);
        List<ResourcesetResourceRel> rels = this.resourcesetResourceRelDao.findAll(specification, Sort.unsorted());
        if (CollectionUtils.isEmpty(rels)) {
            return new ArrayList();
        }
        List<Long> relIds = rels.stream().map(ResourcesetResourceRel::getResourceId).collect(Collectors.toList());
        Specification<Resource> finalSpecification = QueryUtils.getSpecification("resourceId", relIds);
        List<Resource> resources = resourceService.list(finalSpecification, Sort.unsorted());
        if (CollectionUtils.isEmpty(resources)) {
            return new ArrayList();
        }
        Map<Long, Integer> resourcesetIdAndSetIndex = getResourcesetIdAndSetIndex(list);
        Map<Long, Integer> resourceIdAndResourceIndex = getResourceIdAndResourceIndex(resources);
        for (ResourcesetResourceRel resourcesetResourceRel : rels) {
            int resourceIndex = resourceIdAndResourceIndex.get(resourcesetResourceRel.getResourceId());
            Resource resource = resources.get(resourceIndex - 1);
            ResourceExtendDto resourceExtendDto = new ResourceExtendDto();
            BeanUtils.copyProperties(resource, resourceExtendDto);
            Integer relSetIndex = resourcesetIdAndSetIndex.get(resourcesetResourceRel.getResourcesetId());
            if (relSetIndex != null) {
                resourceExtendDto.setRelSetIndex(relSetIndex);
            }
            resourceList.add(resourceExtendDto);
        }
        return resourceList;
    }

    private List<ResourceExtendDto> getModuleResource(Set<Long> resourceIds) {
        if (CollectionUtils.isEmpty(resourceIds)) {
            return new ArrayList<>();
        }
        List<ResourceExtendDto> resourceList = new ArrayList<>(resourceIds.size());
        Specification<Resource> specification = QueryUtils.getSpecification("resourceId", Lists.newArrayList(resourceIds));
        List<Resource> resources = resourceService.list(specification, Sort.unsorted());
        for (Resource resource : resources) {
            ResourceExtendDto resourceExtendDto = new ResourceExtendDto();
            BeanUtils.copyProperties(resource, resourceExtendDto);
            resourceList.add(resourceExtendDto);
        }
        return resourceList;
    }

    private Map<Long, Integer> getResourceIdAndSetIndex(List<ResourcesetResourceRel> rels, Map<Long, Integer> resourcesetIdAndIndex) {
        Map<Long, Integer> map = new HashMap<>(rels.size());
        if (rels.isEmpty()) {
            return map;
        }
        for (ResourcesetResourceRel resourcesetResourceRel : rels) {
            Integer index = resourcesetIdAndIndex.get(resourcesetResourceRel.getResourcesetId());
            map.put(resourcesetResourceRel.getResourceId(), index);
        }
        return map;
    }

    private Map<Long, Integer> getResourcesetIdAndSetIndex(List<Resourceset> list) {
        Map<Long, Integer> map = new HashMap<>(list.size());
        if (list.isEmpty()) {
            return map;
        }
        for (int i = 0; i < list.size(); i++) {
            ResourcesetDto resourcesetDto = list.get(i);
            map.put(resourcesetDto.getResourcesetId(), i + 1);
        }
        return map;
    }

    private Map<Long, Integer> getResourceIdAndResourceIndex(List<Resource> list) {
        Map<Long, Integer> map = new HashMap<>(list.size());
        if (list.isEmpty()) {
            return map;
        }
        for (int i = 0; i < list.size(); i++) {
            Resource resource = list.get(i);
            map.put(resource.getResourceId(), i + 1);
        }
        return map;
    }

    public ResourcesetWrapper saveResourcesetData(ResourcesetWrapper wrapper) {
        ResourcesetWrapper result = new ResourcesetWrapper();
        List<ResourceExtendDto> resourceExtendDtoList = wrapper.getResources();
        List<String> resourceCodes = resourceExtendDtoList.stream().map(ResourceDto::getResourceCode).collect(Collectors.toList());
        List<Resource> resources = resourcePortService.getResourceByCodes(resourceCodes);
        Map<String, Resource> codeAndResourceMap = this.getCodeAndResourceMap(resources);
        List<ResourcesetExtendDto> resourcesets = this.saveResourceset(wrapper.getResourcesets(), wrapper.getResources(), codeAndResourceMap);
        List<ResourceExtendDto> resourceExtends = this.saveResourceRel(wrapper.getResourcesets(), wrapper.getResources(), codeAndResourceMap);
        result.setResourcesets(resourcesets);
        result.setResources(resourceExtends);
        return result;
    }

    private List<ResourceExtendDto> saveResourceRel(List<ResourcesetExtendDto> resourcesets, List<ResourceExtendDto> resources, Map<String, Resource> codeAndResourceMap) {
        List<ResourceExtendDto> existList = new ArrayList<>();
        for (ResourceExtendDto item : resources) {

            if (null == item.getAppId() || !appDao.existsById(item.getAppId())) {
                item.setReason("无效的appId。");
                existList.add(item);
                continue;
            }

            if (item.getRelSetIndex() == null || item.getRelSetIndex() <= 0) {
                continue;
            }
            ResourcesetExtendDto resourcesetExtendDto = resourcesets.get(item.getRelSetIndex() - 1);
            ResourceDto resourceDto = codeAndResourceMap.get(this.getResourceCodeKey(item.getResourceCode(), item.getAppId()));
            if (resourceDto == null) {
                item.setReason("数据库没有对应资源码，请先导入改资源码。");
                existList.add(item);
                log.info("没有找到资源码id=={}", item.getResourceCode());
                continue;
            }
            boolean exists = resourcesetService.saveResourceRel(resourcesetExtendDto.getResourcesetId(), resourceDto.getResourceId(), operateUserName);
            if (exists) {
                item.setReason("该功能集已经和该资源码关联,。");
                existList.add(item);
            }
        }
        return existList;
    }

    private Map<String, Resource> getCodeAndResourceMap(List<Resource> resources) {
        Map<String, Resource> map = new HashMap<>(resources.size());
        if (resources.isEmpty()) {
            return map;
        }
        for (Resource resource : resources) {
            map.put(this.getResourceCodeKey(resource.getResourceCode(), resource.getAppId()), resource);
        }
        return map;
    }

    private String getResourceCodeKey(String resourceCode, Long appId) {
        String appIdStr = (appId == null ? "" : String.valueOf(appId));
        return resourceCode + "_" + appIdStr;
    }

    private List<ResourcesetExtendDto> saveResourceset(List<ResourcesetExtendDto> resourcesets, List<ResourceExtendDto> resources, Map<String, Resource> codeAndResourceMap) {
        List<ResourcesetExtendDto> existLists = new ArrayList<>();
        for (ResourcesetExtendDto item : resourcesets) {
            if (null == item.getAppId() || !appDao.existsById(item.getAppId())) {
                item.setReason("无效的appId。");
                existLists.add(item);
                continue;
            }
            item.setResourceId(0L);
            if (item.getRelResourceIndex() != null && item.getRelResourceIndex() > 0) {
                ResourceExtendDto resourceExtendDto = resources.get(item.getRelResourceIndex() - 1);
                ResourceDto resourceDto = codeAndResourceMap.get(getResourceCodeKey(resourceExtendDto.getResourceCode(), resourceExtendDto.getAppId()));
                if (resourceDto != null) {
                    item.setResourceId(resourceDto.getResourceId());
                } else {
                    log.info("没有找到模块资源码id=={}", resourceExtendDto.getResourceCode());
                }
            }
            ResourcesetDto resourcesetDto = this.isResourcesetExists(item);
            if (resourcesetDto != null) {
                BeanUtils.copyProperties(resourcesetDto, item);
                item.setReason("该功能集已经存在。id==" + resourcesetDto.getResourcesetId());
                existLists.add(item);
            } else {
                resourcesetDto = resourcesetService.save(item);
                item.setResourcesetId(resourcesetDto.getResourcesetId());
            }
        }
        return existLists;
    }

    private ResourcesetDto isResourcesetExists(ResourcesetExtendDto item) {
        if (StringUtils.isBlank(item.getResourcesetName()) || item.getResourceId() == null) {
            return null;
        }
        List<Resourceset> list = resourcesetService.list(new Specification<Resourceset>() {
            @Override
            public Predicate toPredicate(Root<Resourceset> root, CriteriaQuery<?> query, CriteriaBuilder builder) {
                List<Predicate> predicates = new ArrayList<>();
                predicates.add(builder.equal(root.<String>get("resourcesetName"), item.getResourcesetName()));
                if (item.getAppId() != null && item.getAppId() > 0L) {
                    predicates.add(builder.equal(root.<Long>get("appId"), item.getAppId()));
                }
                if (!predicates.isEmpty()) {
                    query.where(predicates.stream().toArray(Predicate[]::new));
                }
                return query.getRestriction();
            }
        }, Sort.unsorted());
        if (CollectionUtils.isEmpty(list)) {
            return null;
        }
        return list.get(0);
    }


}
