package com.xforceplus.business.resource.service;

import com.xforceplus.api.model.ResourcesetModel.Request.BindResources;
import com.xforceplus.api.model.ResourcesetModel.Request.Create;
import com.xforceplus.api.model.ResourcesetModel.Request.Query;
import com.xforceplus.api.model.ResourcesetModel.Request.Save;
import com.xforceplus.bo.ResourceSetQueryBo;
import com.xforceplus.dao.*;
import com.xforceplus.domain.resource.ResourcesetDto;
import com.xforceplus.dto.resource.ResourceSetDTO;
import com.xforceplus.dto.resource.ResourceSetResourceDTO;
import com.xforceplus.entity.Resource;
import com.xforceplus.entity.Resourceset;
import com.xforceplus.entity.ResourcesetResourceRel;
import com.xforceplus.query.ResourcesetQueryHelper;
import io.geewit.core.utils.reflection.BeanUtils;
import io.geewit.data.jpa.essential.domain.EntityGraphs;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Service
@Slf4j
public class ResourcesetService {
    private final static Logger logger = LoggerFactory.getLogger(ResourcesetService.class);

    private final ResourcesetResourceRelDao resourcesetResourceRelDao;

    private final ResourcesetDao resourcesetDao;

    private final ResourceDao resourceDao;

    private final AppDao appDao;

    private final ResourceService resourceService;

    private final ServiceResourcesetRelDao serviceResourcesetRelDao;

    /**
     * 扩展Dao类
     */
    private final ResourceSetExtendDao resourceSetExtendDao;

    public ResourcesetService(ResourcesetDao resourcesetDao, ResourcesetResourceRelDao resourcesetResourceRelDao, ResourceDao resourceDao, AppDao appDao, ResourceService resourceService, ServiceResourcesetRelDao serviceResourcesetRelDao, ResourceSetExtendDao resourceSetExtendDao) {
        this.resourcesetDao = resourcesetDao;
        this.resourcesetResourceRelDao = resourcesetResourceRelDao;
        this.resourceDao = resourceDao;
        this.appDao = appDao;
        this.resourceService = resourceService;
        this.serviceResourcesetRelDao = serviceResourcesetRelDao;
        this.resourceSetExtendDao = resourceSetExtendDao;
    }

    public Page<Resourceset> page(Query query, Pageable pageable) {
        Specification<Resourceset> specification = ResourcesetQueryHelper.querySpecification(query);
        return resourcesetDao.findAll(specification, pageable, EntityGraphs.named(Resourceset.NAMED_ENTITY_GRAPH_DEFAULT));
    }

    public List<Resourceset> list(Query query, Sort sort) {
        Specification<Resourceset> specification = ResourcesetQueryHelper.querySpecification(query);
        List<Resourceset> list = resourcesetDao.findAll(specification, sort);
        for (Resourceset resourceset : list) {
            List<Resource> resources = resourceService.listByResourcesetId(resourceset.getResourcesetId(), query.getStatus());
            resourceset.setResources(new HashSet<>(resources));
        }
        return list;
    }

    public Page<Resourceset> page(Specification<Resourceset> specification, Pageable pageable) {
        pageable.getSort().and(Sort.by("create_time").ascending());
        return resourcesetDao.findAll(specification, pageable, EntityGraphs.named(Resourceset.NAMED_ENTITY_GRAPH_DEFAULT));
    }

    public long count(Query query) {
        Specification<Resourceset> specification = ResourcesetQueryHelper.querySpecification(query);
        return resourcesetDao.count(specification);
    }

    public Optional<Resourceset> findOne(Query query) {
        Specification<Resourceset> specification = ResourcesetQueryHelper.queryOneSpecification(query);
        return resourcesetDao.findOne(specification);
    }

    public long countOne(Query query) {
        Specification<Resourceset> specification = ResourcesetQueryHelper.queryOneSpecification(query);
        return resourcesetDao.count(specification);
    }

    @Transactional(rollbackFor = Exception.class)
    public Resourceset create(Create model) {

        if (null == model.getValidatedAppId() || model.getValidatedAppId()) {
            if (model.getAppId() != null && model.getAppId() > 0) {
                if (!appDao.existsById(model.getAppId())) {
                    throw new IllegalArgumentException("无效的appId");
                }
            } else {
                throw new IllegalArgumentException("无效的参数");
            }
        } else {
            model.setAppId(0L);
        }

        if (StringUtils.isNotBlank(model.getResourcesetName())) {
            Query query = new Query();
            query.setResourcesetNameEqual(model.getResourcesetName());
            query.setAppId(model.getAppId());
            query.setRoleId(model.getRoleId());
            long count = this.count(query);
            if (count > 0) {
                String message = "已经存在该功能集名称(" + model.getResourcesetName() + ")";
                logger.warn(message);
                throw new IllegalArgumentException(message);
            }
        }
        if (StringUtils.isNotBlank(model.getResourcesetCode())) {
            Query query = new Query();
            query.setResourcesetCode(model.getResourcesetCode());
            query.setRoleId(model.getRoleId());
            long count = this.countOne(query);
            if (count > 0) {
                String message = "已经存在该功能集代码(" + model.getResourcesetCode() + ")";
                logger.warn(message);
                throw new IllegalArgumentException(message);
            }
        }
        Resourceset entity = new Resourceset();
        BeanUtils.copyProperties(model, entity);
        entity = resourcesetDao.saveAndFlush(entity);
        if (model.getBindResources() != null) {
            this.bindResources(entity, model.getBindResources());
        }
        return entity;
    }

    @Transactional(rollbackFor = Exception.class)
    public <T extends ResourcesetDto> Resourceset save(T model) {
        Resourceset entity = new Resourceset();
        BeanUtils.copyProperties(model, entity, Stream.of("roleResourcesetRels", "resourcesetResourceRels", "serviceResourcesetRels", "resources", "app").toArray(String[]::new));
        entity = resourcesetDao.saveAndFlush(entity);
        return entity;
    }

    @Transactional(rollbackFor = Exception.class)
    public Resourceset update(long resourcesetId, Save model) {
        Resourceset existEntity = this.findById(resourcesetId);
        if (model.getAppId() != null && model.getAppId() > 0) {
            if (!existEntity.getAppId().equals(model.getAppId())) {
                if (!appDao.existsById(model.getAppId())) {
                    throw new IllegalArgumentException("无效的参数");
                }
            }
        } else {
            model.setAppId(null);
        }
        if (StringUtils.isNotBlank(model.getResourcesetCode())) {
            if (!model.getResourcesetCode().equals(existEntity.getResourcesetCode())) {
                long count = resourcesetDao.countByNotCurrentResourcesetIdAndCode(resourcesetId, model.getResourcesetCode());
                if (count > 0) {
                    throw new IllegalArgumentException("已存在该功能集代码");
                }
            }
        } else {
            model.setResourcesetCode(null);
        }
        if (StringUtils.isNotBlank(model.getResourcesetName())) {
            if (!model.getResourcesetName().equals(existEntity.getResourcesetName())) {
                long count = resourcesetDao.countByNotCurrentResourcesetIdAndName(resourcesetId, model.getResourcesetName());
                if (count > 0) {
                    throw new IllegalArgumentException("已存在该功能集名称");
                }
            }
        } else {
            model.setResourcesetName(null);
        }
        BeanUtils.copyProperties(model, existEntity);
        existEntity = resourcesetDao.saveAndFlush(existEntity);
        if (model.getBindResources() != null) {
            this.bindResources(existEntity, model.getBindResources());
        }
        return existEntity;
    }

    public Resourceset findById(Long resourcesetId) {
        Resourceset resourceset = resourcesetDao.findById(resourcesetId).orElseThrow(() -> new IllegalArgumentException("未找到功能集"));
        List<Resource> resources = resourceService.listByResourcesetId(resourcesetId, null);
        resourceset.setResources(resources.stream().filter(Objects::nonNull).collect(Collectors.toSet()));
        return resourceset;
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteById(long resourcesetId) {
        Resourceset resourceset = this.findById(resourcesetId);
        if (resourceset.getStatus() != null && resourceset.getStatus() == 1) {
            throw new IllegalArgumentException("禁用的功能集才可以被删除");
        }
        long count = serviceResourcesetRelDao.countByResourcesetId(resourcesetId);
        if (count > 0) {
            throw new IllegalArgumentException("该功能集已经被服务包绑定过不能删除");
        }
        resourcesetDao.deleteById(resourceset.getResourcesetId());
        resourcesetResourceRelDao.deleteByResourcesetId(resourcesetId);
        serviceResourcesetRelDao.deleteByResourcesetId(resourcesetId);
    }

    public List<Resourceset> list(Specification<Resourceset> specification, Sort sort) {
        return resourcesetDao.findAll(specification, sort);
    }

    /**
     * 解绑资源码
     *
     * @param resourceSetId
     * @param resourceId
     */
    @Transactional(rollbackFor = Exception.class)
    public void unbindResources(Long resourceSetId, Long resourceId) {
        List<ResourcesetResourceRel> rels = resourcesetResourceRelDao.findByResourcesetIdAndResourceId(resourceSetId, resourceId);
        if (CollectionUtils.isEmpty(rels)) {
            throw new IllegalArgumentException("功能集与资源资源码未绑定");
        }
        //删除绑定关系
        try {
            resourcesetResourceRelDao.deleteById(rels.get(0).getId());
        } catch (Exception e) {
            logger.warn(e.getMessage(), e);
        }
    }

    /**
     * 功能集绑定资源码列表(* 此方法会解绑 resourceIds 中不存在但是数据库中有的功能集-资源码关系)
     *
     * @param resourceset
     * @param bindResources
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public void bindResources(Resourceset resourceset, BindResources bindResources) {
        if (bindResources == null || CollectionUtils.isEmpty(bindResources.getResourceIds())) {
            resourcesetResourceRelDao.deleteByResourcesetId(resourceset.getResourcesetId());
        } else {
            List<Long> bindResourceIds = bindResources.getResourceIds();
            logger.info("bindResourceIds = " + bindResourceIds.stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.joining(",")));
            List<ResourcesetResourceRel> rels = resourcesetResourceRelDao.findByResourcesetId(resourceset.getResourcesetId());
            logger.info("exist resourceset-resource-rels.size = " + rels.size());
            //region 往绑定不在数据库中但是报文 resourceIds 中有的功能集-资源码关系
            Set<ResourcesetResourceRel> insertingRels = bindResourceIds.stream().filter(Objects::nonNull).filter(resourceId -> rels.stream().map(ResourcesetResourceRel::getResourceId).noneMatch(resourceDtoId -> resourceDtoId.equals(resourceId))).map(resourceId -> {
                Optional<Resource> resourceOptional = resourceDao.findById(resourceId);
                if (resourceOptional.isPresent()) {
                    ResourcesetResourceRel rel = new ResourcesetResourceRel();
                    rel.setResourcesetId(resourceset.getResourcesetId());
                    rel.setResourceId(resourceId);
                    return rel;
                } else {
                    return null;
                }
            }).filter(Objects::nonNull).collect(Collectors.toSet());
            this.bindResources(insertingRels);
            //endregion
            if (bindResources.isOverwrite()) {
                //region 从数据库中解绑不在报文 resourcesetIds 中的服务包-功能集关系
                rels.stream().filter(resource -> bindResourceIds.stream().noneMatch(resourceId -> resourceId.equals(resource.getResourceId()))).forEach(rel -> {
                    logger.info("deleting Resourceset-Resource-Rel record, {}", rel);
                    try {
                        resourcesetResourceRelDao.deleteById(rel.getId());
                    } catch (Exception e) {
                        logger.warn(e.getMessage(), e);
                    }
                });
                //endregion
            }
        }
    }

    public void bindResources(Collection<ResourcesetResourceRel> rels) {
        if (!CollectionUtils.isEmpty(rels)) {
            resourcesetResourceRelDao.saveAllAndFlush(rels);
        }
    }

    public boolean saveResourceRel(Long resourcesetId, Long resourceId, String operateUserName) {
        List<ResourcesetResourceRel> list = resourcesetResourceRelDao.findAll(new Specification<ResourcesetResourceRel>() {
            @Override
            public Predicate toPredicate(Root<ResourcesetResourceRel> root, CriteriaQuery<?> query, CriteriaBuilder builder) {
                List<Predicate> predicates = new ArrayList<>();
                predicates.add(builder.equal(root.<Long>get("resourcesetId"), resourcesetId));
                predicates.add(builder.equal(root.<Long>get("resourceId"), resourceId));
                if (!predicates.isEmpty()) {
                    query.where(predicates.stream().toArray(Predicate[]::new));
                }
                return query.getRestriction();
            }
        });

        if (list.isEmpty()) {
            ResourcesetResourceRel resourcesetResourceRel = new ResourcesetResourceRel();
            resourcesetResourceRel.setResourcesetId(resourcesetId);
            resourcesetResourceRel.setResourceId(resourceId);
            resourcesetResourceRel.setCreateUserName(operateUserName);
            resourcesetResourceRelDao.saveAndFlush(resourcesetResourceRel);
            return false;
        } else {
            return true;
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateStatus(long resourcesetId, int status) {
        Resourceset existEntity = this.findById(resourcesetId);
        existEntity.setStatus(status);
        resourcesetDao.saveAndFlush(existEntity);
    }

    public List<Resourceset> listByRoleId(long roleId) {
        Query query = new Query();
        query.setRoleId(roleId);
        query.setStatus(1);
        Specification<Resourceset> specification = ResourcesetQueryHelper.querySpecification(query);
        List<Resourceset> resourcesets = resourcesetDao.findAll(specification);
        return resourcesets;
    }

    public List<Resourceset> listByServicePackageId(long servicePackageId, Integer status) {
        Query query = new Query();
        query.setServicePackageId(servicePackageId);
        query.setStatus(status);
        Specification<Resourceset> specification = ResourcesetQueryHelper.querySpecification(query);
        List<Resourceset> resourcesets = resourcesetDao.findAll(specification);
        return resourcesets;
    }

    public List<Resourceset> listByServicePackageIds(Collection<Long> packageIds) {
        Query query = new Query();
        query.setPackageIds(new HashSet<>(packageIds));
        query.setStatus(1);
        Specification<Resourceset> specification = ResourcesetQueryHelper.querySpecification(query);
        List<Resourceset> resourcesets = resourcesetDao.findAll(specification);
        return resourcesets;
    }

    public Map<Long, List<Resourceset>> mapByServicePackageIds(Collection<Long> packageIds) {
        Query query = new Query();
        query.setPackageIds(new HashSet<>(packageIds));
        query.setStatus(1);
        query.setAttributes(Stream.of("packageId").collect(Collectors.toSet()));
        List<Resourceset> resourcesets = resourcesetDao.findAttributes(query, Sort.unsorted());
        Map<Long, List<Resourceset>> map = resourcesets.stream().collect(Collectors.groupingBy(Resourceset::getPackageId));
        return map;
    }

    /**
     * 分页查询
     *
     * @param pageable
     * @param queryBo
     * @return
     */
    
    public Page<ResourceSetDTO> pagingBy(Pageable pageable, ResourceSetQueryBo queryBo) {
        return this.resourceSetExtendDao.pagingBy(pageable, queryBo);
    }

    /**
     * 分页查询
     *
     * @param pageable
     * @param queryBo
     * @return
     */
    
    public Page<ResourceSetResourceDTO> pagingByResource(Pageable pageable, ResourceSetQueryBo queryBo) {
        return this.resourceSetExtendDao.pagingByResource(pageable, queryBo);
    }

    public Optional<Resourceset> findByResourceSetCode(String resourceSetCode) {
        List<Resourceset> resourceSetList = this.resourcesetDao.findByCode(resourceSetCode);
        if (CollectionUtils.isEmpty(resourceSetList)) {
            return Optional.empty();
        }
        //如果
        if (resourceSetList.size() > 1) {
            throw new IllegalArgumentException("【" + resourceSetCode + "】功能集数据存在不唯一记录");
        }
        return Optional.of(resourceSetList.get(0));
    }
}
