package com.xforceplus.business.resource.service;

import com.xforceplus.api.model.ServiceApiModel.Request.BindResources;
import com.xforceplus.api.model.ServiceApiModel.Request.Query;
import com.xforceplus.api.model.ServiceApiModel.Request.Save;
import com.xforceplus.api.model.ServiceApiModel.Request.Update;
import com.xforceplus.dao.ResourceApiRelDao;
import com.xforceplus.dao.ResourceDao;
import com.xforceplus.dao.ServiceApiDao;
import com.xforceplus.dao.ServiceApiExtendDao;
import com.xforceplus.domain.resource.RequestUri;
import com.xforceplus.domain.resource.RequestUriAuthz;
import com.xforceplus.dto.resource.ServiceApiResourceCodeVo;
import com.xforceplus.entity.Resource;
import com.xforceplus.entity.ResourceApiRel;
import com.xforceplus.entity.ServiceApi;
import com.xforceplus.query.ServiceApiQueryHelper;
import io.geewit.core.utils.reflection.BeanUtils;
import io.geewit.data.jpa.essential.domain.EntityGraphs;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.RequestMethod;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Service
public class ServiceApiService {
    private final static Logger logger = LoggerFactory.getLogger(ServiceApiService.class);

    private final ServiceApiDao serviceApiDao;

    private final ResourceApiRelDao resourceApiRelDao;

    private final ResourceService resourceService;

    private final ResourceDao resourceDao;

    private final ServiceApiExtendDao serviceApiExtendDao;

    public ServiceApiService(ServiceApiDao serviceApiDao, ResourceApiRelDao resourceApiRelDao,
                             ResourceService resourceService, ResourceDao resourceDao,
                             ServiceApiExtendDao serviceApiExtendDao) {
        this.serviceApiDao = serviceApiDao;
        this.resourceApiRelDao = resourceApiRelDao;
        this.resourceService = resourceService;
        this.resourceDao = resourceDao;
        this.serviceApiExtendDao = serviceApiExtendDao;
    }

    public Page<ServiceApi> page(Query query, Pageable pageable) {
        Specification<ServiceApi> specification = ServiceApiQueryHelper.querySpecification(query);
        return serviceApiDao.findAll(specification, pageable, EntityGraphs.named(ServiceApi.NAMED_ENTITY_GRAPH_DEFAULT));
    }

    public Page<ServiceApi> page(Specification<ServiceApi> specification, Pageable pageable) {
        return serviceApiDao.findAll(specification, pageable);
    }

    public List<ServiceApi> list(Query query, Sort sort) {
        Specification<ServiceApi> specification = ServiceApiQueryHelper.querySpecification(query);
        List<ServiceApi> list = serviceApiDao.findAll(specification, sort);
        return list;
    }

    public List<ServiceApi> list(Specification<ServiceApi> specification, Sort sort) {
        return serviceApiDao.findAll(specification, sort);
    }

    public long count(Query query) {
        Specification<ServiceApi> specification = ServiceApiQueryHelper.querySpecification(query);
        return serviceApiDao.count(specification);
    }

    @Transactional(rollbackFor = Exception.class)
    public ServiceApi save(Save model) {
        ServiceApi entity = new ServiceApi();
        BeanUtils.copyProperties(model, entity);
        entity = serviceApiDao.saveAndFlush(entity);
        this.bindResources(entity, model.getBindResources());
        return entity;
    }

    @Transactional(rollbackFor = Exception.class)
    public ServiceApi update(long id, Update model) {
        ServiceApi existEntity = this.findById(id);
        if (StringUtils.isBlank(model.getServiceApiName())) {
            model.setServiceApiName(null);
        }
        if (StringUtils.isBlank(model.getServiceApiPath())) {
            model.setServiceApiPath(null);
        }
        if (StringUtils.isBlank(model.getServiceApiUrl())) {
            model.setServiceApiUrl(null);
        }
        BeanUtils.copyProperties(model, existEntity);
        existEntity = serviceApiDao.saveAndFlush(existEntity);
        this.bindResources(existEntity, model.getBindResources());
        return existEntity;
    }

    /**
     * @param routeId
     * @param serviceApiPath
     * @param requestMethod
     * @return
     */
    public Optional<ServiceApi> findByRouteIdApiUrlMethod(Long routeId, String serviceApiUrl, RequestMethod requestMethod) {
        List<ServiceApi> serviceApis = this.serviceApiDao.findByRouteIdAndServiceApiUrlAndRequestMethod(routeId, serviceApiUrl, requestMethod);
        if (CollectionUtils.isEmpty(serviceApis)) {
            return Optional.empty();
        }
        if (serviceApis.size() > 1) {
            throw new IllegalArgumentException("路径ID:[" + routeId + "], Path:[" + serviceApiUrl + "], RequestMethod:[" + requestMethod.name() + "]不唯一数据，请先修正数据");
        }
        return Optional.of(serviceApis.get(0));
    }

    public ServiceApi findById(Long id) {
        return serviceApiDao.findById(id).orElseThrow(() -> new IllegalArgumentException("未找到实体"));
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteById(long id) {
        Optional<ServiceApi> optionalServiceApi = serviceApiDao.findById(id);
        if (!optionalServiceApi.isPresent()) {
            String message = "不存在id:" + id + "的服务接口";
            throw new IllegalArgumentException(message);
        }

        long countRel = resourceApiRelDao.countByServiceApiId(id);
        if (countRel > 0) {
            throw new IllegalArgumentException("该服务接口(" + id + ")已经关联资源码不能删除");
        }
        serviceApiDao.deleteById(id);
        resourceApiRelDao.deleteByServiceApiId(id);
    }

    @Transactional(rollbackFor = Exception.class)
    public boolean saveResourceApiRel(Long resourceId, Long serviceApiId, String operateUserName) {
        List<ResourceApiRel> rels = resourceApiRelDao.findAll(new Specification<ResourceApiRel>() {
            @Override
            public Predicate toPredicate(Root<ResourceApiRel> root, CriteriaQuery<?> query, CriteriaBuilder builder) {
                List<Predicate> predicates = Stream.of(
                        builder.equal(root.<Long>get("serviceApiId"), serviceApiId),
                        builder.equal(root.<Long>get("resourceId"), resourceId)
                ).collect(Collectors.toList());
                if (!predicates.isEmpty()) {
                    query.where(predicates.stream().toArray(Predicate[]::new));
                }
                return query.getRestriction();
            }
        });
        if (rels.isEmpty()) {
            ResourceApiRel resourceApiRel = new ResourceApiRel();
            resourceApiRel.setResourceId(resourceId);
            resourceApiRel.setServiceApiId(serviceApiId);
            resourceApiRel.setCreateUserName(operateUserName);
            resourceApiRelDao.saveAndFlush(resourceApiRel);
            return true;
        }
        return false;
    }

    /**
     * @param id            接口id
     * @param bindResources
     */
    @Transactional(rollbackFor = Exception.class)
    public void bindResources(long id, BindResources bindResources) {
        ServiceApi serviceApi = this.findById(id);
        this.bindResources(serviceApi, bindResources);
    }

    @Transactional(rollbackFor = Exception.class)
    public void bindResources(ServiceApi serviceApi, BindResources bindResources) {
        if (bindResources == null) {
            return;
        }
        List<Long> resourceIds = bindResources.getResourceIds();
        if (resourceIds == null) {
            return;
        }
        logger.info("resourceIds = " + resourceIds.stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.joining(",")));
        List<ResourceApiRel> existRels = resourceApiRelDao.findByServiceApiId(serviceApi.getServiceApiId());
        logger.info("exist api-resource-rels.size = " + existRels.size());
        //region 往绑定不在数据库中但是报文 resourcesetIds 中有的服务包-功能集关系
        Set<ResourceApiRel> insertingRels = resourceIds.stream().filter(resourceId -> existRels.stream().map(ResourceApiRel::getResourceId).noneMatch(existId -> existId.equals(resourceId))).map(resourceId -> {
            Optional<Resource> resourceOptional = resourceDao.findById(resourceId);
            if (resourceOptional.isPresent()) {
                ResourceApiRel rel = new ResourceApiRel();
                rel.setResourceId(resourceId);
                rel.setServiceApiId(serviceApi.getServiceApiId());
                return rel;
            } else {
                return null;
            }
        }).filter(Objects::nonNull).collect(Collectors.toSet());
        this.bindResources(insertingRels);
        //endregion
        if (bindResources.isOverwrite()) {
            //region 从数据库中解绑不在报文 resourcesetIds 中的服务包-功能集关系
            existRels.stream().filter(rel -> resourceIds.stream().noneMatch(resourceId -> resourceId.equals(rel.getResourceId()))).forEach(rel -> {
                logger.info("deleting Resource-Api-Rel record, {}", rel);
                resourceApiRelDao.deleteById(rel.getId());
            });
            //endregion
        }
    }

    /**
     * 根据serviceApiId查询数据
     *
     * @param serviceApiId serviceApiId
     * @return List<ResourceApiRel>
     */
    public List<ResourceApiRel> findResourceApiRelByServiceApiId(Long serviceApiId) {
        return this.resourceApiRelDao.findByServiceApiId(serviceApiId);
    }

    /**
     * 根据resoruceId查询数据
     *
     * @param resoruceId resoruceId
     * @return List<ResourceApiRel>
     */
    public List<ResourceApiRel> findResourceApiRelByResourceId(Long resoruceId) {
        return this.resourceApiRelDao.findByResourceId(resoruceId);
    }

    @Transactional(rollbackFor = Exception.class)
    public void bindResources(Collection<ResourceApiRel> rels) {
        if (!rels.isEmpty()) {
            rels.forEach(resourceApiRelDao::saveAndFlush);
        }
    }

    public List<ServiceApi> findByRouteId(long routeId) {
        Query query = new Query();
        query.setRouteId(routeId);
        Specification<ServiceApi> specification = ServiceApiQueryHelper.querySpecification(query);
        return serviceApiDao.findAll(specification);
    }

    public Map<RequestUri, RequestUriAuthz> getRequestUriAndResourceCodesMapByRouteId(long routeId) {
        List<ServiceApi> serviceApis = this.findByRouteId(routeId);
        Map<RequestUri, RequestUriAuthz> map = new HashMap<>();
        for (ServiceApi serviceApi : serviceApis) {
            RequestUri requestUri = new RequestUri(serviceApi.getServiceApiUrl(), serviceApi.getRequestMethod());
            List<Resource> resources = resourceService.listByServiceApiId(serviceApi.getServiceApiId());
            Set<String> resourceCodes = resources.stream().map(Resource::getResourceCode).filter(Objects::nonNull).collect(Collectors.toSet());
            RequestUriAuthz requestUriAuthz = new RequestUriAuthz(resourceCodes, serviceApi.getSkipAuthentication(), serviceApi.getSkipAuthorization());
            map.put(requestUri, requestUriAuthz);
        }
        return map;
    }

    public List<ServiceApi> findAllSkipAuthorizationServiceApi() {
        return serviceApiDao.findBySkipAuthorizationEqualsAndStatusEquals(true, 1);
    }

    public List<ServiceApi> findServiceApiUrlsByResourceCode(String resourceCode) {
        if (StringUtils.isBlank(resourceCode)) {
            return new ArrayList<>();
        }
        return serviceApiDao.findServiceApiUrlsByResourceCode(resourceCode);
    }

    @Transactional(rollbackFor = Exception.class)
    public void unbindServiceApiResourceRel(long serviceApiId, long resourceId) {
        resourceApiRelDao.deleteByServiceApiIdAndResourceId(serviceApiId, resourceId);
    }

    public Map<String, List<ServiceApiResourceCodeVo>> getExactServiceApiResourceMap() {
        //1 个api 只应该有一个resource code 匹配， 但是以前已经有这种数据了，这里要兼容一下
        List<ServiceApiResourceCodeVo> serviceApiLists = serviceApiExtendDao.getExactServiceApiResourceList();

        Map<String, List<ServiceApiResourceCodeVo>> map = new HashMap<>(serviceApiLists.size());
        for (ServiceApiResourceCodeVo item : serviceApiLists) {
            List<ServiceApiResourceCodeVo> listValue = map.get(item.getServiceApiUrl());
            if (listValue == null) {
                listValue = new ArrayList<>();
                map.put(item.getServiceApiUrl(), listValue);
            }
            listValue.add(item);
        }
        return map;
    }

    public List<ServiceApiResourceCodeVo> getPathVariableServiceApiResourceList() {
        return serviceApiExtendDao.getPathVariableServiceApiResourceList();
    }

}
