/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: ServicePackageExcelExportServiceImpl.java   2020-10-20 16-21-34
 * Author: Evan
 */
package com.xforceplus.business.resource.service;

import com.alibaba.excel.event.AnalysisEventListener;
import com.xforceplus.api.model.ServicePackageModel;
import com.xforceplus.api.model.ServicePackageModel.Request.BindResourceSets;
import com.xforceplus.api.model.ServicePackageModel.Request.Query;
import com.xforceplus.bo.ServicePackageQueryBo;
import com.xforceplus.business.app.service.AppService;
import com.xforceplus.business.excel.BusinessType;
import com.xforceplus.business.excel.ExcelSheet;
import com.xforceplus.business.excel.SimpleExcelWriter;
import com.xforceplus.business.excel.reader.Context;
import com.xforceplus.business.excel.reader.MessageRow;
import com.xforceplus.business.excel.reader.SimpleDataReadListener;
import com.xforceplus.business.excel.writer.ExcelConfigBusinessType;
import com.xforceplus.business.resource.dto.ServicePackageExcelDTO;
import com.xforceplus.business.resource.dto.ServicePackageResourceSetExcelDTO;
import com.xforceplus.business.service.ExcelReaderService;
import com.xforceplus.business.service.ExcelWriteService;
import com.xforceplus.dto.resource.ServicePackageDTO;
import com.xforceplus.dto.resource.ServicePackageResourceSetDTO;
import com.xforceplus.entity.App;
import com.xforceplus.entity.Resourceset;
import com.xforceplus.entity.ServicePackage;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.NumberUtils;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import static com.xforceplus.business.excel.ExcelFile.*;
import static com.xforceplus.business.resource.service.ResourceExcelConfig.ServicePackageConfig.*;
import static com.xforceplus.business.resource.service.ServicePackageExcelExportServiceImpl.Message.*;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.springframework.data.domain.Sort.Direction.DESC;

/**
 * <p>
 * Title: 产品服务包Excel导出服务
 * </p>
 * <p>
 * Description: 产品服务包Excel导出服务（产品服务，产品&功能集）
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-10-20 16-21-34
 * @since V1.0
 */
@Service
public class ServicePackageExcelExportServiceImpl implements ExcelWriteService, ExcelReaderService {
    /**
     * 日志
     */
    private static final Logger log = LoggerFactory.getLogger(ResourceExcelServiceImpl.class);

    /**
     *
     */
    private final ServicePackageService servicePackageService;

    /**
     *
     */
    private final ResourcesetService resourcesetService;

    /**
     * AppService
     */
    private final AppService appService;

    public ServicePackageExcelExportServiceImpl(ServicePackageService servicePackageService,
                                                ResourcesetService resourcesetService, AppService appService) {
        this.servicePackageService = servicePackageService;
        this.resourcesetService = resourcesetService;
        this.appService = appService;
    }

    /**
     * 获取导入类型，用于Event事件调整导入方法
     * @return ImportBusinessType
     */
    @Override
    public BusinessType getBusinessType() {
        return ExcelConfigBusinessType.SERVICE_PACKAGE_EXCEL_FILE;
    }

    /**
     * 导入定义
     *
     * @param context 上下文
     * @return Context
     */
    @Override
    public Context importExcel(Context context) {
        List<ExcelSheet> sheets = context.getExcelBook().getExcelSheets();
        for (ExcelSheet sheet : sheets) {
            String sheetName = sheet.getSheetName();
            //等于公司服务配置信息，则分页读取公司信息
            if (SERVICE_PACKAGE_SHEET.equals(sheetName)) {
                MessageRow messageRows = new MessageRow(sheetName);
                List<Long> appIds = this.findAllByApp();
                //构建监听器
                AnalysisEventListener<ServicePackageExcelDTO> excelDataListener = SimpleDataReadListener.listener(context, (rows) -> this.saveServicePackage(messageRows, context, rows, appIds));
                //开始处理
                context.getSimpleExcelReader().read(ServicePackageExcelDTO.class, excelDataListener, sheetName);
            } else if (RESOURCE_SET_SHEET.equals(sheetName)) {
                MessageRow messageRows = new MessageRow(sheetName);
                //构建监听器
                AnalysisEventListener<ServicePackageResourceSetExcelDTO> excelDataListener = SimpleDataReadListener.listener(context, (rows) -> this.saveServicePackageResourceSet(messageRows, context, rows));
                //开始处理
                context.getSimpleExcelReader().read(ServicePackageResourceSetExcelDTO.class, excelDataListener, sheetName);
            }
        }
        return context;
    }

    /**
     * 绑定和解绑功能集
     *
     * @param messageRows MessageRow
     * @param context     Context
     * @param rows        List<ServicePackageResourceSetExcelDTO>
     */
    private void saveServicePackageResourceSet(MessageRow messageRows, Context context, List<ServicePackageResourceSetExcelDTO> rows) {
        //为空则不处理
        if (CollectionUtils.isEmpty(rows)) {
            return;
        }
        for (ServicePackageResourceSetExcelDTO row : rows) {
            try {
                if (!row.getValidatedStatus()) {
                    messageRows.fail(row.getRowIndex(), row.getValidatedMessage());
                    continue;
                }
                if (SERVICE_PACKAGE_RESOURCE_SET_ACTION_BIND.equals(row.getAction())) {
                    //绑定
                    this.bindResourceSets(messageRows, row);
                    continue;
                } else if (SERVICE_PACKAGE_RESOURCE_SET_ACTION_UN_BIND.equals(row.getAction())) {
                    //解绑
                    this.unBindResourceSets(messageRows, row);
                    continue;
                }
            } catch (Exception ex) {
                log.error(ex.getMessage(), ex);
                messageRows.fail(row.getRowIndex(), row.getAction() + "数据失败");
            }
        }
        context.messageRow(messageRows.getSheetName(), messageRows);
    }

    /**
     * 解绑功能集
     *
     * @param messageRows MessageRow
     * @param row         ServicePackageResourceSetExcelDTO
     */
    private void unBindResourceSets(MessageRow messageRows, ServicePackageResourceSetExcelDTO row) {
        String servicePackageCode = row.getServicePackageCode();
        Optional<ServicePackage> servicePackageOptional;
        try {
            servicePackageOptional = this.findByServicePackageCode(servicePackageCode);
            if (!servicePackageOptional.isPresent()) {
                messageRows.fail(row.getRowIndex(), "【" + servicePackageCode + "】" + SERVICE_PACKAGE_NOT_FIND);
                return;
            }
        } catch (IllegalArgumentException e) {
            messageRows.fail(row.getRowIndex(), e.getMessage());
            return;
        }
        String resourceSetCode = row.getResourceSetCode();
        Optional<Resourceset> resourceSetOptional;
        try {
            resourceSetOptional = this.findByResourceSetCode(resourceSetCode);
            if (!resourceSetOptional.isPresent()) {
                messageRows.fail(row.getRowIndex(), "【" + resourceSetCode + "】" + RESOURCE_SET_NOT_FIND);
                return;
            }
        } catch (IllegalArgumentException e) {
            messageRows.fail(row.getRowIndex(), e.getMessage());
            return;
        }
        if (!resourceSetOptional.isPresent()) {
            messageRows.fail(row.getRowIndex(), "【" + resourceSetCode + "】" + RESOURCE_SET_NOT_FIND);
            return;
        }
        try {
            Long servicePackageId = servicePackageOptional.get().getServicePackageId();
            Long resourceSetId = resourceSetOptional.get().getResourcesetId();
            this.servicePackageService.unBindResourceSets(servicePackageId, resourceSetId);
            messageRows.success(row.getRowIndex());
        } catch (Exception e) {
            messageRows.fail(row.getRowIndex(), "解绑:" + e.getMessage());
        }
    }

    /**
     * 绑定功能集
     *
     * @param messageRows 功能集
     * @param row         ServicePackageResourceSetExcelDTO
     */
    private void bindResourceSets(MessageRow messageRows, ServicePackageResourceSetExcelDTO row) {
        String servicePackageCode = row.getServicePackageCode();
        Optional<ServicePackage> servicePackageOptional;
        try {
            servicePackageOptional = this.findByServicePackageCode(servicePackageCode);
            if (!servicePackageOptional.isPresent()) {
                messageRows.fail(row.getRowIndex(), "【" + servicePackageCode + "】" + SERVICE_PACKAGE_NOT_FIND);
                return;
            }
        } catch (IllegalArgumentException e) {
            messageRows.fail(row.getRowIndex(), e.getMessage());
            return;
        }
        String resourceSetCode = row.getResourceSetCode();
        Optional<Resourceset> resourceSetOptional;
        try {
            resourceSetOptional = this.findByResourceSetCode(resourceSetCode);
            if (!resourceSetOptional.isPresent()) {
                messageRows.fail(row.getRowIndex(), "【" + resourceSetCode + "】" + RESOURCE_SET_NOT_FIND);
                return;
            }
        } catch (IllegalArgumentException e) {
            messageRows.fail(row.getRowIndex(), e.getMessage());
            return;
        }
        BindResourceSets bindResourceSets = new BindResourceSets();
        bindResourceSets.setOverwrite(false);
        bindResourceSets.setResourcesetIds(Arrays.asList(resourceSetOptional.get().getResourcesetId()));
        try {
            this.servicePackageService.bindResourceSets(servicePackageOptional.get(), bindResourceSets);
            messageRows.success(row.getRowIndex());
        } catch (Exception e) {
            messageRows.fail(row.getRowIndex(), "绑定:" + e.getMessage());
        }
    }

    /**
     * 按功能集Code查询数据
     *
     * @param resourceSetCode
     * @return Optional<Resourceset>
     */
    private Optional<Resourceset> findByResourceSetCode(String resourceSetCode) {
        return this.resourcesetService.findByResourceSetCode(resourceSetCode);
    }

    /**
     * 根据产品服务包Code查询
     *
     * @param servicePackageCode
     * @return Optional<ServicePackage>
     */
    private Optional<ServicePackage> findByServicePackageCode(String servicePackageCode) {
        return this.servicePackageService.findByServicePackageCode(servicePackageCode);
    }

    /**
     * 保存servicePackage
     *
     * @param messageRows
     * @param context
     * @param rows
     */
    private void saveServicePackage(MessageRow messageRows, Context context, List<ServicePackageExcelDTO> rows, List<Long> appIds) {
        //为空则不处理
        if (CollectionUtils.isEmpty(rows)) {
            return;
        }
        for (ServicePackageExcelDTO row : rows) {
            try {
                if (!row.getValidatedStatus()) {
                    messageRows.fail(row.getRowIndex(), row.getValidatedMessage());
                    continue;
                }
                if (SERVICE_PACKAGE_ACTION_ADD.equals(row.getAction())) {
                    //保存
                    this.saveServicePackage(messageRows, row, appIds);
                    continue;
                } else if (SERVICE_PACKAGE_ACTION_UPDATE.equals(row.getAction())) {
                    //更新
                    this.updateServicePackage(messageRows, row, appIds);
                    continue;
                }
            } catch (Exception ex) {
                log.error(ex.getMessage(), ex);
                messageRows.fail(row.getRowIndex(), row.getAction() + "数据失败");
            }
        }
        context.messageRow(messageRows.getSheetName(), messageRows);
    }

    /**
     * 保存资源服务包
     *
     * @param messageRows
     * @param row
     */
    private void saveServicePackage(MessageRow messageRows, ServicePackageExcelDTO row, List<Long> appIds) {
        Long appId = NumberUtils.parseNumber(row.getAppId(), Long.class);
        if (!appIds.contains(appId)) {
            messageRows.fail(row.getRowIndex(), "【" + appId + "】" + APP_NOT_FIND);
            return;
        }
        String servicePackageCode = row.getServicePackageCode();
        Optional<ServicePackage> servicePackageOptional;
        try {
            servicePackageOptional = this.findByServicePackageCode(servicePackageCode);
            if (servicePackageOptional.isPresent()) {
                messageRows.fail(row.getRowIndex(), "【" + servicePackageCode + "】产品服包数据已存在");
                return;
            }
        } catch (IllegalArgumentException e) {
            messageRows.fail(row.getRowIndex(), e.getMessage());
            return;
        }

        Integer status = NumberUtils.parseNumber(row.getStatus(), Integer.class);
        ServicePackageModel.Request.Save save = new ServicePackageModel.Request.Save();
        save.setAppId(appId);
        save.setServicePackageName(row.getServicePackageName());
        save.setServicePackageCode(row.getServicePackageCode());
        save.setServicePackageDesc(row.getServicePackageName());
        save.setStatus(status);
        try {
            servicePackageService.create(save);
            messageRows.success(row.getRowIndex());
        } catch (Exception e) {
            messageRows.fail(row.getRowIndex(), e.getMessage());
        }

    }

    /**
     * 查询所有APP
     *
     * @return Optional<App>
     */
    private List<Long> findAllByApp() {
        List<App> appList = this.appService.list();
        if (CollectionUtils.isEmpty(appList)) {
            return Collections.emptyList();
        }
        return appList.stream().map(App::getAppId).collect(Collectors.toList());
    }

    /**
     * 更新产品服务包
     *
     * @param messageRows
     * @param row
     * @param appIds
     */
    private void updateServicePackage(MessageRow messageRows, ServicePackageExcelDTO row, List<Long> appIds) {
        Long appId = NumberUtils.parseNumber(row.getAppId(), Long.class);
        if (!appIds.contains(appId)) {
            messageRows.fail(row.getRowIndex(), "【" + appId + "】" + APP_NOT_FIND);
            return;
        }
        String servicePackageCode = row.getServicePackageCode();
        Optional<ServicePackage> servicePackageOptional;
        try {
            servicePackageOptional = this.findByServicePackageCode(servicePackageCode);
            if (!servicePackageOptional.isPresent()) {
                messageRows.fail(row.getRowIndex(), "【" + servicePackageCode + "】" + SERVICE_PACKAGE_NOT_FIND);
                return;
            }
        } catch (IllegalArgumentException e) {
            messageRows.fail(row.getRowIndex(), e.getMessage());
            return;
        }
        Integer status = NumberUtils.parseNumber(row.getStatus(), Integer.class);
        ServicePackageModel.Request.Save save = new ServicePackageModel.Request.Save();
        save.setAppId(appId);
        save.setServicePackageName(row.getServicePackageName());
        save.setServicePackageCode(row.getServicePackageCode());
        save.setServicePackageDesc(row.getServicePackageName());
        save.setStatus(status);
        try {
            servicePackageService.update(servicePackageOptional.get().getServicePackageId(), save);
            messageRows.success(row.getRowIndex());
        } catch (Exception e) {
            messageRows.fail(row.getRowIndex(), e.getMessage());
        }
    }

    /**
     * 导出公共方法
     *
     * @param context 上下文
     * @return Context
     */
    @Override
    public void write(Context context) {
        Query query = context.getParam(PARAMS_QUERY, Query.class);
        Sort sort = context.getParam(PARAMS_SORT, Sort.class);
        List<String> sheetNames = Arrays.asList(query.getSheets());
        SimpleExcelWriter simpleExcelWriter = context.getSimpleExcelWriter();
        Integer maxPageSize = context.getMaxPageSize();
        ServicePackageQueryBo queryBo = new ServicePackageQueryBo();
        BeanUtils.copyProperties(query, queryBo);
        for (String sheetName : sheetNames) {
            //功能集
            if (SERVICE_PACKAGE_SHEET.equals(sheetName)) {
                Page<ServicePackageDTO> page;
                Pageable pageable = PageRequest.of(START_PAGE, PAGE_SIZE, sort.and(Sort.by(DESC, SORT_FIELD)));
                do {
                    page = this.servicePackageService.pagingBy(pageable, queryBo);
                    //对象转换
                    Page<ServicePackageExcelDTO> packageExcelPage = page.map(e -> {
                        ServicePackageExcelDTO dto = new ServicePackageExcelDTO();
                        dto.setAppId(e.getAppId() == null ? EMPTY : String.valueOf(e.getAppId()));
                        dto.setServicePackageCode(StringUtils.trimToEmpty(e.getServicePackageCode()));
                        dto.setServicePackageName(StringUtils.trimToEmpty(e.getServicePackageName()));
                        dto.setStatus(e.getStatus() == null ? EMPTY : String.valueOf(e.getStatus()));
                        return dto;
                    });
                    simpleExcelWriter.fill(sheetName, packageExcelPage.getContent());
                } while (page.hasNext() && (pageable = pageable.next()) != null && page.getNumber() < maxPageSize);
                //计算当前导出数据  this.calcSuccessSize(page,)
                simpleExcelWriter.param(SUCCESS_SIZE, this.calcSuccessSize(page));
            }
            //功能集与资源码导入
            if (RESOURCE_SET_SHEET.equals(sheetName)) {
                Page<ServicePackageResourceSetDTO> page;
                Pageable pageable = PageRequest.of(START_PAGE, PAGE_SIZE, sort.and(Sort.by(DESC, SORT_FIELD)));
                do {
                    page = this.servicePackageService.pagingByResourceSet(pageable, queryBo);
                    //对象转换
                    Page<ServicePackageResourceSetExcelDTO> excelPage = page.map(e -> {
                        ServicePackageResourceSetExcelDTO dto = new ServicePackageResourceSetExcelDTO();
                        dto.setResourceSetCode(StringUtils.trimToEmpty(e.getResourceSetCode()));
                        dto.setServicePackageCode(StringUtils.trimToEmpty(e.getServicePackageCode()));
                        return dto;
                    });
                    simpleExcelWriter.fill(sheetName, excelPage.getContent());
                } while (page.hasNext() && (pageable = pageable.next()) != null && page.getNumber() < maxPageSize);
                //计算当前导出数据  this.calcSuccessSize(page,)
                simpleExcelWriter.param(SUCCESS_SIZE, this.calcSuccessSize(page));
            }
        }
    }

    protected static final class Message {
        /**
         * 产品服务:{@value}
         */
        public static final String SERVICE_PACKAGE_NOT_FIND = "产品服包数据不存在";
        /**
         * 功能集数据不存在提示信息 :{@value}
         */
        public static final String RESOURCE_SET_NOT_FIND = "功能集数据不存在";
        /**
         * APP_NOT_FIND:{@value}
         */
        public static final String APP_NOT_FIND = "APP的数据不存在";
    }
}
