package com.xforceplus.business.resource.service;

import com.google.common.base.Joiner;
import com.xforceplus.api.model.ServicePackageModel.Request.BindResourceSets;
import com.xforceplus.api.model.ServicePackageModel.Request.Query;
import com.xforceplus.api.model.ServicePackageModel.Request.Save;
import com.xforceplus.api.utils.Separator;
import com.xforceplus.bo.ServicePackageQueryBo;
import com.xforceplus.dao.*;
import com.xforceplus.domain.TreeNode;
import com.xforceplus.domain.resource.ResourceType;
import com.xforceplus.dto.resource.ServicePackageDTO;
import com.xforceplus.dto.resource.ServicePackageResourceSetDTO;
import com.xforceplus.entity.*;
import com.xforceplus.query.ServicePackageQueryHelper;
import io.geewit.core.utils.reflection.BeanUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import java.math.BigInteger;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

import static com.xforceplus.domain.resource.ResourceType.SERVICE_PACKAGE_PARENT_KEY;

@Service
public class ServicePackageService {
    private final static Logger logger = LoggerFactory.getLogger(ServicePackageService.class);

    private final ServiceResourcesetRelDao serviceResourcesetRelDao;

    private final ServicePackageDao servicePackageDao;

    private final ResourcesetDao resourcesetDao;

    private final CompanyServiceRelDao companyServiceRelDao;

    private final ResourcesetService resourcesetService;

    /**
     * 服务查询扩展Dao
     */
    private final ServicePackageExtendDao servicePackageExtendDao;

    private final RoleResourcesetRelDao roleResourcesetRelDao;

    private final AppDao appDao;


    public ServicePackageService(ServicePackageDao servicePackageDao, ServiceResourcesetRelDao serviceResourcesetRelDao, ResourcesetDao resourcesetDao, CompanyServiceRelDao companyServiceRelDao, ResourcesetService resourcesetService, ServicePackageExtendDao servicePackageExtendDao, RoleResourcesetRelDao roleResourcesetRelDao, AppDao appDao) {
        this.servicePackageDao = servicePackageDao;
        this.serviceResourcesetRelDao = serviceResourcesetRelDao;
        this.resourcesetDao = resourcesetDao;
        this.companyServiceRelDao = companyServiceRelDao;
        this.resourcesetService = resourcesetService;
        this.servicePackageExtendDao = servicePackageExtendDao;
        this.roleResourcesetRelDao = roleResourcesetRelDao;
        this.appDao = appDao;
    }

    public Page<ServicePackage> page(Query query, Pageable pageable) {
        Specification<ServicePackage> specification = ServicePackageQueryHelper.querySpecification(query);
        Page<ServicePackage> page = servicePackageDao.findAll(specification, pageable);
        for (ServicePackage servicePackage : page) {
            List<Resourceset> resourcesets = resourcesetService.listByServicePackageId(servicePackage.getServicePackageId(), query.getStatus());
            servicePackage.setResourcesets(new HashSet<>(resourcesets));
        }
        this.fillServicePackageExtensions(query, page);
        return page;
    }

    public List<ServicePackage> list(Specification<ServicePackage> specification, Sort sort) {
        return servicePackageDao.findAll(specification, sort);
    }

    public List<ServicePackage> list(Query query, Sort sort) {
        Specification<ServicePackage> specification = ServicePackageQueryHelper.querySpecification(query);
        List<ServicePackage> list = servicePackageDao.findAll(specification, sort);
        this.fillServicePackageExtensions(query, list);
        return list;
    }

    private void fillServicePackageExtensions(Query query, Iterable<ServicePackage> servicePackages) {
        if (servicePackages == null || servicePackages.iterator() == null || !servicePackages.iterator().hasNext()) {
            return;
        }
        boolean needAppName = false;
        boolean needResourcesets = false;
        if (StringUtils.isNotBlank(query.getWithExtendParams())) {
            String[] withExtendParams = StringUtils.split(query.getWithExtendParams(), ",");
            for (String withExtendParam : withExtendParams) {
                if ("appName".equalsIgnoreCase(withExtendParam)) {
                    needAppName = true;
                    break;
                } else if ("resourcesets".equalsIgnoreCase(withExtendParam)) {
                    needResourcesets = true;
                    break;
                }
            }
        }
        if (needAppName || needResourcesets) {
            Map<Long, String> appNameMap;
            if (needAppName) {
                Set<Long> appIds = new HashSet<>();
                for (ServicePackage servicePackage : servicePackages) {
                    appIds.add(servicePackage.getAppId());
                }
                List<App> apps = appDao.findAppsByAppIds(appIds);
                appNameMap = apps.stream().collect(Collectors.toMap(App::getAppId, App::getAppName));
            } else {
                appNameMap = Collections.emptyMap();
            }

            if (needResourcesets) {
                Set<Long> packageIds = StreamSupport.stream(servicePackages.spliterator(), false)
                        .map(ServicePackage::getServicePackageId)
                        .collect(Collectors.toSet());

            }

            for (ServicePackage servicePackage : servicePackages) {
                if (appNameMap != null && !appNameMap.isEmpty()) {
                    servicePackage.setAppName(appNameMap.get(servicePackage.getAppId()));
                }
                if (needResourcesets) {
                    List<Resourceset> resourcesets = resourcesetService.listByServicePackageId(servicePackage.getServicePackageId(), 1);
                    if (resourcesets != null && !resourcesets.isEmpty()) {
                        servicePackage.setResourcesets(new HashSet<>(resourcesets));
                    }
                }

            }
        }
    }

    public Optional<ServicePackage> findOne(Query query) {
        Specification<ServicePackage> specification = ServicePackageQueryHelper.queryOneSpecification(query);
        return servicePackageDao.findOne(specification);
    }

    public long count(Query query) {
        Specification<ServicePackage> specification = ServicePackageQueryHelper.queryOneSpecification(query);
        return servicePackageDao.count(specification);
    }

    @Transactional(rollbackFor = Exception.class)
    public ServicePackage create(Save model) {
        ServicePackage entity = new ServicePackage();
        BeanUtils.copyProperties(model, entity);
        entity = servicePackageDao.saveAndFlush(entity);
        if (model.getBindResourceSets() != null) {
            this.bindResourceSets(entity, model.getBindResourceSets());
        }

        return entity;
    }

    @Transactional(rollbackFor = Exception.class)
    public ServicePackage update(long servicePackageId, Save model) {
        ServicePackage existEntity = this.findById(servicePackageId);
        if (StringUtils.isBlank(model.getServicePackageCode())) {
            model.setServicePackageCode(null);
        }
        if (StringUtils.isBlank(model.getServicePackageName())) {
            model.setServicePackageName(null);
        }
        BeanUtils.copyProperties(model, existEntity);
        existEntity = servicePackageDao.saveAndFlush(existEntity);
        if (model.getBindResourceSets() != null) {
            this.bindResourceSets(existEntity, model.getBindResourceSets());
        }


        return existEntity;
    }


    /**
     * 产品服务包与功能集解邦
     *
     * @param servicePackageId
     * @param resourceSetId
     */
    @Transactional(rollbackFor = Exception.class)
    public void unBindResourceSets(Long servicePackageId, Long resourceSetId) {
        final List<ServiceResourcesetRel> rels = serviceResourcesetRelDao.findByServicePackageIdAndResourcesetId(servicePackageId, resourceSetId);
        if (CollectionUtils.isEmpty(rels)) {
            throw new IllegalArgumentException("服务包与功能未绑定");
        }
        try {
            this.serviceResourcesetRelDao.delete(rels.get(0));
        } catch (RuntimeException e) {
            throw new IllegalArgumentException("解绑数据失败");
        }
    }

    /**
     * 服务包绑定功能集列表(* 此方法会解绑 resourcesetIds 中不存在但是数据库中有的服务包-功能集关系)
     *
     * @param servicePackage
     * @param bindResourceSets
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public void bindResourceSets(ServicePackage servicePackage, BindResourceSets bindResourceSets) {
        if (bindResourceSets == null) {
            return;
        }
        List<Long> bindResourcesetIds = bindResourceSets.getResourcesetIds();
        if (bindResourcesetIds == null) {
            return;
        }
        logger.info("bindResourcesetIds = " + bindResourcesetIds.stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.joining(",")));
        final List<ServiceResourcesetRel> rels = serviceResourcesetRelDao.findByServicePackageId(servicePackage.getServicePackageId());
        logger.info("exist sericepackage-resourceset-rels.size = " + rels.size());
        //region 往绑定不在数据库中但是报文 resourcesetIds 中有的服务包-功能集关系
        List<ServiceResourcesetRel> insertingRels = bindResourcesetIds.stream().filter(Objects::nonNull).filter(resourcesetId -> rels.stream().map(ServiceResourcesetRel::getResourcesetId).noneMatch(resourcesetDtoId -> resourcesetDtoId.equals(resourcesetId))).map(resourcesetId -> {
            Optional<Resourceset> resourceOptional = resourcesetDao.findById(resourcesetId);
            if (resourceOptional.isPresent()) {
                ServiceResourcesetRel rel = new ServiceResourcesetRel();
                rel.setResourcesetId(resourcesetId);
                rel.setServicePackageId(servicePackage.getServicePackageId());
                return rel;
            } else {
                return null;
            }
        }).filter(Objects::nonNull).collect(Collectors.toList());
        this.bindResourceSets(insertingRels);
        //endregion
        if (bindResourceSets.isOverwrite()) {
            //region 从数据库中解绑不在报文 resourcesetIds 中的服务包-功能集关系
            rels.stream().filter(rel -> !bindResourcesetIds.contains(rel.getResourcesetId())).forEach(rel -> {
                logger.info("deleting Service-Resourceset-Rel record, {}", rel);
                try {
                    serviceResourcesetRelDao.deleteById(rel.getId());
                } catch (Exception e) {
                    logger.warn(e.getMessage(), e);
                }
            });
            //endregion
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void bindResourceSets(Collection<ServiceResourcesetRel> rels) {
        if (!CollectionUtils.isEmpty(rels)) {
            rels.forEach(serviceResourcesetRelDao::saveAndFlush);
        }
    }

    public ServicePackage findById(long servicePackageId) {
        ServicePackage servicePackage = servicePackageDao.findById(servicePackageId).orElseThrow(() -> new IllegalArgumentException("未找到实体"));
        if (servicePackage != null) {
            List<Resourceset> resourcesets = resourcesetService.listByServicePackageId(servicePackageId, null);
            servicePackage.setResourcesets(resourcesets.stream().filter(Objects::nonNull).collect(Collectors.toSet()));
        }

        return servicePackage;
    }


    @Transactional(rollbackFor = Exception.class)
    public void deleteById(long servicePackageId) {
        Optional<ServicePackage> optionalServicePackage = servicePackageDao.findOne((Specification<ServicePackage>) (root, query1, builder) -> builder.equal(root.<Long>get("servicePackageId"), servicePackageId));
        if (!optionalServicePackage.isPresent()) {
            throw new IllegalArgumentException("错误的服务包id");
        }
        ServicePackage servicePackage = optionalServicePackage.get();
        if (servicePackage.getStatus() != null && servicePackage.getStatus() == 1) {
            throw new IllegalArgumentException("禁用的服务包才可以被删除");
        }
        long count = companyServiceRelDao.countByServicePackageId(servicePackageId);
        if (count > 0) {
            throw new IllegalArgumentException("该服务包已经绑定过公司服务包不能删除");
        }

        try {
            servicePackageDao.deleteById(servicePackageId);
            serviceResourcesetRelDao.deleteByPackageId(servicePackageId);
            companyServiceRelDao.deleteByPackageId(servicePackageId);

        } catch (Exception e) {
            logger.warn(e.getMessage(), e);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateStatus(long servicePackageId, int status) {
        Optional<ServicePackage> optionalServicePackage = servicePackageDao.findById(servicePackageId);
        if (!optionalServicePackage.isPresent()) {
            throw new IllegalArgumentException("错误的服务包id");
        }
        ServicePackage servicePackage = optionalServicePackage.get();
        servicePackage.setStatus(status);
        servicePackageDao.saveAndFlush(servicePackage);

    }

    public boolean saveResourcesetRel(Long servicePackageId, Long resourcesetId, String operateUserName) {
        List<ServiceResourcesetRel> list = serviceResourcesetRelDao.findAll(new Specification<ServiceResourcesetRel>() {
            @Override
            public Predicate toPredicate(Root<ServiceResourcesetRel> root, CriteriaQuery<?> query, CriteriaBuilder builder) {
                List<Predicate> predicates = new ArrayList<>();
                predicates.add(builder.equal(root.<Long>get("servicePackageId"), servicePackageId));
                predicates.add(builder.equal(root.<Long>get("resourcesetId"), resourcesetId));
                if (!predicates.isEmpty()) {
                    query.where(predicates.stream().toArray(Predicate[]::new));
                }
                return query.getRestriction();
            }
        });
        if (list.isEmpty()) {
            ServiceResourcesetRel serviceResourcesetRel = new ServiceResourcesetRel();
            serviceResourcesetRel.setServicePackageId(servicePackageId);
            serviceResourcesetRel.setResourcesetId(resourcesetId);
            serviceResourcesetRel.setCreateUserName(operateUserName);
            serviceResourcesetRelDao.saveAndFlush(serviceResourcesetRel);
            return false;
        } else {
            return true;
        }
    }

    public Page<ServicePackage> page(long tenantId, Query query, Pageable pageable) {
        query.setTenantId(tenantId);
        Specification<ServicePackage> specification = ServicePackageQueryHelper.querySpecification(query);
        Page<ServicePackage> page = servicePackageDao.findAll(specification, pageable);
        for (ServicePackage servicePackage : page) {
            List<Resourceset> resourcesets = this.resourcesetService.listByServicePackageId(servicePackage.getServicePackageId(), query.getStatus());
            servicePackage.setResourcesets(new HashSet<>(resourcesets));
        }
        this.fillServicePackageExtensions(query, page);
        return page;
    }

    public List<ServicePackage> listByTenantIdAndCompanyId(long tenantId, long companyId) {
        Query query = new Query();
        query.setTenantId(tenantId);
        query.setCompanyId(companyId);
        query.setStatus(1);
        Specification<ServicePackage> specification = ServicePackageQueryHelper.querySpecification(query);
        List<ServicePackage> packages = servicePackageDao.findAll(specification);

        return packages;
    }

    /**
     * 服务包以功能集分页查询
     *
     * @param pageable
     * @param queryBo
     * @return
     */
    
    public Page<ServicePackageResourceSetDTO> pagingByResourceSet(Pageable pageable, ServicePackageQueryBo queryBo) {
        return servicePackageExtendDao.pagingByResourceSet(pageable, queryBo);
    }

    /**
     * 服务包分页查询
     *
     * @param pageable
     * @param queryBo
     * @return
     */
    
    public Page<ServicePackageDTO> pagingBy(Pageable pageable, ServicePackageQueryBo queryBo) {
        return servicePackageExtendDao.pagingBy(pageable, queryBo);
    }

    
    public List<TreeNode> tree(Long tenantId, Query query) {
        List<TreeNode> packageNodes = servicePackageDao.treePackageNodeList(tenantId, query);
        //如果产品服务为则直接返回空数据
        if (CollectionUtils.isEmpty(packageNodes)) {
            return Collections.emptyList();
        }
        //产品服务ID
        List<Long> packageIds = packageNodes.stream().map(node -> node.getKey()).collect(Collectors.toList());
        List<TreeNode> resourcesetNodes = servicePackageDao.treeResourcesetNodeList(packageIds);
        //如果数据为空
        if (CollectionUtils.isEmpty(resourcesetNodes)) {
            return Collections.emptyList();
        }
        Set<Long> resourcesetIds = resourcesetNodes.stream().map(pack -> pack.getKey()).collect(Collectors.toSet());
        List<TreeNode> resourceNodes = servicePackageDao.treeResourceNodeList(resourcesetIds);
        if (CollectionUtils.isEmpty(resourceNodes)) {
            return Collections.emptyList();
        }

        int treeNodeSize = packageNodes.size() + resourcesetNodes.size() + resourceNodes.size();
        List<TreeNode> treeNodes = new ArrayList<>(treeNodeSize);
        treeNodes.addAll(packageNodes);
        treeNodes.addAll(resourcesetNodes);
        treeNodes.addAll(resourceNodes);
        treeNodes = this.markResoruceTreeRecursiveCall(treeNodes, SERVICE_PACKAGE_PARENT_KEY, StringUtils.EMPTY);
        return treeNodes;


    }

    /**
     * 生成树结构
     *
     * @param treeNodes  上级路径
     * @param parentKey  上级KEY
     * @param parentPath 上级路径
     * @return List<TreeNode>
     */
    private List<TreeNode> markResoruceTreeRecursiveCall(List<TreeNode> treeNodes, String parentKey, String parentPath) {
        if (CollectionUtils.isEmpty(treeNodes)) {
            return Collections.emptyList();
        }
        List<TreeNode> nodes = new ArrayList<>();
        for (TreeNode treeNode : treeNodes) {
            //等于null呀pid==此时循环的pid
            if (parentKey.equals(treeNode.getParentKey())) {
                if (ResourceType.RESOURCE.equals(treeNode.getType())) {
                    String targetPath = this.setResourcePath(parentPath, treeNode.getId());
                    treeNode.setPath(targetPath);
                    nodes.add(treeNode);
                    continue;
                }
                //路径ID
                String targetPath = this.setResourcePath(parentPath, treeNode.getId());
                //路径ID
                List<TreeNode> children = this.markResoruceTreeRecursiveCall(treeNodes, treeNode.getId(), targetPath);
                treeNode.setChildren(children);
                treeNode.setPath(targetPath);
                //服务包且子节点为空的情况，不显示
                if (ResourceType.SERVICE_PACKAGE.equals(treeNode.getType()) && CollectionUtils.isEmpty(children)) {
                    continue;
                }
                nodes.add(treeNode);
            }
        }
        //以ID升序
        nodes.sort(Comparator.comparing(TreeNode::getId));
        return nodes;
    }

    ;

    /**
     * 构建以服务包、功能集、资源码的树型结构
     *
     * @param packageNodes       服务包
     * @param resourceSetNodeMap 功能集(按)
     * @param resourceMap        资源码
     * @return List<TreeNode>
     */
    private List<TreeNode> markResoruceTree(List<TreeNode> packageNodes, Map<String, Set<TreeNode>> resourceSetNodeMap, Map<String, Set<TreeNode>> resourceMap) {
        if (CollectionUtils.isEmpty(packageNodes)) {
            return Collections.EMPTY_LIST;
        }
        List<TreeNode> treeNodes = new ArrayList<>(packageNodes.size());
        for (TreeNode packageNode : packageNodes) {
            //过滤服务包下级功能集为空的数据
            if (!resourceSetNodeMap.containsKey(packageNode.getId())) {
                continue;
            }
            String targetPath = this.setResourcePath(StringUtils.EMPTY, packageNode.getId());
            packageNode.setPath(targetPath);

            Set<TreeNode> resourceSetNodesSource = resourceSetNodeMap.get(packageNode.getId());
            List<TreeNode> resourceSetNodes = new ArrayList<>(resourceSetNodesSource.size());
            for (TreeNode resourceSetNode : resourceSetNodesSource) {
                String resourceSetPath = this.setResourcePath(targetPath, resourceSetNode.getId());
                Set<TreeNode> resoureNodes = resourceMap.get(resourceSetNode.getId());
                //设置资源码节点
                List<TreeNode> resourceNodeList = this.setResources(resoureNodes, resourceSetPath);
                resourceSetNode.setChildren(resourceNodeList);
                resourceSetNode.setPath(resourceSetPath);
                resourceSetNodes.add(resourceSetNode);
            }
            resourceSetNodes.sort(Comparator.comparing(TreeNode::getId));
            packageNode.setChildren(resourceSetNodes);
            treeNodes.add(packageNode);
        }
        return treeNodes;
    }

    /**
     * @param treeNodes       资源
     * @param resourceSetPath 功能集的路径
     * @return Set<TreeNode>
     */
    private List<TreeNode> setResources(Set<TreeNode> treeNodes, String resourceSetPath) {
        if (CollectionUtils.isEmpty(treeNodes)) {
            return Collections.EMPTY_LIST;
        }
        List<TreeNode> list = treeNodes.stream().map(
                e -> {
                    this.setResourcePath(resourceSetPath, e.getId());
                    return e;
                }
        ).sorted(Comparator.comparing(TreeNode::getId)).collect(Collectors.toList());
        return list;
    }


    /**
     * 设置资源的Path
     *
     * @param path
     * @param id
     * @return String
     */
    private String setResourcePath(String path, String id) {
        return Joiner.on(Separator.DOUBLE_VERTICAL_BAR).skipNulls().join(path, id);
    }


    /**
     * 按资源
     *
     * @param servicePackageCode
     * @return
     */
    public Optional<ServicePackage> findByServicePackageCode(String servicePackageCode) {
        List<ServicePackage> servicePackageList = this.servicePackageDao.findByServicePackageCode(servicePackageCode);
        if (CollectionUtils.isEmpty(servicePackageList)) {
            return Optional.empty();
        }
        //如果
        if (servicePackageList.size() > 1) {
            throw new IllegalArgumentException("【" + servicePackageCode + "】产品服务包数据存在不唯一记录");
        }
        return Optional.of(servicePackageList.get(0));
    }

    public Map<Long, List<ServicePackage>> listByTenantId(Long tenantId) {
        if (tenantId == null && tenantId <= 0) {
            throw new IllegalArgumentException("租户id不能为空或不大于0");
        }
        List<CompanyServiceRel> rels = companyServiceRelDao.findRelsByTenantId(tenantId);
        Map<Long, List<ServicePackage>> companyPackagesMap = new HashMap<>();
        for (CompanyServiceRel rel : rels) {
            List<ServicePackage> companyPackages = companyPackagesMap.getOrDefault(rel.getCompanyId(), new ArrayList<>());
            companyPackages.add(rel.getServicePackage());
            companyPackagesMap.put(rel.getCompanyId(), companyPackages);
        }
        return companyPackagesMap;
    }

    public Map<Long, Boolean> validResourcesetsBound(long packageId, Set<Long> resourcesetIdSet) {
        List<Long> existRresourcesetIds = serviceResourcesetRelDao.findResourcesetIdByServicePackageId(packageId);
        Set<Long> deletingResourcesetIds = resourcesetIdSet.stream().filter(id -> existRresourcesetIds.stream().noneMatch(existId -> id.equals(existId))).collect(Collectors.toSet());
        List<Map<String, Object>> pairs = roleResourcesetRelDao.countResourcesetsWereBoundByRole(packageId, deletingResourcesetIds);
        Map<Long, Boolean> validation = pairs.stream().collect(Collectors.toMap(pair -> ((BigInteger) pair.get("resourcesetId")).longValue(), pair -> ((BigInteger) pair.get("cnt")).longValue() > 0));

        return validation;
    }
}
