package com.xforceplus.business.settle.service;

import com.xforceplus.api.model.OrgModel;
import com.xforceplus.api.model.SettleFlowModel.Request.Query;
import com.xforceplus.api.model.SettleFlowPackageModel;
import com.xforceplus.business.company.service.CompanyService;
import com.xforceplus.business.tenant.service.OrgService;
import com.xforceplus.dao.*;
import com.xforceplus.entity.*;
import com.xforceplus.query.SettleFlowPackageQueryHelper;
import com.xforceplus.query.SettleFlowQueryHelper;
import com.xforceplus.tenant.security.core.domain.OrgType;
import io.geewit.data.jpa.essential.domain.EntityGraphs;
import org.apache.commons.collections4.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import javax.persistence.criteria.Predicate;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

@Service
public class SettleFlowService {
    private final static Logger logger = LoggerFactory.getLogger(SettleFlowService.class);

    private final SettleFlowDao flowDao;

    private final SettleTemplateService templateService;

    private final SettleFlowPackageDao settleFlowPackageDao;

    private final OrgService orgService;
    private final CompanyService companyService;

    private final ServiceResourcesetRelDao serviceResourcesetRelDao;

    private final SettleStepDao settleStepDao;

    @Resource
    private SettleTemplateDao templateDao;

    @Resource
    private SettleFlowTemplateRelDao flowTemplateRelDao;

    @Resource
    private SettleServiceDao serviceDao;

    @Resource
    private SettleTemplateServiceRelDao templateServiceRelDao;

    @Resource
    private SettleTemplateAttributeDao templateAttributeDao;

    public SettleFlowService(SettleFlowDao flowDao, SettleTemplateService templateService, OrgService orgService, CompanyService companyService, ServiceResourcesetRelDao serviceResourcesetRelDao,
                             SettleStepDao settleStepDao, SettleFlowPackageDao settleFlowPackageDao) {
        this.flowDao = flowDao;
        this.templateService = templateService;
        this.orgService = orgService;
        this.companyService = companyService;
        this.serviceResourcesetRelDao = serviceResourcesetRelDao;
        this.settleFlowPackageDao = settleFlowPackageDao;
        this.settleStepDao = settleStepDao;
    }


    public Page<SettleFlow> page(Query query, Pageable pageable) {
        Specification<SettleFlow> specification = SettleFlowQueryHelper.querySpecification(query);
        Page<SettleFlow> page = flowDao.findAll(specification, pageable, EntityGraphs.named(SettleFlowPackage.NAMED_ENTITY_GRAPH_DEFAULT));
        return page;
    }

    public SettleFlow findByTenantId(Long tenantId) {
        return flowDao.findByTenantId(tenantId).orElseThrow(() -> new IllegalArgumentException("未找到入驻页面流"));
    }

    public SettleFlow findById(String flowId) {
        SettleFlow flow;
        try {
            long id = Long.parseLong(flowId);
            flow = this.findByFlowId(id);
        } catch (NumberFormatException e) {
            String message = e.getMessage() + ", value: " + flowId;
            logger.warn(message);
            flow = this.findByFlowCode(flowId);
        }
        List<Long> packageIds = flowDao.findPackageIdsByFlowId(flow.getFlowId());
        flow.setPackageIds(packageIds);
        return flow;
    }

    public SettleFlow findByFlowId(long flowId) {
        SettleFlow flow = flowDao.findById(flowId).orElseThrow(() -> new IllegalArgumentException("未找到入驻页面流"));
        List<SettleTemplate> templates = templateService.findByFlowId(flow.getFlowId());
        flow.setTemplates(templates);
        return flow;
    }

    public SettleFlow findByFlowCode(String flowCode) {
        SettleFlow flow = flowDao.findByFlowCode(flowCode).orElseThrow(() -> new IllegalArgumentException("未找到入驻页面流"));
        List<SettleTemplate> templates = templateService.findByFlowId(flow.getFlowId());
        flow.setTemplates(templates);
        return flow;
    }

    @Transactional(rollbackFor = Exception.class)
    public void delete(String flowId) {
        SettleFlow flow = this.findById(flowId);
        if (flow == null) {
            return;
        }
        List<Long> templateIds = templateDao.findTemplateIdByFlowId(flow.getFlowId());
        for (Long templateId : templateIds) {
            templateDao.deleteById(templateId);
            templateAttributeDao.deleteByTemplateId(templateId);
            List<Long> serviceIds = templateServiceRelDao.findServiceIdByTemplateId(templateId);
            templateServiceRelDao.deleteByTemplateId(templateId);
            for (Long serviceId : serviceIds) {
                serviceDao.deleteById(serviceId);
            }
        }
        flowTemplateRelDao.deleteByFlowId(flow.getFlowId());
        flowDao.deleteById(flow.getFlowId());
    }

    public SettleStep getStep(String taxNum, String flowId) {
        SettleFlow flow;
        try {
            long id = Long.parseLong(flowId);
            flow = findByFlowId(id);
        } catch (NumberFormatException e) {
            String message = e.getMessage() + ", value: " + flowId;
            logger.warn(message);
            flow = findByFlowCode(flowId);
        }
        if (null == flow) {
            throw new IllegalArgumentException("未找到入驻页面流");
        }
        List<SettleStep> settleSteps = settleStepDao.findByTaxNumAndFlowId(taxNum, flow.getFlowId());
        if (CollectionUtils.isEmpty(settleSteps) || settleSteps.size() > 1) {
            throw new IllegalArgumentException("当前流程税号为空或者暂时不支持重复的税号");
        }
        return settleSteps.get(0);
    }

    private Long findServicePackage(Long flowId) {
        List<SettleFlowPackage> flowServicePackages = settleFlowPackageDao.findByFlowId(flowId);
        if (CollectionUtils.isEmpty(flowServicePackages)) {
            throw new IllegalArgumentException("流程未查询到公司服务包");
        }
        return flowServicePackages.get(0).getPackageId();
    }

    private OrgStruct findOrgs(String taxNum) {
        List<Company> company = companyService.findByTaxNum(taxNum);
        if (CollectionUtils.isEmpty(company)) {
            throw new IllegalArgumentException("税号未查询到公司");
        }
        Long companyId = company.get(0).getCompanyId();
        OrgModel.Request.Query query = new OrgModel.Request.Query();
        query.setCompanyId(companyId);
        query.setStatus(1);
        query.setOrgType(String.valueOf(OrgType.COMPANY.value()));
        List<OrgStruct> orgs = orgService.list(query, Sort.unsorted());
        if (CollectionUtils.isEmpty(orgs)) {
            throw new IllegalArgumentException("税号未查询到组织");
        }
        return orgs.get(0);
    }

    public Page<SettleFlowPackage> packages(SettleFlowPackageModel.Request.Query query, Pageable pageable) {
        Specification<SettleFlowPackage> specification = SettleFlowPackageQueryHelper.querySpecification(query);
        return settleFlowPackageDao.findAll(specification, pageable);
    }

    public List<SettleFlowPackage> packageList(SettleFlowPackageModel.Request.Query query, Sort sort) {
        Specification<SettleFlowPackage> specification = SettleFlowPackageQueryHelper.querySpecification(query);
        return settleFlowPackageDao.findAll(specification, sort);
    }

    public Page<SettleStep> taxNum(String flowId, Long tenantId, Pageable pageable) {
        SettleFlow flow;
        try {
            long id = Long.parseLong(flowId);
            flow = this.findByFlowId(id);
        } catch (NumberFormatException e) {
            String message = e.getMessage() + ", value: " + flowId;
            logger.warn(message);
            flow = this.findByFlowCode(flowId);
        }
        if (null == flow) {
            throw new IllegalArgumentException("入住流程不存在 ");
        }
        Specification<SettleStep> query = (Specification<SettleStep>) (root, criteriaQuery, builder) -> {
            List<Predicate> predicates = new ArrayList<>();
            predicates.add(builder.equal(root.get("tenantId"), tenantId));
            criteriaQuery.where(predicates.stream().toArray(Predicate[]::new));
            return criteriaQuery.getRestriction();
        };
        return settleStepDao.findAll(query, pageable);
    }

    public List<Long> findResourceByPackage(List<SettleFlowPackage> content) {
        Set<Long> servicePackageIds = content.stream().map(SettleFlowPackage::getPackageId).collect(Collectors.toSet());
        Specification<ServiceResourcesetRel> query = (Specification<ServiceResourcesetRel>) (root, criteriaQuery, builder) -> {
            List<Predicate> predicates = new ArrayList<>();
            if (servicePackageIds.size() == 1) {
                predicates.add(builder.equal(root.<Long>get("servicePackageId"), servicePackageIds.stream().findFirst().get()));
            } else {
                predicates.add(root.<Long>get("servicePackageId").in(servicePackageIds));
            }
            criteriaQuery.where(predicates.stream().toArray(Predicate[]::new));
            return criteriaQuery.getRestriction();
        };
        List<ServiceResourcesetRel> serviceResourceset = serviceResourcesetRelDao.findAll(query);
        if (CollectionUtils.isEmpty(serviceResourceset)) {
            return new ArrayList<>();
        }
        return serviceResourceset.stream().map(ServiceResourcesetRel::getResourcesetId).collect(Collectors.toList());
    }
}
