/*
 * Copyright (c)  2015~2025, Xforceplus
 * All rights reserved.
 * Project: tenant-service
 * ClassName: GradingRoleController   
 * Date: 2021-08-23 16:43:18
 * Author: zhouxin
 */
package com.xforceplus.business.tenant.controller;

import com.fasterxml.jackson.annotation.JsonView;
import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.current.user.GradingRoleApi;
import com.xforceplus.api.model.RoleModel.Request.BindOrgs;
import com.xforceplus.api.model.RoleModel.Request.BindResourceSets;
import com.xforceplus.api.model.RoleModel.Request.BindUsers;
import com.xforceplus.api.model.RoleModel.Request.GradingRoleCreate;
import com.xforceplus.api.model.RoleModel.Request.Query;
import com.xforceplus.api.model.RoleModel.Request.UnbindUsers;
import com.xforceplus.api.model.RoleModel.Request.Update;
import com.xforceplus.api.model.UserModel.Request;
import com.xforceplus.business.common.controller.AbstractController;
import com.xforceplus.business.tenant.service.GradingRoleService;
import com.xforceplus.domain.tenant.GradingRoleDTO;
import com.xforceplus.domain.tenant.GradingRoleOrg;
import com.xforceplus.domain.validation.ValidationGroup;
import com.xforceplus.entity.Resourceset;
import com.xforceplus.entity.Role;
import com.xforceplus.entity.User;
import com.xforceplus.exception.AccessDeniedException;
import com.xforceplus.tenant.security.autoscan.annotation.AuthorizedDefinition;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.tenant.security.core.domain.view.RoleView;
import com.xforceplus.tenant.security.core.domain.view.UserView;
import io.geewit.data.jpa.essential.domain.PageableFactory;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import java.util.List;
import javax.validation.Valid;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Controller;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.PathVariable;

/**
 * @Description: 分级角色控制器
 * @CopyRight: 上海云砺信息科技有限公司
 * @Author: zhouxin
 * @Email: yf_zhouxin@xforceplus.com
 * @Date: 2021/8/23 16:43:18
 */
@Api(value = "分级角色管理", tags = "分级角色管理")
@Controller
@Validated
@SuppressWarnings("all")
public class GradingRoleController extends AbstractController implements GradingRoleApi {

    @Autowired
    private GradingRoleService gradingRoleService;

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:roleclass:read"})
    @ApiOperation(value = "当前用户的分级管理角色分页", tags = {"role"})
    public ResponseEntity<Page<GradingRoleDTO>> currentPage(Query query, Pageable pageable) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        if(!authorizedUser.isAdmin() && !authorizedUser.isGradingManager()){
            throw new AccessDeniedException("操作受限，请确保有权限操作！");
        }
        if(!authorizedUser.isAdmin() && authorizedUser.isGradingManager()) {
            query.setUserId(authorizedUser.getId());
        }
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        query.setTenantId(authorizedUser.getTenantId());
        Page<GradingRoleDTO> page = gradingRoleService.page(query, currentPageable);
        return ResponseEntity.ok(page);
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:roleclass:save"})
    @ApiOperation(value = "新增分级管理角色", tags = {"role"})
    public ResponseEntity<Role> currentCreate(@ApiParam(value = "model", required = true) GradingRoleCreate model) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        if(!authorizedUser.isAdmin()){
            throw new AccessDeniedException("操作受限，请确保有权限操作！");
        }
        model.setType(1);
        model.setTenantId(authorizedUser.getTenantId());
        Role result = gradingRoleService.create(model);
        return ResponseEntity.ok(result);
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:roleclass:read"})
    @ApiOperation(value = "分级管理角色详情", tags = {"role"})
    @JsonView(RoleView.RoleInfo.class)
    public ResponseEntity<GradingRoleDTO> currentInfo(Long roleId) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        if(!authorizedUser.isAdmin() && !authorizedUser.isGradingManager()){
            throw new AccessDeniedException("操作受限，请确保有权限操作！");
        }
        GradingRoleDTO result = gradingRoleService.findByRoleId(roleId);
        return ResponseEntity.ok(result);
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:roleclass:save"})
    @ApiOperation(value = "分级管理角色编辑", tags = {"role"})
    @JsonView(RoleView.RoleInfo.class)
    public ResponseEntity<GradingRoleDTO> currentUpdate(Long roleId, Update model) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        if(!authorizedUser.isAdmin()){
            throw new AccessDeniedException("操作受限，请确保有权限操作！");
        }

        long tenantId = authorizedUser.getTenantId();
        GradingRoleDTO result = gradingRoleService.updateByTenantId(tenantId, roleId, model);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:roleclass:save"})
    @ApiOperation(value = "启用分级角色", tags = {"role"})
    public ResponseEntity<String> currentEnable(@PathVariable("roleId") Long roleId) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        if(!authorizedUser.isAdmin()){
            throw new AccessDeniedException("操作受限，请确保有权限操作！");
        }
        roleService.findByRoleId(authorizedUser.getTenantId(),roleId);
        gradingRoleService.enable(roleId);
        return ResponseEntity.ok("启用成功");
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:roleclass:save"})
    @ApiOperation(value = "禁用分级角色", tags = {"role"})
    public ResponseEntity<String> currentDisable(@PathVariable("roleId") Long roleId) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        if(!authorizedUser.isAdmin()){
            throw new AccessDeniedException("操作受限，请确保有权限操作！");
        }
        gradingRoleService.disable(authorizedUser.getTenantId(), roleId);
        return ResponseEntity.ok("禁用成功");
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:roleclass:save"})
    @ApiOperation(value = "分级管理角色删除", tags = {"role"})
    public ResponseEntity<String> currentDelete(Long roleId) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        if(!authorizedUser.isAdmin()){
            throw new AccessDeniedException("操作受限，请确保有权限操作！");
        }
        long tenantId = authorizedUser.getTenantId();
        gradingRoleService.deleteByTenantIdAndId(tenantId, roleId);
        return ResponseEntity.ok("删除成功");
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:roleclass:read"})
    @ApiOperation(value = "分级管理角色已绑定的用户分页列表", tags = {"role"})
    @JsonView(UserView.UserInfo.class)
    public ResponseEntity<Page<User>> currentBindedUsers(
        Long roleId, Request.Query query, Pageable pageable) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        if(!authorizedUser.isAdmin()){
            throw new AccessDeniedException("操作受限，请确保有权限操作！");
        }
        long tenantId = authorizedUser.getTenantId();
        query.setTenantId(tenantId);
        query.setRoleId(roleId);
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        Page<User> result = userService.page(query, currentPageable);
        return ResponseEntity.ok(result);
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:roleclass:save"})
    @ApiOperation(value = "分级管理角色绑定用户", tags = {"role"})
    public ResponseEntity<String> currentBindRoles(Long roleId, BindUsers bindUsers) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        if(!authorizedUser.isAdmin()){
            throw new AccessDeniedException("操作受限，请确保有权限操作！");
        }
        long tenantId = authorizedUser.getTenantId();
        gradingRoleService.bindUsers(tenantId, roleId, bindUsers);
        return ResponseEntity.ok("绑定成功");
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:roleclass:save"})
    @ApiOperation(value = "分级管理角色解绑用户", tags = {"role"})
    public ResponseEntity<String> currentUnbindRoles(Long roleId, UnbindUsers unbindUsers) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        if(!authorizedUser.isAdmin()){
            throw new AccessDeniedException("操作受限，请确保有权限操作！");
        }
        long tenantId = authorizedUser.getTenantId();
        roleService.unbindUsers(tenantId, roleId, unbindUsers);
        return ResponseEntity.ok("反绑定成功");
    }

    @Override
    @ApiOperation(value = "获取分级管理角色的组织范围", tags = {"role"})
    @AuthorizedDefinition(resources = {"xforce:ucenter:roleclass:read"})
    public ResponseEntity<List<GradingRoleOrg>> currentBindedOrgs(Long roleId) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        if(!authorizedUser.isAdmin() && !authorizedUser.isGradingManager()){
            throw new AccessDeniedException("操作受限，请确保有权限操作！");
        }
        return ResponseEntity.ok(gradingRoleService.findBindedOrgsByRoleId(roleId));
    }

    @Override
    @ApiOperation(value = "更新分级管理角色的组织范围", tags = {"role"})
    @AuthorizedDefinition(resources = {"xforce:ucenter:roleclass:save"})
    public ResponseEntity<String> currentBindOrgs(Long roleId, BindOrgs bindOrgs) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        if(!authorizedUser.isAdmin()){
            throw new AccessDeniedException("操作受限，请确保有权限操作！");
        }
        gradingRoleService.saveGradingRoleOrgs(roleId, bindOrgs);
        return ResponseEntity.ok("操作成功");
    }

    @Override
    @ApiOperation(value = "获取分级管理角色的功能集列表", tags = {"role"})
    @AuthorizedDefinition(resources = {"xforce:ucenter:roleclass:read"})
    public ResponseEntity<List<Resourceset>> currentBindedResourcesets(Long roleId) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        if(!authorizedUser.isAdmin() && !authorizedUser.isGradingManager()){
            throw new AccessDeniedException("操作受限，请确保有权限操作！");
        }
        return ResponseEntity.ok(gradingRoleService.findBindedResourcesetsByRoleId(roleId));
    }

    @Override
    @ApiOperation(value = "更新分级管理角色的功能集", tags = {"role"})
    @AuthorizedDefinition(resources = {"xforce:ucenter:roleclass:save"})
    public ResponseEntity<String> currentBindResourcesets(Long roleId, BindResourceSets bindResourceSets) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        if(!authorizedUser.isAdmin()){
            throw new AccessDeniedException("操作受限，请确保有权限操作！");
        }
        gradingRoleService.saveGradingRoleResourcesets(roleId, bindResourceSets);
        return ResponseEntity.ok("操作成功");
    }
}
