package com.xforceplus.business.tenant.controller;

import com.fasterxml.jackson.annotation.JsonView;
import com.xforceplus.api.common.response.ResponseBoolenEntity;
import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.global.org.OrgApi;
import com.xforceplus.api.model.OrgExtensionModel;
import com.xforceplus.api.model.OrgModel.Request;
import com.xforceplus.api.model.OrgModel.Request.*;
import com.xforceplus.api.model.OrgModel.Response;
import com.xforceplus.api.model.RoleModel;
import com.xforceplus.api.model.TreeModel;
import com.xforceplus.api.model.UserModel;
import com.xforceplus.api.model.org.OrgCompanyModel;
import com.xforceplus.bo.org.OrgCompanyQueryBo;
import com.xforceplus.business.common.controller.AbstractController;
import com.xforceplus.business.company.service.CompanyApplyService;
import com.xforceplus.business.enums.CmsAppEnum;
import com.xforceplus.business.excel.ExcelFile;
import com.xforceplus.business.excel.ExcelService;
import com.xforceplus.business.excel.ExcelSheet;
import com.xforceplus.business.excel.QueryParam;
import com.xforceplus.business.excel.writer.ExcelConfigBusinessType;
import com.xforceplus.business.file.controller.vo.ImportFileRespVo;
import com.xforceplus.business.file.service.ExportFileService;
import com.xforceplus.business.file.service.ImportFileService;
import com.xforceplus.business.tenant.excel.OrgExportExcel;
import com.xforceplus.business.tenant.service.OrgExtensionService;
import com.xforceplus.business.tenant.service.OrgParentIdsService;
import com.xforceplus.business.tenant.service.WrapperOrgService;
import com.xforceplus.domain.account.AccountDto;
import com.xforceplus.domain.org.OrgDto;
import com.xforceplus.domain.org.OrgExtensionDto;
import com.xforceplus.domain.tenant.RoleDto;
import com.xforceplus.domain.user.UserDto;
import com.xforceplus.domain.validation.ValidationGroup;
import com.xforceplus.dto.org.OrgCompanyDTO;
import com.xforceplus.entity.*;
import com.xforceplus.tenant.core.exception.UnknownException;
import com.xforceplus.tenant.security.autoscan.annotation.AuthorizedDefinition;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.tenant.security.core.domain.OrgType;
import com.xforceplus.tenant.security.core.domain.view.OrgView;
import com.xforceplus.tenant.security.core.domain.view.UserView;
import io.geewit.core.jackson.view.View;
import io.geewit.core.utils.reflection.BeanUtils;
import io.geewit.core.utils.tree.TreeUtils;
import io.geewit.data.jpa.essential.domain.PageableFactory;
import io.geewit.data.jpa.essential.domain.SortFactory;
import io.geewit.data.jpa.essential.search.DynamicSpecifications;
import io.geewit.data.jpa.essential.search.SearchFilter;
import io.geewit.data.jpa.essential.web.servlet.Servlets;
import io.swagger.annotations.*;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.validator.constraints.Range;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Controller;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.context.request.WebRequest;
import org.springframework.web.multipart.MultipartFile;
import springfox.documentation.annotations.ApiIgnore;

import javax.validation.Valid;
import javax.validation.constraints.Min;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.xforceplus.api.global.org.OrgApi.Path.API_PREFIX;

@SuppressWarnings("all")
@Api(value = "组织相关接口", description = "组织相关接口")
@Validated
@Controller
public class OrgController extends AbstractController implements OrgApi, com.xforceplus.api.tenant.org.OrgApi, com.xforceplus.api.current.org.OrgApi {
    private final static Logger logger = LoggerFactory.getLogger(OrgController.class);


    private final OrgParentIdsService orgParentIdsService;

    private final ExcelService excelService;

    private final ImportFileService importFileService;

    private final WrapperOrgService wrapperOrgService;

    private final ExportFileService exportFileService;

    private final OrgExtensionService orgExtensionService;

    /**
     * 公司申请
     */
    private final CompanyApplyService companyApplyService;

    @Value("${params.max-size.orgIds:100}")
    private int maxSizeOrgIds;

    public OrgController(OrgParentIdsService orgParentIdsService, ExcelService excelService,
                         ImportFileService importFileService, WrapperOrgService wrapperOrgService,
                         ExportFileService exportFileService, OrgExtensionService orgExtensionService,
                         CompanyApplyService companyApplyService) {
        this.orgParentIdsService = orgParentIdsService;
        this.excelService = excelService;
        this.importFileService = importFileService;
        this.wrapperOrgService = wrapperOrgService;
        this.exportFileService = exportFileService;
        this.orgExtensionService = orgExtensionService;
        this.companyApplyService = companyApplyService;
    }

    /**
     * 获取组织分页列表
     *
     * @param request
     * @param pageable
     * @return
     */
    @ApiIgnore
    @ApiOperation(value = "获取组织分页", notes = "通过分页参数，获取组织分页")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "pageable", value = "分页对象", dataType = "Pageable")
    })
    @JsonView(View.List.class)
    @ResponseBody
    @RequestMapping(name = "组织分页列表查询", value = {API_PREFIX + "/query"}, method = RequestMethod.GET)
    public Page<OrgStruct> page(WebRequest request, @ApiParam(value = "pageable") Pageable pageable) {
        Map<String, Object> searchParams = Servlets.getParametersStartingWith(request, "search_");
        Collection<SearchFilter> filters = SearchFilter.parse(searchParams);
        Specification<OrgStruct> specification = DynamicSpecifications.bySearchFilter(filters);
        Page<OrgStruct> page = orgService.page(specification, pageable);
        return page;
    }

    @ApiOperation(value = "获取组织分页", notes = "通过分页参数，获取组织分页")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "pageable", value = "分页对象", dataType = "Pageable")
    })
    @JsonView(View.List.class)
    @Override
    public ResponseEntity<Page<OrgStruct>> page(@ApiParam(value = "request") Request.Query query,
                                                @ApiParam(value = "pageable") Pageable pageable) {
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "orgId"));
        Page<OrgStruct> result = orgService.page(query, currentPageable);
        return ResponseEntity.ok(result);
    }

    @ApiOperation(value = "新增组织")
    @JsonView(OrgView.OrgInfo.class)
    @Override
    public ResponseEntity<OrgStruct> create(@ApiParam(value = "org") Request.Save model) {
        model.setOrgId(null);
        OrgStruct result = wrapperOrgService.create(model);
        return ResponseEntity.ok(result);
    }

    @ApiOperation(value = "查询租户公司列表")
    @ResponseBody
    @RequestMapping(name = "查询租户公司列表", value = OrgApi.Path.COMPANIES, method = RequestMethod.GET)
    public ResponseEntity<List<Response.CmsTenantCompany>> companies(Request.QueryForCms query) {
        if (StringUtils.isEmpty(query.getTenantId()) && StringUtils.isEmpty(query.getTaxNum())) {
            throw new IllegalArgumentException("tenantId or taxNum 不能都为空");
        }
        List<Response.CmsTenantCompany> companyList = orgService.listFromCms(query.getTenantId(), query.getTaxNum());
        Optional<CmsAppEnum> optional = Stream.of(CmsAppEnum.values()).filter(x -> x.getKey().equals(query.getSource())).findAny();
        if (optional.isPresent()) {
            CmsAppEnum app = optional.get();
            companyList.forEach(c -> {
                c.setSource(app.getDesc());
                c.setIndustry(app.getDesc());
                if (app.equals(CmsAppEnum.P40) || app.equals(CmsAppEnum.P40_CRC)
                        || app.equals(CmsAppEnum.P40_YANGO) || app.equals(CmsAppEnum.UC_PORTAL)) {
                    c.setClientCategory("中心用户");
                }
            });

        } else {
            companyList.forEach(c -> {
                c.setSource("unknown");
                c.setIndustry("unknown");
            });
        }

        return ResponseEntity.ok(companyList);
    }

    @Override
    public ResponseEntity<List<OrgStruct>> byTaxNum(@ApiParam(value = "taxNum", required = true) String taxNum,
                                                    @ApiParam(value = "companyId") Long companyId,
                                                    @ApiParam(value = "tenantId") Long tenantId,
                                                    @ApiParam(value = "orgId") Long orgId,
                                                    @ApiParam(value = "tenantCode") String tenantCode,
                                                    @ApiParam(value = "withHost, default=true") boolean withHost,
                                                    @ApiParam(value = "hostTenantOnly, default=null") Boolean hostTenantOnly,
                                                    @ApiParam(value = "status, default=null") Integer status) {
        List<OrgStruct> orgs = orgService.findByTaxNumAndCompanyIdAndTenantId(taxNum, companyId, tenantId, orgId, tenantCode, withHost, hostTenantOnly, status);

        if (hostTenantOnly != null && hostTenantOnly) {
            orgs = orgs.stream().filter(item -> item.getCompanyId() != null && OrgType.COMPANY.equals(item.getOrgType()) && item.getThisIsHost() != null && item.getThisIsHost()).collect(Collectors.toList());
        }
        return ResponseEntity.ok(orgs);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:orgs:read"})
    @Override
    public ResponseEntity<Page<OrgExtension>> extensionsPage(OrgExtensionModel.Request.Query query, Pageable pageable) {
        Page<OrgExtension> page = orgExtensionService.page(query, pageable);
        return ResponseEntity.ok(page);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:orgs:save"})
    @Override
    public ResponseEntity<OrgExtension> createExtension(long orgId, OrgExtensionModel.Request.Create mode) {
        mode.setOrgId(orgId);
        OrgExtension entity = orgExtensionService.create(mode);
        return ResponseEntity.ok(entity);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:orgs:save"})
    @Override
    public ResponseEntity<OrgExtensionDto> updateExtension(long extensionId, OrgExtensionModel.Request.Update model) {
        model.setExtensionId(extensionId);
        OrgExtension entity = orgExtensionService.update(model);
        return ResponseEntity.ok(entity);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:orgs:save"})
    @Override
    public ResponseEntity<String> deleteExtension(long extensionId) {
        orgExtensionService.deleteById(extensionId);
        return ResponseEntity.ok("删除成功");
    }

    @AuthorizedDefinition(resources = {"xforce:operation:orgs:save"})
    @Override
    public ResponseEntity<String> deleteOrgExtensions(long orgId) {
        orgExtensionService.deleteByOrgId(orgId);
        return ResponseEntity.ok("删除成功");
    }

    @ApiOperation(value = "更新组织")
    @JsonView(OrgView.OrgInfo.class)
    @Override
    public ResponseEntity<OrgStruct> update(@ApiParam(value = "orgId", required = true) long orgId,
                                            @ApiParam(value = "org") Request.Save model) {
        OrgStruct result = orgService.update(orgId, model);
        return ResponseEntity.ok(result);
    }

    @ApiOperation(value = "修改组织状态")
    @Override
    public ResponseEntity<String> updateStatus(long orgId, int status) {
        orgService.updateStatus(orgId, status);
        return ResponseEntity.ok("修改成功");
    }

    @ApiOperation(value = "获取组织详情")
    @JsonView(OrgView.OrgInfo.class)
    @Override
    public ResponseEntity<OrgStruct> info(@ApiParam(value = "orgId", required = true) long orgId,
                                          @ApiParam(value = "extraInfo", defaultValue = "0") int extraInfo) {
        OrgStruct result = orgService.findById(orgId, extraInfo);
        return ResponseEntity.ok(result);
    }

    @JsonView(View.List.class)
    @ApiOperation(value = "组织用户列表")
    @Override
    public ResponseEntity<Page<User>> users(@ApiParam(value = "orgId", required = true) long orgId,
                                            Pageable pageable) {
        UserModel.Request.Query query = new UserModel.Request.Query();
        query.setOrgId(orgId);
        Page<User> result = userService.page(query, pageable);
        return ResponseEntity.ok(result);
    }

    @ApiOperation(value = "获取组织子节点")
    @JsonView(View.Tree.class)
    @Override
    public ResponseEntity<List<OrgStruct>> children(@ApiParam(value = "orgId", required = true) long orgId,
                                                    @ApiParam(value = "userId", required = false) Long userId,
                                                    @ApiParam(value = "withUserBoundFlag", required = false) Boolean withUserBoundFlag) {
        Request.Query query = new Request.Query();
        query.setParentId(orgId);
        query.setWithExtendParams("childrenCount");
        query.setStatus(1);
        query.setWithUserBoundFlag(withUserBoundFlag);
        query.setUserId(userId);
        List<OrgStruct> children = orgService.list(query, Sort.unsorted());
        return ResponseEntity.ok(children);
    }

    @ApiOperation(value = "获取组织子孙列表", notes = "通过id，获取组织子孙列表")
    @JsonView(View.List.class)
    @Override
    public ResponseEntity<List<OrgStruct>> descendants(@ApiParam(value = "orgId", required = true) long orgId) {
        List<OrgStruct> result = orgService.findDescendantsById(orgId);
        return ResponseEntity.ok(result);
    }

    @ApiOperation(value = "获取组织树", notes = "通过tenantid，获取组织树")
    @JsonView(View.Tree.class)
    @Override
    public ResponseEntity<OrgStruct> globalTree(Request.TreeQuery query) {
        query.setAttributes(Stream.of("orgId", "orgName", "orgCode", "parentId", "orgType", "status", "parentIds").collect(Collectors.toSet()));
        List<OrgStruct> result = orgService.findTreeByTenantId(query);
        return ResponseEntity.ok(result.stream().findFirst().orElse(null));
    }

    @ApiOperation(value = "获取组织子孙列表", notes = "通过id，获取组织子孙列表")
    @JsonView(View.List.class)
    @Override
    public ResponseEntity<Page<OrgStruct>> page(@ApiParam(value = "tenantKey", required = true) String tenantKey,
                                                @ApiParam(value = "request") Request.Query query,
                                                @ApiParam(value = "pageable") Pageable pageable) {
        long tenantId = super.toTenantId(tenantKey, query.getByTenantCode());
        query.setTenantId(tenantId);
        //益海嘉里特别需求， 需要用orgCode 升序排列！！！！ 不要乱动！！！
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.ASC, "orgCode"));
        Set<String> withExtendParams = Stream.of("company", "tenant").collect(Collectors.toSet());
        if (StringUtils.isNotBlank(query.getWithExtendParams())) {
            withExtendParams.addAll(Arrays.stream(StringUtils.split(query.getWithExtendParams(), ",")).collect(Collectors.toSet()));
        }
        query.setWithExtendParams(withExtendParams.stream().collect(Collectors.joining(",")));
        Page<OrgStruct> result = orgService.page(query, currentPageable);
        return ResponseEntity.ok(result);
    }

    @ApiIgnore
    @ApiOperation(value = "获取组织列表", notes = "通过id，获取组织列表")
    @JsonView(View.List.class)
    @Override
    public ResponseEntity<List<OrgStruct>> list(@ApiParam(value = "tenantKey", required = true) String tenantKey,
                                                @ApiParam(value = "request") Request.Query query,
                                                @ApiParam(value = "sort") Sort sort) {
        long tenantId = super.toTenantId(tenantKey, query.getByTenantCode());
        query.setTenantId(tenantId);
        Sort currentSort = SortFactory.ofDefaultSort(sort, Sort.by(Sort.Direction.DESC, "orgId"));
        List<OrgStruct> result = orgService.list(query, currentSort);
        return ResponseEntity.ok(result);
    }

    @ApiOperation(value = "创建组织", notes = "创建组织")
    @JsonView(OrgView.OrgInfo.class)
    @Override
    public ResponseEntity<OrgStruct> create(@ApiParam(value = "tenantKey", required = true) String tenantKey,
                                            @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                            @ApiParam(value = "model", required = true) Request.Save model) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        model.setTenantId(tenantId);
        model.setWithApplication(true);

        OrgStruct result = wrapperOrgService.create(model);
        return ResponseEntity.ok(result);
    }

    @ApiOperation(value = "向指定父组织批量导入组织", notes = "向指定父组织批量导入组织")
    @Override
    public ResponseEntity<Boolean> batchImportChildrenIntoParent(@ApiParam(value = "租户id", required = true) String tenantKey,
                                                                 @ApiParam(value = "上级组织id", required = true) String parentKey,
                                                                 @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                                 @ApiParam(value = "byOrgCode", required = false) Boolean byOrgCode,
                                                                 @ApiParam(value = "组织集合", required = true) List<Save> models) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long parentId = super.toOrgId(tenantId, parentKey, byOrgCode);
        wrapperOrgService.save(tenantId, parentId, models);
        return ResponseEntity.ok(Boolean.TRUE);
    }

    @ApiOperation(value = "批量导入组织", notes = "批量导入组织")
    @Override
    public ResponseEntity<Boolean> batchImport(@ApiParam(value = "租户id", required = true) String tenantKey,
                                               @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                               @ApiParam(value = "组织集合", required = true) List<Save> models) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        wrapperOrgService.save(tenantId, models);
        return ResponseEntity.ok(Boolean.TRUE);
    }

    @ApiOperation(value = "批量创建公司和组织2.0-3", notes = "并且绑定公司租户和组织租户关系2.0")
    @Override
    public <O extends OrgDto<O>> ResponseBoolenEntity<Boolean, O> batchImportWithResult(String tenantKey,
                                                                                        @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                                                        List<Save> models) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        List<O> orgStructs = (List<O>) orgService.saveWithResult(tenantId, models);
        return ResponseBoolenEntity.ok(true, orgStructs);

    }

    @ApiOperation(value = "更新组织", notes = "通过id，更新组织")
    @JsonView(OrgView.OrgInfo.class)
    @Override
    public ResponseEntity<OrgStruct> update(@ApiParam(value = "tenantKey", required = true) String tenantKey,
                                            @ApiParam(value = "orgId", required = true) String orgKey,
                                            @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                            @ApiParam(value = "byOrgCode", required = false) Boolean byOrgCode,
                                            @ApiParam(value = "model", required = true) Request.Save model) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long orgId = super.toOrgId(tenantId, orgKey, byOrgCode);
        OrgStruct result = orgService.updateByTenantId(tenantId, orgId, model);
        return ResponseEntity.ok(result);
    }

    @ApiOperation(value = "租户更新组织状态")
    @Override
    public ResponseEntity<String> updateStatus(String tenantKey,
                                               String orgKey,
                                               int status,
                                               @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                               @ApiParam(value = "byOrgCode", required = false) Boolean byOrgCode) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long orgId = super.toOrgId(tenantId, orgKey, byOrgCode);
        orgService.updateStatus(tenantId, orgId, status);
        return ResponseEntity.ok("修改成功");
    }

    @ApiOperation(value = "删除租户组织")
    @Override
    public ResponseEntity<String> delete(String tenantKey, String orgKey, Boolean byTenantCode, Boolean byOrgCode) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long orgId = super.toOrgId(tenantId, orgKey, byOrgCode);
        orgService.deleteByTenantIdAndId(tenantId, orgId);
        return ResponseEntity.ok("删除组织成功");
    }

    @ApiOperation(value = "租户组织下的用户分页列表")
    @JsonView(View.List.class)
    @Override
    public ResponseEntity<Page<User>> users(String tenantKey, String orgKey, UserModel.Request.Query query, Pageable pageable) {
        long tenantId = super.toTenantId(tenantKey, query.getByTenantCode());
        long orgId = super.toOrgId(tenantId, orgKey, query.getByOrgCode());
        query.setTenantId(tenantId);
        query.setOrgId(orgId);
        Page<User> page = userService.page(query, pageable);
        return ResponseEntity.ok(page);
    }

    @ApiOperation(value = "租户根组织列表")
    @JsonView(View.Tree.class)
    @Override
    public ResponseEntity<List<OrgStruct>> roots(@ApiParam(value = "tenantKey", required = true) String tenantKey,
                                                 @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                 @ApiParam(value = "userId", required = false) Long userId,
                                                 @ApiParam(value = "withUserBoundFlag", required = false) Boolean withUserBoundFlag) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        Request.Query query = new Request.Query();
        query.setRootOrg(true);
        query.setWithExtendParams("childrenCount");
        query.setStatus(1);
        query.setUserId(userId);
        if (withUserBoundFlag != null) {
            query.setWithUserBoundFlag(withUserBoundFlag);
        } else {
            query.setWithUserBoundFlag(true);
        }
        query.setTree(true);
        return this.list(tenantId, query, Sort.unsorted());
    }

    @ApiOperation(value = "租户组织子节点")
    @JsonView(View.Tree.class)
    @Override
    public ResponseEntity<List<OrgStruct>> children(@ApiParam(value = "tenantKey", required = true) String tenantKey,
                                                    @ApiParam(value = "orgId", required = true) String orgKey,
                                                    @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                    @ApiParam(value = "byOrgCode", required = false) Boolean byOrgCode,
                                                    @ApiParam(value = "userId", required = false) Long userId,
                                                    @ApiParam(value = "withUserBoundFlag", required = false) Boolean withUserBoundFlag) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long orgId = super.toOrgId(tenantId, orgKey, byOrgCode);
        return this.children(orgId, userId, withUserBoundFlag);
    }

    @ApiOperation(value = "租户通过id获取组织")
    @JsonView(OrgView.OrgInfo.class)
    @Override
    public ResponseEntity<OrgStruct> info(@ApiParam(value = "tenantKey", required = true) String tenantKey,
                                          @ApiParam(value = "orgId", required = true) String orgKey,
                                          @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                          @ApiParam(value = "byOrgCode", required = false) Boolean byOrgCode) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long orgId = super.toOrgId(tenantId, orgKey, byOrgCode);
        OrgStruct result = orgService.info(tenantId, orgId);
        return ResponseEntity.ok(result);
    }

    @ApiOperation(value = "组织树")
    @JsonView(View.Tree.class)
    @Override
    public ResponseEntity<OrgStruct> tree(@ApiParam(value = "tenantKey", required = true) String tenantKey,
                                          @ApiParam(value = "treeQuery") Request.TreeQuery treeQuery) {
        long tenantId = super.toTenantId(tenantKey, treeQuery.getByTenantCode());
        treeQuery.setTenantId(tenantId);
        treeQuery.setAttributes(Stream.of("orgId", "orgName", "parentId", "orgType", "parentIds").collect(Collectors.toSet()));
        List<OrgStruct> result = orgService.findTreeByTenantId(treeQuery);
        return ResponseEntity.ok(result.stream().findFirst().orElse(null));
    }

    @ApiOperation(value = "获取组织分页列表", notes = "通过参数，获取组织分页列表")
    @JsonView(View.List.class)
    @Override
    public ResponseEntity<List<OrgStruct>> descendants(@ApiParam(value = "tenantKey", required = true) String tenantKey,
                                                       @ApiParam(value = "orgId", required = true) String orgKey,
                                                       @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                       @ApiParam(value = "byOrgCode", required = false) Boolean byOrgCode) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long orgId = super.toOrgId(tenantId, orgKey, byOrgCode);
        List<OrgStruct> result = orgService.findDescendantsById(orgId);
        return ResponseEntity.ok(result);
    }

    @Override
    @ApiOperation(value = "当前租户下组织分页查询")
    @JsonView(View.List.class)
    public ResponseEntity<Page<OrgStruct>> currentPage(Request.Query query, Pageable pageable) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        Set<String> withExtendParams = Stream.of("userCount", "parentName", "companyNos").collect(Collectors.toSet());
        if (StringUtils.isNotBlank(query.getWithExtendParams())) {
            withExtendParams.addAll(Arrays.stream(StringUtils.split(query.getWithExtendParams(), ",")).collect(Collectors.toSet()));
        }
        query.setWithExtendParams(withExtendParams.stream().collect(Collectors.joining(",")));

        String modules = authorizedUser.getModules();
        query.setModules(modules);
        return this.page(tenantId, query, pageable);
    }

    @Override
    @ApiOperation(value = "当前租户下新建组织")
    @JsonView(OrgView.OrgInfo.class)
    public ResponseEntity<OrgStruct> currentCreate(Request.Save model) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        return this.create(tenantId, model);
    }

    @Override
    @ApiOperation(value = "当前租户下组织详情")
    @JsonView(OrgView.OrgInfo.class)
    public ResponseEntity<OrgStruct> currentInfo(Long orgId) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        return this.info(tenantId, orgId);
    }

    @Override
    @ApiOperation(value = "当前租户下更新组织详情")
    @JsonView(OrgView.OrgInfo.class)
    public ResponseEntity<OrgStruct> currentUpdate(Long orgId, @Validated(ValidationGroup.OnUpdate.class) Request.Save model) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        model.setTenantId(tenantId);
        return this.update(tenantId, orgId, model);
    }

    @ApiIgnore
    @ApiOperation(value = "获取当前租户组织列表", notes = "通过id，获取组织列表")
    @JsonView(View.List.class)
    @Override
    public ResponseEntity<List<OrgStruct>> currentList(Request.Query query, Sort sort) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        return this.list(tenantId, query, sort);
    }

    @Override
    @ApiOperation(value = "当前租户下删除组织")
    public ResponseEntity<String> currentDelete(Long orgId) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        return this.delete(tenantId, orgId);
    }

    @Override
    @ApiOperation(value = "当前租户下组织启用")
    public ResponseEntity<String> currentEnable(Long orgId) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        return this.updateStatus(tenantId, orgId, 1);
    }

    @Override
    @ApiOperation(value = "当前租户下组织禁用")
    public ResponseEntity<String> currentDisable(Long orgId) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        return this.updateStatus(tenantId, orgId, 0);
    }

    @Override
    @ApiOperation(value = "当前租户下检查组织名称")
    public ResponseEntity<Boolean> currentCheckName(String orgName) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        boolean result = orgService.checkName(tenantId, orgName);
        return ResponseEntity.ok(result);
    }

    @Override
    @ApiOperation(value = "当前租户下指定组织的用户分页列表")
    @JsonView(UserView.UserInfo.class)
    public ResponseEntity<Page<User>> currentOrgUsers(String orgKey, UserModel.Request.Query query, Pageable pageable) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        query.setTenantId(tenantId);
        long orgId = super.toOrgId(tenantId, orgKey, query.getByOrgCode());
        query.setOrgId(orgId);
        query.setDetail(true);
        query.setIncludeIndependents(false);
        //id 倒序排列，避免慢查询
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "id"));
        Page<User> result = userService.page(query, currentPageable);
        return ResponseEntity.ok(result);
    }

    @Override
    @ApiOperation(value = "当前租户下的指定组织下创建用户")
    public <U extends UserDto<O, R, A>, O extends OrgDto<O>, R extends RoleDto, A extends AccountDto> ResponseEntity<U> currentCreateOrgUser(String orgKey, Boolean byOrgCode, UserModel.Request.SimpleCreate model) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        OrgStruct org = super.toOrg(tenantId, orgKey, byOrgCode);
        UserModel.Request.Create save = new UserModel.Request.Create();
        BeanUtils.copyProperties(model, save);
        save.setIsRoleOverwrite(false);
        save.setIsOrgOverwrite(false);
        save.setIsTagOverwrite(false);
        save.setIsAppOverwrite(false);
        save.setIsMergeAccount(true);
        save.setIsStrict(true);
        save.setTenantId(tenantId);
        UserModel.Request.SaveUserOutput<U, O, R, A> saveUserOutput = userService.save(tenantId, org, save, save.isRoleOverwrite(), save.isOrgOverwrite(), save.isTagOverwrite(), save.isAppOverwrite(), save.isMergeAccount(), save.isStrict());
        if (saveUserOutput.getUserResultMap().isEmpty()) {
            throw new UnknownException("未知错误");
        }
        UserModel.Request.SaveUserContext<U, O, R, A> saveUserContext = saveUserOutput.getUserResultMap().values().stream().findFirst().orElseThrow(() -> new UnknownException("未知错误"));
        U user = saveUserContext.getUser();
        return ResponseEntity.ok(user);
    }

    @Override
    @ApiOperation(value = "当前租户下的指定组织下更新用户")
    public <U extends UserDto<O, R, A>, O extends OrgDto<O>, R extends RoleDto, A extends AccountDto> ResponseEntity<U> currentUpdateOrgUser(String orgKey, long userId, Boolean byOrgCode, UserModel.Request.SimpleUpdate model) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        OrgStruct org = super.toOrg(tenantId, orgKey, byOrgCode);
        UserModel.Request.Update save = new UserModel.Request.Update();
        BeanUtils.copyProperties(model, save);
        save.setUserId(userId);
        save.setIsRoleOverwrite(false);
        save.setIsOrgOverwrite(false);
        save.setIsTagOverwrite(false);
        save.setIsAppOverwrite(false);
        save.setIsMergeAccount(true);
        save.setIsStrict(true);
        save.setTenantId(tenantId);

        UserModel.Request.SaveUserOutput<U, O, R, A> saveUserOutput = userService.save(tenantId, org, save, save.isRoleOverwrite(), save.isOrgOverwrite(), save.isTagOverwrite(), save.isAppOverwrite(), save.isMergeAccount(), save.isStrict());
        Map<Long, UserModel.Request.SaveUserContext<U, O, R, A>> userContextMap = saveUserOutput.getUserResultMap();
        if (userContextMap.values().isEmpty()) {
            throw new IllegalArgumentException("没有返回期望的值");
        }
        U result = userContextMap.values().stream().findFirst().orElseThrow(() -> new IllegalArgumentException("没有返回期望的值")).getUser();
        return ResponseEntity.ok(result);
    }

    @Override
    @ApiOperation(value = "删除当前租户下的指定组织下指定用户")
    public ResponseEntity<String> currentDeleteOrgUser(String orgKey, long userId, Boolean byOrgCode) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        long orgId = super.toOrgId(tenantId, orgKey, byOrgCode);
        userService.deleteByOrgIdAndId(orgId, userId);
        return ResponseEntity.ok("删除成功");
    }

    @ApiOperation("当前租户下绑定指定组织和用户列表")
    @Override
    public ResponseEntity<String> currentOrgBindUsers(String orgKey, Boolean byOrgCode, Request.BindUsers bindUsers) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        long orgId = super.toOrgId(tenantId, orgKey, byOrgCode);
        orgService.bindUsers(tenantId, orgId, bindUsers);
        return ResponseEntity.ok("绑定成功");
    }

    @ApiOperation("当前租户下反绑定指定组织和用户列表")
    @Override
    public ResponseEntity<String> currentOrgUnbindUsers(String orgKey, Boolean byOrgCode, Request.UnbindUsers unbindUsers) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        long orgId = super.toOrgId(tenantId, orgKey, byOrgCode);
        orgService.unbindUsers(tenantId, orgId, unbindUsers);
        return ResponseEntity.ok("反绑定成功");
    }

    @ApiOperation("当前租户下的指定组织的组织角色分页列表")
    @Override
    public ResponseEntity<Page<Role>> currentOrgRoles(String orgKey, RoleModel.Request.Query query, Pageable pageable) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        long orgId = super.toOrgId(tenantId, orgKey, query.getByOrgCode());
        query.setTenantId(tenantId);
        if (query.getStatus() == null) {
            query.setStatus(1);
        }
        query.setType(2);
        query.setOrgId(orgId);
        Page<Role> result = roleService.page(query, pageable);
        return ResponseEntity.ok(result);
    }

    @ApiOperation("当前租户下获取用户绑定组织角色列表")
    @Override
    public ResponseEntity<Page<Role>> currentBindedOrgRoles(String orgKey, long userId, RoleModel.Request.Query query, Pageable pageable) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        long orgId = super.toOrgId(tenantId, orgKey, query.getByOrgCode());
        query.setTenantId(tenantId);
        query.setUserId(userId);
        if (query.getStatus() == null) {
            query.setStatus(1);
        }
        query.setType(2);
        query.setOrgId(orgId);
        Page<Role> result = roleService.page(query, pageable);
        return ResponseEntity.ok(result);
    }

    @ApiOperation("当前租户下绑定指定用户和组织角色列表")
    @Override
    public ResponseEntity<String> currentBindRoles(String orgKey, long userId, Boolean byOrgCode, UserModel.Request.BindRoles bindRoles) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        long orgId = super.toOrgId(tenantId, orgKey, byOrgCode);
        userService.bindRoles(tenantId, orgId, userId, null, null, null, bindRoles.getOrgRoleIds(), bindRoles.isOverwrite(), false, false);
        return ResponseEntity.ok("绑定成功");
    }

    @ApiOperation("当前用户下组织树")
    @JsonView(View.Tree.class)
    @Override
    public ResponseEntity<OrgStruct> currentUserOrgTree(Request.TreeQuery treeQuery) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        treeQuery.setTenantId(tenantId);
        Long userId = authorizedUser.getId();
        treeQuery.setUserId(userId);
        treeQuery.setStatus(1);
        treeQuery.setAttributes(Stream.of("parentIds").collect(Collectors.toSet()));
        List<OrgStruct> list = orgService.findTreeByTenantId(treeQuery);
        List<OrgStruct> result = TreeUtils.buildTree(list);
        return ResponseEntity.ok(result.stream().findFirst().orElse(null));
    }

    @ApiOperation("当前用户下的根组织集合")
    @JsonView(View.Tree.class)
    @Override
    public ResponseEntity<List<OrgStruct>> currentUserOrgRoots() {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        Request.Query query = new Request.Query();
        query.setTenantId(tenantId);
        List<OrgStruct> roots = orgService.lazyLoadCurrentUserOrgRoots(query);
        return ResponseEntity.ok(roots);
    }

    @ApiOperation("当前用户下的组织子节点")
    @JsonView(View.Tree.class)
    @Override
    public ResponseEntity<List<OrgStruct>> currentUserOrgChildren(String orgKey, Boolean byOrgCode) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        long orgId = super.toOrgId(tenantId, orgKey, byOrgCode);
        Request.Query query = new Request.Query();
        query.setTenantId(tenantId);
        query.setParentId(orgId);
        List<OrgStruct> children = orgService.lazyLoadCurrentUserOrgChildren(query);
        return ResponseEntity.ok(children);
    }

    @ApiOperation("当前用户下的组织子树节点")
    @JsonView(View.Tree.class)
    @Override
    public ResponseEntity<OrgStruct> currentUserOrgChildrenTree(String orgKey, Boolean byOrgCode, Request.TreeQuery treeQuery) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        long orgId = super.toOrgId(tenantId, orgKey, byOrgCode);
        treeQuery.setTenantId(tenantId);
        treeQuery.setRootId(orgId);
        int status = treeQuery.getStatus() == null ? 1 : treeQuery.getStatus();
        treeQuery.setStatus(status);

        List<OrgStruct> roots = orgService.lazyLoadCurrentUserOrgDescendants(treeQuery);
        roots = TreeUtils.buildTree(roots);
        OrgStruct root = roots.stream().filter(o -> o.getOrgId().equals(orgId)).findFirst().orElse(null);
        return ResponseEntity.ok(root);
    }

    @Override
    @ApiOperation(value = "当前租户下的根组织集合")
    @JsonView(View.Tree.class)
    public ResponseEntity<List<OrgStruct>> currentRoots(@ApiParam(value = "userId", required = false) Long userId,
                                                        @ApiParam(value = "withUserBoundFlag", required = false) Boolean withUserBoundFlag) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        return this.roots(tenantId, userId, withUserBoundFlag);
    }

    @Override
    @ApiOperation(value = "当前租户组织子节点")
    @JsonView(View.Tree.class)
    public ResponseEntity<List<OrgStruct>> currentChildren(@ApiParam(value = "orgId", required = true) String orgKey,
                                                           @ApiParam(value = "byOrgCode", required = false) Boolean byOrgCode,
                                                           @ApiParam(value = "userId", required = false) Long userId,
                                                           @ApiParam(value = "withUserBoundFlag", required = false) Boolean withUserBoundFlag) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        long orgId = super.toOrgId(tenantId, orgKey, byOrgCode);
        return this.children(tenantId, orgId, userId, withUserBoundFlag);
    }

    @Override
    @ApiOperation(value = "当前租户下的指定组织的子孙组织")
    public ResponseEntity<List<OrgStruct>> currentDescendants(@ApiParam(value = "orgId", required = true) String orgKey,
                                                              @ApiParam(value = "byOrgCode", required = false) Boolean byOrgCode) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        long orgId = super.toOrgId(tenantId, orgKey, byOrgCode);
        return this.descendants(tenantId, orgId);
    }

    @Override
    @ApiOperation(value = "当前租户下的组织树")
    @JsonView(View.Tree.class)
    public ResponseEntity<OrgStruct> currentTree(Request.TreeQuery treeQuery) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        int status = treeQuery.getStatus() == null ? 1 : treeQuery.getStatus();
        String modules = authorizedUser.getModules();
        treeQuery.setStatus(status);
        treeQuery.setModules(modules);
        treeQuery.setWithUserBound(true);
        return this.tree(tenantId, treeQuery);
    }

    /**
     * 重建组织paths
     *
     * @return
     */
    @ResponseBody
    @ApiOperation(value = "重建组织paths", notes = "重建组织paths")
    @RequestMapping(name = "重建组织paths", value = API_PREFIX + "/paths/rebuild", method = RequestMethod.POST)
    public ResponseEntity<String> rebuildAllPaths() {
        orgParentIdsService.rebuildAllParentIds();
        return ResponseEntity.ok("重建成功");
    }

    /**
     * 重建组织paths
     *
     * @return
     */
    @ResponseBody
    @ApiOperation(value = "重建组织paths", notes = "重建组织paths")
    @RequestMapping(name = "重建组织paths", value = com.xforceplus.api.tenant.org.OrgApi.Path.API_PREFIX + "/paths/rebuild", method = RequestMethod.POST)
    public ResponseEntity<String> rebuildAllPaths(@PathVariable("tenantKey") String tenantKey,
                                                  @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        orgParentIdsService.rebuildParentIds(tenantId);
        return ResponseEntity.ok("重建成功");
    }

    @ApiOperation(value = "组织导入excel")
    @ResponseBody
    @RequestMapping(name = "批量导入组织excel", value = com.xforceplus.api.tenant.org.OrgApi.Path.BATCH_IMPORT_2, produces = {"application/json"}, method = RequestMethod.POST)
    public ResponseEntity batchImport(@PathVariable("tenantKey") String tenantKey,
                                      @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                      @RequestParam("file") MultipartFile file) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        return excelService.processExcel(tenantId, file);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:orgs:read"})
    @ResponseBody
    @ApiOperation(value = "批量当前租户组织导出excel", notes = "批量当前租户组织导出excel")
    @RequestMapping(name = "批量当前租户组织导出excel", value = com.xforceplus.api.current.org.OrgApi.Path.BATCH_EXPORT_EXCEL, method = RequestMethod.POST)
    public ResponseEntity<ImportFileRespVo> currentExportExcel(@ApiParam(value = "request") @RequestBody Request.Query query) {
        if (CollectionUtils.isEmpty(query.getSheets())) {
            query.setSheets(OrgExportExcel.getSheets().stream().map(ExcelSheet::getSheetName).collect(Collectors.toList()));
        }
        //设置租户
        IAuthorizedUser authorizedUser = UserInfoHolder.get();
        query.setTenantId(authorizedUser.getTenantId());
        //与查询逻辑保持一致
        query.setStatus(query.getStatus() == null ? 1 : query.getStatus());
        orgService.buildCurrentQuery(query, authorizedUser);
        QueryParam queryParam = QueryParam.builder()
                .param(ExcelFile.PARAMS_QUERY, query)
                .build();
        ExcelConfigBusinessType businessType = ExcelConfigBusinessType.ORG_IMPORT;
        //异步执行
        ExcelFileStore excelFileStore = exportFileService.asyncExcelExport(queryParam.params(), businessType);
        ImportFileRespVo respVo = new ImportFileRespVo();
        respVo.setId(excelFileStore.getId());
        respVo.setBusinessType(businessType.getBusinessName());
        //检查ExcelFile文件是否正确
        return ResponseEntity.ok(respVo);
    }

    //    @ResourceInfo("xforce:ucenter:orgs:save")
    @ResponseBody
    @ApiOperation(value = "批量当前租户组织导入excel", notes = "批量当前租户组织导入excel")
    @RequestMapping(name = "批量当前租户组织导入excel", value = com.xforceplus.api.current.org.OrgApi.Path.BATCH_IMPORT_EXCEL, method = RequestMethod.POST)
    public ResponseEntity<ImportFileRespVo> currentImportExcel(@RequestParam("file") MultipartFile file) {
        IAuthorizedUser authorizedUser = UserInfoHolder.get();
        if (authorizedUser==null){
            throw new IllegalArgumentException("没有合法上下文");
        }
        Long tenantId = authorizedUser.getTenantId();
        Long userId = authorizedUser.getId();
        String businessType = ExcelConfigBusinessType.ORG_IMPORT.name();
        //保存数据和上传文件，并转为异步处理
        ExcelFileStore excelFileStore = importFileService.createAndSaveFile(file, businessType, userId, tenantId);

        ImportFileRespVo respVo = new ImportFileRespVo();
        respVo.setBusinessType(businessType);
        respVo.setFileId(excelFileStore.getSourceFileId());
        respVo.setId(excelFileStore.getId());
        //检查ExcelFile文件是否正确
        return ResponseEntity.ok(respVo);
    }

    @ApiOperation(value = "组织扩展字段 excel 批量导入")
    @ResponseBody
    @RequestMapping(name = "组织扩展字段 excel 批量导入", value = com.xforceplus.api.current.org.OrgApi.Path.BATCH_IMPORT_ORG_EXTENSION, produces = {"application/json"}, method = RequestMethod.POST)
    public ResponseEntity<ImportFileRespVo> batchImportOrgExtensions(@RequestParam("file") MultipartFile file) {

        Long tenantId = 0L;
        Long userId = 0L;
        IAuthorizedUser authorizedUser = UserInfoHolder.get();
        if (null != authorizedUser) {
            tenantId = authorizedUser.getTenantId();
            userId = authorizedUser.getId();
        }

        String businessType = ExcelConfigBusinessType.ORG_EXTENSION_IMPORT.name();
        //保存数据和上传文件，并转为异步处理
        ExcelFileStore excelFileStore = importFileService.createAndSaveFile(file, businessType, userId, tenantId);

        ImportFileRespVo respVo = new ImportFileRespVo();
        respVo.setBusinessType(businessType);
        respVo.setFileId(excelFileStore.getSourceFileId());
        respVo.setId(excelFileStore.getId());
        //检查ExcelFile文件是否正确
        return ResponseEntity.ok(respVo);
    }

    /**
     * 根据税号查询
     *
     * @param req 请求参数科号必须
     * @return List<OrgCompanyModel.Request>
     */
    @Override
    public ResponseEntity<List<OrgCompanyModel.Response>> orgCompaniesByTaxNum(OrgCompanyModel.Request req) {

        OrgCompanyQueryBo queryBo = new OrgCompanyQueryBo();
        BeanUtils.copyProperties(req, queryBo);
        List<OrgCompanyDTO> companyDTOList = orgService.findOrgCompanyByTaxNum(queryBo);
        //数据转换
        List<OrgCompanyModel.Response> resps = companyDTOList.stream()
                .map(e -> {
                    OrgCompanyModel.Response resp = new OrgCompanyModel.Response();
                    BeanUtils.copyProperties(e, resp);
                    resp.setCompanyNos(e.getCompanyNos());
                    return resp;
                }).collect(Collectors.toList());
        return ResponseEntity.ok(resps);
    }

    @Override
    public ResponseEntity<String> delete(long orgId) {
        orgService.deleteById(orgId);
        return ResponseEntity.ok("删除组织成功");
    }

    @Override
    @JsonView(View.List.class)
    public ResponseEntity<List<OrgStruct>> listByOrgKeys(String tenantKey,
                                                         @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                         TreeModel.OrgScope scope,
                                                         boolean byId,
                                                         String orgKeys) {
        String[] keyArr = orgKeys.split(",");
        if (keyArr.length > maxSizeOrgIds) {
            throw new IllegalArgumentException("orgIds条数限制为" + maxSizeOrgIds);
        }
        Set<String> keySet = Arrays.stream(keyArr).collect(Collectors.toSet());
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        List<OrgStruct> list = orgService.listByOrgKeys(tenantId, scope, byId, keySet);
        return ResponseEntity.ok(list);
    }

    @AuthorizedDefinition(resources = {"xforce:business:user:read"})
    @ApiOperation(value = "当前租户下导出指定组织下用户列表Excel")
    @ResponseBody
    @RequestMapping(name = "当前租户下导出指定组织下用户列表Excel", value = com.xforceplus.api.current.org.OrgApi.Path.EXPORT_USER_EXCEL, method = RequestMethod.GET)
    public ResponseEntity currentUsersExport(@PathVariable("orgKey") String orgKey, UserModel.Request.Query query) {

        if (CollectionUtils.isEmpty(query.getSheets())) {
            throw new IllegalArgumentException("请选择SheetName");
        }
        //租户Id
        IAuthorizedUser authorizedUser = UserInfoHolder.get();
        Long tenantId = authorizedUser.getTenantId();
        long orgId = super.toOrgId(tenantId, orgKey, query.getByOrgCode());
        query.setTenantId(tenantId);
        query.setOrgId(orgId);
        userService.buildCurrentQuery(query, authorizedUser);
        //异步执行
        QueryParam queryParam = QueryParam.builder()
                .param(ExcelFile.PARAMS_QUERY, query)
                .build();
        ExcelConfigBusinessType businessType = ExcelConfigBusinessType.USER_IMPORT;
        ExcelFileStore excelFileStore = exportFileService.asyncExcelExport(queryParam.params(), businessType);
        ImportFileRespVo respVo = new ImportFileRespVo();
        respVo.setId(excelFileStore.getId());
        respVo.setBusinessType(businessType.getBusinessName());
        //检查ExcelFile文件是否正确
        return ResponseEntity.ok(respVo);
    }

}
