/*
 * Copyright (c)  2015~2025, Xforceplus
 * All rights reserved.
 * Project: tenant-service
 * ClassName: GradingRoleController   
 * Date: 2021-08-23 16:43:18
 * Author: zhouxin
 */
package com.xforceplus.business.tenant.controller;

import com.fasterxml.jackson.annotation.JsonView;
import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.current.user.OrgRoleApi;
import com.xforceplus.api.model.RoleModel.Request.BindUsers;
import com.xforceplus.api.model.RoleModel.Request.OrgRoleCreate;
import com.xforceplus.api.model.RoleModel.Request.OrgRoleUpdate;
import com.xforceplus.api.model.RoleModel.Request.Query;
import com.xforceplus.api.model.RoleModel.Request.UnbindUsers;
import com.xforceplus.api.model.UserModel.Request;
import com.xforceplus.business.common.controller.AbstractController;
import com.xforceplus.business.tenant.service.OrgRoleService;
import com.xforceplus.domain.tenant.OrgRoleDTO;
import com.xforceplus.entity.Resourceset;
import com.xforceplus.entity.Role;
import com.xforceplus.entity.User;
import com.xforceplus.tenant.security.autoscan.annotation.AuthorizedDefinition;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.tenant.security.core.domain.view.UserView;
import io.geewit.data.jpa.essential.domain.PageableFactory;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import java.util.List;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Controller;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.PathVariable;

/**
 * @Description: 组织角色控制器
 * @CopyRight: 上海云砺信息科技有限公司
 * @Author: zhouxin
 * @Email: yf_zhouxin@xforceplus.com
 * @Date: 2021/8/23 16:43:18
 */
@Api(value = "组织角色管理", tags = "组织角色管理")
@Controller
@Validated
@SuppressWarnings("all")
public class OrgRoleController extends AbstractController implements OrgRoleApi {

    @Autowired
    private OrgRoleService orgRoleService;

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:roleclass:read"})
    @ApiOperation(value = "指定分级角色的组织角色分页", tags = {"role"})
    public ResponseEntity<Page<OrgRoleDTO>> currentPage(Query query, Pageable pageable) {
        if(null == query.getFromRoleId()) {
            throw new IllegalArgumentException("参数错误， 必须指定来源分级角色id");
        }
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        query.setTenantId(tenantId);
        Page<OrgRoleDTO> page = orgRoleService.page(query, pageable);
        return ResponseEntity.ok(page);
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:roleclass:save"})
    @ApiOperation(value = "新增组织角色", tags = {"role"})
    public ResponseEntity<Role> currentCreate(@ApiParam(value = "model", required = true) OrgRoleCreate model) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        model.setTenantId(tenantId);
        model.setType(2);
        Role role = orgRoleService.create(model);
        return ResponseEntity.ok(role);
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:roleclass:read"})
    @ApiOperation(value = "组织角色详情", tags = {"role"})
    public ResponseEntity<OrgRoleDTO> currentInfo(Long roleId) {
        return ResponseEntity.ok(orgRoleService.findByRoleId(roleId));
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:roleclass:save"})
    @ApiOperation(value = "组织角色编辑", tags = {"role"})
    public ResponseEntity<OrgRoleDTO> currentUpdate(Long roleId, OrgRoleUpdate model) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        OrgRoleDTO result = orgRoleService.updateByTenantId(tenantId, roleId, model);
        return ResponseEntity.ok(result);
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:roleclass:save"})
    @ApiOperation(value = "组织角色删除", tags = {"role"})
    public ResponseEntity<String> currentDelete(Long roleId) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        orgRoleService.deleteByTenantIdAndId(tenantId, roleId);
        return ResponseEntity.ok("删除成功");
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:roleclass:read"})
    @ApiOperation(value = "组织角色已绑定的用户分页列表", tags = {"role"})
    @JsonView(UserView.UserInfo.class)
    public ResponseEntity<Page<User>> currentBindedUsers(
        Long roleId, Request.Query query, Pageable pageable) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        query.setTenantId(tenantId);
        query.setRoleId(roleId);
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        Page<User> result = userService.page(query, currentPageable);
        return ResponseEntity.ok(result);
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:roleclass:save"})
    @ApiOperation(value = "组织角色绑定用户", tags = {"role"})
    public ResponseEntity<String> currentBindRoles(Long roleId, BindUsers bindUsers) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        roleService.bindUsers(tenantId, roleId, bindUsers, 2);
        return ResponseEntity.ok("绑定成功");
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:ucenter:roleclass:save"})
    @ApiOperation(value = "组织角色解绑用户", tags = {"role"})
    public ResponseEntity<String> currentUnbindRoles(Long roleId, UnbindUsers unbindUsers) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        roleService.unbindUsers(tenantId, roleId, unbindUsers);
        return ResponseEntity.ok("反绑定成功");
    }

    @Override
    @ApiOperation(value = "获取组织角色的功能集列表", tags = {"role"})
    @AuthorizedDefinition(resources = {"xforce:ucenter:roleclass:read"})
    public ResponseEntity<List<Resourceset>> currentBindedResourcesets(Long roleId) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        return ResponseEntity.ok(orgRoleService.findBindedResourcesetsByRoleId(roleId));
    }

    @Override
    @ApiOperation(value = "组织角色启用", tags = {"role"})
    @AuthorizedDefinition(resources = {"xforce:ucenter:roleclass:save"})
    public ResponseEntity<String> currentEnable(@PathVariable("roleId") Long roleId) {
        orgRoleService.enable(roleId);
        return ResponseEntity.ok("操作成功");
    }

    @Override
    @ApiOperation(value = "组织角色禁用", tags = {"role"})
    @AuthorizedDefinition(resources = {"xforce:ucenter:roleclass:save"})
    public ResponseEntity<String> currentDisable(@PathVariable("roleId") Long roleId) {
        orgRoleService.disable(roleId);
        return ResponseEntity.ok("操作成功");
    }
}
