/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: OrgUserController.java   2020-12-28 15-04-42
 * Author: Evan
 */
package com.xforceplus.business.tenant.controller;

import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.global.user.OrgUserApi;
import com.xforceplus.api.model.user.OrgUserModel;
import com.xforceplus.api.model.user.OrgUserModel.OrgUserComplexRequest;
import com.xforceplus.api.model.user.OrgUserModel.OrgUserRequest;
import com.xforceplus.api.model.user.OrgUserModel.OrgUserResponse;
import com.xforceplus.bo.user.OrgUserBO;
import com.xforceplus.bo.user.OrgUserComplexBO;
import com.xforceplus.business.tenant.service.OrgUserService;
import com.xforceplus.dto.user.OrgUserDTO;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.springframework.beans.BeanUtils;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.RestController;

import javax.validation.Valid;

import static com.xforceplus.api.model.user.OrgUserModel.RangeType.CURRENT_NODE;

/**
 * <p>
 * Title:
 * </p>
 * <p>
 * Description:
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-12-28 15-04-42
 * @since V1.0
 */
@Slf4j
@Api(value = "组织用户查询")
@RestController
public class OrgUserController implements OrgUserApi {

    private final OrgUserService orgUserService;

    public OrgUserController(OrgUserService orgUserService) {
        this.orgUserService = orgUserService;
    }

    /**
     * 组织复杂条件分页查询用户列表
     *
     * @param request  OrgUserRequest 分页请求对象
     * @param pageable 分页对象
     * @return ResponseEntity<Page < OrgUserModel.OrgUserResponse>>
     */
    @Override
    @ApiOperation(value = "根据组件(ID、Code、名称)多条件分页查询用户列表")
    public ResponseEntity<Page<OrgUserResponse>> pageComplex(OrgUserComplexRequest request, Pageable pageable) {
        //每页记录数不能大于100条
        Assert.isTrue(pageable.getPageSize() <= 100, "每页记录数不能大于100条");
        pageable = PageRequest.of(pageable.getPageNumber(), pageable.getPageSize(), Sort.unsorted());
        //校验查询类型
        this.checkQueryType(request);
        //如查询范围为空，则查询范围为当前节点
        if (null == request.getRangeType()) {
            request.setRangeType(CURRENT_NODE);
        }
        log.info("OrgUserRequest.request:{},page:{}", request, pageable);
        //检验时间并补充时间

        this.checkCreatedTimeBeginEnd(request);
        OrgUserComplexBO orgUserBO = new OrgUserComplexBO();
        BeanUtils.copyProperties(request, orgUserBO);
        //查询当前节点数据
        Page<OrgUserDTO> page;
        if (CURRENT_NODE == request.getRangeType()) {
            page = orgUserService.pageByCurrentOrgQueryType(pageable, orgUserBO);
        } else {
            //查询当前组织节点及以下节点，每页记录数不能大于50条
            page = orgUserService.pageByBelongOrgQueryType(pageable, orgUserBO);
        }
        Page<OrgUserResponse> responsePage = page.map(e -> {
            OrgUserResponse response = new OrgUserResponse();
            BeanUtils.copyProperties(e, response);
            return response;
        });
        return ResponseEntity.ok(responsePage);
    }

    /**
     * 校验日期
     *
     * @param request OrgUserComplexRequest
     */
    private void checkCreatedTimeBeginEnd(OrgUserComplexRequest request) {

        if (request.getCreatedDateBegin() != null) {
            Assert.notNull(request.getCreatedDateEnd(), "按时间范围，用户创建时间结束时间不能为空");
        }
        DateTime tempDate = new DateTime(request.getCreatedDateBegin());
        DateTime beginDate = new DateTime(tempDate.getYear(), tempDate.getMonthOfYear(), tempDate.getDayOfMonth(), 0, 0, 0);
        log.info("beginDate:{}", beginDate.toDateTime());
        tempDate = new DateTime(request.getCreatedDateEnd());
        tempDate = tempDate.plusDays(1);
        request.setCreatedDateBegin(beginDate.toDate());
        request.setCreatedDateEnd(tempDate.toDate());
        Assert.isTrue(tempDate.isAfter(beginDate), "用户创建时间的查询结束时间应该大于查询开始时间");
    }

    /**
     * 校验查询类型
     *
     * @param request OrgUserComplexRequest
     */
    private void checkQueryType(OrgUserComplexRequest request) {
        if (OrgUserModel.OrgQueryType.ID == request.getQueryType()) {
            request.setOrgCode(null);
            request.setOrgName(null);
            Assert.isTrue(request.getOrgId() != null && request.getOrgId() >= 0, "查询类型为：组织ID查询，组织ID为必填且ID大于等于0");
            return;
        }

        if (OrgUserModel.OrgQueryType.CODE == request.getQueryType()) {
            request.setOrgId(null);
            request.setOrgName(null);
            Assert.hasText(request.getOrgCode(), "查询类型为：组织Code查询，组织Code为必填");
            return;
        }

        if (OrgUserModel.OrgQueryType.NAME == request.getQueryType()) {
            request.setOrgCode(null);
            request.setOrgId(null);
            Assert.hasText(request.getOrgName(), "查询类型为：组织名称，组织名称为必填");
            Assert.isTrue(request.getRangeType() != null, "查询类型为：组织名称时，查询范围不能为空且仅支持当前节点查询");
            Assert.isTrue(request.getRangeType() == CURRENT_NODE, "查询类型为：组织名称时，查询范围仅支持当前节点查询");
        }
    }


    /**
     * 用户主要分页查询
     *
     * @param request  OrgUserRequest
     * @param pageable 分页对象
     * @return ResponseEntity<Page < OrgUserModel.OrgUserResponse>>
     */
    @ApiOperation(value = "根据组织ID分页查询用户列表主要信息")
    @Override
    public ResponseEntity<Page<OrgUserResponse>> page(OrgUserRequest request, Pageable pageable) {
        //每页记录数不能大于100条
        Assert.isTrue(pageable.getPageSize() <= 100, "每页记录数不能大于100条");
        pageable = PageRequest.of(pageable.getPageNumber(), pageable.getPageSize(), Sort.unsorted());
        //如查询范围为空，则查询范围为当前节点
        if (null == request.getRangeType()) {
            request.setRangeType(CURRENT_NODE);
        }
        log.info("OrgUserRequest.request:{},page:{}", request, pageable);
        Page<OrgUserDTO> page;
        OrgUserBO orgUserBO = new OrgUserBO();
        BeanUtils.copyProperties(request, orgUserBO);
        //查询当前节点数据
        if (CURRENT_NODE == request.getRangeType()) {
            page = this.orgUserService.pageByCurrentOrg(pageable, orgUserBO);
        } else {
            //查询当前组织节点及以下节点，每页记录数不能大于100条
            page = this.orgUserService.pageByBelongOrg(pageable, orgUserBO);
        }
        Page<OrgUserResponse> responsePage = page.map(e -> {
            OrgUserResponse response = new OrgUserResponse();
            BeanUtils.copyProperties(e, response);
            return response;
        });
        return ResponseEntity.ok(responsePage);
    }
}
