package com.xforceplus.business.tenant.controller;


import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.global.user.PreRoleApi;
import com.xforceplus.api.model.RoleModel;
import com.xforceplus.api.model.RoleModel.Request.*;
import com.xforceplus.api.model.UserModel;
import com.xforceplus.business.common.controller.AbstractController;
import com.xforceplus.business.tenant.service.PreRoleService;
import com.xforceplus.business.tenant.service.RoleService;
import com.xforceplus.constants.RoleTypeEnum;
import com.xforceplus.domain.tenant.PreRoleDTO;
import com.xforceplus.domain.validation.ValidationGroup;
import com.xforceplus.entity.Role;
import com.xforceplus.entity.User;
import com.xforceplus.tenant.security.autoscan.annotation.AuthorizedDefinition;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.utils.ObjectCheckAndExcuteUtils;
import io.geewit.data.jpa.essential.domain.PageableFactory;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Controller;
import org.springframework.util.CollectionUtils;
import org.springframework.validation.annotation.Validated;

import javax.validation.Valid;
import javax.validation.constraints.Min;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;


@SuppressWarnings("all")
@Api(value = "预置角色相关接口", description = "预置角色相关接口", tags = {"role"})
@Validated
@Controller
public class PreRoleController extends AbstractController implements PreRoleApi, com.xforceplus.api.current.user.PreRoleApi {
    private final PreRoleService preRoleService;
    private final RoleService roleService;
    private final static Logger logger = LoggerFactory.getLogger(PreRoleController.class);

    public PreRoleController(PreRoleService preRoleService, RoleService roleService) {
        this.preRoleService = preRoleService;
        this.roleService = roleService;
    }

    /**
     * 用户查询可以用的预置角色
     * @param roleId
     * @param <T>
     * @return
     */
    @ApiOperation(value = "当前用户角色信息", tags = {"role"})
    @Override
    public ResponseEntity<Role> detailInfo(long roleId) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        int res = preRoleService.isVisiable(tenantId, roleId);
        if (res <= 0) {
            throw new IllegalArgumentException("当前用户无法访问此角色信息");
        }
        return ResponseEntity.ok(roleService.findById(roleId));
    }

    @ApiOperation(value = "当前用户角色分页列表", tags = {"role"})
    @Override
    public   ResponseEntity<Page<PreRoleDTO>> currentPage(PreRoleQuery query, Pageable pageable) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        String code = Objects.isNull(query) ? null : StringUtils.isEmpty(query.getRoleCode()) ? null : query.getRoleCode();
        String name = Objects.isNull(query) ? null : StringUtils.isEmpty(query.getRoleName()) ? null : query.getRoleName();
        Page<PreRoleDTO> roles = preRoleService.queryPreRoleListByTenantId(tenantId,query.getId(), code, name, pageable);
        return ResponseEntity.ok(roles);
    }

    @ApiOperation(value = "当前用户角色批量绑定用户", tags = {"role"})
    @Override
    public ResponseEntity<String> bindUsers(long roleId, BindUsers bindUsers) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();

        //pre check user
        if(bindUsers==null || CollectionUtils.isEmpty(bindUsers.getUserIds())){
            throw new IllegalArgumentException("用户信息错误！");
        }
        if(bindUsers.getUserIds().size()>500){
            throw new IllegalArgumentException("用户数量超过 500限制！");
        }

        List<Long> userIds = preRoleService.checkUser(bindUsers.getUserIds(),  tenantId);
        List<Long> wrongUserIds = bindUsers.getUserIds().stream().filter(item -> userIds.stream().noneMatch(userId -> userId.equals(item))).collect(
                Collectors.toList());

        //pre check role
        int roleRes = preRoleService.isVisiable(tenantId, roleId);
        if (roleRes <= 0) {
            throw new IllegalArgumentException("当前用户无法访问此角色信息");
        }

        Role role = roleService.findById(roleId);
        ObjectCheckAndExcuteUtils.docheckAndExcute(role, x -> {
            return Objects.nonNull(x) && x.getType().compareTo(RoleTypeEnum.PRE.getType()) == 0;
        }, x -> x, new IllegalArgumentException("只支持预置角色的绑定"));
        roleService.bindUsers(roleId, userIds, bindUsers.isOverwrite());
        if (wrongUserIds.isEmpty()) {
            return ResponseEntity.ok("绑定成功");
        } else {
            return ResponseEntity.fail("0", "包含错误的userId:" + wrongUserIds.toString());
        }
    }

    @ApiOperation(value = "当前用户角色批量解绑用户", tags = {"role"})
    @Override
    public ResponseEntity<String> unBindUsers(long roleId, UnbindUsers bindUsers) {
        Role role = roleService.findById(roleId);
        //解绑用户 需要使用用户的所在租户，解除用户-角色关系
        Long tenant_id = UserInfoHolder.currentUser().getTenantId();
        ObjectCheckAndExcuteUtils.docheckAndExcute(role, x -> {
            return Objects.nonNull(x) && x.getType().compareTo(RoleTypeEnum.PRE.getType()) == 0;
        }, x -> x, new IllegalArgumentException("只支持预置角色的解绑"));
        roleService.unbindUsers(tenant_id, roleId, bindUsers);
        return ResponseEntity.ok("解绑成功");
    }

    /**
     * 查询当前预置角色绑定的用户列表
     * @param roleId
     * @param query
     * @param pageable
     * @return
     */
    @ApiOperation(value = "当前用户获取角色绑定用户列表", tags = {"role"})
    @Override
    public ResponseEntity<Page<User>> currentBindedRoles(Long roleId, UserModel.Request.Query query, Pageable pageable) {
        Role role = roleService.findById(roleId);
        ObjectCheckAndExcuteUtils.docheckAndExcute(role, x -> {
            return Objects.nonNull(x) && x.getType().compareTo(RoleTypeEnum.PRE.getType()) == 0;
        }, x -> x, new IllegalArgumentException("只支持预置角色的解绑"));
        long tenantId = PreRoleService.PER_TENANT_ID;
        // 查询sys_role 需要保证租户id = -1 否则无法查询到预置角色
        query.setTenantId(tenantId);
        query.setRoleId(roleId);
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        Page<User> result = userService.page(query, currentPageable);
        return ResponseEntity.ok(result);

    }

    @ApiOperation(value = "预置角色分页列表", tags = {"role"})
    @AuthorizedDefinition(resources = {"xforce:operation:PreDefineRole:view"})
    @Override
    public ResponseEntity<Page<PreRoleDTO>> page(PreRoleQuery query, Pageable pageable) {
        Page<PreRoleDTO> page = preRoleService.queryPreRoleListByAdmin(query, pageable);
        return ResponseEntity.ok(page);
    }

    @ApiOperation(value = "新增预置角色", tags = {"role"})
    @AuthorizedDefinition(resources = {"xforce:operation:PreDefineRole:save"})
    @Override
    public ResponseEntity<Role> create(PreCreate model) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Role role = preRoleService.bindServicePackageRole(model, authorizedUser.getId());
        return ResponseEntity.ok(role);
    }

    @ApiOperation(value = "修改预置角色", tags = {"role"})
    @AuthorizedDefinition(resources = {"xforce:operation:PreDefineRole:save"})
    @Override
    public ResponseEntity<Role> update(long roleId, BaseCreate model) {
        Role tmp = roleService.findById(roleId);
        //修改的角色必须是预置角色 否则提示错误
        ObjectCheckAndExcuteUtils.docheckAndExcute(tmp, x -> {
            return Objects.nonNull(x) && x.getType().compareTo(RoleTypeEnum.PRE.getType()) == 0;
        }, x -> x, new IllegalArgumentException("修改的角色类型非法"));
        model.setType(RoleTypeEnum.PRE.getType());
        model.setRoleCode(null);
        RoleModel.Request.Create create = new RoleModel.Request.Create();
        BeanUtils.copyProperties(model, create);
        Role role = roleService.update(roleId, create);
        return ResponseEntity.ok(role);
    }

    @ApiOperation(value = "预置角色启用", tags = {"role"})
    @AuthorizedDefinition(resources = {"xforce:operation:PreDefineRole:save"})
    @Override
    public ResponseEntity<String> enable(Long roleId) throws Throwable {
        preRoleService.updatePreRoleStatus(roleId, 1, x -> {
            return x.getType().compareTo(RoleTypeEnum.PRE.getType()) == 0;
        }, new IllegalArgumentException("只能启用预置角色"));
        return ResponseEntity.ok("启用成功");
    }

    @ApiOperation(value = "预置角色禁用", tags = {"role"})
    @AuthorizedDefinition(resources = {"xforce:operation:PreDefineRole:save"})
    @Override
    public ResponseEntity<String> disable(Long roleId) throws Throwable {
        preRoleService.updatePreRoleStatus(roleId, 0, x -> {
            return x.getType().compareTo(RoleTypeEnum.PRE.getType()) == 0;
        }, new IllegalArgumentException("只能禁用预置角色"));
        return ResponseEntity.ok("禁用成功");
    }

    @ApiOperation(value = "删除预置角色", tags = {"role"})
    @AuthorizedDefinition(resources = {"xforce:operation:PreDefineRole:save"})
    @Override
    public ResponseEntity<String> delete(long roleId) {
        preRoleService.delete(roleId);
        return ResponseEntity.ok("删除成功");
    }

    @ApiOperation(value = "预置角色详情", tags = {"role"})
    @AuthorizedDefinition(resources = {"xforce:operation:PreDefineRole:view"})
    @Override
    public ResponseEntity<Role> info(long roleId) {
        Role role = roleService.findById(roleId);
        return ResponseEntity.ok(role);
    }

}
