package com.xforceplus.business.tenant.controller;

import com.fasterxml.jackson.annotation.JsonView;
import com.xforceplus.api.common.response.ResponseBoolenEntity;
import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.global.user.RoleApi;
import com.xforceplus.api.model.ResourceModel;
import com.xforceplus.api.model.RoleModel.Request.*;
import com.xforceplus.api.model.UserModel;
import com.xforceplus.business.common.controller.AbstractController;
import com.xforceplus.business.excel.ExcelFile;
import com.xforceplus.business.excel.ExcelService;
import com.xforceplus.business.excel.QueryParam;
import com.xforceplus.business.excel.writer.ExcelConfigBusinessType;
import com.xforceplus.business.file.controller.vo.ImportFileRespVo;
import com.xforceplus.business.file.service.ExportFileService;
import com.xforceplus.business.file.service.ImportFileService;
import com.xforceplus.business.resource.service.ResourceService;
import com.xforceplus.business.tenant.service.PreRoleService;
import com.xforceplus.business.tenant.service.TenantPolicyService;
import com.xforceplus.constants.RoleTypeEnum;
import com.xforceplus.domain.tenant.RoleDto;
import com.xforceplus.domain.tenant.view.SysRole;
import com.xforceplus.domain.validation.ValidationGroup;
import com.xforceplus.entity.ExcelFileStore;
import com.xforceplus.entity.Resource;
import com.xforceplus.entity.Role;
import com.xforceplus.entity.User;
import com.xforceplus.exception.AccessDeniedException;
import com.xforceplus.tenant.security.autoscan.annotation.AuthorizedDefinition;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.tenant.security.core.domain.view.RoleView;
import com.xforceplus.tenant.security.core.domain.view.UserView;
import com.xforceplus.utils.ObjectCheckAndExcuteUtils;
import io.geewit.data.jpa.essential.domain.PageableFactory;
import io.geewit.data.jpa.essential.domain.SortFactory;
import io.geewit.data.jpa.essential.search.DynamicSpecifications;
import io.geewit.data.jpa.essential.search.SearchFilter;
import io.geewit.data.jpa.essential.web.servlet.Servlets;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.ArrayUtils;
import org.hibernate.validator.constraints.Range;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cloud.openfeign.SpringQueryMap;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Controller;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.context.request.WebRequest;
import org.springframework.web.multipart.MultipartFile;
import springfox.documentation.annotations.ApiIgnore;

import javax.validation.Valid;
import javax.validation.constraints.Min;
import java.util.*;
import java.util.concurrent.atomic.AtomicReference;


@SuppressWarnings("all")
@Api(value = "全局角色相关接口", description = "全局角色相关接口")
@Validated
@Controller
public class RoleController extends AbstractController implements RoleApi, com.xforceplus.api.tenant.user.RoleApi, com.xforceplus.api.current.user.RoleApi {
    private final static Logger logger = LoggerFactory.getLogger(RoleController.class);

    private final ResourceService resourceService;

    private final ExcelService excelService;

    private final ImportFileService importFileService;

    /**
     * 文件导出服务
     */
    private final ExportFileService exportFileService;

    private final TenantPolicyService tenantPolicyService;

    private final PreRoleService preRoleService;

    @Value("${xforce.global.grading-managment.enable:true}")
    private boolean globalGradingEnabled;


    public RoleController(ResourceService resourceService, ExcelService excelService, ImportFileService importFileService,
                          ExportFileService exportFileService, TenantPolicyService tenantPolicyService, PreRoleService preRoleService) {
        this.resourceService = resourceService;
        this.excelService = excelService;
        this.importFileService = importFileService;
        this.exportFileService = exportFileService;
        this.tenantPolicyService = tenantPolicyService;
        this.preRoleService = preRoleService;
    }

    /**
     * 获取用户分页列表
     *
     * @param request
     * @param pageable
     * @return
     */
    @AuthorizedDefinition(resources = {"xforce:ucenter:role:read"})
    @ApiIgnore
    @ApiOperation(value = "获取用户分页", notes = "通过分页参数，获取用户分页", tags = {"role"})
    @ResponseBody
    @RequestMapping(name = "用户分页列表查询", value = {"/api/global/roles/query"}, method = RequestMethod.GET)
    public Page<Role> page(WebRequest request, @ApiParam(value = "pageable") Pageable pageable) {
        Map<String, Object> searchParams = Servlets.getParametersStartingWith(request, "search_");
        Collection<SearchFilter> filters = SearchFilter.parse(searchParams);
        Specification<Role> specification = DynamicSpecifications.bySearchFilter(filters);
        Page<Role> page = roleService.page(specification, pageable);
        return page;
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role:read"})
    @ApiOperation(value = "获取系统全局角色分页列表", notes = "通过分页参数，获取系统全局角色分页列表", tags = {"role"})
    @JsonView(RoleView.RoleInfo.class)
    @Override
    public ResponseEntity<Page<Role>> page(@ApiParam(value = "request") Query query,
                                           @ApiParam(value = "pageable") Pageable pageable) {
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        if (query.getType() == null) {
            query.setType(0);
        }
        Page<Role> result = roleService.page(query, currentPageable);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role:read"})
    @ApiIgnore
    @ApiOperation(value = "获取全局角色列表", notes = "通过参数，获取全局角色列表", tags = {"role"})
    @JsonView(RoleView.RoleInfo.class)
    @ResponseBody
    @RequestMapping(name = "全局角色列表", value = RoleApi.Path.LIST, method = RequestMethod.GET)
    public ResponseEntity<List<Role>> list(@ApiParam(value = "request") Query query,
                                           @ApiParam(value = "sort") Sort sort) {
        Sort currentSort = SortFactory.ofDefaultSort(sort, Sort.by(Sort.Direction.DESC, "id"));
        if (query.getType() == null) {
            query.setType(0);
        }
        List<Role> result = roleService.list(query, currentSort);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role:save"})
    @ApiOperation(value = "新增全局角色", tags = {"role"})
    @Override
    public ResponseEntity<Role> create(@ApiParam(value = "model", required = true) Create model) {
        ObjectCheckAndExcuteUtils.docheckAndExcute(model.getType(), x->{return Objects.nonNull(x) && x.compareTo(RoleTypeEnum.PRE.getType()) <0;}, x -> x, new IllegalArgumentException("新增角色的类型不合法"));
        Role result = roleService.create(model);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role:save"})
    @ApiOperation(value = "更新全局角色", tags = {"role"})
    @Override
    public ResponseEntity<Role> update(@ApiParam(value = "roleId", required = true) long roleId,
                                       @ApiParam(value = "model", required = true) Update model)   {
        Role result = roleService.update(roleId, model);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role:save"})
    @ApiOperation(value = "更新全局角色状态", tags = {"role"})
    @Override
    public ResponseEntity<String> updateStatus(long roleId, int status) {
        roleService.updateStatus(roleId, status);
        return ResponseEntity.ok("更新成功");
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role:save"})
    @ApiIgnore
    @ApiOperation(value = "启用全局角色", tags = {"role"})
    @ResponseBody
    @RequestMapping(name = "启用全局角色", value = RoleApi.Path.ENABLE, method = RequestMethod.PUT)
    public ResponseEntity<String> enable(@PathVariable("roleId") long roleId) {
        roleService.enable(roleId);
        return ResponseEntity.ok("启用成功");
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role:save"})
    @ApiIgnore
    @ApiOperation(value = "禁用全局角色", tags = {"role"})
    @ResponseBody
    @RequestMapping(name = "禁用全局角色", value = RoleApi.Path.DISABLE, method = RequestMethod.PUT)
    public ResponseEntity<String> disable(@PathVariable("roleId") long roleId) {
        roleService.disable(roleId);
        return ResponseEntity.ok("禁用成功");
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role:read"})
    @JsonView(RoleView.RoleInfo.class)
    @ApiOperation(value = "获取全局角色信息", tags = {"role"})
    @Override
    public ResponseEntity<Role> info(@ApiParam(value = "roleId", required = true) long roleId) {
        Role result = roleService.findById(roleId);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role:save"})
    @ApiOperation(value = "根据id删除全局角色", tags = {"role"})
    @Override
    public ResponseEntity<String> delete(@ApiParam(value = "roleId", required = true) long roleId) {
        roleService.deleteById(roleId);
        return ResponseEntity.ok("删除成功");
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role:read"})
    @ApiOperation(value = "全局角色用户列表", tags = {"role"})
    @Override
    @JsonView(UserView.UserInfo.class)
    public ResponseEntity<Page<User>> users(@ApiParam(value = "roleId", required = true) long roleId,
                                            Pageable pageable) {
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "id"));
        UserModel.Request.Query query = new UserModel.Request.Query();
        query.setRoleId(roleId);
        Page<User> result = userService.page(query, currentPageable);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role-resourceset:read"})
    @ApiOperation(value = "全局角色资源码列表", tags = {"role"})
    @Override
    public ResponseEntity<Page<Resource>> resources(@ApiParam(value = "roleId", required = true) long roleId,
                                                    Pageable pageable) {
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "resourceId"));
        ResourceModel.Request.Query query = new ResourceModel.Request.Query();
        query.setRoleId(roleId);
        query.setStatus(1);
        Page<Resource> result = resourceService.page(query, currentPageable);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role:users:save"})
    @ApiOperation(value = "全局角色批量绑定用户", notes = "全局角色批量绑定用户", tags = {"role"})
    @Override
    public ResponseEntity<String> bindUsers(@ApiParam(value = "roleId", required = true) long roleId,
                                            BindUsers bindUsers) {
        roleService.bindUsers(roleId, bindUsers);
        return ResponseEntity.ok("绑定成功");
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role:users:save"})
    @ApiOperation(value = "全局角色批量绑定用户(废弃)", tags = {"role"})
    @Deprecated
    @ResponseBody
    @RequestMapping(name = "全局角色批量绑定用户(废弃)", value = "/api/global/roles/{roleId}/users/legacy", method = RequestMethod.POST)
    public ResponseEntity<String> legacyBindRoles(@PathVariable("roleId") long roleId, @RequestBody List<Long> userIds) {
        roleService.bindUsers(roleId, userIds, true);
        return ResponseEntity.ok("绑定成功");
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role-resourceset:save"})
    @ApiOperation(value = "全局角色批量绑定功能集", notes = "全局角色批量绑定功能集", tags = {"role"})
    @Override
    public ResponseEntity<String> bindResourceSets(@ApiParam(value = "roleId", required = true) long roleId,
                                                   BindResourceSets bindResourceSets) {
        roleService.bindResourceSets(roleId, bindResourceSets);
        return ResponseEntity.ok("绑定成功");
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role:read"})
    @JsonView(RoleView.RoleInfo.class)
    @ApiOperation(value = "获取全局角色分页", notes = "通过分页参数，获取全局角色分页", tags = {"role"})
    @Override
    public ResponseEntity<Page<Role>> page(@ApiParam(value = "tenantKey", required = true) String tenantKey,
                                           @ApiParam(value = "request") Query query,
                                           @ApiParam(value = "pageable") Pageable pageable) {
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));

        long tenantId = super.toTenantId(tenantKey, query.getByTenantCode());
        query.setTenantId(tenantId);
        query.setType(0);
        Page<Role> result = roleService.page(query, currentPageable);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role:read"})
    @ApiIgnore
    @ApiOperation(value = "获取全局角色列表", notes = "通过参数，获取全局角色列表", tags = {"role"})
    @ResponseBody
    @RequestMapping(name = "全局角色列表", value = com.xforceplus.api.tenant.user.RoleApi.Path.LIST, method = RequestMethod.GET)
    public ResponseEntity<List<Role>> list(@ApiParam(value = "tenantKey", required = true) @PathVariable("tenantKey") String tenantKey,
                                           @ApiParam(value = "request") @SpringQueryMap Query query,
                                           @ApiParam(value = "sort") Sort sort) {
        Sort currentSort = SortFactory.ofDefaultSort(sort, Sort.by(Sort.Direction.DESC, "createTime"));
        long tenantId = super.toTenantId(tenantKey, query.getByTenantCode());
        query.setTenantId(tenantId);
        query.setType(0);
        List<Role> result = roleService.list(query, currentSort);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role:save"})
    @ApiOperation(value = "租户新增全局角色", tags = {"role"})
    @Override
    public ResponseEntity<Role> create(@ApiParam(value = "tenantKey", required = true) String tenantKey,
                                       @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                       @ApiParam(value = "model", required = true) Create model)  {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        model.setTenantId(tenantId);
        Role result;
        if (model.getId() != null && model.getId() > 0) {
            result = roleService.updateByTenantId(tenantId, model.getId(), model);
        } else {
            ObjectCheckAndExcuteUtils.docheckAndExcute(model.getType(), x->{return Objects.nonNull(x) && x.compareTo(RoleTypeEnum.PRE.getType()) <0;}, x -> x, new IllegalArgumentException("新增角色的类型不合法"));
            result = roleService.create(model);
        }
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role:save"})
    @ApiOperation(value = "批量创建全局角色2.0-2", tags = {"role"})
    @Override
    public <R extends RoleDto> ResponseEntity<List<R>> batchCreate(String tenantKey,
                                                                   @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                                   List<Save> models)   {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        List<R> roles = new ArrayList<>();
        for (Save save : models) {
            try {
                save.setTenantId(tenantId);
                ObjectCheckAndExcuteUtils.docheckAndExcute(save.getType(), x->{return Objects.nonNull(x) && x.compareTo(RoleTypeEnum.PRE.getType()) <0;}, x -> x, new IllegalArgumentException("新增角色的类型不合法"));
                roles.add((R) roleService.save(save));
            } catch (Exception e) {
                logger.error("新增角色异常batchCreate：{}", e);
            }
        }
        return ResponseEntity.ok(roles);
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role-resourceset:save"})
    @ApiOperation(value = "批量绑定全局角色和功能集合2.0-6", tags = {"role"})
    @Override
    public ResponseBoolenEntity<Boolean, BindResourceSet> bindResourceSets(String tenantKey,
                                                                           @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                                           List<BindResourceSet> bindResourceSets) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        return roleService.bindResourceSets(bindResourceSets);
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role:save"})
    @ApiOperation(value = "租户更新全局角色", tags = {"role"})
    @Override
    public ResponseEntity<Role> update(@ApiParam(value = "tenantKey", required = true) String tenantKey,
                                       @ApiParam(value = "roleKey", required = true) String roleKey,
                                       @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                       @ApiParam(value = "byRoleCode", required = false) Boolean byRoleCode,
                                       @ApiParam(value = "model", required = true) Update model)   {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long roleId = super.toRoleId(tenantId, roleKey, byRoleCode);

        Role result = roleService.updateByTenantId(tenantId, roleId, model);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role:save"})
    @ApiOperation(value = "租户更新全局角色状态", tags = {"role"})
    @Override
    public ResponseEntity<String> updateStatus(@ApiParam(value = "tenantKey", required = true) String tenantKey,
                                               @ApiParam(value = "roleKey", required = true) String roleKey,
                                               @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                               @ApiParam(value = "byRoleCode", required = false) Boolean byRoleCode,
                                               @ApiParam(value = "status", required = true) int status) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long roleId = super.toRoleId(tenantId, roleKey, byRoleCode);
        roleService.updateStatus(tenantId, roleId, status);
        return ResponseEntity.ok("更新成功");
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role:read"})
    @JsonView(RoleView.RoleInfo.class)
    @ApiOperation(value = "根据id获取全局角色信息", tags = {"role"})
    @Override
    public ResponseEntity<Role> info(@ApiParam(value = "tenantKey", required = true) String tenantKey,
                                     @ApiParam(value = "roleKey", required = true) String roleKey,
                                     @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                     @ApiParam(value = "byRoleCode", required = false) Boolean byRoleCode) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long roleId = super.toRoleId(tenantId, roleKey, byRoleCode);
        Role result = roleService.findByTenantId(tenantId, roleId);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role:save"})
    @ApiOperation(value = "租户根据id删除全局角色", tags = {"role"})
    @Override
    public ResponseEntity<String> delete(@ApiParam(value = "tenantKey", required = true) String tenantKey,
                                         @ApiParam(value = "roleKey", required = true) String roleKey,
                                         @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                         @ApiParam(value = "byRoleCode", required = false) Boolean byRoleCode) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long roleId = super.toRoleId(tenantId, roleKey, byRoleCode);
        roleService.deleteByTenantIdAndId(tenantId, roleId);
        return ResponseEntity.ok("删除成功");
    }

    @ApiOperation(value = "租户全局角色用户列表", tags = {"role"})
    @Override
    public ResponseEntity<Page<User>> users(
            @ApiParam(value = "tenantKey", required = true) String tenantKey,
            @ApiParam(value = "roleKey", required = true) String roleKey,
            UserModel.Request.Query query,
            Pageable pageable) {
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        long tenantId = super.toTenantId(tenantKey, query.getByTenantCode());
        long roleId = super.toRoleId(tenantId, roleKey, query.getByRoleCode());
        query.setTenantId(tenantId);
        query.setRoleId(roleId);
        Page<User> result = userService.page(query, currentPageable);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role-resourceset:read"})
    @ApiOperation(value = "租户全局角色资源码列表", tags = {"role"})
    @Override
    public ResponseEntity<Page<Resource>> resources(String tenantKey,
                                                    String roleKey,
                                                    @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                    @ApiParam(value = "byRoleCode", required = false) Boolean byRoleCode,
                                                    Pageable pageable) {
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long roleId = super.toRoleId(tenantId, roleKey, byRoleCode);
        ResourceModel.Request.Query query = new ResourceModel.Request.Query();
        query.setTenantId(tenantId);
        query.setRoleId(roleId);
        //添加状态为1，
        query.setStatus(1);
        Page<Resource> result = (Page<Resource>) resourceService.page(query, currentPageable);
        return ResponseEntity.ok(result);
    }


    @ApiOperation(value = "租户角色绑定用户列表", tags = {"role"})
    @AuthorizedDefinition(resources = {"xforce:ucenter:role:save"})
    @Override
    public ResponseEntity<String> bindUsers(String tenantKey,
                                            String roleKey,
                                            @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                            @ApiParam(value = "byRoleCode", required = false) Boolean byRoleCode,
                                            BindUsers bindUsers) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long roleId = super.toRoleId(tenantId, roleKey, byRoleCode);
        roleService.bindUsers(tenantId, roleId, bindUsers, 0);
        return ResponseEntity.ok("绑定成功");
    }

    @ApiOperation(value = "全局角色批量绑定用户(废弃)", tags = {"role"})
    @AuthorizedDefinition(resources = {"xforce:ucenter:role:save"})
    @Deprecated
    @ResponseBody
    @RequestMapping(name = "全局角色批量绑定用户(废弃)", value = "/api/{tenantKey}/roles/{roleKey}/users/legacy", method = RequestMethod.POST)
    public ResponseEntity<String> legacyBindUsers(@PathVariable("tenantKey") String tenantKey,
                                                  @PathVariable("roleKey") String roleKey,
                                                  @RequestParam(value = "byTenantCode", required = false) @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                  @RequestParam(value = "byRoleCode", required = false) @ApiParam(value = "byRoleCode", required = false) Boolean byRoleCode,
                                                  @RequestBody List<Long> userIds) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long roleId = super.toRoleId(tenantId, roleKey, byRoleCode);
        roleService.bindUsers(tenantId, roleId, userIds, false, 0);
        return ResponseEntity.ok("绑定成功");
    }

    @ApiOperation(value = "租户角色反绑定用户列表", tags = {"role"})
    @AuthorizedDefinition(resources = {"xforce:ucenter:role:save"})
    @Override
    public ResponseEntity<String> unbindUsers(String tenantKey,
                                              String roleKey,
                                              @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                              @ApiParam(value = "byRoleCode", required = false) Boolean byRoleCode,
                                              UnbindUsers unbindUsers) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long roleId = super.toRoleId(tenantId, roleKey, byRoleCode);
        roleService.unbindUsers(tenantId, roleId, unbindUsers);
        return ResponseEntity.ok("反绑定成功");
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role:save"})
    @ApiOperation(value = "创建系统全局角色", tags = {"role"})
    @Override
    public ResponseEntity<String> sysSave(SysCreate model)  {
        ObjectCheckAndExcuteUtils.docheckAndExcute(model.getType(), x->{return Objects.nonNull(x) && x.compareTo(RoleTypeEnum.PRE.getType()) <0;}, x -> x, new IllegalArgumentException("新增角色的类型不合法"));
        roleService.sysSave(model);
        return ResponseEntity.ok();
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role:save"})
    @ApiOperation(value = "更新系统全局角色", tags = {"role"})
    @Override
    public ResponseEntity<String> sysUpdate(long roleId, SysCreate model) {
        roleService.sysUpdate(roleId, model);
        return ResponseEntity.ok();
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role:read"})
    @JsonView(SysRole.class)
    @ApiOperation(value = "系统全局角色信息", tags = {"role"})
    @Override
    public ResponseEntity<Role> sysInfo(long roleId) {
        Role result = roleService.sysInfo(roleId);
        return ResponseEntity.ok(result);
    }

    /**
     * 批量导入关联账号和全局角色.
     *
     * @return
     */
    @AuthorizedDefinition(resources = {"xforce:ucenter:role:save"})
    @ApiOperation(value = "批量导入关联账号和全局角色", tags = {"role"})
    @Override
    public ResponseEntity<Long> batchImportBind(String tenantKey,
                                                @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                MultipartFile file) {
        Long tenantId = super.toTenantId(tenantKey, byTenantCode);
        return excelService.batchImportRoles(tenantId, file);
    }

    @AuthorizedDefinition(resources = {"xforce:ucenter:role:save"})
    @ApiOperation(value = "导入全局角色租户功能集", tags = {"role"})
    @ResponseBody
    @RequestMapping(name = "导入系统全局角色租户功能集", value = RoleApi.Path.SYS_UPLOAD, method = RequestMethod.POST)
    public ResponseEntity<Long> importSysData(@PathVariable("roleId") @Valid @Min(1) long roleId,
                                              @RequestParam(name = "file", required = false) MultipartFile file) {

        return roleService.importSysData(roleId, file);

    }

    @ApiOperation(value = "当前用户的全局角色分页", tags = {"role"})
    @Override
    public ResponseEntity<Page<Role>> currentPage(Query query, Pageable pageable) {
        //TODO 添加查询预置角色的逻辑
        AtomicReference<Pageable> currentPageable = new AtomicReference<>(PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime")));
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();

        boolean tenantGradingEnabled = tenantPolicyService.tenantGradingManagementEnabled(authorizedUser.getTenantId());
        if (globalGradingEnabled && tenantGradingEnabled && !authorizedUser.isAdmin()) {
            throw new AccessDeniedException("操作受限，请确保有权限操作！");
        }

        query.setTenantId(authorizedUser.getTenantId());
        if (query.getExcludeBoundCurrent() != null && query.getExcludeBoundCurrent()) {
            if (query.getUserId() == null) {
                String message = "缺少参数: userId";
                throw new IllegalArgumentException(message);
            }
        }
        /**
         * 查询预置角色前置条件
         */
        if (Objects.isNull(query.getType()) || query.getType().equals(RoleTypeEnum.PRE.getType())) {
            List<Long> preRoleids = preRoleService.queryUserPreRole(authorizedUser.getTenantId(), query.getUserId(), query.getExcludeBoundCurrent());
            ObjectCheckAndExcuteUtils.docheckAndExcute(preRoleids, org.apache.commons.collections.CollectionUtils::isNotEmpty, x -> {   query.setOrRoleIds(x); currentPageable.set(PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "type","createTime") )); return Boolean.TRUE;} );
        }
        query.setType(0);
        Page<Role> page = roleService.page(query, currentPageable.get());
        return ResponseEntity.ok(page);
    }

    @ApiOperation(value = "当前用户新建角色", tags = {"role"})
    @JsonView(RoleView.RoleInfo.class)
    @Override
    public ResponseEntity<Role> currentCreate(Create model)   {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();

        boolean tenantGradingEnabled = tenantPolicyService.tenantGradingManagementEnabled(authorizedUser.getTenantId());
        if (globalGradingEnabled && tenantGradingEnabled && !authorizedUser.isAdmin()) {
            throw new AccessDeniedException("操作受限，请确保有权限操作！");
        }

        model.setTenantId(authorizedUser.getTenantId());
        Role result;
        if (model.getId() != null && model.getId() > 0) {
            result = roleService.updateByTenantId(authorizedUser.getTenantId(), model.getId(), model);
        } else {
            result = roleService.create(model);
        }
        return ResponseEntity.ok(result);
    }

    @ApiOperation(value = "当前用户角色详情", tags = {"role"})
    @JsonView(RoleView.RoleInfo.class)
    @Override
    public ResponseEntity<Role> currentInfo(Long roleId) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        Role result = roleService.findByTenantId(tenantId, roleId);
        return ResponseEntity.ok(result);
    }

    @ApiOperation(value = "当前用户更新角色", tags = {"role"})
    @JsonView(RoleView.RoleInfo.class)
    @Override
    public ResponseEntity<Role> currentUpdate(Long roleId, @Validated(ValidationGroup.OnCurrentTenantCreate.class) Update model)   {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();

        boolean tenantGradingEnabled = tenantPolicyService.tenantGradingManagementEnabled(authorizedUser.getTenantId());
        if (globalGradingEnabled && tenantGradingEnabled && !authorizedUser.isAdmin()) {
            throw new AccessDeniedException("操作受限，请确保有权限操作！");
        }

        long tenantId = authorizedUser.getTenantId();
        Role result = roleService.updateByTenantId(tenantId, roleId, model);
        return ResponseEntity.ok(result);
    }

    @ApiOperation(value = "当前用户删除角色", tags = {"role"})
    @Override
    public ResponseEntity<String> currentDelete(Long roleId) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();

        boolean tenantGradingEnabled = tenantPolicyService.tenantGradingManagementEnabled(authorizedUser.getTenantId());
        if (globalGradingEnabled && tenantGradingEnabled && !authorizedUser.isAdmin()) {
            throw new AccessDeniedException("操作受限，请确保有权限操作！");
        }

        long tenantId = authorizedUser.getTenantId();
        roleService.deleteByTenantIdAndId(tenantId, roleId);
        return ResponseEntity.ok("删除成功");
    }

    @ApiOperation(value = "当前用户获取角色绑定用户列表", tags = {"role"})
    @JsonView(UserView.UserInfo.class)
    @Override
    public ResponseEntity<Page<User>> currentBindedRoles(Long roleId,
                                                         UserModel.Request.Query query,
                                                         Pageable pageable) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();

        boolean tenantGradingEnabled = tenantPolicyService.tenantGradingManagementEnabled(authorizedUser.getTenantId());
        if (globalGradingEnabled && tenantGradingEnabled && !authorizedUser.isAdmin()) {
            throw new AccessDeniedException("操作受限，请确保有权限操作！");
        }

        long tenantId = authorizedUser.getTenantId();
        query.setTenantId(tenantId);
        query.setRoleId(roleId);
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        Page<User> result = userService.page(query, currentPageable);
        return ResponseEntity.ok(result);
    }

    @ApiOperation(value = "角色绑定用户列表", tags = {"role"})
    @Override
    public ResponseEntity<String> currentBindRoles(Long roleId, BindUsers bindUsers) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();

        boolean tenantGradingEnabled = tenantPolicyService.tenantGradingManagementEnabled(authorizedUser.getTenantId());
        if (globalGradingEnabled && tenantGradingEnabled && !authorizedUser.isAdmin()) {
            throw new AccessDeniedException("操作受限，请确保有权限操作！");
        }

        long tenantId = authorizedUser.getTenantId();
        roleService.bindUsers(tenantId, roleId, bindUsers, 0);
        return ResponseEntity.ok("绑定成功");
    }

    @ApiOperation(value = "当前用户角色反绑定用户列表", tags = {"role"})
    @Override
    public ResponseEntity<String> currentUnbindRoles(Long roleId, UnbindUsers unbindUsers) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();

        boolean tenantGradingEnabled = tenantPolicyService.tenantGradingManagementEnabled(authorizedUser.getTenantId());
        if (globalGradingEnabled && tenantGradingEnabled && !authorizedUser.isAdmin()) {
            throw new AccessDeniedException("操作受限，请确保有权限操作！");
        }

        long tenantId = authorizedUser.getTenantId();
        roleService.unbindUsers(tenantId, roleId, unbindUsers);
        return ResponseEntity.ok("反绑定成功");
    }

    @ApiOperation(value = "当前用户角色启用", tags = {"role"})
    @Override
    public ResponseEntity<String> currentEnable(Long roleId) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();

        boolean tenantGradingEnabled = tenantPolicyService.tenantGradingManagementEnabled(authorizedUser.getTenantId());
        if (globalGradingEnabled && tenantGradingEnabled && !authorizedUser.isAdmin()) {
            throw new AccessDeniedException("操作受限，请确保有权限操作！");
        }

        long tenantId = authorizedUser.getTenantId();
        roleService.updateStatus(tenantId, roleId, 1);
        return ResponseEntity.ok("更新成功");
    }

    @ApiOperation(value = "当前用户角色禁用", tags = {"role"})
    @Override
    public ResponseEntity<String> currentDisable(Long roleId) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();

        boolean tenantGradingEnabled = tenantPolicyService.tenantGradingManagementEnabled(authorizedUser.getTenantId());
        if (globalGradingEnabled && tenantGradingEnabled && !authorizedUser.isAdmin()) {
            throw new AccessDeniedException("操作受限，请确保有权限操作！");
        }

        long tenantId = authorizedUser.getTenantId();
        roleService.updateStatus(tenantId, roleId, 0);
        return ResponseEntity.ok("更新成功");
    }

    @ApiOperation(value = "当前用户角色异步导出", tags = {"role"})
    @Override
    public ResponseEntity<ImportFileRespVo> exportRoles(@RequestBody Export query) {
        if (ArrayUtils.isEmpty(query.getSheets())) {
            throw new IllegalArgumentException("请先选择导出模板");
        }
        //异步执行
        QueryParam queryParam = QueryParam.builder()
                .param(ExcelFile.PARAMS_QUERY, query)
                .build();
        ExcelConfigBusinessType businessType = ExcelConfigBusinessType.ROLE_EXCEL_FILE;
        ExcelFileStore excelFileStore = this.exportFileService.asyncExcelExport(queryParam.params(), businessType);
        ImportFileRespVo respVo = new ImportFileRespVo();
        respVo.setId(excelFileStore.getId());
        respVo.setBusinessType(businessType.getBusinessName());
        //检查ExcelFile文件是否正确
        return ResponseEntity.ok(respVo);
    }

    @ApiOperation(value = "当前用户全局角色批量绑定用户(废弃)")
    @Deprecated
    @ResponseBody
    @RequestMapping(name = "当前用户全局角色批量绑定用户(废弃)", value = "/api/current/roles/{roleId}/users/legacy", method = RequestMethod.POST)
    public ResponseEntity<String> currentLegacyBindRoles(@PathVariable("roleId") long roleId, @RequestBody List<Long> userIds) {
        long tenantId = UserInfoHolder.get().getTenantId();
        roleService.bindUsers(tenantId, roleId, userIds, true, 0);
        return ResponseEntity.ok("绑定成功");
    }


    @AuthorizedDefinition(resources = {"xforce:ucenter:role:save"})
    @ApiOperation(value = "全局角色批量绑定账号导入excel")
    @ResponseBody
    @RequestMapping(name = "全局角色批量绑定账号导入excel", value = com.xforceplus.api.current.user.RoleApi.Path.ROLE_BIND_ACCOUNT_IMPORT, produces = {"application/json"}, method = RequestMethod.POST)
    public ResponseEntity<ImportFileRespVo> batchImportBind(@RequestParam("file") MultipartFile file) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();

        boolean tenantGradingEnabled = tenantPolicyService.tenantGradingManagementEnabled(authorizedUser.getTenantId());
        if (globalGradingEnabled && tenantGradingEnabled && !authorizedUser.isAdmin()) {
            throw new AccessDeniedException("操作受限，请确保有权限操作！");
        }

        Long tenantId = authorizedUser.getTenantId();
        Long userId = authorizedUser.getId();

        String businessType = ExcelConfigBusinessType.ROLE_BIND_ACCOUNT_IMPORT.name();
        //保存数据和上传文件，并转为异步处理
        ExcelFileStore excelFileStore = this.importFileService.createAndSaveFile(file, businessType, userId, tenantId);

        ImportFileRespVo respVo = new ImportFileRespVo();
        respVo.setBusinessType(businessType);
        respVo.setFileId(excelFileStore.getSourceFileId());
        respVo.setId(excelFileStore.getId());
        //检查ExcelFile文件是否正确
        return ResponseEntity.ok(respVo);
    }
}
