package com.xforceplus.business.tenant.controller;

import com.fasterxml.jackson.annotation.JsonView;
import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.global.tenant.TenantApi;
import com.xforceplus.api.model.ServicePackageModel;
import com.xforceplus.api.model.TenantModel.Request.Create;
import com.xforceplus.api.model.TenantModel.Request.Policies;
import com.xforceplus.api.model.TenantModel.Request.Query;
import com.xforceplus.api.model.TenantModel.Request.Save;
import com.xforceplus.api.model.TenantModel.Response.CreateResult;
import com.xforceplus.business.excel.ExcelFile;
import com.xforceplus.business.excel.QueryParam;
import com.xforceplus.business.excel.writer.ExcelConfigBusinessType;
import com.xforceplus.business.file.controller.vo.ImportFileRespVo;
import com.xforceplus.business.file.service.ExportFileService;
import com.xforceplus.business.resource.service.ServicePackageService;
import com.xforceplus.business.tenant.service.*;
import com.xforceplus.domain.account.AccountDto;
import com.xforceplus.domain.company.CompanyDto;
import com.xforceplus.domain.org.OrgDto;
import com.xforceplus.domain.tenant.RoleDto;
import com.xforceplus.domain.tenant.TenantDto;
import com.xforceplus.domain.tenant.TenantExtensionDto;
import com.xforceplus.domain.tenant.TenantManagerDto;
import com.xforceplus.domain.user.UserDto;
import com.xforceplus.entity.*;
import com.xforceplus.tenant.security.autoscan.annotation.AuthorizedDefinition;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import io.geewit.core.jackson.view.View;
import io.geewit.data.jpa.envers.domain.ComparedRevision;
import io.geewit.data.jpa.essential.domain.PageableFactory;
import io.geewit.data.jpa.essential.domain.SortFactory;
import io.geewit.data.jpa.essential.search.DynamicSpecifications;
import io.geewit.data.jpa.essential.search.SearchFilter;
import io.geewit.data.jpa.essential.web.servlet.Servlets;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Controller;
import org.springframework.util.CollectionUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.context.request.WebRequest;
import springfox.documentation.annotations.ApiIgnore;

import javax.validation.Valid;
import javax.validation.constraints.Min;
import java.util.*;

import static com.xforceplus.api.utils.Separator.COMMA;

@SuppressWarnings("all")
@Api(value = "租户相关接口", description = "租户相关接口")
@Validated
@Controller
public class TenantController implements TenantApi, com.xforceplus.api.current.tenant.TenantApi {
    private final static Logger logger = LoggerFactory.getLogger(TenantController.class);

    private final TenantService tenantService;

    private final ServicePackageService packageService;

    private final TenantExtensionService tenantExtensionService;

    private final TenantPolicyService tenantPolicyService;

    private final TenantRelationService tenantRelationService;

    private final UserService userService;
    /**
     * 文件导出服务
     */
    private final ExportFileService exportFileService;


    public TenantController(TenantService tenantService, ServicePackageService packageService, TenantExtensionService tenantExtensionService,
                            TenantPolicyService tenantPolicyService, ExportFileService exportFileService, TenantRelationService tenantRelationService, UserService userService) {
        this.tenantService = tenantService;
        this.packageService = packageService;
        this.tenantExtensionService = tenantExtensionService;
        this.tenantPolicyService = tenantPolicyService;
        this.exportFileService = exportFileService;
        this.tenantRelationService = tenantRelationService;
        this.userService = userService;
    }

    /**
     * 获取租户分页列表
     *
     * @param request
     * @param pageable
     * @return
     */
    @AuthorizedDefinition(resources = {"xforce:business:tenant:read"})
    @ApiIgnore
    @ApiOperation(value = "获取租户分页列表", notes = "通过分页参数，获取租户分页")
    @ResponseBody
    @RequestMapping(name = "租户分页列表查询", value = {"/api/global/tenants/query"}, method = RequestMethod.GET)
    public Page<Tenant> page(WebRequest request, @ApiParam(value = "pageable") Pageable pageable) {
        Map<String, Object> searchParams = Servlets.getParametersStartingWith(request, "search_");
        Collection<SearchFilter> filters = SearchFilter.parse(searchParams);
        Specification<Tenant> specification = DynamicSpecifications.bySearchFilter(filters);
        Page<Tenant> page = tenantService.page(specification, pageable);
        return page;
    }

    @AuthorizedDefinition(resources = {"xforce:business:tenant:read"})
    @ApiOperation(value = "获取租户分页列表", notes = "通过参数，获取租户分页")
    @Override
    public ResponseEntity<Page<Tenant>> page(@ApiParam(value = "request", required = true) Query query,
                                             @ApiParam(value = "pageable") Pageable pageable) {
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        Page<Tenant> result = tenantService.page(query, currentPageable);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:business:tenant:read"})
    @ApiIgnore
    @ResponseBody
    @RequestMapping(name = "获取租户列表", value = TenantApi.Path.LIST, method = RequestMethod.GET)
    public ResponseEntity<List<Tenant>> list(Query query, Sort sort) {
        Sort currentSort = SortFactory.ofDefaultSort(sort, Sort.by(Sort.Direction.DESC, "createTime"));
        List<Tenant> result = tenantService.list(query, currentSort);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:business:tenant:save"})
    @ApiOperation(value = "新增租户")
    @Override
    public <T extends TenantDto, O extends OrgDto<O>, C extends CompanyDto<O>, U extends UserDto<O, R, A>, R extends RoleDto, A extends AccountDto> ResponseEntity<CreateResult<T, O, C, U, R, A>> create(@ApiParam(value = "tenant", required = true) Create model) {
        CreateResult<T, O, C, U, R, A> result = tenantService.create(model);
        UserService.removeCachedRoleOrgUserRelsThreadLocal();
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:business:tenant:save"})
    @ApiOperation(value = "更新租户信息")
    @Override
    public ResponseEntity<Tenant> update(@ApiParam(value = "tenantId", required = true) long tenantId,
                                         @ApiParam(value = "tenant", required = true) Save model) {
        Tenant result = tenantService.update(tenantId, model);
        UserService.removeCachedRoleOrgUserRelsThreadLocal();
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:business:tenant:read"})
    @ApiOperation(value = "查询租户信息")
    @Override
    public ResponseEntity<Tenant> info(@ApiParam(value = "租户id", required = true) long tenantId,
                                       @ApiParam(value = "扩展信息", required = false) String withExtendParams,
                                       @ApiParam(value = "是否查询租户管理员", required = false) boolean withManager) {
        Tenant result = tenantService.findById(tenantId);
        if (StringUtils.isNotBlank(withExtendParams)) {
            String[] withExtendParamsArray = StringUtils.split(withExtendParams, COMMA);
            boolean enableExtensions = Arrays.stream(withExtendParamsArray)
                    .map(String::trim)
                    .anyMatch("extensions"::equalsIgnoreCase);
            if (enableExtensions) {
                List<TenantExtension> extensions = tenantExtensionService.getListByTenantId(tenantId);
                if (!CollectionUtils.isEmpty(extensions)) {
                    List<TenantExtensionDto> extensionDtos = new ArrayList<>(extensions.size());
                    for (TenantExtension extension : extensions) {
                        extensionDtos.add(extension);
                    }
                    result.setExtensions(extensionDtos);
                }
            }
        }

        if (withManager) {
            //获取租户管理员
            List tenantIds = new ArrayList();
            tenantIds.add(tenantId);
            List<TenantManagerDto> managerDtos = userService.findTenantManager(tenantIds);

            if (!CollectionUtils.isEmpty(managerDtos)) {
                result.setTenantManager(managerDtos.get(0));
            }
        }


        return ResponseEntity.ok(result);
    }

    @ApiOperation(value = "获取已存在租户信息")
    @Override
    public ResponseEntity<Tenant> exists(String tenantCode, String tenantName) {
        if (StringUtils.isBlank(tenantCode) && StringUtils.isBlank(tenantName)) {
            throw new IllegalArgumentException("租户code和租户名称不能同时为空");
        }
        Tenant tenant = tenantService.findByTenantCodeOrTenantName(tenantCode, tenantName);
        return ResponseEntity.ok(tenant);
    }

    @AuthorizedDefinition(resources = {"xforce:business:tenant:read"})
    @ApiOperation(value = "查询租户私钥")
    @Override
    public ResponseEntity<String> secret(@ApiParam(value = "tenantId", required = true) long tenantId) {
        String result = tenantService.findSecretByTenantId(tenantId);
        return ResponseEntity.ok(result);
    }

    @ApiOperation(value = "查询所有租户私钥")
    @Override
    public ResponseEntity<Map<Long, String>> secrets() {
        Map<Long, String> result = tenantService.allTenantSecrets();
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:business:tenant:read"})
    @ApiOperation(value = "租户的公司服务包关联的功能集分页")
    @Override
    public ResponseEntity<Page<ServicePackage>> servicePackages(long tenantId, Pageable pageable) {
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        ServicePackageModel.Request.Query query = new ServicePackageModel.Request.Query();
        query.setTenantId(tenantId);
        Page<ServicePackage> result = packageService.page(query, currentPageable);
        return ResponseEntity.ok(result);
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:business:tenant:save"})
    @ApiOperation(value = "租户绑定多个公司")
    @ResponseBody
    @RequestMapping(name = "租户绑定公司", value = TenantApi.Path.BIND_COMPANIES, method = RequestMethod.POST)
    public ResponseEntity<String> bindCompanies(@ApiParam(value = "tenantId", required = true) long tenantId,
                                                List<Long> companyIds) {
        tenantService.bindCompanies(tenantId, companyIds);
        return ResponseEntity.ok("绑定成功");
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:business:tenant:save"})
    @ApiOperation(value = "租户绑定单个公司")
    @ResponseBody
    public ResponseEntity<String> bindCompany(long tenantId, long companyId) {
        List<Long> companyIds = new ArrayList<Long>();
        companyIds.add(companyId);
        tenantService.bindCompanies(tenantId, companyIds);
        return ResponseEntity.ok("绑定成功");
    }

    @AuthorizedDefinition(resources = {"xforce:business:tenant:read"})
    @ApiOperation(value = "查询租户策略")
    @Override
    public ResponseEntity<Map<String, TenantPolicy>> policies(@ApiParam(value = "tenantId", required = true) long tenantId) {
        Map<String, TenantPolicy> result = tenantPolicyService.policies(tenantId);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:business:tenant:read"})
    @ApiOperation(value = "导出租户")
    @Override
    public ResponseEntity export(Query query, Pageable pageable) {
        if (ArrayUtils.isEmpty(query.getSheets())) {
            throw new IllegalArgumentException("请先选择导出模板");
        }
        //异步执行
        QueryParam queryParam = QueryParam.builder()
                .param(ExcelFile.PARAMS_QUERY, query)
                .param(ExcelFile.PARAMS_PAGEABLE, pageable)
                .build();
        ExcelConfigBusinessType businessType = ExcelConfigBusinessType.TENANT_EXCEL_FILE;
        ExcelFileStore excelFileStore = this.exportFileService.asyncExcelExport(queryParam.params(), businessType);
        ImportFileRespVo respVo = new ImportFileRespVo();
        respVo.setId(excelFileStore.getId());
        respVo.setBusinessType(businessType.getBusinessName());
        //检查ExcelFile文件是否正确
        return ResponseEntity.ok(respVo);
    }

    @AuthorizedDefinition(resources = {"xforce:business:tenant:save"})
    @ApiOperation(value = "保存租户策略")
    @ResponseBody
    @RequestMapping(name = "保存租户策略", value = TenantApi.Path.POLICY, method = RequestMethod.POST)
    public ResponseEntity<String> policies(@ApiParam(value = "tenantId", required = true) @PathVariable("id") long tenantId,
                                           @RequestBody Policies policies) {
        tenantPolicyService.policies(tenantId, policies);
        return ResponseEntity.ok("保存成功");
    }

    @AuthorizedDefinition(resources = {"xforce:business:tenant:save"})
    @Deprecated
    @ApiIgnore
    @ResponseBody
    @RequestMapping(name = "租户修复绑定公司", value = TenantApi.Path.FIX_COMPANIES, method = RequestMethod.GET)
    public ResponseEntity<String> fixCompanies(@ApiParam(value = "tenantId", required = true) @PathVariable("tenantId") @Valid @Min(1) long tenantId) {
        tenantService.fxCompanies(tenantId);
        return ResponseEntity.ok("绑定成功");
    }

    @ApiOperation(value = "当前账户租户列表")
    @Override
    public ResponseEntity<List<Tenant>> currentTenants() {
        List<Tenant> tenants = tenantService.currentTenants();
        return ResponseEntity.ok(tenants);
    }

    @Override
    @ApiOperation(value = "当前租户信息")
    public ResponseEntity<Tenant> currentTenantInfo() {
        return ResponseEntity.ok(tenantService.currentTenantInfo());
    }

    @ApiOperation(value = "当前租户策略")
    @Override
    public ResponseEntity<Map<String, TenantPolicy>> tenantPolicies() {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        Map<String, TenantPolicy> result = tenantPolicyService.policies(tenantId);
        return ResponseEntity.ok(result);
    }

    @ApiOperation(value = "查询跟这个租户有上下级关系的租户")
    @Override
    @JsonView({View.class})
    public ResponseEntity<List<TenantRelation>> getTenantReleationByTenantId(Long tenantId) {
        List<TenantRelation> tenantRelationDtos = tenantRelationService.findTenantRelationByTenantId(tenantId);
        return ResponseEntity.ok(tenantRelationDtos);
    }

    @Override
    @ApiOperation(value = "新增租户间关系")
    public ResponseEntity<String> addTenantReleation(Long parentTenantId, Long childTenantId) {

        tenantRelationService.addTenantReletion(parentTenantId, childTenantId);
        return ResponseEntity.ok("操作成功");
    }

    @Override
    @ApiOperation(value = "删除租户间关系")
    public ResponseEntity<String> deleteTenantReleation(Long parentTenantId, Long childTenantId) {
        tenantRelationService.deleteTenantReletion(parentTenantId, childTenantId);
        return ResponseEntity.ok("操作成功");
    }

    @Override
    public ResponseEntity<Page<ComparedRevision<Tenant, String>>> histories(Long tenantId, Pageable pageable) {
        Tenant tenant = tenantService.findById(tenantId);
        Page<ComparedRevision<Tenant, String>> page = tenantService.findTenantHistories(tenantId, pageable);
        return ResponseEntity.ok(page);
    }
}
