/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: TenantUserController.java   2020-12-29 10-03-44
 * Author: Evan
 */
package com.xforceplus.business.tenant.controller;

import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.global.user.TenantUserApi;
import com.xforceplus.api.model.user.TenantUserModel.TenantUserComplexRequest;
import com.xforceplus.api.model.user.TenantUserModel.TenantUserRequest;
import com.xforceplus.api.model.user.TenantUserModel.TenantUserResponse;
import com.xforceplus.bo.user.TenantUserBo;
import com.xforceplus.business.tenant.service.TenantUserService;
import com.xforceplus.dto.user.TenantUserDTO;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.springframework.beans.BeanUtils;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.RestController;

import javax.validation.Valid;

import static com.xforceplus.api.model.user.TenantUserModel.TenantUserQueryType.*;

/**
 * <p>
 * Title: 租户与用户Controller
 * </p>
 * <p>
 * Description: 租户与用户Controller
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-12-29 10-03-44
 * @since V1.0
 */
@Slf4j
@Api(value = "租户用户查询")
@RestController
public class TenantUserController implements TenantUserApi {

    /**
     * 租户用户服务
     */
    private final TenantUserService tenantUserService;

    public TenantUserController(TenantUserService tenantUserService) {
        this.tenantUserService = tenantUserService;
    }

    /**
     * 租户用户主要信息分页查询
     *
     * @param request  OrgUserRequest 分页请求对象
     * @param pageable 分页对象
     * @return ResponseEntity<Page < OrgUserModel.OrgUserResponse>>
     */
    @ApiOperation(value = "根据租户ID查询用户列表主要信息分页查询")
    @Override
    public ResponseEntity<Page<TenantUserResponse>> page(TenantUserRequest request, Pageable pageable) {
        Assert.isTrue(pageable.getPageSize() <= 100, "每页记录数不能大于100条");
        TenantUserBo tenantUserBo = new TenantUserBo();
        BeanUtils.copyProperties(request, tenantUserBo);
        Page<TenantUserDTO> page = this.tenantUserService.findTenantUsersByTenantId(pageable, tenantUserBo);
        Page<TenantUserResponse> responsePage = page.map(e -> {
            TenantUserResponse response = new TenantUserResponse();
            BeanUtils.copyProperties(e, response);
            return response;
        });
        return ResponseEntity.ok(responsePage);
    }

    /**
     * 租户复杂条件分页查询用户列表
     *
     * @param request  OrgUserRequest 分页请求对象
     * @param pageable 分页对象
     * @return ResponseEntity<Page < OrgUserModel.OrgUserResponse>>
     */
    @ApiOperation(value = "根据租户(ID、Code、名称)复杂条件分页查询用户列表")
    @Override
    public ResponseEntity<Page<TenantUserResponse>> pageComplex(TenantUserComplexRequest request, Pageable pageable) {
        Assert.isTrue(pageable.getPageSize() <= 100, "每页记录数不能大于100条");
        //校验查询类型
        this.checkQueryType(request);
        this.checkCreatedTimeBeginEnd(request);
        TenantUserBo tenantUserBo = new TenantUserBo();
        BeanUtils.copyProperties(request, tenantUserBo);
        Page<TenantUserDTO> page = this.tenantUserService.findTenantUsersByCondition(pageable, tenantUserBo);
        Page<TenantUserResponse> responsePage = page.map(e -> {
            TenantUserResponse response = new TenantUserResponse();
            BeanUtils.copyProperties(e, response);
            return response;
        });
        return ResponseEntity.ok(responsePage);
    }

    /**
     * 校验查询类型
     *
     * @param request TenantUserComplexRequest
     */
    private void checkQueryType(TenantUserComplexRequest request) {
        if (ID == request.getQueryType()) {
            request.setTenantName(null);
            request.setTenantCode(null);
            Assert.isTrue(request.getTenantId() != null && request.getTenantId() >= 0, "查询类型为：租户ID查询，租户ID为必填且ID大于等于0");
            return;
        }

        if (CODE == request.getQueryType()) {
            request.setTenantId(null);
            request.setTenantName(null);
            Assert.hasText(request.getTenantCode(), "查询类型为：租户Code查询时，租户Code为必填");
            return;
        }

        if (NAME == request.getQueryType()) {
            request.setTenantCode(null);
            request.setTenantId(null);
            Assert.hasText(request.getTenantName(), "查询类型为：租户名称时，租户名称为必填");
        }
    }

    /**
     * 校验日期
     *
     * @param request TenantUserComplexRequest
     */
    private void checkCreatedTimeBeginEnd(TenantUserComplexRequest request) {
        if (request.getCreatedDateBegin() != null) {
            Assert.notNull(request.getCreatedDateEnd(), "按时间范围，用户创建时间结束时间不能为空");
        }
        DateTime tempDate = new DateTime(request.getCreatedDateBegin());
        DateTime beginDate = new DateTime(tempDate.getYear(), tempDate.getMonthOfYear(), tempDate.getDayOfMonth(), 0, 0, 0);
        log.info("beginDate:{}", beginDate.toDateTime());
        tempDate = new DateTime(request.getCreatedDateEnd());
        tempDate = tempDate.plusDays(1);
        request.setCreatedDateBegin(beginDate.toDate());
        request.setCreatedDateEnd(tempDate.toDate());
        Assert.isTrue(tempDate.isAfter(beginDate), "用户创建时间的查询结束时间应该大于查询开始时间");
    }
}
