package com.xforceplus.business.tenant.controller;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.fasterxml.jackson.annotation.JsonView;
import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.global.user.UserApi;
import com.xforceplus.api.model.OrgModel;
import com.xforceplus.api.model.RoleModel;
import com.xforceplus.api.model.SettleFlowPackageModel;
import com.xforceplus.api.model.UserModel;
import com.xforceplus.api.model.UserModel.Request.*;
import com.xforceplus.business.common.controller.AbstractController;
import com.xforceplus.business.enums.BaseEnum;
import com.xforceplus.business.enums.SourceTypeEnum;
import com.xforceplus.business.excel.ExcelFile;
import com.xforceplus.business.excel.ExcelService;
import com.xforceplus.business.excel.QueryParam;
import com.xforceplus.business.excel.writer.ExcelConfigBusinessType;
import com.xforceplus.business.file.controller.vo.ImportFileRespVo;
import com.xforceplus.business.file.service.ExportFileService;
import com.xforceplus.business.file.service.FileService;
import com.xforceplus.business.file.service.ImportFileService;
import com.xforceplus.business.reponse.code.Rep;
import com.xforceplus.business.settle.service.SettleFlowService;
import com.xforceplus.business.tenant.service.FixService;
import com.xforceplus.business.tenant.service.UserService;
import com.xforceplus.domain.account.AccountDto;
import com.xforceplus.domain.org.OrgDto;
import com.xforceplus.domain.tenant.RoleDto;
import com.xforceplus.domain.user.UserDto;
import com.xforceplus.domain.user.UserExportDto;
import com.xforceplus.domain.user.view.ExtraInfo;
import com.xforceplus.entity.*;
import com.xforceplus.tenant.core.exception.UnknownException;
import com.xforceplus.tenant.security.autoscan.annotation.AuthorizedDefinition;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.tenant.security.core.domain.view.RoleView;
import com.xforceplus.tenant.security.core.domain.view.UserView;
import com.xforceplus.utils.ObjectCheckAndExcuteUtils;
import io.geewit.core.jackson.view.View;
import io.geewit.core.utils.enums.BinaryUtils;
import io.geewit.core.utils.reflection.BeanUtils;
import io.geewit.data.jpa.essential.domain.PageableFactory;
import io.geewit.data.jpa.essential.domain.SortFactory;
import io.geewit.data.jpa.essential.search.DynamicSpecifications;
import io.geewit.data.jpa.essential.search.SearchFilter;
import io.geewit.data.jpa.essential.web.servlet.Servlets;
import io.geewit.web.utils.JsonUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Controller;
import org.springframework.util.CollectionUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.context.request.WebRequest;
import org.springframework.web.multipart.MultipartFile;
import springfox.documentation.annotations.ApiIgnore;

import javax.validation.Valid;
import javax.validation.Validator;
import javax.validation.constraints.Min;
import java.util.*;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import java.util.stream.Stream;


@SuppressWarnings("all")
@Api(value = "用户相关接口", tags = "用户相关接口")
@Validated
@Controller
public class UserController extends AbstractController implements UserApi, com.xforceplus.api.tenant.user.UserApi, com.xforceplus.api.current.user.UserApi {
    private final static Logger logger = LoggerFactory.getLogger(UserController.class);

    private final ExcelService excelService;

    private final FileService fileService;

    private final FixService fixService;

    private final SettleFlowService settleFlowService;

    /**
     * 文件导出服务
     */
    private final ExportFileService exportFileService;

    /**
     * 文件导入服务
     */
    @Autowired
    private ImportFileService importFileService;

    @Autowired
    private Validator validator;

    public UserController(ExcelService excelService, FileService fileService,
                          FixService fixService, ExportFileService exportFileService,
                          SettleFlowService settleFlowService) {
        this.excelService = excelService;
        this.fileService = fileService;
        this.fixService = fixService;
        this.exportFileService = exportFileService;
        this.settleFlowService = settleFlowService;
    }

    /**
     * 获取用户分页列表
     *
     * @param request
     * @param pageable
     * @return
     */
    @ApiIgnore
    @JsonView(UserView.UserInfo.class)
    @ApiOperation(value = "获取用户分页", notes = "通过分页参数，获取用户分页")
    @ResponseBody
    @RequestMapping(name = "用户分页列表查询", value = {UserApi.Path.API_PREFIX + "/query"}, method = RequestMethod.GET)
    public Page<User> page(WebRequest request, Pageable pageable) {
        Map<String, Object> searchParams = Servlets.getParametersStartingWith(request, "search_");
        Collection<SearchFilter> filters = SearchFilter.parse(searchParams);
        Specification<User> specification = DynamicSpecifications.bySearchFilter(filters);
        Page<User> page = userService.page(specification, pageable);
        return page;
    }


    @JsonView(UserView.UserInfo.class)
    @AuthorizedDefinition(resources = {"xforce:operation:user-accounts:read"})
    @ApiOperation(value = "获取用户分页", notes = "通过分页参数，获取用户分页")
    @Override
    public ResponseEntity<Page<User>> page(Query query, Pageable pageable) {
        //id 倒序排列，避免慢查询
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "id"));
        Page<User> result = userService.page(query, currentPageable);
        return ResponseEntity.ok(result);
    }

    @JsonView(UserView.UserInfo.class)
    @ApiOperation(value = "创建用户")
    @Override
    public <U extends UserDto<O, R, A>, O extends OrgDto<O>, R extends RoleDto, A extends AccountDto> ResponseEntity<U> create(Create model) {
        SaveUserOutput<U, O, R, A> result = userService.save(model, model.isRoleOverwrite(), model.isOrgOverwrite(), model.isTagOverwrite(), model.isAppOverwrite(), model.isMergeAccount(), model.isStrict());
        UserService.removeCachedRoleOrgUserRelsThreadLocal();
        if (result.getUserResultMap().isEmpty()) {
            throw new UnknownException("未知错误");
        }
        SaveUserContext<U, O, R, A> saveUserContext = result.getUserResultMap().values().stream().findFirst().orElseThrow(() -> new UnknownException("未知错误"));
        U user = saveUserContext.getUser();
        A account = saveUserContext.getAccount();
        if (account != null) {
            user.setAccount(account);
        }
        return ResponseEntity.ok(user);
    }

    @JsonView(UserView.UserInfo.class)
    @Override
    public <U extends UserDto<O, R, A>, O extends OrgDto<O>, R extends RoleDto, A extends AccountDto> ResponseEntity<U> simpleCreate(SimpleCreate model) {
        Create save = new Create();
        BeanUtils.copyProperties(model, save);
        save.setIsRoleOverwrite(false);
        save.setIsOrgOverwrite(false);
        save.setIsTagOverwrite(false);
        save.setIsAppOverwrite(false);
        save.setIsMergeAccount(true);
        save.setIsStrict(true);
        SaveUserOutput<U, O, R, A> result = userService.save(save, save.isRoleOverwrite(), save.isOrgOverwrite(), save.isTagOverwrite(), save.isAppOverwrite(), save.isMergeAccount(), save.isStrict());
        UserService.removeCachedRoleOrgUserRelsThreadLocal();
        if (result.getUserResultMap().isEmpty()) {
            throw new UnknownException("未知错误");
        }
        SaveUserContext<U, O, R, A> saveUserContext = result.getUserResultMap().values().stream().findFirst().orElseThrow(() -> new UnknownException("未知错误"));
        U user = saveUserContext.getUser();
        A account = saveUserContext.getAccount();
        if (account != null) {
            user.setAccount(account);
        }
        return ResponseEntity.ok(user);
    }

    @ApiOperation(value = "批量创建用户和账户2.0-1")
    @Override
    public ResponseEntity<List<BatchSaveVo>> createBatch(List<BatchSave> models) {
        List<BatchSaveVo> result = userService.batchCreate(models);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:user-accounts:read"})
    @ApiOperation(value = "获取用户详情")
    @JsonView(UserView.UserInfo.class)
    @Override
    public ResponseEntity<User> info(long userId, String loginId, String modules, int extraInfoDimension) {
        User result = userService.findByLoginId(userId, loginId, modules, extraInfoDimension);
        return ResponseEntity.ok(result);
    }

    @ApiOperation(value = "根据用户名获取用户详情")
    @JsonView(UserView.UserInfo.class)
    @Override
    public ResponseEntity<User> userInfo(String username,
                                         String tenantCode,
                                         String userCode,
                                         int extraInfoDimension) {
        User result;
        if (StringUtils.isNotBlank(username)) {
            if (StringUtils.isNotBlank(tenantCode)) {
                result = userService.findByTenantCodeAndUsername(tenantCode, username, extraInfoDimension);
            } else {
                result = userService.findByUsername(username, extraInfoDimension);
            }
        } else if (StringUtils.isNotBlank(tenantCode) && StringUtils.isNotBlank(userCode)) {
            long tenantId = super.toTenantId(tenantCode, true);
            Long userId = super.toUserId(tenantId, userCode, true);
            if (userId == null) {
                String message = "找不到对应的用户(tenantCode:" + tenantCode + ", userCode:" + userCode + ")";
                throw new IllegalArgumentException(message);
            }
            result = userService.findByTenantIdAndUserIdAndLoginId(tenantId, userId, null, null, extraInfoDimension);
        } else {
            String message = "错误的参数,username:" + username + ",tenantCode=" + tenantCode + ",userCode=" + userCode;
            throw new IllegalArgumentException(message);
        }
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:user-accounts:save"})
    @JsonView(UserView.UserInfo.class)
    @ApiOperation(value = "修改用户信息")
    @Override
    public <U extends UserDto<O, R, A>, O extends OrgDto<O>, R extends RoleDto, A extends AccountDto> ResponseEntity<U> update(long userId, Update model) {
        model.setIsMergeAccount(true);
        model.setIsStrict(true);
        model.setUserId(userId);
        SaveUserOutput<U, O, R, A> saveUserOutput = userService.save(model, model.isRoleOverwrite(), model.isOrgOverwrite(), model.isTagOverwrite(), model.isAppOverwrite(), model.isMergeAccount(), model.isStrict());
        UserService.removeCachedRoleOrgUserRelsThreadLocal();
        Map<Long, SaveUserContext<U, O, R, A>> userContextMap = saveUserOutput.getUserResultMap();
        if (userContextMap.values().isEmpty()) {
            throw new IllegalArgumentException("没有返回期望的值");
        }
        SaveUserContext<U, O, R, A> saveUserContext = userContextMap.values().stream().findFirst().orElseThrow(() -> new IllegalArgumentException("没有返回期望的值"));
        U user = saveUserContext.getUser();
        A account = saveUserContext.getAccount();
        if (account != null) {
            user.setAccount(account);
        }
        if ((((User)user).getTenant() == null || ((User)user).getTenant().getCreateTime() == null) && user.getTenantId() != null && user.getTenantId() > 0) {
            Tenant tenant = tenantService.findById(user.getTenantId());
            ((User) user).setTenant(tenant);
        }
        return ResponseEntity.ok(user);
    }

    @JsonView(UserView.UserInfo.class)
    @Override
    public <U extends UserDto<O, R, A>, O extends OrgDto<O>, R extends RoleDto, A extends AccountDto> ResponseEntity<U> simpleUpdate(long userId, SimpleUpdate model) {
        Update save = new Update();
        BeanUtils.copyProperties(model, save);
        save.setUserId(userId);
        save.setIsRoleOverwrite(false);
        save.setIsOrgOverwrite(false);
        save.setIsTagOverwrite(false);
        save.setIsAppOverwrite(false);
        save.setIsMergeAccount(true);
        save.setIsStrict(true);
        SaveUserOutput<U, O, R, A> saveUserOutput = userService.save(save, save.isRoleOverwrite(), save.isOrgOverwrite(), save.isTagOverwrite(), save.isAppOverwrite(), save.isMergeAccount(), save.isStrict());
        UserService.removeCachedRoleOrgUserRelsThreadLocal();
        Map<Long, SaveUserContext<U, O, R, A>> userContextMap = saveUserOutput.getUserResultMap();
        if (userContextMap.values().isEmpty()) {
            throw new IllegalArgumentException("没有返回期望的值");
        }
        SaveUserContext<U, O, R, A> saveUserContext = userContextMap.values().stream().findFirst().orElseThrow(() -> new IllegalArgumentException("没有返回期望的值"));
        U user = saveUserContext.getUser();
        A account = saveUserContext.getAccount();
        if (account != null) {
            user.setAccount(account);
        }
        return ResponseEntity.ok(user);
    }

    @ApiOperation(value = "更新用户状态")
    @Override
    public ResponseEntity<String> updateStatus(long userId, int status) {
        userService.updateStatus(userId, status);

        return ResponseEntity.ok("更新状态成功");
    }

    @ApiOperation(value = "删除用户")
    @Override
    public ResponseEntity<String> delete(long id) {
        userService.deleteById(id);

        return ResponseEntity.ok("删除成功");
    }

    @ApiOperation(value = "获取用户组织树", notes = "获取指定用户的组织树")
    @JsonView(View.Tree.class)
    @Override
    public ResponseEntity<OrgStruct> userOrgs(long userId, String modules) {
        List<OrgStruct> orgs = orgService.findTreeByUserId(userId, modules);
        OrgStruct result = orgs.stream().findFirst().orElse(null);
        return ResponseEntity.ok(result);
    }

    @ApiOperation(value = "获取用户资源码集合", notes = "获取指定用户资源码集合")
    @Override
    public ResponseEntity<Set<String>> userResources(long userId, String loginId, String modules) {
        User user = userService.findByLoginId(userId, loginId, modules, BinaryUtils.toBinary(ExtraInfo.resources));
        Set<String> resources = user.getResourceCodes();
        return ResponseEntity.ok(resources);
    }

    @ApiOperation(value = "用户的应用资源码集合")
    @Override
    public ResponseEntity<Set<String>> appResources(long userId, long appId, String loginId, String modules) {
        User user = userService.findByLoginId(userId, loginId, modules, BinaryUtils.toBinary(ExtraInfo.resourceDetail));
        Map<Long, Set<String>> appResources = user.getAppResources();
        Set<String> resources = appResources == null ? new HashSet<>() : appResources.get(appId);
        if (resources == null) {
            resources = new HashSet<>();
        }
        return ResponseEntity.ok(resources);
    }

    @ApiOperation(value = "用户的多个应用资源码集合")
    @Override
    public ResponseEntity<Map<Long, Set<String>>> appsResources(long userId, String appIds, String loginId, String modules) {
        User user = userService.findByLoginId(userId, loginId, modules, BinaryUtils.toBinary(ExtraInfo.resourceDetail));
        Map<Long, Set<String>> appResources = user.getAppResources();
        if (appResources != null && !appResources.isEmpty()) {
            String[] appIdsArray = StringUtils.split(appIds, ",");
            Set<Long> removingIds = appResources.keySet().stream().filter(Objects::nonNull).filter(key -> !ArrayUtils.contains(appIdsArray, String.valueOf(key))).collect(Collectors.toSet());
            removingIds.forEach(appResources::remove);
        } else {
            return ResponseEntity.ok(new HashMap<>(0));
        }
        return ResponseEntity.ok(appResources);
    }

    @ApiOperation(value = "用户绑定角色列表", tags = {"role"})
    @Override
    public ResponseEntity<String> bindRoles(long userId, BindRoles bindRoles) {
        userService.bindRoles(null, null, userId, bindRoles.getAllRoleIds(), bindRoles.getRoleIds(), bindRoles.getGradingRoleIds(), bindRoles.getOrgRoleIds(), bindRoles.isOverwrite(), false, false);

        return ResponseEntity.ok("绑定成功");
    }

    @ApiOperation(value = "绑定用户角色列表(废弃)", tags = {"role"})
    @Deprecated
    @ResponseBody
    @RequestMapping(name = "批量绑定用户角色(废弃)", value = UserApi.Path.API_PREFIX + "/{userId}/roles/legacy", method = RequestMethod.POST)
    public ResponseEntity<String> legacyBindRoles(@PathVariable("userId") @Valid @Min(1) long userId, @RequestBody List<Long> roleIds) {
        userService.bindRoles(null, null, userId, null, roleIds, null, null, true, false, false);
        return ResponseEntity.ok("绑定成功");
    }

    @ApiIgnore
    @ResponseBody
    @RequestMapping(name = "清理用户缓存", value = UserApi.Path.API_PREFIX + "/{userId}/cache/clear", method = RequestMethod.DELETE)
    public ResponseEntity<String> clearCache(@PathVariable("userId") long userId) {
        userService.clearCache(userId);
        return ResponseEntity.ok("清理成功");
    }

    @ApiOperation(value = "绑定角色和人员2.0-5", notes = "绑定角色和人员", tags = {"role"})
    @Override
    public ResponseEntity<List<BindRole>> bindUsersAndRoles(List<BindRole> bindRoles) {
        return userService.bindUsersAndRoles(bindRoles);
    }

    @ApiOperation(value = "用户绑定组织列表")
    @Override
    public ResponseEntity<String> bindOrgs(long userId, BindOrgs bindOrgs) {
        userService.bindOrgs(null, userId, bindOrgs.getOrgIds(), bindOrgs.getModules(), bindOrgs.isOverwrite(), false, false);
        return ResponseEntity.ok("绑定成功");
    }

    @ApiOperation(value = "绑定组织和人员2.0-4", notes = "绑定组织和人员")
    @Override
    public ResponseEntity<List<BindOrg>> bindUserAndOrg(@RequestBody List<BindOrg> bindOrgs) {

        return userService.bindUserOrg(bindOrgs);
    }

    @ApiOperation(value = "检查用户是否有接口访问权限")
    @Override
    public ResponseEntity<Boolean> checkUriAuthz(long userId, CheckUriAuthzQuery checkUriAuthzQuery) {
        checkUriAuthzQuery.setUserId(userId);
        boolean result = userService.checkUriAuthz(checkUriAuthzQuery);
        return ResponseEntity.ok(result);
    }

    @JsonView(UserView.UserInfo.class)
    @ApiOperation(value = "租户下获取用户分页", notes = "通过分页参数，获取用户分页")
    @Override
    public ResponseEntity<Page<User>> page(String tenantKey, Query query, Pageable pageable) {
        //id 倒序排列，避免慢查询
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "id"));
        query.setTenantId(super.toTenantId(tenantKey, query.getByTenantCode()));
        Page<User> result = userService.page(query, currentPageable);
        return ResponseEntity.ok(result);
    }

    @ApiOperation(value = "获取用户列表", notes = "通过参数，获取用户列表")
    @Override
    public ResponseEntity<List<User>> list(String tenantKey, Query query, Sort sort) {
        Sort currentSort = SortFactory.ofDefaultSort(sort, Sort.by(Sort.Direction.DESC, "id"));
        query.setTenantId(super.toTenantId(tenantKey, query.getByTenantCode()));
        List<User> result = userService.list(query, currentSort);
        return ResponseEntity.ok(result);
    }

    @JsonView(UserView.UserInfo.class)
    @ApiOperation(value = "租户下创建用户")
    @Override
    public <U extends UserDto<O, R, A>, O extends OrgDto<O>, R extends RoleDto, A extends AccountDto> ResponseEntity<U> create(String tenantKey,
                                                                                                                               @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                                                                                               Create model) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        model.setTenantId(tenantId);
        userService.checkUserModel(model);
        SaveUserOutput<U, O, R, A> saveUserOutput = userService.save(tenantId, 0, Stream.of(model).collect(Collectors.toList()), model.isRoleOverwrite(), model.isOrgOverwrite(), model.isTagOverwrite(), model.isAppOverwrite(), model.isMergeAccount(), model.isStrict());
        UserService.removeCachedRoleOrgUserRelsThreadLocal();
        if (saveUserOutput.getUserResultMap().isEmpty()) {
            throw new UnknownException("未知错误");
        }
        SaveUserContext<U, O, R, A> saveUserContext = saveUserOutput.getUserResultMap().values().stream().findFirst().orElseThrow(() -> new UnknownException("未知错误"));
        U user = saveUserContext.getUser();
        A account = saveUserContext.getAccount();
        if (account != null) {
            user.setAccount(account);
        }
        return ResponseEntity.ok(user);
    }

    @JsonView(UserView.UserInfo.class)
    @Override
    public <U extends UserDto<O, R, A>, O extends OrgDto<O>, R extends RoleDto, A extends AccountDto> ResponseEntity<U> simpleCreate(String tenantKey, Boolean byTenantCode, SimpleCreate model) {
        Create save = new Create();
        BeanUtils.copyProperties(model, save);
        save.setIsRoleOverwrite(false);
        save.setIsOrgOverwrite(false);
        save.setIsTagOverwrite(false);
        save.setIsAppOverwrite(false);
        save.setIsMergeAccount(true);
        save.setIsStrict(true);
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        save.setTenantId(tenantId);
        SaveUserOutput<U, O, R, A> result = userService.save(save, save.isRoleOverwrite(), save.isOrgOverwrite(), save.isTagOverwrite(), save.isAppOverwrite(), save.isMergeAccount(), save.isStrict());
        UserService.removeCachedRoleOrgUserRelsThreadLocal();
        if (result.getUserResultMap().isEmpty()) {
            throw new UnknownException("未知错误");
        }
        SaveUserContext<U, O, R, A> saveUserContext = result.getUserResultMap().values().stream().findFirst().orElseThrow(() -> new UnknownException("未知错误"));
        U user = saveUserContext.getUser();
        A account = saveUserContext.getAccount();
        if (account != null) {
            user.setAccount(account);
        }
        return ResponseEntity.ok(user);
    }

    @ApiOperation(value = "批量导入用户")
    @Override
    public ResponseEntity<Boolean> batchImport(String tenantKey,
                                               @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                               BatchImport batchImport) {
        logger.info("batchImport.isRoleOverwrite = {}, isOrgOverwrite = {}, isTagOverwrite = {}, isAppOverwrite = {}, isOverwrite = {}, isMergeAccount = {}, isStrict = {}", batchImport.isRoleOverwrite(), batchImport.isOrgOverwrite(), batchImport.isTagOverwrite(), batchImport.isAppOverwrite(), batchImport.isOverwrite(), batchImport.isMergeAccount(), batchImport.isStrict());


        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        SaveUserOutput result = userService.save(tenantId,0, batchImport.getModels(), batchImport.isRoleOverwrite(), batchImport.isOrgOverwrite(), batchImport.isTagOverwrite(), batchImport.isAppOverwrite(), batchImport.isMergeAccount(), batchImport.isStrict());
        UserService.removeCachedRoleOrgUserRelsThreadLocal();
        return ResponseEntity.ok(true);
    }

    @Override
    public ResponseEntity<List<UserModel.Response.BatchSyncResult>> batchSync(BatchSync batchSync) {
        List<UserModel.Response.BatchSyncResult> result = userService.batchCreate(batchSync.getTenantId(), batchSync);
        UserService.removeCachedRoleOrgUserRelsThreadLocal();
        return ResponseEntity.ok(result);
    }

    @Override
    public ResponseEntity<User> changeTenant(Long userId, Long tenantId) {
        User user = userService.changeTenant(userId, tenantId);
        return ResponseEntity.ok(user);
    }

    @ApiOperation(value = "批量往组织导入用户")
    @Override
    public <U extends UserDto<O, R, A>, O extends OrgDto<O>, R extends RoleDto, A extends AccountDto> ResponseEntity<SaveUserOutput<U, O, R, A>> batchImportIntoOrg(String tenantKey,
                                                                                                                                                                    String orgKey,
                                                                                                                                                                    @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                                                                                                                                    @ApiParam(value = "byOrgCode", required = false) Boolean byOrgCode,
                                                                                                                                                                    List<Create> models) {
        if (CollectionUtils.isEmpty(models)) {
            throw new IllegalArgumentException("无效的报文");
        }
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long orgId = super.toOrgId(tenantId, orgKey, byOrgCode);
        Create firstModel = models.get(0);
        SaveUserOutput<U, O, R, A> result = userService.save(tenantId, orgId, models, firstModel.isRoleOverwrite(), firstModel.isOrgOverwrite(), firstModel.isTagOverwrite(), firstModel.isTagOverwrite(), firstModel.isMergeAccount(), firstModel.isStrict());
        UserService.removeCachedRoleOrgUserRelsThreadLocal();
        return ResponseEntity.ok(result);
    }

    @ApiOperation(value = "一站式入住批量往组织导入用户")
    @JsonView(View.Page.class)
    @Override
    public <U extends UserDto<O, R, A>, O extends OrgDto<O>, R extends RoleDto, A extends AccountDto> ResponseEntity<SaveUserOutput<U, O, R, A>> batchCreateUser(String flowId, Long companyId, List<Create> models) {

        SettleFlow flow;
        try {
            long id = Long.parseLong(flowId);
            flow = settleFlowService.findByFlowId(id);
        } catch (NumberFormatException e) {
            String message = e.getMessage() + ", value: " + flowId;
            logger.warn(message);
            flow = settleFlowService.findByFlowCode(flowId);
        }
        if (null == flow) {
            throw new IllegalArgumentException("未找到入驻页面流");
        }
        if (null == flow.getFlowId()) {
            return ResponseEntity.fail("0", "租户不存在流程");
        }
        logger.info("一站式入住:开始查询角色 {}", JsonUtils.toJson(flow));
        //判断租户角色是不是存在，存在绑定，不存在根据现有角色功能集合创建
        RoleModel.Request.Query roleQuery = new RoleModel.Request.Query();
        roleQuery.setTenantId(models.get(0).getTenantId());
        List<Role> tenantRoles = roleService.list(roleQuery, Sort.unsorted());
        tenantRoles = getRoles(models, flow, tenantRoles);
        logger.info("一站式入住:创建角色成功 {}", JsonUtils.toJson(tenantRoles));
        if (!CollectionUtils.isEmpty(tenantRoles)) {
            final List<Long> ids = tenantRoles.stream().map(Role::getId).collect(Collectors.toList());
            models.forEach(model -> {
                Set<Long> roleIds = model.getRoleIds();
                if (CollectionUtils.isEmpty(roleIds)) {
                    roleIds = new HashSet<>();
                }
                roleIds.addAll(ids);
                model.setRoleIds(roleIds);
            });
        } else {
            return ResponseEntity.fail(Rep.CommonCode.FAIL, "角色创建失败");
        }
        logger.info("一站式入住:查询组织 开始 ");
        OrgModel.Request.Query query = new OrgModel.Request.Query();
        query.setCompanyId(companyId);
        query.setTenantId(models.get(0).getTenantId());
        query.setStatus(1);
        List<OrgStruct> org = orgService.list(query, Sort.unsorted());
        if (null == org || org.size() != 1) {
            logger.error(JsonUtils.toJson(org));
            return ResponseEntity.fail(Rep.CommonCode.FAIL, "公司为空或者有多个");
        }

        userService.filterUser(models);
        logger.info("一站式入住:保存用户{}", JsonUtils.toJson(org));
        SaveUserOutput<U, O, R, A> result = userService.save(org.get(0).getTenantId(), org.get(0).getOrgId(), models, true, true, true, true, false, true);
        UserService.removeCachedRoleOrgUserRelsThreadLocal();
        return ResponseEntity.ok(result);
    }

    public List<Role> getRoles(List<Create> models, SettleFlow flow, List<Role> tenantRoles) {
        if (CollectionUtils.isEmpty(tenantRoles)) {
            logger.info("一站式入住:创建租户角色 {}", models.get(0).getTenantId());
            SettleFlowPackageModel.Request.Query packageQuery = new SettleFlowPackageModel.Request.Query();
            packageQuery.setFlowId(flow.getFlowId());

            List<SettleFlowPackage> packages = settleFlowService.packageList(packageQuery, Sort.unsorted());
            if (CollectionUtils.isEmpty(packages)) {
                throw new IllegalArgumentException("服务包没配置");
//                return ResponseEntity.fail(Rep.CommonCode.FAIL, "服务包没配置");
            }
            //一站式入住查询功能集合和资源包
            List<Long> resourceSetIds = settleFlowService.findResourceByPackage(packages);
            RoleModel.Request.Create create = new RoleModel.Request.Create();
            RoleModel.Request.BindResourceSets bindResourceSets = new RoleModel.Request.BindResourceSets();
            bindResourceSets.setOverwrite(true);
            bindResourceSets.setResourcesetIds(resourceSetIds);
            create.setBindResourceSets(bindResourceSets);
            create.setRoleCode(String.valueOf(models.get(0).getTenantId()));
            create.setRoleName("管理员");
            create.setRoleDesc("易快报");
            create.setStatus(1);
            create.setTenantId(models.get(0).getTenantId());
            Role res = roleService.create(create);
            tenantRoles = Stream.of(res).collect(Collectors.toList());
        }
        return tenantRoles;
    }

    @JsonView(UserView.UserInfo.class)
    @ApiOperation(value = "租户修改用户信息")
    @Override
    public <U extends UserDto<O, R, A>, O extends OrgDto<O>, R extends RoleDto, A extends AccountDto> ResponseEntity<U> update(String tenantKey,
                                                                                                                               String userKey,
                                                                                                                               @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                                                                                               @ApiParam(value = "byUserCode", required = false) Boolean byUserCode,
                                                                                                                               Update model) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long userId = super.toUserId(tenantId, userKey, byUserCode);
        model.setTenantId(tenantId);
        model.setUserId(userId);
        model.setIsOverwrite(true);
        model.setIsMergeAccount(true);
        model.setIsStrict(true);
        model.setSourceType(Objects.isNull(model.getSourceType())?null:(Objects.isNull(BaseEnum.getEnum(SourceTypeEnum.class,model.getSourceType())) ? SourceTypeEnum.INTERNAL.getSourceType() : model.getSourceType()));
        SaveUserOutput<U, O, R, A> saveUserOutput = userService.save(model, model.isRoleOverwrite(), model.isOrgOverwrite(), model.isTagOverwrite(), model.isAppOverwrite(), model.isMergeAccount(), model.isStrict());
        UserService.removeCachedRoleOrgUserRelsThreadLocal();
        Map<Long, SaveUserContext<U, O, R, A>> userContextMap = saveUserOutput.getUserResultMap();
        if (userContextMap.values().isEmpty()) {
            throw new IllegalArgumentException("没有返回期望的值");
        }
        U result = userContextMap.values().stream().findFirst().orElseThrow(() -> new IllegalArgumentException("没有返回期望的值")).getUser();
        return ResponseEntity.ok(result);
    }

    @JsonView(UserView.UserInfo.class)
    @Override
    public <U extends UserDto<O, R, A>, O extends OrgDto<O>, R extends RoleDto, A extends AccountDto> ResponseEntity<U> simpleUpdate(String tenantKey, String userKey, Boolean byTenantCode, Boolean byUserCode, SimpleUpdate model) {
        Update save = new Update();
        BeanUtils.copyProperties(model, save);
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long userId = super.toUserId(tenantId, userKey, byUserCode);
        save.setTenantId(tenantId);
        save.setUserId(userId);
        save.setIsRoleOverwrite(false);
        save.setIsOrgOverwrite(false);
        save.setIsTagOverwrite(false);
        save.setIsAppOverwrite(false);
        save.setIsMergeAccount(true);
        save.setIsStrict(true);
        SaveUserOutput<U, O, R, A> saveUserOutput = userService.save(save, save.isRoleOverwrite(), save.isOrgOverwrite(), save.isTagOverwrite(), save.isAppOverwrite(), save.isMergeAccount(), save.isStrict());
        UserService.removeCachedRoleOrgUserRelsThreadLocal();
        Map<Long, SaveUserContext<U, O, R, A>> userContextMap = saveUserOutput.getUserResultMap();
        if (userContextMap.values().isEmpty()) {
            throw new IllegalArgumentException("没有返回期望的值");
        }
        U result = userContextMap.values().stream().findFirst().orElseThrow(() -> new IllegalArgumentException("没有返回期望的值")).getUser();

        return ResponseEntity.ok(result);
    }

    @JsonView(UserView.UserInfo.class)
    @ApiOperation(value = "租户获取用户信息")
    @Override
    public ResponseEntity<User> info(String tenantKey,
                                     String userKey,
                                     @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                     @ApiParam(value = "byUserCode", required = false) Boolean byUserCode,
                                     String loginId,
                                     String modules,
                                     int extraInfoDimension) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long userId = super.toUserId(tenantId, userKey, byUserCode);
        User result = userService.findByTenantIdAndUserIdAndLoginId(tenantId, userId, loginId, modules, extraInfoDimension);
        return ResponseEntity.ok(result);
    }
    @Deprecated
    @ApiOperation(value = "租户删除用户")
    @Override
    public ResponseEntity<String> delete(String tenantKey,
                                         String userKey,
                                         @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                         @ApiParam(value = "byUserCode", required = false) Boolean byUserCode) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long userId = super.toUserId(tenantId, userKey, byUserCode);
        userService.deleteByTenantIdAndUserId(tenantId, userId);
        return ResponseEntity.ok("删除成功");
    }

    @ApiOperation(value = "租户绑定用户角色列表")
    @Override
    public ResponseEntity<String> bindRoles(String tenantKey,
                                            String userKey,
                                            @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                            @ApiParam(value = "byUserCode", required = false) Boolean byUserCode,
                                            BindRoles bindRoles) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long userId = super.toUserId(tenantId, userKey, byUserCode);
        userService.bindRoles(tenantId, null, userId, bindRoles.getAllRoleIds(), bindRoles.getRoleIds(), bindRoles.getGradingRoleIds(), bindRoles.getOrgRoleIds(), bindRoles.isOverwrite(), false, false);
        return ResponseEntity.ok("绑定成功");
    }

    @ApiOperation(value = "租户绑定用户角色列表(废弃)", tags = {"role"})
    @Deprecated
    @RequestMapping(name = "租户批量绑定用户角色(废弃)", value = com.xforceplus.api.tenant.user.UserApi.Path.API_PREFIX + "/{userKey}/roles/legacy", method = RequestMethod.POST)
    public ResponseEntity<String> legacyBindRoles(@PathVariable("tenantKey") String tenantKey,
                                                  @PathVariable("userKey") String userKey,
                                                  @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                  @RequestParam(value = "byUserCode", required = false) Boolean byUserCode,
                                                  @RequestBody List<Long> roleIds) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long userId = super.toUserId(tenantId, userKey, byUserCode);
        userService.bindRoles(tenantId, null, userId, null, roleIds, null, null, false, false, false);
        return ResponseEntity.ok("绑定成功");
    }

    @ApiOperation(value = "租户反绑定用户角色列表", tags = {"role"})
    @Override
    public ResponseEntity<String> unbindRoles(String tenantKey,
                                              String userKey,
                                              @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                              @ApiParam(value = "byUserCode", required = false) Boolean byUserCode,
                                              UnbindRoles unbindRoles) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long userId = super.toUserId(tenantId, userKey, byUserCode);
        userService.unbindRoles(tenantId, userId, unbindRoles.getRoleIds());
        return ResponseEntity.ok("反绑定成功");
    }

    @ApiOperation(value = "租户绑定用户组织列表")
    @Override
    public ResponseEntity<String> bindOrgs(String tenantKey,
                                           String userKey,
                                           @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                           @ApiParam(value = "byUserCode", required = false) Boolean byUserCode,
                                           BindOrgs bindOrgs) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long userId = super.toUserId(tenantId, userKey, byUserCode);
        userService.bindOrgs(tenantId, userId, bindOrgs.getOrgIds(), bindOrgs.getModules(), bindOrgs.isOverwrite(), false, false);
        return ResponseEntity.ok("绑定成功");
    }

    @ApiOperation(value = "租户反绑定用户组织列表")
    @Override
    public ResponseEntity<String> unbindOrgs(String tenantKey,
                                             String userKey,
                                             @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                             @ApiParam(value = "byUserCode", required = false) Boolean byUserCode,
                                             UnbindOrgs unbindOrgs) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long userId = super.toUserId(tenantId, userKey, byUserCode);
        userService.unbindOrgs(tenantId, userId, unbindOrgs.getModules(), unbindOrgs.getOrgIds());
        return ResponseEntity.ok("反绑定成功");
    }

    @ApiOperation(value = "租户更新用户状态")
    @Override
    public ResponseEntity<String> updateStatus(String tenantKey,
                                               String userKey,
                                               int status,
                                               @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                               @ApiParam(value = "byUserCode", required = false) Boolean byUserCode) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long userId = super.toUserId(tenantId, userKey, byUserCode);
        userService.updateStatus(tenantId, userId, status);
        return ResponseEntity.ok("更新成功");
    }

    @ApiOperation(value = "租户下获取用户资源码集合", notes = "租户下获取用户资源码集合")
    @Override
    public ResponseEntity<Set<String>> userResources(String tenantKey,
                                                     String userKey,
                                                     @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                     @ApiParam(value = "byUserCode", required = false) Boolean byUserCode,
                                                     String loginId,
                                                     String modules) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long userId = super.toUserId(tenantId, userKey, byUserCode);
        User user = userService.findByLoginId(userId, loginId, modules, BinaryUtils.toBinary(ExtraInfo.resources));
        Set<String> resources = user.getResourceCodes();
        if (resources == null) {
            resources = new HashSet<>();
        }
        return ResponseEntity.ok(resources);
    }

    @Override
    public ResponseEntity<Map<Long, Set<String>>> appsResources(String tenantKey, String userKey, Boolean byTenantCode, Boolean byUserCode, String appIds, String loginId, String modules) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        long userId = super.toUserId(tenantId, userKey, byUserCode);
        return this.appsResources(userId, appIds, loginId, modules);
    }

    @ApiOperation(value = "批量导入用户Excel.")
    @Override
    public ResponseEntity<Boolean> batchImportFromExcel(String tenantKey,
                                                        @ApiParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                        MultipartFile file) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        return excelService.batchImportUsers(tenantId, file);
    }

    @Override
    public ResponseEntity<String> tenantDisable(String tenantKey, Boolean byTenantCode, long userId) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        userService.updateStatus(tenantId, userId, 0);
        return ResponseEntity.ok("禁用成功");
    }

    @JsonView(UserView.UserInfo.class)
    @ApiOperation(value = "当前租户下获取用户分页", notes = "通过分页参数，获取用户分页")
    @Override
    public ResponseEntity<Page<User>> currentPage(Query query, Pageable pageable) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        query.setIncludeIndependents(true);
        return this.page(tenantId, query, pageable);
    }

    @Override
    @JsonView(UserView.UserInfo.class)
    @ApiOperation(value = "获取当前用户信息")
    public ResponseEntity<User> currentInfo(String loginId, int extraInfoDimension) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        long userId = authorizedUser.getId();
        if (loginId == null) {
            loginId = authorizedUser.getLoginId();
        }
        String modules = authorizedUser.getModules();
        return this.info(tenantId, userId, loginId, modules, extraInfoDimension);
    }

    @Override
    public ResponseEntity<Set<String>> currentResources() {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        long userId = authorizedUser.getId();
        String loginId = authorizedUser.getLoginId();
        String modules = authorizedUser.getModules();
        return this.userResources(tenantId, userId, loginId, modules);
    }

    @JsonView(UserView.UserInfo.class)
    @Override
    public ResponseEntity<User> currentUserinfo(long userId, int extraInfo) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        String modules = authorizedUser.getModules();
        return this.info(tenantId, userId, null, modules, extraInfo);
    }

    @JsonView(UserView.UserInfo.class)
    @Override
    public ResponseEntity<User> currentCreate(Create model) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        return this.create(tenantId, model);
    }

    @JsonView(UserView.UserInfo.class)
    @Override
    public ResponseEntity<User> currentUpdate(long userId, Update model) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        return this.update(tenantId, userId, model);
    }

    @Override
    public ResponseEntity<User> currentSimpleCreate(SimpleCreate model) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        return this.simpleCreate(tenantId, model);
    }

    @ApiOperation("当前租户下更新人员信息")
    @Override
    public ResponseEntity<User> currentSimpleUpdate(long userId, SimpleUpdate model) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        return this.simpleUpdate(tenantId, userId, model);
    }

    @Override
    public ResponseEntity<String> currentEnable(long userId) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        userService.updateStatus(tenantId, userId, 1);
        return ResponseEntity.ok("启用成功");
    }

    @Override
    public ResponseEntity<String> currentDisable(long userId) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        userService.updateStatus(tenantId, userId, 0);
        return ResponseEntity.ok("禁用成功");
    }

    @Override
    public ResponseEntity<String> currentDeleteUser(Long userId) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        return this.delete(tenantId, userId);
    }

    @ApiOperation(value = "当前租户下获取用户绑定角色列表", tags = {"role"})
    @JsonView(RoleView.RoleInfo.class)
    @Override
    public ResponseEntity<Page<Role>> currentBindedRoles(Long userId, RoleModel.Request.Query query, Pageable pageable) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        AtomicReference<Pageable> currentPageable = new AtomicReference<>(PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime")));
        query.setTenantId(tenantId);
        query.setUserId(userId);
        if (query.getStatus() == null) {
            query.setStatus(1);
        }
        List<Long> preRoleids = preRoleService.queryUserPreRole(tenantId, userId, query.getExcludeBoundCurrent());
        ObjectCheckAndExcuteUtils.docheckAndExcute(preRoleids, org.apache.commons.collections.CollectionUtils::isNotEmpty, x -> {   query.setOrRoleIds(x); currentPageable.set(PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "type","createTime") )); return Boolean.TRUE;} );
        Page<Role> result = roleService.page(query, currentPageable.get());
        return ResponseEntity.ok(result);
    }

    @ApiOperation(value = "当前租户下用户绑定角色列表", tags = {"role"})
    @Override
    public ResponseEntity<String> currentBindRoles(Long userId, BindRoles bindRoles) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        return this.bindRoles(tenantId, userId, bindRoles);
    }

    @ApiOperation(value = "当前租户下用户反绑定角色列表", tags = {"role"})
    @Override
    public ResponseEntity<String> currentUnbindRoles(Long userId, UnbindRoles unbindRoles) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        return this.unbindRoles(tenantId, userId, unbindRoles);
    }

    @Override
    public ResponseEntity<String> currentBindOrgs(Long userId, BindOrgs bindOrgs) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        return this.bindOrgs(tenantId, userId, bindOrgs);
    }

    @Override
    public ResponseEntity<String> currentUnbindOrgs(Long userId, UnbindOrgs unbindOrgs) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        return this.unbindOrgs(tenantId, userId, unbindOrgs);
    }

    @Override
    public ResponseEntity<Page<OrgStruct>> bindedOrgsPage(Long userId, OrgModel.Request.Query query, Pageable pageable) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        query.setTenantId(tenantId);
        query.setUserId(userId);
        if (query.getStatus() == null) {
            query.setStatus(1);
        }
        query.setModules(authorizedUser.getModules());
        Page<OrgStruct> result = orgService.page(query, pageable);
        return ResponseEntity.ok(result);
    }

    @JsonView(View.List.class)
    @Override
    public ResponseEntity<List<OrgStruct>> bindedOrgs(Long userId, OrgModel.Request.Query query, Sort sort) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        query.setTenantId(tenantId);
        query.setUserId(userId);
        if (query.getStatus() == null) {
            query.setStatus(1);
        }
        query.setModules(authorizedUser.getModules());
        List<OrgStruct> result = orgService.list(query, sort);
        return ResponseEntity.ok(result);
    }


    @ApiOperation(value = "当前用户绑定用户角色列表(废弃)")
    @Deprecated
    @ResponseBody
    @RequestMapping(name = "当前用户批量绑定用户角色(废弃)", value = com.xforceplus.api.current.user.UserApi.Path.API_PREFIX + "/{userKey}/roles/legacy", method = RequestMethod.POST)
    public ResponseEntity<String> currentBindRolesLegacy(@PathVariable("userKey") String userKey,
                                                         @RequestParam(value = "byUserCode", required = false) Boolean byUserCode,
                                                         @RequestBody List<Long> roleIds) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long tenantId = authorizedUser.getTenantId();
        long userId = super.toUserId(tenantId, userKey, byUserCode);
        userService.bindRoles(tenantId, null, userId, null, roleIds, null, null, false, false, false);
        return ResponseEntity.ok("绑定成功");
    }

    /**
     * 修复用户组织
     *
     * @return
     */
    @ResponseBody
    @ApiOperation(value = "修复用户组织", notes = "修复用户组织")
    @RequestMapping(name = "修复用户组织", value = UserApi.Path.API_PREFIX + "/orgs/fix", method = RequestMethod.POST)
    public ResponseEntity<String> fixAllPaths() {
        fixService.fixAllUserOrgs();
        return ResponseEntity.ok("修复成功");
    }

    @Override
    public ResponseEntity export(@RequestBody UserModel.Export export) {
        if (CollectionUtils.isEmpty(export.getUserIds())) {
            throw new IllegalArgumentException("未选择人员");
        }

        List<UserExportDto> users = userService.getExportData(export.getUserIds());
        if (CollectionUtils.isEmpty(users)) {
            throw new IllegalArgumentException("导出失败，没有获取到人员");
        }

        LinkedHashMap<String, String> rules = new LinkedHashMap<>();

        rules.put("userName", "姓名");
        rules.put("userCode", "人员代码");
        rules.put("userNumber", "工号");
        rules.put("status", "状态");
        rules.put("email", "邮箱账号");
        rules.put("telPhone", "手机账号");
        rules.put("accountName", "普通账号");
        rules.put("invoices", "发票类型");
        rules.put("ticketOpeningTerminal", "开票终端");
        rules.put("printingEquipment", "税控设备");
        rules.put("roleNames", "角色名称");
        rules.put("roleCodes", "角色代码");
        rules.put("orgNames", "组织名称");
        rules.put("orgCodes", "组织代码");

        Long fileId;
        try {
//            JsonUtils.toJson(users);
            String json = JSON.toJSONString(users, SerializerFeature.WriteMapNullValue);
            fileId = fileService.exportDataExcel(json, rules, "人员导出列表");
        } catch (Exception e) {
            logger.error("上传文件失败", e);
            throw new IllegalArgumentException("人员导出失败");
        }

        return ResponseEntity.ok(fileId);
    }

    @AuthorizedDefinition(resources = {"xforce:business:user:read"})
    @ApiOperation(value = "批量导出用户关联信息toExcel")
    @Override
    public ResponseEntity batchExport(Query query) {

        if (CollectionUtils.isEmpty(query.getSheets())) {
            throw new IllegalArgumentException("请选择SheetName");
        }
        //租户Id
        IAuthorizedUser authorizedUser = UserInfoHolder.get();
        Long tenantId = authorizedUser.getTenantId();
        query.setTenantId(tenantId);
        userService.buildCurrentQuery(query, authorizedUser);
        //异步执行
        QueryParam queryParam = QueryParam.builder()
                .param(ExcelFile.PARAMS_QUERY, query)
                .build();
        ExcelConfigBusinessType businessType = ExcelConfigBusinessType.USER_IMPORT;
        ExcelFileStore excelFileStore = this.exportFileService.asyncExcelExport(queryParam.params(), businessType);
        ImportFileRespVo respVo = new ImportFileRespVo();
        respVo.setId(excelFileStore.getId());
        respVo.setBusinessType(businessType.getBusinessName());
        //检查ExcelFile文件是否正确
        return ResponseEntity.ok(respVo);
    }

    @AuthorizedDefinition(resources = {"xforce:business:user:save"})
    @ApiOperation(value = "批量从Excel导入用户关联信息")
    @Override
    public ResponseEntity<ImportFileRespVo> batchImportData(MultipartFile file) {
        IAuthorizedUser currentUser = UserInfoHolder.currentUser();
        Long tenantId = currentUser.getTenantId();
        Long userId = currentUser.getId();

        //保存数据和上传文件，并转为异步处理
        ExcelFileStore excelFileStore = this.importFileService.createAndSaveFile(file, ExcelConfigBusinessType.USER_IMPORT.name(), userId, tenantId);

        ImportFileRespVo respVO = new ImportFileRespVo();
        respVO.setBusinessType(ExcelConfigBusinessType.USER_IMPORT.name());
        respVO.setFileId(excelFileStore.getSourceFileId());
        respVO.setId(excelFileStore.getId());
        //检查ExcelFile文件是否正确
        return ResponseEntity.ok(respVO);
    }

    @Override
    public ResponseEntity<Map<Long, Set<String>>> appsResources(String appIds) {
        IAuthorizedUser currentUser = UserInfoHolder.currentUser();
        Long tenantId = currentUser.getTenantId();
        Long userId = currentUser.getId();
        String loginId = currentUser.getLoginId();
        String modules = currentUser.getModules();
        return this.appsResources(tenantId, userId, appIds, loginId, modules);
    }

    @ApiOperation(value = "拷贝指定用户权限相关权限")
    @Override
    public ResponseEntity copyPrivilege(CopyPrivilege privilege) {
        Long tenantId = UserInfoHolder.get().getTenantId();
        userService.copyPrivilege(tenantId, privilege);
        return ResponseEntity.ok();
    }


    @ApiOperation(value = "查询税号")
    @Override
    public ResponseEntity<Set<String>> taxNums(Long userId, String loginId, String modules) {
        User result = userService.findByLoginId(userId, loginId, modules, BinaryUtils.toBinary(ExtraInfo.companies));
        Set<String> set = new HashSet<>();
        if (null != result) {
            Set<OrgStruct> orgs = result.getCompanies();
            if (!CollectionUtils.isEmpty(orgs)) {
                set = orgs.stream().map(OrgStruct::getTaxNum).collect(Collectors.toSet());
            }
        }
        return ResponseEntity.ok(set);
    }

    @Override
    public ResponseEntity<Set<Company>> companies(Long userId, String loginId, String modules) {
        User result = userService.findByLoginId(userId, loginId, modules, ExtraInfo.companies.value());
        Set<Company> companies = new HashSet<>();
        if (null != result && !CollectionUtils.isEmpty(result.getCompanies())) {
            companies = result.getCompanies().stream().filter(orgStruct -> null != orgStruct.getCompany()).map(OrgStruct::getCompany).collect(Collectors.toSet());
        }
        return ResponseEntity.ok(companies);
    }

    @Override
    @ApiOperation(value = "新增绑定设备")
    public ResponseEntity<String> bindPrintingEquipment(Long tenantId, Long userId, List<String> equipmentList) {
        User user = userService.findById(userId);
        if (!user.getTenantId().equals(tenantId)) {
            throw new IllegalArgumentException("用户信息和租户不匹配！");
        }
        userService.bindPrintingEquipment(user, equipmentList);
        return ResponseEntity.ok("绑定成功");
    }

    @Override
    @ApiOperation(value = "解綁设备")
    public ResponseEntity<String> unbindPrintingEquipment(
            Long tenantId, Long userId, List<String> equipmentList) {
        User user = userService.findById(userId);
        if (!user.getTenantId().equals(tenantId)) {
            throw new IllegalArgumentException("用户信息和租户不匹配！");
        }
        userService.unbindPrintingEquipment(user, equipmentList);
        return ResponseEntity.ok("解绑成功");
    }

    @Override
    @ApiOperation(value = "新增绑定终端")
    public ResponseEntity<String> bindTicketTerminal(
            Long tenantId, Long userId, List<String> terminalList) {
        User user = userService.findById(userId);
        if (!user.getTenantId().equals(tenantId)) {
            throw new IllegalArgumentException("用户信息和租户不匹配！");
        }
        userService.bindTicketTerminal(user, terminalList);
        return ResponseEntity.ok("绑定成功");
    }

    @Override
    @ApiOperation(value = "解绑终端")
    public ResponseEntity<String> unbindTicketTerminal(Long tenantId, Long userId, List<String> terminalList) {
        User user = userService.findById(userId);
        if (!user.getTenantId().equals(tenantId)) {
            throw new IllegalArgumentException("用户信息和租户不匹配！");
        }
        userService.unbindTicketTerminal(user, terminalList);
        return ResponseEntity.ok("解绑成功");
    }

    @Override
    @AuthorizedDefinition(resources = {"xforce:business:user:tenant:manager"})
    public ResponseEntity<String> changeTenantManager(String username, Long tenantId) {
        userService.changeTenantManager(username, tenantId);
        return ResponseEntity.ok("管理员更换成功");
    }

    @Override
    public ResponseEntity<String> logoff(Long userId, String loginId) {
        userService.logoff(userId, loginId);
        return ResponseEntity.ok("登出成功");
    }

    @ApiOperation(value = "根据用户名获取用户详情")
    @JsonView(UserView.UserInfo.class)
    @Override
    public ResponseEntity<User> userInfo(String username) {
        Optional<User> result = userService.findByTenantIdAndUsername(UserInfoHolder.get().getTenantId(), username);
        if(!result.isPresent()){
            throw new IllegalArgumentException("找不到对应的用户(用户名:" + username + ")");
        }
        return ResponseEntity.ok(result.get());
    }

    @Override
    public ResponseEntity<String> userTag(String tenantKey, String userKey, Boolean byTenantCode,
        Boolean byUserCode, String tagName) {
        long tenantId = super.toTenantId(tenantKey, byTenantCode);
        Long userId = super.toUserId(tenantId, userKey, byUserCode);
        return this.userTag(userId,tagName);
    }

    @Override
    public ResponseEntity<String> userTag(Long userId, String tagName) {
        return ResponseEntity.ok(userService.getUserTag(userId, tagName));

    }
}
