package com.xforceplus.business.tenant.service;

import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import lombok.Builder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.util.concurrent.ThreadPoolExecutor;


/**
 * 保存 组织-用户关联表
 *
 * @author geewit
 */
@Service
public class AsyncOrgUserService implements InitializingBean, DisposableBean {
    private static final Logger logger = LoggerFactory.getLogger(AsyncOrgUserService.class);

    @Value("${xforce.tenant.auto-bind-org-user-sleep-mills:3000}")
    private Long autoBindOrgUserSleepMills;
    /**
     * 导入消息
     */
    private AsyncEventBus asyncEventBus;

    /**
     * 创建线程池
     */
    @Autowired
    private ThreadPoolExecutor threadPoolExecutor;

    @Autowired
    private OrgUserService orgUserService;

    /**
     * 初始化Guava EventBus
     */
    @Override
    public void afterPropertiesSet() {
        //region 构建异步线程池
        asyncEventBus = new AsyncEventBus(threadPoolExecutor, (exception, context) -> {
            logger.error(exception.getMessage(), exception);
        });
        //endregion
        //region 异步注册
        asyncEventBus.register(new AsyncProcessListener());
        //endregion
    }

    /**
     * 异步自动根据上级组织关联的用户集合关联新建组织
     * @param tenantId 租户id
     * @param orgId   组织id
     */
    public void autoBindParentUsers(Long tenantId, Long orgId) {
        logger.info("tenantId:{}, orgId: {}", tenantId, orgId);
        if (tenantId == null || tenantId == 0 || orgId == null || orgId == 0) {
            logger.info("tenantId == 0 || orgId == 0, return");
            return;
        }
        //region 构造上下文
        Context context = Context.builder()
                .tenantId(tenantId)
                .orgId(orgId)
                .startTime(System.currentTimeMillis())
                .build();
        //endregion

        //region 异步处理
        try {
            this.asyncEventBus.post(context);
        } catch (Exception e) {
            logger.error("asyncExcelProcess:" + e.getMessage(), e);
            throw new IllegalArgumentException("导入数据失败，当前服务器繁忙，请稍后重试");
        }
        //endregion
    }

    @Override
    public void destroy() {
        //region 关闭线程
//        this.threadPoolExecutor.shutdown();
        //endregion
    }

    @Builder
    static class Context {
        long tenantId;
        long orgId;
        long startTime;
    }

    /**
     * 事件监控处理器
     */
    protected class AsyncProcessListener {
        /**
         * 执行异步操作
         *
         * @param context 上下文信息
         */
        @Subscribe
        public void doProcess(Context context) {
            logger.info("AsyncOrgUserService:current active thread={}, max thread count={},current queue size={},task count={},completed task count={}",
                threadPoolExecutor.getActiveCount(), threadPoolExecutor.getLargestPoolSize(), threadPoolExecutor.getQueue().size(),
                threadPoolExecutor.getTaskCount(),threadPoolExecutor.getCompletedTaskCount());

            logger.info("AsyncProcessListener.doProcess, context.tenantId = {}, context.orgId = {}, waitTime={}",
                context.tenantId, context.orgId, System.currentTimeMillis()-context.startTime);
            if (context.orgId > 0) {
                try {
                    Thread.sleep(autoBindOrgUserSleepMills);
                } catch (InterruptedException e) {
                    logger.warn(e.getMessage());
                    Thread.currentThread().interrupt();
                } finally {
                    try {
                        orgUserService.autoBindUsers(context.tenantId, context.orgId, context.startTime);
                    } catch (Exception e) {
                        logger.warn(e.getMessage());
                    }
                }
            }
        }
    }
}
