/*
 * Copyright (c)  2015~2025, Xforceplus
 * All rights reserved.
 * Project: tenant-service
 * ClassName: GradingRoleService
 * Date: 2021-08-31 09:57:55
 * Author: zhouxin
 */
package com.xforceplus.business.tenant.service;

import com.xforceplus.api.model.RoleModel.Request.*;
import com.xforceplus.dao.*;
import com.xforceplus.domain.resource.ResourcesetDto;
import com.xforceplus.domain.tenant.GradingRoleDTO;
import com.xforceplus.domain.tenant.GradingRoleOrg;
import com.xforceplus.domain.tenant.GradingRoleUser;
import com.xforceplus.domain.tenant.OrgRoleCntDTO;
import com.xforceplus.entity.*;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @Description: 分级角色服务
 * @CopyRight: 上海云砺信息科技有限公司
 * @Author: zhouxin
 * @Email: yf_zhouxin@xforceplus.com
 * @Date: 2021/8/31 09:57:55
 */
@Service
public class GradingRoleService {

    @Autowired
    private RoleService roleService;

    @Autowired
    private OrgStructDao orgStructDao;

    @Autowired
    private OrgUserRelDao orgUserRelDao;

    @Autowired
    private RoleUserRelDao roleUserRelDao;

    @Autowired
    private RoleResourcesetRelDao roleResourcesetRelDao;

    @Autowired
    private UserDao userDao;

    @Autowired
    private ResourcesetDao resourcesetDao;

    @Value("${xforce.tenant.grading.role.user.maxCnt:5}")
    private Integer maxUserCnt;

    /**
     * 分级角色创建
     *
     * @param model
     * @return
     */
    @Transactional(rollbackFor = {Exception.class})
    public Role create(GradingRoleCreate model) {
        Create role = new Create();
        BeanUtils.copyProperties(model, role);
        role.setType(1);
        // 创建分级角色时  如果未关联功能集，状态设置为未启用
        if (CollectionUtils.isEmpty(model.getResourcesetIds())) {
            role.setStatus(0);
        }
        Role base = roleService.create(role);
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();

        if (!CollectionUtils.isEmpty(model.getOrgIds())) {
            List<OrgStruct> orgStructs = orgStructDao.findAllById(model.getOrgIds());
            boolean orgAlreadyBinded = orgStructs.stream().anyMatch(o -> o.getGradingRoleId() != null);
            if (orgAlreadyBinded) {
                throw new IllegalArgumentException("组织id非法，一个组织只能绑定一个分级管理角色!");
            }
            Set<Long> orgIds = model.getOrgIds().stream().collect(Collectors.toSet());
            this.batchSetOrgGradingRoles(base.getId(), orgIds);
        }
        if (!CollectionUtils.isEmpty(model.getResourcesetIds())) {
            this.batchRelGradingRoleResourcesets(base, model.getResourcesetIds());
        }
        return base;
    }

    /**
     * 分级角色分页列表查询
     *
     * @param query
     * @param currentPageable
     * @return
     */
    @SuppressWarnings("AlibabaLowerCamelCaseVariableNaming")
    public Page<GradingRoleDTO> page(Query query, Pageable currentPageable) {
        // 只查分级角色
        query.setType(1);
        Page<Role> rolePage = roleService.page(query, currentPageable);
        if (!CollectionUtils.isEmpty(rolePage.getContent())) {
            Set<Long> roleIds = rolePage.getContent().stream().map(Role::getId).collect(Collectors.toSet());
            List<OrgRoleCntDTO> orgRoleCntDTOS = roleService.getOrgRoleCntByGradingRoleIds(query.getTenantId(), roleIds);

            Map<Long, Long> map = new HashMap<>();
            if (!CollectionUtils.isEmpty(orgRoleCntDTOS)) {
                for (OrgRoleCntDTO orgRoleCntDTO : orgRoleCntDTOS) {
                    map.put(orgRoleCntDTO.getRoleId(), orgRoleCntDTO.getOrgRoleCnt());
                }
            }

            List<GradingRoleOrg> orgs = orgStructDao.findOrgsByGradingRoleIds(roleIds);
            Map<Long, List<GradingRoleOrg>> orgMap = new HashMap<>();
            if (!CollectionUtils.isEmpty(orgs)) {
                orgMap = orgs.stream().collect(Collectors.groupingBy(GradingRoleOrg::getRoleId));
            }

            Map<Long, List<GradingRoleOrg>> finalOrgMap = orgMap;
            Page<GradingRoleDTO> page = rolePage.map(role -> {
                GradingRoleDTO dto = new GradingRoleDTO();
                BeanUtils.copyProperties(role, dto);
                dto.setOrgRoleCnt(map.get(role.getId()) == null ? 0L : map.get(role.getId()));
                dto.setOrgs(finalOrgMap.get(role.getId()));
                return dto;
            });
            return page;
        }

        return new PageImpl<>(new ArrayList<GradingRoleDTO>());
    }

    /**
     * 分级角色详情
     *
     * @param roleId
     * @return
     */
    public GradingRoleDTO findByRoleId(Long roleId) {
        Role role = roleService.findById(roleId);
        GradingRoleDTO dto = new GradingRoleDTO();
        BeanUtils.copyProperties(role, dto);

        Map<Long, List<ResourcesetDto>> resourcesetMap = new HashMap<>();
        if (!CollectionUtils.isEmpty(role.getResourcesets())) {
            resourcesetMap = role.getResourcesets().stream().collect(Collectors.groupingBy(ResourcesetDto::getAppId));
            dto.setResourcesetMap(resourcesetMap);
        }

        List<GradingRoleOrg> gradingRoleOrgs = orgStructDao.findOrgsByGradingRoleIds(Stream.of(roleId).collect(Collectors.toSet()));
        dto.setOrgs(gradingRoleOrgs);

        List<Long> userIds = roleUserRelDao.findUserIdsByRoleId(roleId);
        if (!CollectionUtils.isEmpty(userIds)) {
            List<GradingRoleUser> gradingRoleUsers = userDao.findGradingRoleUserByUserIds(userIds);
            dto.setUsers(gradingRoleUsers);
        } else {
            dto.setUsers(new ArrayList<>());
        }

        return dto;
    }

    /**
     * 分级角色更新
     *
     * @param tenantId
     * @param roleId
     * @param model
     * @return
     */
    @Transactional(rollbackFor = {Exception.class})
    public GradingRoleDTO updateByTenantId(long tenantId, Long roleId, Update model) {
        model.setType(1);
        model.setTenantId(tenantId);
        Role role = roleService.updateByTenantId(tenantId, roleId, model);

        GradingRoleDTO dto = new GradingRoleDTO();
        BeanUtils.copyProperties(role, dto);
        return dto;
    }

    /**
     * 查询分级角色关联的组织
     *
     * @param roleId
     * @return
     */
    public List<GradingRoleOrg> findBindedOrgsByRoleId(Long roleId) {
        return orgStructDao.findOrgsByGradingRoleIds(Stream.of(roleId).collect(Collectors.toSet()));
    }

    /**
     * 查询分级角色关联的功能集
     *
     * @param roleId
     * @return
     */
    public List<Resourceset> findBindedResourcesetsByRoleId(Long roleId) {
        Role role = roleService.findById(roleId);
        List<RoleResourcesetRel> rels = role.getRoleResourcesetRels();
        if (!CollectionUtils.isEmpty(rels)) {
            List<Long> resourcesetIds = rels.stream().map(RoleResourcesetRel::getResousesetId).collect(Collectors.toList());
            return resourcesetDao.findByResourcesetIds(resourcesetIds);
        }
        return Collections.emptyList();
    }

    /**
     * 设置分级角色关联的组织
     *
     * @param roleId
     * @param bindOrgs
     */
    @Transactional(rollbackFor = {Exception.class})
    public void saveGradingRoleOrgs(Long roleId, BindOrgs bindOrgs) {
        List<GradingRoleOrg> gradingRoleOrgs = orgStructDao.findOrgsByGradingRoleIds(Stream.of(roleId).collect(Collectors.toSet()));
        // 如果未绑定过组织, 传了则新增
        if (CollectionUtils.isEmpty(gradingRoleOrgs)) {
            if (bindOrgs != null && !CollectionUtils.isEmpty(bindOrgs.getOrgIds())) {
                batchSetOrgGradingRoles(roleId, bindOrgs.getOrgIds());
            }
        } else {
            // 没传(批量置空)
            Set<Long> bindedOrgIds = gradingRoleOrgs.stream().filter(Objects::nonNull).map(GradingRoleOrg::getOrgId).collect(Collectors.toSet());
            if (bindOrgs == null || CollectionUtils.isEmpty(bindOrgs.getOrgIds())) {
                batchSetOrgGradingRoles(null, bindedOrgIds);
            } else {
                // 新减旧 差集为新增的
                Set<Long> toBindOrgIds = new HashSet<>(bindOrgs.getOrgIds());
                toBindOrgIds.removeAll(bindedOrgIds);
                if (!CollectionUtils.isEmpty(toBindOrgIds)) {
                    batchSetOrgGradingRoles(roleId, toBindOrgIds);
                }
                // 旧减新 差集为要删除的
                toBindOrgIds.clear();
                toBindOrgIds = new HashSet<>(bindOrgs.getOrgIds());
                bindedOrgIds.removeAll(toBindOrgIds);
                if (!CollectionUtils.isEmpty(bindedOrgIds)) {
                    batchSetOrgGradingRoles(null, bindedOrgIds);
                }
            }
        }
    }

    /**
     * 批量设置组织分级角色
     *
     * @param roleId
     * @param orgIds
     */
    private void batchSetOrgGradingRoles(Long roleId, Set<Long> orgIds) {
        List<OrgStruct> orgStructs = orgStructDao.findAllById(orgIds);
        for (OrgStruct orgStruct : orgStructs) {
            Long originGradingRoleId = orgStruct.getGradingRoleId();
            orgStruct.setGradingRoleId(roleId);
            // 如果解绑分级管理角色，也解绑默认组织角色
            if (null == roleId || !Objects.equals(originGradingRoleId, roleId)) {
                Long defaultOrgRoleId = orgStruct.getDefaultOrgRoleId();
                // 同时需要解绑默认组织角色和组织用户关系
                if (null != defaultOrgRoleId) {
                    orgStruct.setDefaultOrgRoleId(null);
                    Set<OrgUserRel> orgUserRels = orgUserRelDao.findRelsByOrgIds(Stream.of(orgStruct.getOrgId())
                            .collect(Collectors.toSet()));
                    if (!CollectionUtils.isEmpty(orgUserRels)) {
                        Set<Long> userIds = orgUserRels.stream().map(OrgUserRel::getUserId).collect(Collectors.toSet());
                        roleUserRelDao.deleteByRoleIdAndUserIds(defaultOrgRoleId, userIds);
                    }
                }
            }
        }
        orgStructDao.saveAllAndFlush(orgStructs);
    }

    /**
     * 设置分级角色关联的功能集
     *
     * @param roleId
     * @param bindResourceSets
     */
    @Transactional(rollbackFor = {Exception.class})
    public void saveGradingRoleResourcesets(Long roleId, BindResourceSets bindResourceSets) {
        Role role = roleService.findById(roleId);
        List<Resourceset> resourcesets = Collections.EMPTY_LIST;
        List<RoleResourcesetRel> rels = role.getRoleResourcesetRels();
        if (!CollectionUtils.isEmpty(rels)) {
            List<Long> resourcesetIds = rels.stream().map(RoleResourcesetRel::getResousesetId).collect(Collectors.toList());
            resourcesets = resourcesetDao.findByResourcesetIds(resourcesetIds);
        }
        // 如果未绑定过功能集, 传了则新增
        if (CollectionUtils.isEmpty(resourcesets)) {
            if (bindResourceSets != null && !CollectionUtils.isEmpty(bindResourceSets.getResourcesetIds())) {
                batchRelGradingRoleResourcesets(role, new HashSet<>(bindResourceSets.getResourcesetIds()));
            } else {
                roleService.disable(roleId);
                roleService.disableOrgRolesByGradingRoleId(roleId);
            }
        } else {
            Set<Long> bindedResourcesetIds = resourcesets.stream().filter(Objects::nonNull).map(ResourcesetDto::getResourcesetId).collect(Collectors.toSet());
            // 没传(批量删除)
            if (bindResourceSets == null || CollectionUtils.isEmpty(bindResourceSets.getResourcesetIds())) {
                unbindGradingRoleResourcesets(roleId, bindedResourcesetIds);
                // 置空功能集时，分级角色和分级角色创建出来的组织角色的状态都需要设置为 未启用
                roleService.disable(roleId);
                roleService.disableOrgRolesByGradingRoleId(roleId);
            } else {
                // 新减旧 差集为新增的
                Set<Long> toBindResourcesetIds = new HashSet<>(bindResourceSets.getResourcesetIds());
                toBindResourcesetIds.removeAll(bindedResourcesetIds);
                if (!CollectionUtils.isEmpty(toBindResourcesetIds)) {
                    batchRelGradingRoleResourcesets(role, toBindResourcesetIds);
                }
                // 旧减新 差集为要删除的
                toBindResourcesetIds.clear();
                toBindResourcesetIds = new HashSet<>(bindResourceSets.getResourcesetIds());
                bindedResourcesetIds.removeAll(toBindResourcesetIds);
                if (!CollectionUtils.isEmpty(bindedResourcesetIds)) {
                    unbindGradingRoleResourcesets(roleId, bindedResourcesetIds);
                    // 旧的功能集如果已经绑定了组织角色，则解绑掉
                    List<Role> orgRoles = roleService.findByTenantIdAndFromRoleId(role.getTenantId(), roleId);
                    if (!CollectionUtils.isEmpty(orgRoles)) {
                        for (Role orgRole : orgRoles) {
                            List<RoleResourcesetRel> rrs = roleResourcesetRelDao.findByRoleId(orgRole.getId());
                            if (!CollectionUtils.isEmpty(rrs)) {
                                Set<Long> brrIds = rrs.stream().map(RoleResourcesetRel::getResousesetId).collect(Collectors.toSet());
                                for (Long brrId : brrIds) {
                                    if (bindedResourcesetIds.contains(brrId)) {
                                        roleResourcesetRelDao.deleteByRoleIdAndResourcesetId(orgRole.getId(), brrId);
                                    }
                                }
                            }
                            List<RoleResourcesetRel> list = roleResourcesetRelDao.findByRoleId(orgRole.getId());
                            // 如果组织角色已经没有任何功能集关联了，则禁用掉
                            if (CollectionUtils.isEmpty(list)) {
                                roleService.disable(orgRole.getId());
                            }
                        }
                    }
                }
            }
        }
    }

    private void batchRelGradingRoleResourcesets(Role role, Set<Long> bindResourcesetIds) {
        Set<RoleResourcesetRel> roleResourcesetRels = new HashSet<>();
        for (Long resourcesetId : bindResourcesetIds) {
            Resourceset r = resourcesetDao.findById(resourcesetId).orElseThrow(() -> new IllegalArgumentException("功能集id非法!"));
            RoleResourcesetRel rel = new RoleResourcesetRel();
            rel.setRoleId(role.getId());
            rel.setTenantId(role.getTenantId());
            rel.setResousesetId(resourcesetId);
            roleResourcesetRels.add(rel);
        }
        roleResourcesetRelDao.saveAllAndFlush(roleResourcesetRels);
    }

    /**
     * 批量解绑分级角色功能集关联
     *
     * @param roleId
     * @param bindedResourcesetIds
     */
    private void unbindGradingRoleResourcesets(Long roleId, Set<Long> bindedResourcesetIds) {
        roleResourcesetRelDao.unbindGradingRoleResourcesets(roleId, bindedResourcesetIds);
    }

    /**
     * 分级角色绑定用户
     *
     * @param tenantId
     * @param roleId
     * @param bindUsers
     */
    @Transactional(rollbackFor = Exception.class)
    public void bindUsers(long tenantId, Long roleId, BindUsers bindUsers) {
        if (null == bindUsers) {
            return;
        }

        if (!CollectionUtils.isEmpty(bindUsers.getUserIds()) && bindUsers.getUserIds().size() > maxUserCnt) {
            throw new IllegalArgumentException("一个分级角色最多只能绑定" + maxUserCnt + "个用户！");
        }

        // 查询当前角色绑定了多少用户
        long count = roleUserRelDao.countByRoleId(roleId);
        if ((count + bindUsers.getUserIds().size()) > maxUserCnt) {
            throw new IllegalArgumentException("一个分级角色最多只能绑定" + maxUserCnt + "个用户！");
        }

        // 最多循环 maxUserCnt 次， 默认5次
        for (Long userId : bindUsers.getUserIds()) {
            Set<Long> gradingRoleIds = roleUserRelDao.findGradingRoleIdsByUserId(userId);
            if (!CollectionUtils.isEmpty(gradingRoleIds) && (gradingRoleIds.size() > 1 || gradingRoleIds.size() == 1 && !gradingRoleIds.contains(roleId))) {
                throw new IllegalArgumentException("一个用户只能绑定一个分级管理员角色，用户[" + userId + "]已绑定其他分级管理员角色！");
            }
        }

        roleService.bindUsers(tenantId, roleId, bindUsers, 1);
    }

    /**
     * 删除分级角色
     *
     * @param tenantId
     * @param roleId
     */
    @Transactional(rollbackFor = {Exception.class})
    public void deleteByTenantIdAndId(long tenantId, Long roleId) {
        Role role = roleService.findById(roleId);
        if (role.getStatus() != 0) {
            throw new IllegalArgumentException("只能删除未启用的分级角色!");
        }
        roleService.deleteByTenantIdAndId(tenantId, roleId);

        // 清除组织分级角色和默认组织角色
        List<GradingRoleOrg> orgs = orgStructDao.findOrgsByGradingRoleIds(Stream.of(roleId).collect(Collectors.toSet()));
        if (!CollectionUtils.isEmpty(orgs)) {
            Set<Long> orgIds = orgs.stream().map(GradingRoleOrg::getOrgId).collect(Collectors.toSet());
            batchSetOrgGradingRoles(null, orgIds);
        }

        // 清除创建的组织角色，同时解绑所有组织角色关联的用户和功能集
        List<Role> orgRoles = roleService.findByTenantIdAndFromRoleId(tenantId, roleId);
        if (!CollectionUtils.isEmpty(orgRoles)) {
            List<Long> ids = orgRoles.stream().map(Role::getId).collect(Collectors.toList());

            roleUserRelDao.deleteByTenantIdAndRoleIds(tenantId, ids);

            roleResourcesetRelDao.deleteByTenantIdAndRoleIds(tenantId, ids);

            roleService.deleteByTenantIdAndIds(tenantId, ids);

        }

        // 解绑分级角色关联的功能集
        roleResourcesetRelDao.deleteByTenantIdAndRoleIds(tenantId, Stream.of(roleId).collect(Collectors.toList()));
    }

    /**
     * 启用分级角色
     *
     * @param roleId
     */
    @Transactional(rollbackFor = {Exception.class})
    public void enable(long roleId) {
        Role role = roleService.findById(roleId);
        List<Resourceset> resourcesets = Collections.EMPTY_LIST;
        List<RoleResourcesetRel> rels = role.getRoleResourcesetRels();
        if (!CollectionUtils.isEmpty(rels)) {
            List<Long> resourcesetIds = rels.stream().map(RoleResourcesetRel::getResousesetId).collect(Collectors.toList());
            resourcesets = resourcesetDao.findByResourcesetIds(resourcesetIds);
        }
        if (CollectionUtils.isEmpty(resourcesets)) {
            throw new IllegalArgumentException("该角色关联的功能集为空，不能启用!");
        }
        boolean anyEnabled = resourcesets.stream().anyMatch(r -> r.getStatus() == 1);
        if (!anyEnabled) {
            throw new IllegalArgumentException("该角色关联的功能集都是停用状态，不能启用!");
        }
        roleService.enable(roleId);
    }

    /**
     * 禁用分级角色
     *
     * @param roleId
     */
    @Transactional(rollbackFor = {Exception.class})
    public void disable(long tenantId, long roleId) {
        roleService.updateStatus(tenantId, roleId, 0);
        roleService.disableOrgRolesByGradingRoleId(roleId);
    }

}
