package com.xforceplus.business.tenant.service;

import com.xforceplus.api.model.OrgModel.Request.Query;
import com.xforceplus.business.company.service.CompanyExtensionService;
import com.xforceplus.business.excel.BusinessType;
import com.xforceplus.business.excel.SimpleExcelWriter;
import com.xforceplus.business.excel.reader.Context;
import com.xforceplus.business.excel.writer.ExcelConfigBusinessType;
import com.xforceplus.business.service.ExcelWriteService;
import com.xforceplus.business.tenant.excel.CompanyExportData;
import com.xforceplus.config.ImportExportThreadPool;
import com.xforceplus.dao.OrgCompanynoDao;
import com.xforceplus.dao.OrgStructDao;
import com.xforceplus.entity.OrgCompanyRel;
import com.xforceplus.entity.OrgStruct;
import com.xforceplus.query.OrgQueryHelper;
import com.xforceplus.tenant.security.core.domain.OrgType;
import io.geewit.core.utils.reflection.BeanUtils;
import io.geewit.data.jpa.essential.domain.EntityGraphs;
import org.apache.commons.collections4.ListUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.persistence.criteria.Predicate;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.xforceplus.business.company.service.CompanyConfigExcel.SORT_FIELD;
import static com.xforceplus.business.excel.ExcelFile.*;
import static com.xforceplus.business.tenant.excel.OrgExportExcel.SHEET_NAME_1;
import static com.xforceplus.business.tenant.excel.OrgExportExcel.SHEET_NAME_2;
import static org.springframework.data.domain.Sort.Direction.DESC;

/**
 * 组织批量导出excel Service
 *
 * @author geewit
 * @since 2020-09-27
 */
@Service
public class OrgExcelExportService implements ExcelWriteService {
    private final static Logger logger = LoggerFactory.getLogger(OrgExcelExportService.class);

    private final static String COMPANYNOS_DELIMITER = "/";

    @Autowired
    private OrgStructDao orgStructDao;

    @Autowired
    private OrgCompanynoDao orgCompanynoDao;

    @Autowired
    private CompanyExtensionService companyExtensionService;

    /**
     * 获取导入类型，用于Event事件调整导入方法
     * @return ImportBusinessType
     */
    @Override
    public BusinessType getBusinessType() {
        return ExcelConfigBusinessType.ORG_IMPORT;
    }

    /**
     * 导出公共方法
     *
     * @param context 上下文
     * @return Context
     */
    @Override
    public void write(Context context) {
        Query query = context.getParam(PARAMS_QUERY, Query.class);
        SimpleExcelWriter simpleExcelWriter = context.getSimpleExcelWriter();
        List<String> sheetNames = query.getSheets();
        Integer maxPageSize = context.getMaxPageSize();
        logger.info("maxPageSize:{}", maxPageSize);
        //region 分页
        Pageable pageable = PageRequest.of(START_PAGE, PAGE_SIZE, Sort.by(DESC, SORT_FIELD));
        Page<OrgStruct> page;
        do {
            page = this.excelPage(query, pageable);
            List<OrgStruct> orgStructList = page.getContent();
            List<OrgStruct> resultList = new ArrayList<>();
            //2022-03-24 默认导出扩展信息，避免前端没有传扩展参数而要求导出结果包含扩展字段
            resultList = this.fillOrg(orgStructList);
            resultList.forEach(org -> sheetNames.forEach(sheetName -> this.writeSheet(simpleExcelWriter, sheetName, org)));
        } while (page.hasNext() && (pageable = pageable.next()) != null && page.getNumber() < maxPageSize);
        //endregion

        //region 统计总数
        simpleExcelWriter.param(SUCCESS_SIZE, page.getTotalElements());
    }

    /**
     * 导出excel专用分页方法
     *
     * @param query
     * @param pageable
     * @return
     */
    private Page<OrgStruct> excelPage(Query query, Pageable pageable) {
        Specification<OrgStruct> specification = OrgQueryHelper.querySpecification(query);
        Page<OrgStruct> page = orgStructDao.findAll(specification, pageable, EntityGraphs.named(OrgStruct.NAMED_ENTITY_GRAPH_DEFAULT));
        //查询组织和公司关系使用多线程来处理
        return page;
    }

    /**
     * 分批次补充组织信息
     * @param orgStructList
     * @return 填充扩展信息的org列表
     */
    private List<OrgStruct> fillOrg(List<OrgStruct> orgStructList){
        int partitionSize = orgStructList.size() > ImportExportThreadPool.CORE_POOL_SIZE ? orgStructList.size()/ImportExportThreadPool.CORE_POOL_SIZE : 1;
        List<CompletableFuture> futureList = new ArrayList<>();
        List<List<OrgStruct>> partitionList = ListUtils.partition(orgStructList,partitionSize);
        partitionList.forEach(list -> futureList.add(CompletableFuture.runAsync(()->this.batchSetExtend(list),ImportExportThreadPool.get())));
        try {
            CompletableFuture.allOf(futureList.toArray(new CompletableFuture[]{})).get();
            List<OrgStruct> resultList = new ArrayList<>();
            partitionList.forEach(item->resultList.addAll(item));
            return resultList;
        } catch (InterruptedException  | ExecutionException e) {
            logger.error("error to async fill org extend:{}",e.getMessage());
            throw new IllegalArgumentException(e.getMessage());
        }catch (UnsupportedOperationException unsupportedOperationException){
            logger.error("error to handle query result:{}",unsupportedOperationException.getMessage());
            throw new IllegalArgumentException(unsupportedOperationException.getMessage());
        }
    }

    /**
     * 批量补充组织信息
     * @param list
     */
    private void batchSetExtend(List<OrgStruct> list){
        list.forEach(org->{
            if (org != null && org.getOrgId() != null && org.getOrgId() > 0) {
                List<OrgCompanyRel> orgCompanyRels = orgCompanynoDao.findByOrgId(org.getOrgId());
                Set<String> companyNos = orgCompanyRels.stream().map(OrgCompanyRel::getCompanyNo).filter(StringUtils::isNoneBlank).collect(Collectors.toSet());
                org.setCompanyNos(companyNos);
                if (OrgType.COMPANY.equals(org.getOrgType())) {
                    org.setCompanyCode(org.getCompanyCode());
                    org.setCompanyName(org.getCompanyName());
                    org.setTaxNum(org.getTaxNum());
                }
            }
        });
    }

    private void writeSheet(SimpleExcelWriter simpleExcelWriter, String sheetName, OrgStruct org) {
        switch (sheetName) {
            case SHEET_NAME_1: {
                if (OrgType.COMPANY.equals(org.getOrgType())) {
                    logger.info("writeSheet1");
                    this.writeSheet1(simpleExcelWriter, sheetName, org);
                }
                break;
            }
            case SHEET_NAME_2: {
                if (org.getOrgType() != null && !OrgType.GROUP.equals(org.getOrgType())) {
                    this.writeSheet2(simpleExcelWriter, sheetName, org);
                }
                break;
            }
            default: {
                break;
            }
        }
    }

    /**
     * 写公司批量导出Sheet
     *
     * @param simpleExcelWriter
     * @param sheetName
     * @param org
     */
    private void writeSheet1(SimpleExcelWriter simpleExcelWriter, String sheetName, OrgStruct org) {
        if (org.getCompany() == null || org.getCompany().getTaxNum() == null) {
            logger.info("org.company = null, return");
            return;
        }
        CompanyExportData row = new CompanyExportData();
        BeanUtils.copyProperties(org.getCompany(), row);
        simpleExcelWriter.fill(sheetName, Stream.of(row).collect(Collectors.toList()));
    }

    /**
     * 写组织批量导出Sheet
     *
     * @param simpleExcelWriter
     * @param sheetName
     * @param org
     */
    private void writeSheet2(SimpleExcelWriter simpleExcelWriter, String sheetName, OrgStruct org) {
        Map<String, Object> row = new HashMap<>(30);
        BeanUtils.copyProperties(org, row, Stream.of("companyNos").toArray(String[]::new));
        //region 设置 parentCode
        if (org.getParentId() != null && org.getParentId() > 0) {
            Specification<OrgStruct> specification = (Specification<OrgStruct>) (root, criteriaQuery, builder) -> {
                List<Predicate> predicates = new ArrayList<>();
                predicates.add(builder.equal(root.<Long>get("orgId"), org.getParentId()));
                criteriaQuery.where(predicates.stream().toArray(Predicate[]::new));
                return criteriaQuery.getRestriction();
            };
            Optional<OrgStruct> parentOptional = orgStructDao.findOne(specification);
            parentOptional.ifPresent(orgStruct -> row.put("parentCode", orgStruct.getOrgCode()));
        }
        //endregion
        if (org.getOrgType() != null) {
            row.put("orgType", org.getOrgType().value());
        }
        if (!CollectionUtils.isEmpty(org.getCompanyNos())) {
            String companyNos = String.join(COMPANYNOS_DELIMITER, org.getCompanyNos());
            row.put("companyNos", companyNos);
        }
        //补充终端鉴权信息
        if (OrgType.COMPANY.equals(org.getOrgType())){
            if (org.getCompanyId()!=null) {
                //处理部分org表中类型为公司但是companyId为空的情况
                int authSwitch = companyExtensionService.getCompanyTerminalSwitch(org.getTenantId(), org.getCompanyId());
                row.put("authSwitch", authSwitch);
            }else {
                row.put("authSwitch", 0);
            }
        }
        simpleExcelWriter.fill(sheetName, Stream.of(row).collect(Collectors.toList()));
    }
}
