package com.xforceplus.business.tenant.service;

import com.xforceplus.api.model.OrgExtensionModel;
import com.xforceplus.api.model.OrgExtensionModel.Request.Query;
import com.xforceplus.api.model.OrgModel;
import com.xforceplus.constants.ExtensionStatus;
import com.xforceplus.dao.OrgExtensionDao;
import com.xforceplus.entity.OrgExtension;
import com.xforceplus.query.OrgExtensionQueryHelper;
import io.geewit.core.utils.reflection.BeanUtils;
import io.geewit.data.jpa.essential.domain.EntityGraphs;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.validation.annotation.Validated;

import java.util.*;
import java.util.stream.Collectors;

@Validated
@Service
public class OrgExtensionService {
    private final static Logger logger = LoggerFactory.getLogger(OrgExtensionService.class);

    public OrgExtensionService(OrgExtensionDao orgExtensionDao) {
        this.orgExtensionDao = orgExtensionDao;
    }

    private final OrgExtensionDao orgExtensionDao;

    public Page<OrgExtension> page(Query query, Pageable pageable) {
        Specification<OrgExtension> specification = OrgExtensionQueryHelper.querySpecification(query);
        return orgExtensionDao.findAll(specification, pageable, EntityGraphs.named(OrgExtension.NAMED_ENTITY_GRAPH_DEFAULT));
    }

    @Transactional(rollbackFor = Exception.class)
    public OrgExtension create(OrgExtensionModel.Request.Create model) {
        List<OrgExtension> list = orgExtensionDao.findByOrgIdAndKey(model.getOrgId(), model.getExtensionKey());
        if (CollectionUtils.isEmpty(list)) {
            OrgExtension entity = new OrgExtension();
            entity.setOrgStructId(model.getOrgId());
            BeanUtils.copyProperties(model, entity);
            entity.setStatus(ExtensionStatus.ENABLE);
            return orgExtensionDao.saveAndFlush(entity);
        } else {
            String message = "已存在orgId(" + model.getOrgId() + ", key(" + model.getExtensionKey() + ")的组织扩展信息";
            logger.warn(message);
            throw new IllegalArgumentException(message);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public OrgExtension update(OrgExtensionModel.Request.Update model) {
        Optional<OrgExtension> existOrgExtensionOptional = orgExtensionDao.findById(model.getExtensionId());
        if (!existOrgExtensionOptional.isPresent()) {
            String message = "不存在Id(" + model.getExtensionId() + ")的组织扩展信息";
            logger.warn(message);
            throw new IllegalArgumentException(message);
        } else {
            OrgExtension entity = existOrgExtensionOptional.get();
            entity.setExtensionValue(model.getExtensionValue());
            entity.setStatus(ExtensionStatus.ENABLE);
            return orgExtensionDao.saveAndFlush(entity);
        }
    }

    public Set<OrgExtension> batchSave(Long orgStructId, List<OrgModel.Request.Extension> extensions, boolean isOverwrite) {
        if (orgStructId == null || CollectionUtils.isEmpty(extensions)) {
            return null;
        }
        List<OrgExtension> existExtensions = this.findByOrgId(orgStructId);
        Set<OrgExtension> savingExtensions = extensions.stream().filter(Objects::nonNull).map(extension -> {
            List<OrgExtension> exists = orgExtensionDao.findByOrgIdAndKey(orgStructId, extension.getExtensionKey());
            OrgExtension savingExtension;
            if (CollectionUtils.isEmpty(exists)) {
                savingExtension = new OrgExtension();
                savingExtension.setOrgStructId(orgStructId);
                savingExtension.setExtensionKey(extension.getExtensionKey());
            } else {
                savingExtension = exists.get(0);
            }
            savingExtension.setExtensionValue(extension.getExtensionValue());
            return savingExtension;
        }).collect(Collectors.toSet());
        if (!savingExtensions.isEmpty()) {
            orgExtensionDao.saveAllAndFlush(savingExtensions);
        }
        Set<OrgExtension> result = new HashSet<>(savingExtensions);
        result.addAll(existExtensions);
        if (isOverwrite) {
            existExtensions.stream().filter(existExtension -> extensions.stream().noneMatch(extension -> extension.getExtensionKey().equals(existExtension.getExtensionKey()))).forEach(extension -> {
                logger.info("deleting OrgExtension record, {}", extension);
                try {
                    orgExtensionDao.deleteById(extension.getOrgExtensionId());
                } catch (Exception e) {
                    logger.warn(e.getMessage(), e);
                }
            });
        } else {
            result.addAll(existExtensions);
        }
        return result;
    }

    public List<OrgExtension> findByOrgId(Long orgStructId) {
        Query query = new Query();
        query.setOrgStructId(orgStructId);
        Specification<OrgExtension> specification = OrgExtensionQueryHelper.querySpecification(query);
        return orgExtensionDao.findAll(specification, Sort.unsorted());
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteByOrgIdAndKey(long orgId, String extensionKey) {
        int result = orgExtensionDao.deleteByOrgIdAndKey(orgId, extensionKey);
        if (result <= 0) {
            String message = "不存在orgId(" + orgId + "), key(" + extensionKey + ")的组织扩展信息";
            logger.warn(message);
            throw new IllegalArgumentException(message);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteByOrgId(long orgId) {
        orgExtensionDao.deleteByOrgId(orgId);
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteById(long extensionId) {
        orgExtensionDao.deleteById(extensionId);
    }

    public List<OrgExtension> findByOrgIdAndKey(long orgId, String extensionKey) {
        return orgExtensionDao.findByOrgIdAndKey(orgId, extensionKey);
    }

}
