package com.xforceplus.business.tenant.service;

import com.xforceplus.api.model.OrgModel;
import com.xforceplus.dao.OrgStructDao;
import com.xforceplus.entity.OrgStruct;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.util.List;

@Service
public class OrgParentIdsService {
    private final static Logger logger = LoggerFactory.getLogger(OrgParentIdsService.class);

    private final OrgService orgService;

    private final OrgStructDao orgStructDao;

    public OrgParentIdsService(OrgService orgService, OrgStructDao orgStructDao) {
        this.orgService = orgService;
        this.orgStructDao = orgStructDao;
    }

    @Async("threadPoolExecutor")
    public void rebuildAllParentIds() {
        Pageable pageable = PageRequest.of(0, 100);
        Page<OrgStruct> roots;
        do {
            roots = orgService.findRoots(pageable);
            if (roots.isEmpty()) {
                return;
            }
            roots.forEach(this::rebuildOneOrg);
            if (!roots.isLast()) {
                pageable = pageable.next();
            }
        } while (!roots.isLast());
    }

    @Async("threadPoolExecutor")
    public void rebuildParentIds(long tenantId) {
        Pageable pageable = PageRequest.of(0, 100);
        Page<OrgStruct> page;
        do {
            page = orgService.findRoots(tenantId, pageable);
            if (page.isEmpty()) {
                return;
            }
            page.forEach(this::rebuildOneOrg);
            if (!page.isLast()) {
                pageable = pageable.next();
            }
        } while (!page.isLast());
    }

    public void rebuildOneOrg(OrgStruct org) {
        if (org == null) {
            return;
        }
        String parentIds = this.buildParentIds(org);
        orgService.updateParentIds(org.getOrgId(), parentIds);
        OrgModel.Request.Query query = new OrgModel.Request.Query();
        query.setParentId(org.getOrgId());
        List<OrgStruct> children = orgService.list(query, Sort.unsorted());
        children.forEach(this::rebuildOneOrg);
    }

    private String buildParentIds(OrgStruct entity) {
        StringBuilder builder = new StringBuilder();
        if (entity.getParentId() != null && entity.getParentId() > 0) {
            try {
                String parentIds = orgStructDao.findCommittedParentIdsByOrgId(entity.getParentId());
                if (StringUtils.isNotBlank(parentIds)) {
                    builder.append(parentIds);
                }
            } catch (Exception e) {
                logger.warn(e.getMessage(), e);
            }
        }
        if (entity.getOrgId() != null) {
            builder.append(entity.getOrgId());
        }
        if (entity.getOrgType() != null) {
            builder.append(entity.getOrgType().separator());
        }
        return builder.toString();
    }
}
