/*
 * Copyright (c)  2015~2025, Xforceplus
 * All rights reserved.
 * Project: tenant-service
 * ClassName: GradingRoleService
 * Date: 2021-08-31 09:57:55
 * Author: zhouxin
 */
package com.xforceplus.business.tenant.service;

import com.xforceplus.api.model.RoleModel;
import com.xforceplus.api.model.RoleModel.Request.*;
import com.xforceplus.dao.*;
import com.xforceplus.domain.resource.ResourcesetDto;
import com.xforceplus.domain.tenant.OrgRoleDTO;
import com.xforceplus.entity.Resourceset;
import com.xforceplus.entity.Role;
import com.xforceplus.entity.RoleResourcesetRel;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @Description: 组织角色服务
 * @CopyRight: 上海云砺信息科技有限公司
 * @Author: zhouxin
 * @Email: yf_zhouxin@xforceplus.com
 * @Date: 2021/8/31 09:57:55
 */
@Service
public class OrgRoleService {

    @Autowired
    private RoleService roleService;

    @Autowired
    private OrgStructDao orgStructDao;

    @Autowired
    private RoleUserRelDao roleUserRelDao;

    @Autowired
    private RoleResourcesetRelDao roleResourcesetRelDao;

    @Autowired
    private UserDao userDao;

    @Autowired
    private ResourcesetDao resourcesetDao;

    /**
     * 组织角色创建
     *
     * @param model
     * @return
     */
    @Transactional(rollbackFor = {Exception.class})
    public Role create(OrgRoleCreate model) {
        Create role = new Create();
        BeanUtils.copyProperties(model, role);
        role.setType(2);

        if (CollectionUtils.isEmpty(model.getResourcesetIds())) {
            role.setStatus(0);
        }

        Role base = roleService.create(role);

        if (!CollectionUtils.isEmpty(model.getResourcesetIds())) {
            Set<RoleResourcesetRel> roleResourcesetRels = new HashSet<>();
            for (Long resourcesetId : model.getResourcesetIds()) {
                RoleResourcesetRel rel = new RoleResourcesetRel();
                rel.setRoleId(base.getId());
                rel.setTenantId(base.getTenantId());
                rel.setResousesetId(resourcesetId);
                roleResourcesetRels.add(rel);
            }
            roleResourcesetRelDao.saveAllAndFlush(roleResourcesetRels);
        }
        return base;
    }

    /**
     * 组织角色分页列表查询
     *
     * @param query
     * @param currentPageable
     * @return
     */
    public Page<OrgRoleDTO> page(Query query, Pageable currentPageable) {
        // 只查组织角色
        query.setType(2);
        Page<Role> rolePage = roleService.page(query, currentPageable);

        Page<OrgRoleDTO> page = rolePage.map(orgRole -> {
            OrgRoleDTO dto = new OrgRoleDTO();
            BeanUtils.copyProperties(orgRole, dto);
            return dto;
        });

        return page;
    }

    /**
     * 组织角色详情
     *
     * @param roleId
     * @return
     */
    public OrgRoleDTO findByRoleId(Long roleId) {
        Role role = roleService.findById(roleId);
        OrgRoleDTO dto = new OrgRoleDTO();
        BeanUtils.copyProperties(role, dto);
        return dto;
    }

    /**
     * 组织角色更新
     *
     * @param tenantId
     * @param roleId
     * @param model
     * @return
     */
    @Transactional(rollbackFor = {Exception.class})
    public OrgRoleDTO updateByTenantId(long tenantId, Long roleId, OrgRoleUpdate model) {
        RoleModel.Request.Update update = new Update();
        model.setType(2);
        BeanUtils.copyProperties(model, update);
        Role role = roleService.updateByTenantId(tenantId, roleId, update);

        OrgRoleDTO dto = new OrgRoleDTO();
        BeanUtils.copyProperties(role, dto);

        this.saveOrgRoleResourcesets(role, model);

        return dto;
    }

    private void saveOrgRoleResourcesets(Role role, OrgRoleUpdate model) {
        List<Resourceset> resourcesets = Collections.emptyList();
        List<RoleResourcesetRel> rels = role.getRoleResourcesetRels();
        if (!CollectionUtils.isEmpty(rels)) {
            List<Long> resourcesetIds = rels.stream().map(RoleResourcesetRel::getResousesetId).collect(Collectors.toList());
            resourcesets = resourcesetDao.findByResourcesetIds(resourcesetIds);
        }
        // 如果未绑定过功能集
        if (CollectionUtils.isEmpty(resourcesets)) {
            if (model != null && !CollectionUtils.isEmpty(model.getResourcesetIds())) {
                this.batchRelOrgRoleResourcesets(role, model.getResourcesetIds());
            } else {
                roleService.disable(role.getId());
            }
        } else {
            Set<Long> bindedResourcesetIds = resourcesets.stream().filter(Objects::nonNull).map(ResourcesetDto::getResourcesetId).collect(
                    Collectors.toSet());
            // 未传递功能集(批量删除)
            if (model == null || CollectionUtils.isEmpty(model.getResourcesetIds())) {
                this.unbindOrgRoleResourcesets(role.getId(), bindedResourcesetIds);
                // 置空功能集时，组织角色的状态需要设置为 未启用
                roleService.disable(role.getId());
            } else {
                // 新减旧 差集为新增的
                Set<Long> toBindResourcesetIds = new HashSet<>(model.getResourcesetIds());
                toBindResourcesetIds.removeAll(bindedResourcesetIds);
                if (!CollectionUtils.isEmpty(toBindResourcesetIds)) {
                    this.batchRelOrgRoleResourcesets(role, toBindResourcesetIds);
                }
                // 旧减新 差集为要删除的
                toBindResourcesetIds.clear();
                toBindResourcesetIds = new HashSet<>(model.getResourcesetIds());
                bindedResourcesetIds.removeAll(toBindResourcesetIds);
                if (!CollectionUtils.isEmpty(bindedResourcesetIds)) {
                    this.unbindOrgRoleResourcesets(role.getId(), bindedResourcesetIds);
                }
            }
        }
    }

    private void unbindOrgRoleResourcesets(Long roleId, Set<Long> bindedResourcesetIds) {
        roleResourcesetRelDao.unbindGradingRoleResourcesets(roleId, bindedResourcesetIds);
    }

    private void batchRelOrgRoleResourcesets(Role role, Set<Long> resourcesetIds) {
        Set<RoleResourcesetRel> roleResourcesetRels = new HashSet<>();
        for (Long resourcesetId : resourcesetIds) {
            Resourceset r = resourcesetDao.findById(resourcesetId).orElseThrow(() -> new IllegalArgumentException("功能集id非法!"));
            RoleResourcesetRel rel = new RoleResourcesetRel();
            rel.setRoleId(role.getId());
            rel.setTenantId(role.getTenantId());
            rel.setResousesetId(resourcesetId);
            roleResourcesetRels.add(rel);
        }
        roleResourcesetRelDao.saveAllAndFlush(roleResourcesetRels);
    }

    /**
     * 查询组织角色关联的功能集
     *
     * @param roleId
     * @return
     */
    public List<Resourceset> findBindedResourcesetsByRoleId(Long roleId) {
        Role role = roleService.findById(roleId);
        List<Resourceset> resourcesets = Collections.EMPTY_LIST;
        List<RoleResourcesetRel> rels = role.getRoleResourcesetRels();
        if (!CollectionUtils.isEmpty(rels)) {
            List<Long> resourcesetIds = rels.stream().map(RoleResourcesetRel::getResousesetId).collect(Collectors.toList());
            return resourcesetDao.findByResourcesetIds(resourcesetIds);
        }
        return Collections.EMPTY_LIST;
    }

    @Transactional(rollbackFor = {Exception.class})
    public void deleteByTenantIdAndId(long tenantId, Long roleId) {
        Role role = roleService.findById(roleId);
        if (role.getStatus() != 0) {
            throw new IllegalArgumentException("只能删除未启用的组织角色!");
        }
        roleService.deleteByTenantIdAndId(tenantId, roleId);
        orgStructDao.unSetOrgStructDefaultOrgRole(tenantId, roleId);
    }

    @Transactional(rollbackFor = {Exception.class})
    public void enable(Long roleId) {
        Role role = roleService.findById(roleId);
        List<Resourceset> resourcesets = Collections.EMPTY_LIST;
        List<RoleResourcesetRel> rels = role.getRoleResourcesetRels();
        if (!CollectionUtils.isEmpty(rels)) {
            List<Long> resourcesetIds = rels.stream().map(RoleResourcesetRel::getResousesetId).collect(Collectors.toList());
            resourcesets = resourcesetDao.findByResourcesetIds(resourcesetIds);
        }
        if (CollectionUtils.isEmpty(resourcesets)) {
            throw new IllegalArgumentException("该角色关联的功能集为空，不能启用!");
        }
        boolean anyEnabled = resourcesets.stream().anyMatch(r -> r.getStatus() == 1);
        if (!anyEnabled) {
            throw new IllegalArgumentException("该角色关联的功能集都是停用状态，不能启用!");
        }
        roleService.enable(roleId);
    }

    @Transactional(rollbackFor = {Exception.class})
    public void disable(Long roleId) {
        roleService.disable(roleId);
    }
}
