package com.xforceplus.business.tenant.service;

import com.xforceplus.api.model.CompanyModel;
import com.xforceplus.api.model.OrgModel;
import com.xforceplus.api.model.OrgModel.Constant;
import com.xforceplus.api.model.OrgModel.Request.*;
import com.xforceplus.api.model.OrgModel.Response;
import com.xforceplus.api.model.TreeModel;
import com.xforceplus.bo.org.OrgCompanyQueryBo;
import com.xforceplus.bo.org.OrgUserStatisticsQueryBo;
import com.xforceplus.business.company.service.CompanyApplyService;
import com.xforceplus.business.company.service.CompanyService;
import com.xforceplus.business.excel.company.OrgCompanyNumberDTO;
import com.xforceplus.business.messagebus.OrgPubService;
import com.xforceplus.business.org.virtual.service.OrgVirtualOrgStructService;
import com.xforceplus.dao.*;
import com.xforceplus.domain.org.OrgExtensionDto;
import com.xforceplus.domain.org.view.OrgExtraInfo;
import com.xforceplus.domain.validation.ValidationGroup;
import com.xforceplus.dto.org.OrgCompanyDTO;
import com.xforceplus.dto.org.OrgCompanyNoDTO;
import com.xforceplus.dto.org.OrgParentDTO;
import com.xforceplus.dto.org.OrgUserStatisticsDTO;
import com.xforceplus.entity.*;
import com.xforceplus.query.OrgQueryHelper;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.tenant.security.core.domain.OrgType;
import com.xforceplus.tenant.security.token.domain.IRole;
import com.xforceplus.utils.OrgUtils;
import com.xforceplus.utils.RoleUtils;
import com.xforceplus.utils.excel.ExcelUtils;
import com.xforceplus.utils.excel.exception.ImportException;
import io.geewit.core.utils.enums.BinaryUtils;
import io.geewit.core.utils.reflection.BeanUtils;
import io.geewit.core.utils.tree.TreeUtils;
import io.geewit.data.jpa.essential.domain.EntityGraphs;
import io.geewit.utils.uuid.UUIDUtils;
import lombok.*;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.lang3.tuple.Triple;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.validation.annotation.Validated;

import javax.persistence.Tuple;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import javax.validation.ConstraintViolation;
import javax.validation.Valid;
import javax.validation.Validator;
import java.security.InvalidParameterException;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.xforceplus.api.utils.Separator.BACKSLASH;
import static java.util.stream.Collectors.*;
import static org.apache.commons.lang3.StringUtils.EMPTY;


@SuppressWarnings("all")
@Validated
@Service
public class OrgService {
    private final static Logger logger = LoggerFactory.getLogger(OrgService.class);

    private final OrgExtensionService orgExtensionService;

    private final OrgStructDao orgStructDao;

    private final OrgUserRelDao orgUserRelDao;

    private final TenantDao tenantDao;

    private final CompanyDao companyDao;

    private final UserDao userDao;

    private final Validator validator;

    private final CompanyService companyService;

    private final SettleStepDao settleStepDao;

    private final CompanyApplyService companyApplyService;

    private final OrgExtensionDao orgExtensionDao;

    private final OrgCompanynoDao orgCompanynoDao;

    private final OrgVirtualOrgStructService orgVirtualOrgStructService;

    private final TenantPolicyDao tenantPolicyDao;

    private final OrgUserService orgUserService;
    private final OrgVirtualDao orgVirtualDao;

    private final OrgPubService orgPubService;

    private final TenantPolicyService tenantPolicyService;

    private final RoleUserRelDao roleUserRelDao;

    private final RoleService roleService;

    private final RoleDao roleDao;

    private final TenantCompanyRelDao tenantCompanyRelDao;

    @Value("${xforce.global.grading-managment.enable:true}")
    private boolean globalGradingEnabled;

    private static final String SAVE = "save";
    private static final String DELETE = "delete";


    public OrgService(OrgStructDao orgStructDao, OrgUserRelDao orgUserRelDao,
                      TenantDao tenantDao, CompanyDao companyDao, UserDao userDao, OrgExtensionService orgExtensionService,
                      Validator validator, CompanyService companyService, SettleStepDao settleStepDao,
                      CompanyApplyService companyApplyService, OrgExtensionDao orgExtensionDao,
                      OrgCompanynoDao orgCompanynoDao, OrgVirtualOrgStructService orgVirtualOrgStructService,
                      TenantPolicyDao tenantPolicyDao, OrgUserService orgUserService, OrgVirtualDao orgVirtualDao,
                      @Autowired(required = false) OrgPubService orgPubService, TenantPolicyService tenantPolicyService, RoleUserRelDao roleUserRelDao,
                      RoleService roleService, RoleDao roleDao, TenantCompanyRelDao tenantCompanyRelDao) {
        this.orgStructDao = orgStructDao;
        this.orgUserRelDao = orgUserRelDao;
        this.tenantDao = tenantDao;
        this.companyDao = companyDao;
        this.userDao = userDao;
        this.orgExtensionService = orgExtensionService;
        this.validator = validator;
        this.companyService = companyService;
        this.settleStepDao = settleStepDao;
        this.companyApplyService = companyApplyService;
        this.orgExtensionDao = orgExtensionDao;
        this.orgCompanynoDao = orgCompanynoDao;
        this.orgVirtualOrgStructService = orgVirtualOrgStructService;
        this.tenantPolicyDao = tenantPolicyDao;
        this.orgUserService = orgUserService;
        this.orgVirtualDao = orgVirtualDao;
        this.orgPubService = orgPubService;
        this.tenantPolicyService = tenantPolicyService;
        this.roleUserRelDao = roleUserRelDao;
        this.roleService = roleService;
        this.roleDao = roleDao;
        this.tenantCompanyRelDao = tenantCompanyRelDao;
    }

    public Page<OrgStruct> page(Query query, Pageable pageable) {
        if (query.getTenantId() != null && query.getTenantId() > 0 && StringUtils.isNotBlank(query.getModules())) {
            String[] orgCodes = StringUtils.split(query.getModules(), ",");
            if (ArrayUtils.isNotEmpty(orgCodes)) {
                List<String> filteringParentIdsList = orgStructDao.findParentIdsByTenantIdAndOrgCodes(query.getTenantId(), Stream.of(orgCodes).collect(toSet()));
                query.setFilterParentIds(new HashSet<>(filteringParentIdsList));
            }
        }
        IAuthorizedUser authorizedUser = UserInfoHolder.get();
        try {
            this.buildCurrentQuery(query, authorizedUser);
        } catch (InvalidParameterException e) {
            logger.warn(e.getMessage());
            return Page.empty(pageable);
        }
        Page<OrgStruct> page;
        if (query.isTupleSelection()) {
            Page<Tuple> tuples = orgStructDao.findTuples(query, pageable);
            List<OrgStruct> contents = tuples.getContent().stream().map(OrgQueryHelper.tupleMapper(query)).collect(Collectors.toList());
            page = new PageImpl<>(contents, pageable, tuples.getTotalElements());
        } else {
            Specification<OrgStruct> specification = OrgQueryHelper.querySpecification(query);
            page = orgStructDao.findAll(specification, pageable, EntityGraphs.named(OrgStruct.NAMED_ENTITY_GRAPH_DEFAULT));
        }
        if (page == null) {
            return Page.empty(pageable);
        }
        List<OrgStruct> contents = this.fillOrgExtend(page.getContent(), query);
        return new PageImpl<>(contents, pageable, page.getTotalElements());
    }

    public List<OrgStruct> list(Query query, Sort sort) {
        if (query.getTenantId() != null && query.getTenantId() > 0 && StringUtils.isNotBlank(query.getModules())) {
            String[] orgCodes = StringUtils.split(query.getModules(), ",");
            if (ArrayUtils.isNotEmpty(orgCodes)) {
                List<String> filteringParentIdsList = orgStructDao.findParentIdsByTenantIdAndOrgCodes(query.getTenantId(), Stream.of(orgCodes).collect(toSet()));
                Set<String> filterParentIds;
                if (query.getFilterParentIds() == null || query.getFilterParentIds().isEmpty()) {
                    filterParentIds = new HashSet<>(filteringParentIdsList);
                } else {
                    filterParentIds = query.getFilterParentIds();
                    if (filteringParentIdsList != null && !filteringParentIdsList.isEmpty()) {
                        filterParentIds = filterParentIds.stream().filter(i -> filteringParentIdsList.stream().anyMatch(p -> p.equals(i))).collect(toSet());
                    }
                }
                query.setFilterParentIds(filterParentIds);
            }
        }
        IAuthorizedUser authorizedUser = UserInfoHolder.get();
        try {
            this.buildCurrentQuery(query, authorizedUser);
        } catch (InvalidParameterException e) {
            return Collections.emptyList();
        }
        List<OrgStruct> list;
        if (query.isTupleSelection()) {
            List<Tuple> tuples = orgStructDao.findTuples(query, sort);
            list = tuples.stream().map(OrgQueryHelper.tupleMapper(query)).collect(Collectors.toList());
        } else {
            Specification<OrgStruct> specification = OrgQueryHelper.querySpecification(query);
            list = orgStructDao.findAll(specification, sort, EntityGraphs.named(OrgStruct.NAMED_ENTITY_GRAPH_DEFAULT));
        }
        list = this.fillOrgExtend(list, query);
        return list;
    }

    /**
     * @param query
     * @return
     */
    public List<OrgStruct> roots(Query query) {
        query.setRootOrg(true);
        query.setStatus(1);
        query.setWithChildrenCount(true);
        query.setWithUserBoundFlag(true);
        List<OrgStruct> roots = this.list(query, Sort.unsorted());
        return roots;
    }

    private void buildLazyLoadCurrentQuery(Query query) {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        Long tenantId = authorizedUser.getTenantId();
        query.setTenantId(tenantId);
        Set<String> treeAttributes = Stream.of("orgId", "orgName", "parentId", "orgType").collect(toSet());
        if (query.getAttributes() == null || query.getAttributes().isEmpty()) {
            query.setAttributes(treeAttributes);
        } else {
            query.getAttributes().addAll(treeAttributes);
        }
        query.setStatus(1);
        if (query.getRootOrg() == null || !query.getRootOrg()) {
            Long userId = authorizedUser.getId();
            query.setUserId(userId);
            Set<String> filterParentIds = orgStructDao.findParentIdsByTenantIdAndUserId(tenantId, userId);
            if (query.getFilterParentIds() != null && !query.getFilterParentIds().isEmpty()) {

                if (filterParentIds != null && !filterParentIds.isEmpty()) {
                    final Set<String> filterParentIdsSet = filterParentIds;
                    filterParentIds = query.getFilterParentIds().stream().filter(i -> filterParentIdsSet.stream().anyMatch(p -> p.equals(i))).collect(toSet());
                }
            }
            filterParentIds = OrgUtils.compressParentIdsCollection(filterParentIds);
            query.setFilterParentIds(filterParentIds);
        }
        this.buildFilterParentIds(query, authorizedUser);
    }

    /**
     * @param query
     * @return
     */
    public List<OrgStruct> lazyLoadCurrentUserOrgTrees(Query query) {
        query.setTree(true);
        query.setWithChildrenCount(true);
        this.buildLazyLoadCurrentQuery(query);
        List<OrgStruct> list = orgStructDao.findAttributes(query, Sort.unsorted());
        return list;
    }

    /**
     * @param query
     * @return
     */
    public List<OrgStruct> lazyLoadCurrentUserOrgRoots(Query query) {
        query.setRootOrg(true);
        query.setTree(true);
        query.setWithChildrenCount(true);
        int status = query.getStatus() == null ? 1 : query.getStatus();
        query.setStatus(status);
        this.buildLazyLoadCurrentQuery(query);
        List<OrgStruct> list = orgStructDao.findAttributes(query, Sort.unsorted());
        list = this.fillOrgExtend(list, query);
        return list;
    }

    /**
     * @param query
     * @return
     */
    public List<OrgStruct> lazyLoadCurrentUserOrgChildren(Query query) {
        query.setTree(true);
        query.setWithChildrenCount(true);
        this.buildLazyLoadCurrentQuery(query);
        List<OrgStruct> list = orgStructDao.findAttributes(query, Sort.unsorted());
        list = this.fillOrgExtend(list, query);
        return list;
    }

    public List<OrgStruct> lazyLoadCurrentUserOrgDescendants(OrgModel.Request.TreeQuery treeQuery) {
        Query query = new Query();
        query.setTenantId(treeQuery.getTenantId());
        query.setTree(true);
        query.setOrgNameLike(treeQuery.getOrgName());
        query.setWithChildrenCount(treeQuery.getWithChildrenCount());
        query.setWithUserBoundFlag(treeQuery.getWithUserBound());
        Set<Long> excludeParentIds;
        if (treeQuery.getRootId() != null && treeQuery.getRootId() > 0) {
            String parentIds = orgStructDao.findCommittedParentIdsByOrgId(treeQuery.getRootId());
            query.setParentIds(parentIds);
            excludeParentIds = OrgUtils.findOrgIdInParentIds(parentIds).stream().filter(id -> !id.equals(treeQuery.getRootId())).collect(toSet());
        } else {
            excludeParentIds = new HashSet<>();
        }
        Set<String> treeAttributes = Stream.of("orgId", "orgName", "orgType", "parentId", "parentIds").collect(toSet());
        if (query.getAttributes() == null || query.getAttributes().isEmpty()) {
            query.setAttributes(treeAttributes);
        } else {
            query.getAttributes().addAll(treeAttributes);
        }
        this.buildLazyLoadCurrentQuery(query);
        List<OrgStruct> list = orgStructDao.findAttributes(query, Sort.unsorted());

        Set<String> parentIdsSet = list.stream().map(OrgStruct::getParentIds).collect(toSet());
        Set<Long> loadedOrgIds = list.stream().map(OrgStruct::getOrgId).collect(toSet());
        excludeParentIds.addAll(loadedOrgIds);
        Set<Long> orgIds = parentIdsSet.stream().map(parentIds -> OrgUtils.findOrgIdInParentIds(parentIds)).flatMap(Collection::stream).filter(id -> excludeParentIds.stream().noneMatch(excludeParentId -> excludeParentId.equals(id))).collect(toSet());
        if (orgIds != null && !orgIds.isEmpty()) {
            query = new Query();
            query.setTenantId(treeQuery.getTenantId());
            query.setOrgIds(orgIds);
            query.setAttributes(treeAttributes);
            query.setWithChildrenCount(treeQuery.getWithChildrenCount());
            query.setWithUserBoundFlag(treeQuery.getWithUserBound());
            query.setTree(true);
            List<OrgStruct> parentOrgs = orgStructDao.findAttributes(query, Sort.unsorted());
            list.addAll(parentOrgs);
        }
        list = this.fillOrgExtend(list, query);
        return list;
    }

    private void buildFilterParentIds(Query query, IAuthorizedUser authorizedUser) {
        if (authorizedUser == null) {
            return;
        }
        boolean tenantGradingEnabled = tenantPolicyService.tenantGradingManagementEnabled(authorizedUser.getTenantId());
        if (globalGradingEnabled && tenantGradingEnabled) {
            Set<IRole> roles = authorizedUser.getRoles();
            Triple<Boolean, Set<Long>, Set<Long>> adminAndGradingRoles = RoleUtils.calcAdminAndGradingRoles(authorizedUser);
            boolean isAdmin = adminAndGradingRoles.getLeft();
            Set<Long> gradingRoleIds = adminAndGradingRoles.getMiddle();
            Set<Long> orgRoleIds = adminAndGradingRoles.getRight();
            Set<String> filterParentIds;
            if (orgRoleIds != null && !orgRoleIds.isEmpty()) {
                Set<Long> orgIds = roleUserRelDao.findOrgIdsByTenantIdAndRoleIdsAndType(authorizedUser.getTenantId(), orgRoleIds);
                filterParentIds = orgStructDao.findParentIdsByTenantIdAndOrgIds(authorizedUser.getTenantId(), orgIds);
            } else {
                filterParentIds = new HashSet<>();
            }
            if (gradingRoleIds != null && !gradingRoleIds.isEmpty()) {
                List<String> gradingOrgParentIdsList = orgStructDao.listParentIdsByGradingRoleIds(gradingRoleIds);
                filterParentIds.addAll(gradingOrgParentIdsList);
            }
            if (query.getFilterParentIds() != null && !query.getFilterParentIds().isEmpty()) {
                if (filterParentIds != null && !filterParentIds.isEmpty()) {
                    final Set<String> filterParentIdsSet = filterParentIds;
                    filterParentIds = query.getFilterParentIds().stream().filter(i -> filterParentIdsSet.stream().anyMatch(p -> p.equals(i))).collect(toSet());
                }
            }
            if (filterParentIds != null && !filterParentIds.isEmpty()) {
                query.setFilterParentIds(filterParentIds);
            } else if (!isAdmin) {
                throw new InvalidParameterException("非管理员身份, 返回空列表");
            }
        }
    }

    /**
     * 构造查询Query对象
     *
     * @param query
     * @param authorizedUser
     */
    public void buildCurrentQuery(Query query, IAuthorizedUser authorizedUser) {
        if (authorizedUser == null) {
            return;
        }
        this.buildFilterParentIds(query, authorizedUser);

        String modules = authorizedUser.getModules();
        query.setModules(modules);
    }

    private List<OrgStruct> fillOrgExtend(List<OrgStruct> orgs, Query query) {
        if (orgs == null || orgs.isEmpty()) {
            return orgs;
        }
        List<OrgStruct> result = new ArrayList<>(orgs.size());
        boolean enableUserCount = false;
        boolean enableParentName = false;
        boolean enableExtensions = false;
        boolean enableCompanyNos = false;
        boolean enableParentOrgs = false;
        boolean enableChildrenCount = ObjectUtils.defaultIfNull(query.getWithChildrenCount(), false);
        if (StringUtils.isNotBlank(query.getWithExtendParams())) {
            String[] withExtendParamsArray = StringUtils.split(query.getWithExtendParams(), ",");
            for (String withExtendParam : withExtendParamsArray) {
                if ("userCount".equalsIgnoreCase(withExtendParam)) {
                    enableUserCount = true;
                } else if ("parentName".equalsIgnoreCase(withExtendParam)) {
                    enableParentName = true;
                } else if ("extensions".equalsIgnoreCase(withExtendParam)) {
                    enableExtensions = true;
                } else if ("companyNos".equalsIgnoreCase(withExtendParam)) {
                    //默认是查询-批量查询CompanyNos
                    enableCompanyNos = true;
                } else if ("parentOrgs".equalsIgnoreCase(withExtendParam)) {
                    enableParentOrgs = true;
                } else if (!enableChildrenCount && "childrenCount".equalsIgnoreCase(withExtendParam)) {
                    enableChildrenCount = true;
                }
            }
        }
        Set<Long> orgIds = new HashSet<>(orgs.size());
        Set<Long> companyIds = new HashSet<>();
        Set<Long> tenantIds = new HashSet<>();
        Map<Long, Long> orgUserStatisticsDTOMap = Collections.emptyMap();
        Map<Long, String> orgParentDTOMap = Collections.emptyMap();
        Map<Long, Set<String>> orgCompanyNoMap = new HashMap<>();
        Map<Long, List<OrgExtension>> orgExtensionMap = Collections.emptyMap();
        Map<Long, Long> childrenCountMaps = Collections.emptyMap();
        Set<Long> orgParentIds = new HashSet<>(orgs.size());
        Map<Long, Company> companyMap = Collections.emptyMap();
        boolean hasCompanyLoaded = false;
        Map<Long, Tenant> tenantMap = Collections.emptyMap();
        boolean hasTenantLoaded = false;
        for (OrgStruct org : orgs) {
            if (org == null) {
                continue;
            }
            if (org.getOrgId() != null) {
                orgIds.add(org.getOrgId());
            }
            if (org.getParentId() != null) {
                orgParentIds.add(org.getParentId());
            }
            if (!query.isTree()) {
                hasTenantLoaded |= org.isTenantLoaded();
                if (org.getTenantId() != null && org.getTenantId() > 0) {
                    tenantIds.add(org.getTenantId());
                }

                hasCompanyLoaded |= org.isCompanyLoaded();
                if (OrgType.COMPANY.equals(org.getOrgType()) && org.getCompanyId() != null && org.getCompanyId() > 0) {
                    companyIds.add(org.getCompanyId());
                }
            }
        }
        //region 查询用户统计数据
        if (enableUserCount) {
            orgUserStatisticsDTOMap = this.findOrgUserStatisticsByOrgIds(orgIds);
        }
        //endregion

        //region 查询父级组织
        if (enableParentName) {
            orgParentDTOMap = this.findOrgParentNameByOrgParentIds(orgParentIds);
        }
        //endregion

        //region 组织扩展列表信息查询
        if (enableExtensions) {
            orgExtensionMap = this.findOrgExtensionByOrgIds(orgIds);
        }
        //endregion

        //region 组织树节点的children数
        if (enableChildrenCount) {
            List<Pair<Long, Long>> childrenCountPairs = orgStructDao.findChildrenCountMapsByOrgIds(orgIds);
            childrenCountMaps = childrenCountPairs.stream().collect(Collectors.toMap(Pair::getLeft, Pair::getRight));
        }
        //endregion

        //region 设置companyNos
        if (enableCompanyNos) {
            List<OrgCompanyRel> orgCompanyRels = orgCompanynoDao.findCompanyNosByOrgIds(orgIds);
            for (OrgCompanyRel rel : orgCompanyRels) {
                if (StringUtils.isBlank(rel.getCompanyNo())) {
                    continue;
                }
                if (orgCompanyNoMap.containsKey(rel.getOrgStructId())) {
                    Set<String> companyNos = orgCompanyNoMap.get(rel.getOrgStructId());
                    if (companyNos == null) {
                        companyNos = Stream.of(rel.getCompanyNo()).collect(toSet());
                    } else {
                        companyNos.add(rel.getCompanyNo());
                    }
                    orgCompanyNoMap.put(rel.getOrgStructId(), companyNos);
                } else {
                    orgCompanyNoMap.put(rel.getOrgStructId(), Stream.of(rel.getCompanyNo()).collect(toSet()));
                }
            }
        }
        //endregion

        //region 设置 tenantMap
        if (!hasTenantLoaded && tenantIds != null && !tenantIds.isEmpty()) {
            tenantMap = tenantDao.findTenantMapByTenantIds(tenantIds);
        }
        //endregion

        //region 设置 companyMap
        if (!hasCompanyLoaded && companyIds != null && !companyIds.isEmpty()) {
            companyMap = companyDao.findCompanyMapByCompanyIds(companyIds);
        }
        //endregion

        for (OrgStruct org : orgs) {
            if (org != null && org.getOrgId() != null && org.getOrgId() > 0) {
                OrgStruct resultOrg = new OrgStruct();
                Set<String> ignoreProperties = Stream.of("companyNos", "gradingRole", "defaultOrgRole", "orgUserRels").collect(toSet());
                if (!hasCompanyLoaded) {
                    ignoreProperties.add("company");
                }
                if (!hasTenantLoaded) {
                    ignoreProperties.add("tenant");
                }
                BeanUtils.copyProperties(org, resultOrg, ignoreProperties.stream().toArray(String[]::new));
                if (enableUserCount) {
                    if (orgUserStatisticsDTOMap.containsKey(org.getOrgId())) {
                        resultOrg.setUserCount(orgUserStatisticsDTOMap.get(org.getOrgId()));
                    } else {
                        resultOrg.setUserCount(0L);
                    }
                }
                if (enableChildrenCount) {
                    if (childrenCountMaps.containsKey(org.getOrgId())) {
                        resultOrg.setChildrenCount(childrenCountMaps.get(org.getOrgId()));
                    } else {
                        resultOrg.setChildrenCount(0L);
                    }
                }
                if (enableParentName) {
                    if (org.getParentId() != null && org.getParentId() > 0 && orgParentDTOMap.containsKey(org.getParentId())) {
                        resultOrg.setParentName(orgParentDTOMap.get(org.getParentId()));
                    } else {
                        resultOrg.setParentName(EMPTY);
                    }
                }
                if (enableExtensions) {
                    if (orgExtensionMap.containsKey(org.getOrgId())) {
                        List<OrgExtension> extentions = orgExtensionMap.get(org.getOrgId());
                        resultOrg.setExtensions(new HashSet<>(extentions));
                    } else {
                        resultOrg.setExtensions(Collections.emptySet());
                    }
                }
                if (enableParentOrgs) {
                    Set<Long> parentOrgIds = OrgUtils.findOrgIdInParentIds(org.getParentIds());
//                            parentOrgIds = parentOrgIds.stream().filter(id -> !id.equals(org.getOrgId())).collect(toSet());
                    List<OrgStruct> parentOrgs = new ArrayList<>();
                    for (Long parentOrgId : parentOrgIds) {
                        Optional<OrgStruct> orgOptional = orgs.stream().filter(o -> o.getOrgId().equals(parentOrgId)).findAny();
                        if (orgOptional.isPresent()) {
                            OrgStruct parentOrg = new OrgStruct();
                            BeanUtils.copyProperties(orgOptional.get(), parentOrg, Stream.of("parentOrgs", "children").toArray(String[]::new));
                            parentOrgs.add(parentOrg);
                        } else {
                            OrgStruct parentOrg = this.findById(parentOrgId, 0);
                            if (parentOrg != null) {
                                parentOrgs.add(parentOrg);
                            }
                        }
                    }
                    resultOrg.setParentOrgs(parentOrgs);
                }

                if (enableCompanyNos) {
                    if (orgCompanyNoMap.containsKey(org.getOrgId())) {
                        Set<String> companyNos = orgCompanyNoMap.get(org.getOrgId());
                        resultOrg.setCompanyNos(companyNos);
                    } else {
                        resultOrg.setCompanyNos(Collections.emptySet());
                    }
                }
                if (!hasCompanyLoaded) {
                    if (OrgType.COMPANY.equals(org.getOrgType()) && org.getCompanyId() != null && org.getCompanyId() > 0) {
                        Company company = companyMap.get(org.getCompanyId());
                        if (company != null) {
                            resultOrg.setCompany(company);
                        }
                    }
                }

                if (!hasTenantLoaded) {
                    if (org.getTenantId() != null && org.getTenantId() > 0) {
                        Tenant tenant = tenantMap.get(org.getTenantId());
                        if (tenant != null) {
                            resultOrg.setTenant(tenant);
                        }
                    }
                }
                result.add(resultOrg);
            }
        }
        return result;
    }

    /**
     * 根据组织ID列表查询组织扩展信息
     *
     * @param orgIds 组织IDs
     * @return Map<Long, Set < OrgExtension>>
     */
    public Map<Long, List<OrgExtension>> findOrgExtensionByOrgIds(Collection<Long> orgIds) {
        //为空直接返回数据
        if (CollectionUtils.isEmpty(orgIds)) {
            return Collections.emptyMap();
        }
        List<OrgExtension> orgExtensionList = this.orgExtensionDao.findByOrgIds(orgIds);
        return orgExtensionList.stream()
                .collect(groupingBy(OrgExtension::getOrgStructId));
    }

    /**
     * 根据列表查询数据
     *
     * @param orgIds
     * @return Map<Long, Set < String>>
     */
    public Map<Long, Set<String>> findOrgCompanyNosByOrgIds(List<Long> orgIds) {
        //为空直接返回数据
        if (CollectionUtils.isEmpty(orgIds)) {
            return Collections.emptyMap();
        }
        List<OrgCompanyRel> list = orgCompanynoDao.findByOrgIds(orgIds);
        //以组织ID分，并以CompanyNo转化为Set集合
        return list.stream()
                .filter(e -> StringUtils.isNoneBlank(e.getCompanyNo()))
                .collect(groupingBy(OrgCompanyRel::getOrgStructId, Collectors.mapping(OrgCompanyRel::getCompanyNo, toSet())));

    }


    /**
     * 根据组织Ids查询该列表的值
     *
     * @param orgIds orgIds列表
     * @return Map<Long, String>
     */
    public Map<Long, String> findOrgParentNameByOrgParentIds(Set<Long> orgIds) {
        //为空直接返回数据
        if (CollectionUtils.isEmpty(orgIds)) {
            return Collections.emptyMap();
        }
        List<OrgParentDTO> list = orgStructDao.findOrgParentByOrgIds(orgIds);
        if (CollectionUtils.isEmpty(list)) {
            return Collections.emptyMap();
        }
        return list.stream().collect(Collectors.toMap(OrgParentDTO::getOrgId, OrgParentDTO::getOrgName));
    }


    /**
     * 根据用户组织ids或tenantId统计查询用户数量
     *
     * @param tenantId 租户Id
     * @param orgIds   组织Ids
     * @return Map<Long, OrgUserStatisticsDTO>
     */
    public Map<Long, Long> findOrgUserStatisticsByOrgIds(Set<Long> orgIds) {
        if (CollectionUtils.isEmpty(orgIds)) {
            return Collections.emptyMap();
        }
        OrgUserStatisticsQueryBo queryBo = new OrgUserStatisticsQueryBo();
        queryBo.setOrgIds(orgIds);
        List<OrgUserStatisticsDTO> list = orgStructDao.findUserStatisticsByOrgIds(queryBo);
        if (CollectionUtils.isEmpty(list)) {
            return Collections.emptyMap();
        }
        return list.stream().collect(Collectors.toMap(OrgUserStatisticsDTO::getOrgId, OrgUserStatisticsDTO::getUserTotalNum));
    }

    public Page<OrgStruct> page(Specification<OrgStruct> specification, Pageable pageable) {
        return orgStructDao.findAll(specification, pageable, EntityGraphs.named(OrgStruct.NAMED_ENTITY_GRAPH_DEFAULT));
    }

    public List<OrgStruct> findByTaxNumAndCompanyIdAndTenantId(String taxNum, Long companyId,
                                                               Long tenantId, Long orgId, String tenantCode, boolean withHost, Boolean hostTenantOnly,
                                                               Integer status) {
        Query query = new Query();
        query.setTaxNum(taxNum);
        query.setCompanyId(companyId);
        query.setWithIsHost(withHost);
        query.setOrgId(orgId);
        query.setWithExtendParams("companyNos");
        if (tenantId != null && tenantId > 0) {
            query.setTenantId(tenantId);
            query.setTenantCode(null);
        } else if (StringUtils.isNotBlank(tenantCode)) {
            tenantId = tenantDao.findTenantIdByTenantCode(tenantCode);
            query.setTenantId(tenantId);
            query.setTenantCode(null);
        }
        if (status != null) {
            query.setStatus(status);
        }
        query.setAttributes(Stream.of("org", "company", "tenant").collect(toSet()));
        List<OrgStruct> orgs = this.list(query, Sort.unsorted());
        return orgs;
    }

    public Optional<OrgStruct> findOne(Query query) {
        Specification<OrgStruct> specification = OrgQueryHelper.queryOneSpecification(query);
        return orgStructDao.findOne(specification, EntityGraphs.named(OrgStruct.NAMED_ENTITY_GRAPH_TENANT));
    }

    /**
     * cms 单点登录专用.
     *
     * @param tenantId
     * @param accountId
     * @return
     */
    public List<Response.CmsTenantCompany> listFromCms(String tenantId, String taxNum) {
        Query query = new Query();
        if (!StringUtils.isEmpty(tenantId)) {
            query.setTenantId(Long.valueOf(tenantId));
        }
        if (!StringUtils.isEmpty(taxNum)) {
            query.setTaxNum(taxNum);
        }
        query.setOrgType(OrgType.COMPANY.name());
        query.setStatus(1);
        List<Response.CmsTenantCompany> companyRelList = new ArrayList<>();
        Specification<OrgStruct> specification = OrgQueryHelper.querySpecification(query);
        List<OrgStruct> list = orgStructDao.findAll(specification, EntityGraphs.named(OrgStruct.NAMED_ENTITY_GRAPH_DEFAULT));
        for (OrgStruct org : list) {
            Response.CmsTenantCompany companyRelDto = new Response.CmsTenantCompany();
            companyRelDto.setCompanyName(org.getCompanyName());
            companyRelDto.setTaxNo(org.getTaxNum());
            companyRelDto.setCompanyCode(org.getCompanyCode());
            companyRelDto.setTenantName(org.getTenantName());
            companyRelDto.setTenantCode(org.getTenantCode());
            if (CollectionUtils.isNotEmpty(org.getCompanyNos())) {
                companyRelDto.setCompanyNo(org.getCompanyNos().stream().findFirst().orElse(null));
            }
            companyRelList.add(companyRelDto);
        }
        return companyRelList;
    }


    public Long countByTenantIdAndOrgCode(long tenantId, String orgCode) {
        long orgCodeCount = orgStructDao.countByTenantIdAndOrgCode(tenantId, orgCode);
        return orgCodeCount;
    }

    /**
     * 创建 组织 核心方法
     *
     * @param model 请求报文对象
     * @return 组织实体
     */
    @Transactional(rollbackFor = Exception.class)
    public OrgStruct create(@Valid Save model) {
        logger.info("unique:{}, application:{}", model.getUniqueCompany(), model.isWithApplication());

        Long tenantId = model.getTenantId();
        List<String> tenantPolicies = tenantPolicyDao.findPoliciesByTenantIdAndName(tenantId, TenantPolicy.CREATE_COMPANY_AUDIT);
        if (!tenantPolicies.isEmpty()) {
            String policy = tenantPolicies.stream().findFirst().get();
            boolean withApplication = "1".equals(policy);
            model.setWithApplication(withApplication);
        }
        //新建org，要检查orgCode 是否重复
        if ((model.getOrgId() == null || model.getOrgId() == 0) && StringUtils.isNotEmpty(model.getOrgCode())) {
            long orgCodeCount = this.countByTenantIdAndOrgCode(tenantId, model.getOrgCode());
            if (orgCodeCount > 0) {
                String message = "orgCode (" + model.getOrgCode() + ") 已经存在";
                logger.info(message);
                throw new IllegalArgumentException(message);
            }
        }

        if (model.getUniqueCompany() == null) {
            model.setUniqueCompany(true);
        }
        //PLAT-5255
        if (StringUtils.isBlank(model.getOrgCode())) {
            model.setOrgCode(UUIDUtils.randomUUID());
        }

        if (model.getOrgType() == null && model.getCompany() != null && StringUtils.isNotBlank(model.getCompany().getCompanyName()) && StringUtils.isNotBlank(model.getCompany().getTaxNum())) {
            model.setOrgType(OrgType.COMPANY);
        }

        //当公司不为空时，判断公司名称和税号不能为空
        if (OrgType.COMPANY.equals(model.getOrgType())) {
            this.validCompany(model.getCompany());
        }

        //判断公司申请税号是唯一(以租户ID和税号为空)
        if (OrgType.COMPANY.equals(model.getOrgType()) && model.getUniqueCompany() && model.getCompany() != null) {
            if (model.isWithApplication()) {
                //根据税号判断是否已经申请过
                Optional<CompanyApply> optional = companyApplyService.findByTenantIdAndTaxNum(model.getTenantId(), model.getCompany().getTaxNum());
                if (optional.isPresent()) {
                    //直接抛出异常
                    String message = "税号为:" + model.getCompany().getTaxNum() + "公司已申请，待审批";
                    logger.info(message);
                    throw new IllegalArgumentException(message);
                }
            }
            Optional<Company> companyOptional = orgStructDao.findByTenantIdAndTaxNum(model.getTenantId(), model.getCompany().getTaxNum());
            if (companyOptional.isPresent()) {
                //直接抛出异常
                String message = "税号为:" + model.getCompany().getTaxNum() + "公司在租户内已存在";
                logger.info(message);
                throw new IllegalArgumentException(message);
            }
        }
        if (model.getDefaultOrgRoleId() != null) {
            long countDefaultOrgRoleId = roleDao.countRoleByTenantIdAndIdAndType(model.getTenantId(), model.getDefaultOrgRoleId(), 2);
            if (countDefaultOrgRoleId == 0) {
                String message = "指定的默认组织角色(id:" + model.getDefaultOrgRoleId() + ")不存在";
                logger.info(message);
                throw new IllegalArgumentException(message);
            }
        }
        if (model.getGradingRoleId() != null) {
            long countGradingRoleId = roleDao.countRoleByTenantIdAndIdAndType(model.getTenantId(), model.getGradingRoleId(), 1);
            if (countGradingRoleId == 0) {
                String message = "指定的分级管理员角色(id:" + model.getGradingRoleId() + ")不存在";
                logger.info(message);
                throw new IllegalArgumentException(message);
            }
        }
        OrgStruct orgStructEntity = new OrgStruct();

        BeanUtils.copyProperties(model, orgStructEntity, Stream.of("company", "tenant", "companyNos", "extensions").toArray(String[]::new));

        boolean isOverwrite = true;
        logger.info("create Org, isOverwrite = true");
        //验证属性
        //判断如类型为公司，判断客户为空，则返回提示"类型为公司的组织需要输入公司信息"
        if (OrgType.COMPANY.equals(model.getOrgType()) && model.getCompany() == null) {
            String message = "类型为公司的组织需要输入公司信息";
            logger.info(message);
            throw new IllegalArgumentException(message);
        }
        //region 公司类型且不使用" 不使用审核流程"
        Company company = null;
        if (OrgType.COMPANY.equals(model.getOrgType()) && !model.isWithApplication()) {
            company = this.saveCompany(orgStructEntity.getTenantId(), model, isOverwrite);
            if (company != null) {
                orgStructEntity.setCompanyId(company.getCompanyId());
            }
        }
        //endregion

        //region 设置属性设置
        model.setIsAutoBindParentOrgUsers(Boolean.TRUE);
        orgStructEntity = this.buildOrgStructEntity(model, orgStructEntity, company);
        //endregion

        //region 保存组织
        orgStructEntity = this.saveOrgStructEntity(orgStructEntity);
        //endregion

        if (model.isWithApplication() && model.getCompany() != null && OrgType.COMPANY.equals(model.getOrgType())) {
            //region 使用审核流程
            try {
                companyApplyService.apply(orgStructEntity.getTenantId(), orgStructEntity.getOrgId(), model.getCompany());
            } catch (Exception e) {
                String message = "税号为:" + model.getCompany().getTaxNum() + "公司已申请，待审批";
                logger.warn(e.getMessage() + "," + message, e);
                //直接抛出异常
                throw new IllegalArgumentException(message);
            }
            //endregion
        }
        if (company != null && company.getTaxNum() != null && company.getCompanyName() != null) {
            //判断租户是不是一站式入住的
            Optional<SettleStep> SettleStepEntity = settleStepDao.findByTaxNum(company.getTaxNum());
            //判断当前税号是不是在流程中
            if (!SettleStepEntity.isPresent()) {
                SettleStep settleStep = new SettleStep();
                settleStep.setFlowId(0L);
                settleStep.setCompanyName(company.getCompanyName());
                settleStep.setTaxNum(company.getTaxNum());
                settleStep.setStepStatus(0);
                settleStep.setParamsJson(EMPTY);
                settleStep.setTemplateCode(EMPTY);
                settleStep.setTenantId(model.getTenantId());
                settleStepDao.saveAndFlush(settleStep);
            }
        }
        //判断组织类型等于公司，且公司Id不能为空
        if (OrgType.COMPANY.equals(orgStructEntity.getOrgType()) && CollectionUtils.isNotEmpty(model.getCompanyNos())) {
            companyService.saveCompanyNos(orgStructEntity.getOrgId(), model.getCompanyNos(), model.isOverwrite());
        }

        OrgStruct result = new OrgStruct();
        BeanUtils.copyProperties(orgStructEntity, result);

        if (company != null) {
            result.setCompany(company);
            result.postLoad();
        }
        if (CollectionUtils.isNotEmpty(model.getCompanyNos())) {
            result.setCompanyNos(model.getCompanyNos());
        }
        if (CollectionUtils.isNotEmpty(model.getExtensions())) {
            Set<OrgExtension> extensions = this.orgExtensionService.batchSave(orgStructEntity.getOrgId(), model.getExtensions(), isOverwrite);
            Set<OrgExtensionDto> extensionDtos = extensions.stream().filter(Objects::nonNull).collect(toSet());
            result.setExtensions(extensionDtos);
        }
        return result;
    }

    /**
     * 校验公司信息
     *
     * @param company
     */
    private void validCompany(CompanyModel.Request.Save company) {
        if (company == null) {
            return;
        }
        //判断公司名称不为空
        Assert.hasText(company.getCompanyName(), "公司名称不能为空");
        //判断公司税号不能为空
        Assert.hasText(company.getTaxNum(), "公司税号不能为空");
    }


    /**
     * 创建 组织 核心方法
     *
     * @param model 请求报文对象
     * @return 组织实体
     */
    @Transactional(rollbackFor = Exception.class)
    public OrgStruct createAsync(@Valid Save model) {
        OrgStruct orgStructEntity = new OrgStruct();
        BeanUtils.copyProperties(model, orgStructEntity, Stream.of("company", "tenant", "companyNos", "extensions").toArray(String[]::new));

        boolean isOverwrite = true;
        Boolean isAutoBindParentOrgUsers = Boolean.TRUE;
        logger.info("create Org, isOverwrite = true");
        //验证属性
        Company company = null;
        //判断如类型为公司，判断客户为空，则返回提示"类型为公司的组织需要输入公司信息"
        if (OrgType.COMPANY.equals(model.getOrgType()) && model.getCompany() == null) {
            throw new IllegalArgumentException("类型为公司的组织需要输入公司信息");
        }
        //公司类型且不使用" 不使用审核流程"
        if (OrgType.COMPANY.equals(model.getOrgType()) && !model.isWithApplication()) {
            company = this.saveCompany(orgStructEntity.getTenantId(), model, isOverwrite);
            if (company != null) {
                orgStructEntity.setCompanyId(company.getCompanyId());
            }
        }
        // 使用审核流程
        if (model.isWithApplication() && model.getCompany() != null) {
            companyApplyService.apply(orgStructEntity.getTenantId(), orgStructEntity.getOrgId(), model.getCompany());
        }

        //设置属性设置
        orgStructEntity = this.buildOrgStructEntity(model, orgStructEntity, company);
        //保存或更新
        orgStructEntity = this.saveOrgStructEntity(orgStructEntity);

        logger.info("isAutoBindParentOrgUsers = {}", isAutoBindParentOrgUsers);
        if (isAutoBindParentOrgUsers) {
            logger.info("auto_bind_user,orgId={}, startTime:{}", orgStructEntity.getOrgId(),  System.currentTimeMillis());
            orgUserService.autoBindUsers(orgStructEntity.getTenantId(), orgStructEntity.getOrgId(), System.currentTimeMillis());
        }

        if (company != null) {
            //判断租户是不是一站式入住的
            Optional<SettleStep> SettleStepEntity = settleStepDao.findByTaxNum(company.getTaxNum());
            //判断当前税号是不是在流程中
            if (!SettleStepEntity.isPresent()) {
                SettleStep settleStep = new SettleStep();
                settleStep.setFlowId(0L);
                settleStep.setCompanyName(company.getCompanyName());
                settleStep.setTaxNum(company.getTaxNum());
                settleStep.setStepStatus(0);
                settleStep.setParamsJson(EMPTY);
                settleStep.setTemplateCode(EMPTY);
                settleStep.setTenantId(model.getTenantId());
                settleStepDao.saveAndFlush(settleStep);
            }
        }
        //判断组织类型等于公司，且公司Id不能为空
        if (OrgType.COMPANY.equals(orgStructEntity.getOrgType()) && CollectionUtils.isNotEmpty(model.getCompanyNos())) {
            companyService.saveCompanyNos(orgStructEntity.getOrgId(), model.getCompanyNos(), model.isOverwrite());
        }

        OrgStruct result = new OrgStruct();
        BeanUtils.copyProperties(orgStructEntity, result);

        if (company != null) {
            result.setCompany(company);
        }
        if (CollectionUtils.isNotEmpty(model.getCompanyNos())) {
            result.setCompanyNos(model.getCompanyNos());
        }
        if (CollectionUtils.isNotEmpty(model.getExtensions())) {
            Set<OrgExtension> extensions = orgExtensionService.batchSave(orgStructEntity.getOrgId(), model.getExtensions(), isOverwrite);
            Set<OrgExtensionDto> extensionDtos = extensions.stream().filter(Objects::nonNull).collect(toSet());
            result.setExtensions(extensionDtos);
        }
        return result;
    }

    /**
     * 设置OrgStruct属性
     * @param model Save
     * @param entity OrgStruct
     * @param company {@link Company}
     * @param isAutoBindParentOrgUsers
     */
    private OrgStruct buildOrgStructEntity(Save model, OrgStruct entity, Company company) {
        OrgStruct existEntity = null;
        //判断公司是否
        //判断公司是否存在的情况
        if (OrgType.COMPANY.equals(model.getOrgType()) && company != null && company.getCompanyId() != null && company.getCompanyId() > 0) {
            Query query = new Query();
            query.setTenantId(entity.getTenantId());
            query.setCompanyId(entity.getCompanyId());
            List<OrgStruct> companyOrgs = orgStructDao.findAll(OrgQueryHelper.querySpecification(query));
            if (!companyOrgs.isEmpty()) {
                for (OrgStruct companyOrg : companyOrgs) {
                    if (companyOrg.getStatus() != null && companyOrg.getStatus() == 1) {
                        if (model.getUniqueCompany() != null && model.getUniqueCompany()) {
                            String message = "税号为:" + company.getTaxNum() + "公司已存在, 无法重复提交";
                            logger.warn(message);
                            if (model.getIsStrict() != null && model.getIsStrict()) {
                                throw new IllegalArgumentException(message);
                            }
                        }
                        existEntity = companyOrg;
                    } else {
                        companyOrg.setStatus(1);
                        existEntity = companyOrg;
                    }
                    break;
                }
                BeanUtils.copyProperties(entity, existEntity, Stream.of("orgUserRels", "defaultOrgRole", "gradingRole", "companyNos", "companyNos", "tenant", "company").toArray(String[]::new));
                entity = existEntity;
            }
        }
        Long parentId = null;
        Long tenantId = null;
        if (entity != null) {
            tenantId = entity.getTenantId();
        }
        if (tenantId == null && model.getTenantId() != null) {
            tenantId = model.getTenantId();
        }
        if (tenantId == null) {
            String message = "租户id(" + tenantId + ")为空";
            logger.warn(message);
            if (model.getIsStrict() != null && model.getIsStrict()) {
                throw new IllegalArgumentException(message);
            }
        }
        if (model.getParentId() != null && model.getParentId() > 0) {
            parentId = orgStructDao.findOrgIdByTenantIdAndOrgId(tenantId, model.getParentId());
            if (parentId == null) {
                String message = "非法的父级组织id(" + model.getParentId() + ")";
                logger.warn(message);
                if (model.getIsStrict() != null && model.getIsStrict()) {
                    throw new IllegalArgumentException(message);
                }
            }
        }
        if (parentId == null && StringUtils.isNotBlank(model.getParentCode())) {
            List<Long> parentIdList = orgStructDao.findOrgIdByTenantIdAndOrgCode(tenantId, model.getParentCode());
            if (parentIdList.isEmpty()) {
                String message = "非法的父级组织code(" + model.getParentCode() + ")";
                logger.warn(message);
                if (model.getIsStrict() != null && model.getIsStrict()) {
                    throw new IllegalArgumentException(message);
                }
            } else {
                parentId = parentIdList.get(0);
            }
        }
        if (parentId == null && entity.getParentId() == null) {
            List<Long> rootIds = orgStructDao.findRootIdsByTenantId(tenantId);
            if (!rootIds.isEmpty()) {
                parentId = rootIds.get(0);
            }
        }
        if (parentId != null && parentId <= 0) {
            parentId = null;
        }
        if (parentId != null) {
            entity.setParentId(parentId);
            if (OrgType.GROUP.equals(entity.getOrgType()) && entity.getParentId() > 0) {
                entity.setOrgType(OrgType.NORMAL);
            }

            if (OrgType.COMPANY.equals(entity.getOrgType()) && entity.getCompanyId() != null && entity.getCompanyId() > 0) {
                long countCompanies = orgStructDao.countCompaniesByParentIdAndCompanyId(parentId, entity.getCompanyId());
                if (countCompanies > 0) {
                    String message = "父组织下已经存在相同的公司";
                    logger.warn(message);
                    if (model.getIsStrict() != null && model.getIsStrict()) {
                        throw new IllegalArgumentException(message);
                    }
                }
            }
            model.setIsAutoBindParentOrgUsers(true);
        } else {
            if (!OrgType.GROUP.equals(entity.getOrgType())) {
                if (entity.getParentId() == null) {
                    logger.warn("非集团组织的父级组织为空: tenantId:{},orgCode:{},orgName:{}", tenantId, entity.getOrgCode(), entity.getOrgName());
                    //非集团组织的父级组织为空
                    final long finalTenantId = tenantId;
                    List<Long> rootIds = orgStructDao.findRootIdsByTenantId(tenantId);
                    Long rootId = rootIds.stream().findFirst().orElseThrow(() -> new IllegalArgumentException("未找租户(" + finalTenantId + ")的根组织"));
                    logger.info("tenantId:{}, rootId:{}, orgCode:{}, orgName:{}", tenantId, rootId, entity.getOrgCode(), entity.getOrgName());
                    entity.setParentId(rootId);
                }
                model.setIsAutoBindParentOrgUsers(true);
            }
        }
        if (model.isWithApplication() && model.getCompany() != null) {
            entity.setAuditStatus(0);
            entity.setStatus(0);
        }

        return entity;
    }

    private Company saveCompany(long tenantId, Save model, boolean isOverwrite) {
        Optional<Tenant> tenantOptional = tenantDao.findById(tenantId);
        if (!tenantOptional.isPresent()) {
            throw new IllegalArgumentException("没有对应的租户(id:" + tenantId + ")");
        }
        Set<ConstraintViolation<Save>> constraintViolations = validator.validate(model, ValidationGroup.OnCreate.class);
        if (!constraintViolations.isEmpty()) {
            String errorMsgs = constraintViolations.stream().map(violation -> violation.getPropertyPath() + violation.getMessage()).collect(Collectors.joining(","));
            throw new IllegalArgumentException(errorMsgs);
        }
        if (model.getCompany() == null) {
            return null;
        }
        Company company;
        if (model.getCompany().getCompanyId() != null && model.getCompany().getCompanyId() > 0) {
            company = companyService.findById(model.getCompany().getCompanyId());
        } else {
            if (model.getCompany().getHostTenantId() == null) {
                model.getCompany().setHostTenantId(tenantId);
            }
            company = companyService.save(model.getCompany(), isOverwrite);
        }
        Tenant tenant = tenantOptional.get();
        companyService.saveTenantCompany(tenant, company);
        return company;
    }

    @Transactional(rollbackFor = Exception.class)
    public OrgStruct update(long orgId, Save model) {
        OrgStruct existEntity = orgStructDao.findOne((Specification<OrgStruct>) (root, query, criteriaBuilder) -> criteriaBuilder.equal(root.get("orgId"), orgId)).orElseThrow(() -> new IllegalArgumentException("未找到组织实体"));
        OrgStruct result = this.update(existEntity, model);
        return result;
    }

    /**
     * 保存组织核心方法
     *
     * @param existEntity 已存在组织实体
     * @param model       输入的组织报文对象
     * @return
     */
    private OrgStruct update(OrgStruct existEntity, Save model) {
        if (model.getDefaultOrgRoleId() != null && model.getDefaultOrgRoleId() > 0 && !model.getDefaultOrgRoleId().equals(existEntity.getDefaultOrgRoleId())) {
            long countDefaultOrgRoleId = roleDao.countRoleByTenantIdAndIdAndType(existEntity.getTenantId(), model.getDefaultOrgRoleId(), 2);
            if (countDefaultOrgRoleId == 0) {
                throw new IllegalArgumentException("指定的默认组织角色(id:" + model.getDefaultOrgRoleId() + ")不存在");
            }
        }
        if (model.getGradingRoleId() != null && model.getGradingRoleId() > 0 && !model.getGradingRoleId().equals(existEntity.getGradingRoleId())) {
            long countGradingRoleId = roleDao.countRoleByTenantIdAndIdAndType(existEntity.getTenantId(), model.getGradingRoleId(), 1);
            if (countGradingRoleId == 0) {
                throw new IllegalArgumentException("指定的分级管理员角色(id:" + model.getGradingRoleId() + ")不存在");
            }
        }
        //region 设置 defaultOrgRoleId = null, 需要将org_user_rel 的关系清除
        if (existEntity.getDefaultOrgRoleId() != null && existEntity.getDefaultOrgRoleId() > 0 && model.getDefaultOrgRoleId() != null && model.getDefaultOrgRoleId() == -1) {
            roleUserRelDao.deleteByRoleIdAndOrgIdAndType(existEntity.getDefaultOrgRoleId(), existEntity.getId(), 2);
        }
        //endregion
        //region 设置 gradingRoleId = null, 需要将org_user_rel 的关系清除
        if (existEntity.getGradingRoleId() != null && existEntity.getGradingRoleId() > 0 && model.getGradingRoleId() != null && model.getGradingRoleId() == -1) {
            roleUserRelDao.deleteByRoleIdAndOrgIdAndType(existEntity.getGradingRoleId(), existEntity.getId(), 1);
        }
        //endregion
        //region 入参覆盖existEntity
        BeanUtils.copyProperties(model, existEntity, Stream.of("orgId", "company", "tenant", "companyNos", "extensions").toArray(String[]::new));
        //endregion
        if (existEntity.getOrgId() != null && existEntity.getParentId() != null && existEntity.getParentId().equals(existEntity.getOrgId())) {
            throw new IllegalArgumentException("父组织不能关联本身");
        }
        OrgStruct result = new OrgStruct();
        boolean isOverwrite = model.isOverwrite();
        if (model.getCompany() != null) {
            Company company = this.saveCompany(existEntity.getTenantId(), model, isOverwrite);
            if (company != null) {
                //region 拦截开启组织树内公司唯一校验
                if (model.getUniqueCompany() != null && model.getUniqueCompany()) {
                    List<Long> existCompanyOrgIds = orgStructDao.findIdByTenantIdAndCompanyId(existEntity.getTenantId(), company.getCompanyId());
                    if (!existCompanyOrgIds.isEmpty() && !existCompanyOrgIds.contains(existEntity.getOrgId())) {
                        throw new IllegalArgumentException("组织树内已存在公司(" + company.getCompanyName() + ")");
                    }
                }
                //endregion
                existEntity.setCompanyId(company.getCompanyId());
            }
        }
        if (CollectionUtils.isNotEmpty(model.getCompanyNos())) {
            companyService.saveCompanyNos(existEntity.getOrgId(), model.getCompanyNos(), isOverwrite);
            result.setCompanyNos(model.getCompanyNos());
        }
        if (CollectionUtils.isNotEmpty(model.getExtensions())) {
            Set<OrgExtension> extensions = orgExtensionService.batchSave(existEntity.getOrgId(), model.getExtensions(), isOverwrite);
            Set<OrgExtensionDto> extensionDtos = extensions.stream().filter(Objects::nonNull).collect(toSet());
            result.setExtensions(extensionDtos);
        }
        existEntity = this.saveOrgStructEntity(existEntity);
        BeanUtils.copyProperties(existEntity, result, Stream.of("companyNos", "extensions").toArray(String[]::new));
        return result;
    }

    /**
     * 核心方法
     * 获取组织信息
     *
     * @param orgId
     * @param extraInfoDimension
     * @return
     */
    public OrgStruct findById(long orgId, int extraInfoDimension) {
        OrgStruct org = orgStructDao.findById(orgId, EntityGraphs.named(OrgStruct.NAMED_ENTITY_GRAPH_FULL)).orElseThrow(() -> new IllegalArgumentException("未找到组织实体(orgId:" + orgId + ")"));
        if (BinaryUtils.is(OrgExtraInfo.extentions, extraInfoDimension)) {
            List<OrgExtension> extentions = orgExtensionService.findByOrgId(orgId);
            org.setExtensions(new HashSet<>(extentions));
        }
        if (BinaryUtils.is(OrgExtraInfo.children, extraInfoDimension)) {
            if (org.getTenantId() != null && org.getTenantId() > 0) {
                Query childrenQuery = new Query();
                childrenQuery.setTenantId(org.getTenantId());
                childrenQuery.setParentId(org.getOrgId());
                Specification<OrgStruct> childrenSpecification = OrgQueryHelper.querySpecification(childrenQuery);
                List<OrgStruct> children = orgStructDao.findAll(childrenSpecification, EntityGraphs.named(OrgStruct.NAMED_ENTITY_GRAPH_DEFAULT));
                org.setChildren(children);
            }
        }

        return org;
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteById(long orgId) {
        OrgStruct org = orgStructDao.findById(orgId).orElseThrow(() -> new IllegalArgumentException("未找到组织(" + orgId + ")实体"));
        this.deleteOrgWithValidation(org);
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteByTenantIdAndId(long tenantId, long orgId) {
        Query query = new Query();
        query.setTenantId(tenantId);
        query.setOrgId(orgId);
        Specification<OrgStruct> specification = OrgQueryHelper.queryOneSpecification(query);
        Optional<OrgStruct> existOptional = orgStructDao.findOne(specification);
        if (!existOptional.isPresent()) {
            String message = "未找到组织实体(tenantId: " + tenantId + ", orgId:" + orgId + ")";
            logger.warn(message);
            throw new IllegalArgumentException(message);
        }
        OrgStruct existOrg = existOptional.get();
        this.deleteOrgWithValidation(existOrg);
    }

    private void beforeDelOrDisableOrgValidation(long orgId) {
        long childrenCount = orgStructDao.countChildren(orgId);
        if (childrenCount > 0) {
            String message = "该组织(" + orgId + ")存在子组织无法删除或禁用";
            logger.warn(message);
            throw new IllegalArgumentException(message);
        }
    }

    private void deleteOrgWithValidation(OrgStruct org) {
        this.beforeDelOrDisableOrgValidation(org.getOrgId());
        long userCount = orgUserRelDao.countByOrgId(org.getOrgId());
        if (userCount > 0) {
            String message = "已关联人员的组织无法删除，若需删除，请先移除组织内人员";
            logger.warn(message);
            throw new IllegalArgumentException(message);
        }

        // 删除时如果已经基于组织角色关联的用户也应该解绑
        Long defaultOrgRoleId = org.getDefaultOrgRoleId();
        if (null != defaultOrgRoleId) {
            roleUserRelDao.deleteByRoleIdAndOrgId(defaultOrgRoleId, org.getOrgId());
        }

        this.deleteOrg(org);
    }

    private void deleteOrg(OrgStruct org) {
        orgStructDao.deleteById(org.getOrgId());

        orgPubService.sendOrgMsg(DELETE, org);
//        orgUserRelDao.deleteByOrgId(org.getOrgId());
        orgExtensionDao.deleteByOrgId(org.getOrgId());
        if (org.getTenantId() != null && org.getTenantId() > 0 && org.getCompanyId() != null && org.getCompanyId() > 0) {
            tenantCompanyRelDao.deleteByTenantIdAndCompanyId(org.getTenantId(), org.getCompanyId());
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteByTenantIdAndCompanyId(long tenantId, long companyId) {
        List<OrgStruct> orgs = orgStructDao.findAll((Specification<OrgStruct>) (root, query, criteriaBuilder) -> {
            return criteriaBuilder.and(criteriaBuilder.equal(root.get("tenantId"), tenantId), criteriaBuilder.equal(root.get("companyId"), companyId));
        });
        Set<Long> orgIds = orgs.stream().filter(Objects::nonNull).map(OrgStruct::getOrgId).collect(toSet());
        orgStructDao.disableByIds(orgIds);
        orgUserRelDao.deleteByOrgIds(orgIds);
        orgExtensionDao.deleteByOrgIds(orgIds);

        orgs.stream().forEach(org -> orgPubService.sendOrgMsg(SAVE, org));

    }

    @Transactional(rollbackFor = Exception.class)
    public OrgStruct updateByTenantId(long tenantId, long orgId, Save model) {
        Query query = new Query();
        query.setTenantId(tenantId);
        query.setOrgId(orgId);
        Optional<OrgStruct> existOptional = this.findOne(query);
        OrgStruct existEntity = existOptional.orElseThrow(() -> new IllegalArgumentException("未找到组织实体(tenantId:" + tenantId + ", orgId:" + orgId + ")"));
        if (model.getStatus() != null && Constant.STATUS_DISABLE == model.getStatus()) {
            List<OrgVirtual> orgVirtuals = orgVirtualDao.findByOrgStructId(orgId);
            if (CollectionUtils.isNotEmpty(orgVirtuals)) {
                throw new IllegalArgumentException("该组织(" + orgId + ")状态不能停用，已关联虚拟组织");
            }
        }

        //fix bug: 6120. check orgcode
        //如果传入orgcode，并且与原值不一致， 需要查询是否有重复的orgcode
        if (StringUtils.isNotEmpty(model.getOrgCode())) {
            if (!model.getOrgCode().equalsIgnoreCase(existEntity.getOrgCode())) {
                long orgCodeCount = orgStructDao.countByTenantIdAndOrgCode(tenantId, model.getOrgCode());
                if (orgCodeCount > 0) {
                    throw new IllegalArgumentException("orgCode (" + model.getOrgCode() + ") 已经存在");
                }
            }
        }

        OrgStruct result = this.update(existEntity, model);
        return result;
    }

    public OrgStruct info(long tenantId, long orgId) {
        OrgStruct org = this.findByTenantIdAndId(tenantId, orgId);
        if (org.getParentId() != null && org.getParentId() > 0) {
            String parentName = orgStructDao.findNameByOrgId(org.getParentId());
            org.setParentName(parentName);
        }

        List<OrgExtension> extentions = orgExtensionService.findByOrgId(org.getOrgId());
        org.setExtensions(new HashSet<>(extentions));
        return org;
    }

    public OrgStruct findByTenantIdAndId(long tenantId, long orgId) {
        Query query = new Query();
        query.setTenantId(tenantId);
        query.setOrgId(orgId);
        Specification<OrgStruct> specification = OrgQueryHelper.queryOneSpecification(query);
        Optional<OrgStruct> existOptional = orgStructDao.findOne(specification, EntityGraphs.named(OrgStruct.NAMED_ENTITY_GRAPH_FULL));
        String message = "未找到组织实体(tenantId:" + tenantId + ", orgId:" + orgId + ")";
        OrgStruct org = existOptional.orElseThrow(() -> new IllegalArgumentException(message));
        return org;
    }

    /**
     * 获取组织子孙列表
     *
     * @param orgId
     * @return
     */
    public List<OrgStruct> findDescendantsById(long orgId) {
        return orgStructDao.findDescendantsById(orgId);
    }

    /**
     * 根据用户获取该用户的组织树
     *
     * @param userId
     * @return
     */
    public List<OrgStruct> findTreeByUserId(long userId, String modules) {
        Optional<User> existsUserOptional = userDao.findById(userId);
        if (!existsUserOptional.isPresent()) {
            String message = "找不到用户实体(userId:" + userId + ")";
            logger.warn(message);
            throw new IllegalArgumentException(message);
        }
        Query query = new Query();
        query.setTenantId(existsUserOptional.get().getTenantId());
        query.setUserId(userId);
        query.setModules(modules);
        query.setStatus(1);
        List<OrgStruct> orgs = this.list(query, Sort.unsorted());
        return TreeUtils.buildTree(orgs);
    }

    /**
     * 保存/更新组织关联的公司
     *
     * @param orgSaveInput
     * @param model
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public OrgSaveOutput saveOrg(OrgSaveInput orgSaveInput, Save model, boolean isOverwrite) {
        CompanyModel.Request.Save companyCreate = model.getCompany();
        Company company = null;
        Long companyId = null;
        OrgSaveOutput orgSaveOutput = new OrgSaveOutput();
        if (companyCreate != null) {
            logger.info("companyCreate != null");
            companyCreate.setHostTenantId(orgSaveInput.tenantId);
            company = companyService.save(companyCreate, isOverwrite);
            companyId = company.getCompanyId();
        }
        if (companyId != null) {
            Tenant tenant = tenantDao.findById(orgSaveInput.tenantId).get();
            companyService.saveTenantCompany(tenant, company);
        }

        OrgStruct orgStruct = null;
        boolean isNew = false;
        if (model.getOrgId() != null && model.getOrgId() > 0) {
            orgStruct = orgStructDao.findById(model.getOrgId()).orElse(null);
        }
        if (orgStruct == null && StringUtils.isNotBlank(model.getOrgCode())) {
            List<OrgStruct> orgStructs = orgStructDao.findByTenantIdAndOrgCode(orgSaveInput.tenantId, model.getOrgCode());
            if (!orgStructs.isEmpty()) {
                Optional<OrgStruct> orgStructOptional = orgStructs.stream().filter(o -> o.getStatus() != null && o.getStatus() == 1).findFirst();
                if (orgStructOptional.isPresent()) {
                    orgStruct = orgStructOptional.get();
                } else {
                    orgStructOptional = orgStructs.stream().filter(o -> o.getStatus() == null || o.getStatus() == 0).findFirst();
                    if (orgStructOptional.isPresent()) {
                        orgStruct = orgStructOptional.get();
                    }
                }
            }
        }
        //region 如果包含公司信息，则根据是否开启组织树内公司唯一参数判断是否使用已存在的公司组织
        if (companyId != null && companyId > 0 && model.getUniqueCompany() != null && model.getUniqueCompany()) {
            Query query = new Query();
            query.setCompanyId(companyId);
            query.setTenantId(orgSaveInput.tenantId);
            List<OrgStruct> exsitCompanyOrgs = orgStructDao.findAll(OrgQueryHelper.querySpecification(query));
            for (OrgStruct exsitCompanyOrg : exsitCompanyOrgs) {
                if (model.getOrgId() != null && model.getOrgId().equals(exsitCompanyOrg.getOrgId())) {
                    orgStruct = exsitCompanyOrg;
                    break;
                }
                if (exsitCompanyOrg.getStatus() != null && exsitCompanyOrg.getStatus() == 1) {
                    orgStruct = exsitCompanyOrg;
                } else {
                    exsitCompanyOrg.setStatus(1);
                    orgStruct = exsitCompanyOrg;
                }
            }
        }
        //endregion
        if (orgStruct == null) { //没有已存在组织, 需要创建新的
            isNew = true;
            //region 清理输入的不合适的数据
            if (orgSaveInput.parentId > 0) {
                model.setParentId(orgSaveInput.parentId);
            }

            if (model.getOrgId() != null && model.getOrgId() == 0) {
                model.setOrgId(null);
            }
            model.setTenantId(orgSaveInput.tenantId);
            //endregion
            orgStruct = new OrgStruct();
        } else if (orgStruct.getStatus() != null && orgStruct.getStatus() == 0) {
            if (model.getStatus() != null && model.getStatus() == 1) {
                orgStruct.setStatus(1);
            }
        }
        long parentId = 0;
        if (StringUtils.isNotBlank(model.getParentCode())) {
            List<Long> parentIdList = orgStructDao.findOrgIdByTenantIdAndOrgCode(orgSaveInput.tenantId, model.getParentCode());
            if (parentIdList.isEmpty()) {
                String message = "非法的父级组织code(" + model.getParentCode() + ")";
                logger.warn(message);
                throw new IllegalArgumentException(message);
            }
            parentId = parentIdList.get(0);
        } else {
            List<Long> rootIds = orgStructDao.findRootIdsByTenantId(orgSaveInput.tenantId);
            if (!rootIds.isEmpty()) {
                parentId = rootIds.get(0);
            }
        }
        if (parentId > 0) {
            model.setParentId(parentId);
        }
        if (model.getParentId() != null && model.getParentId() == 0) {
            model.setParentId(null);
        }
        if (StringUtils.isBlank(model.getOrgCode())) {
            model.setOrgCode(null);
        }
        if (StringUtils.isBlank(model.getOrgName())) {
            model.setOrgName(null);
        }
        BeanUtils.copyProperties(model, orgStruct, Stream.of("orgId", "company", "tenant", "companyNos", "extensions").toArray(String[]::new));
        if (orgStruct.getCompanyId() == null && companyId != null && companyId > 0) {
            orgStruct.setCompanyId(companyId);
        }

        if (StringUtils.isBlank(orgStruct.getOrgCode())) {
            orgStruct.setOrgCode(UUIDUtils.randomUUID());
        }
        orgStruct = this.saveOrgStructEntity(orgStruct);
        if (company != null) {
            if (orgStruct.getCompany() == null) {
                orgStruct.setCompany(company);
            }
        }
        orgSaveInput.orgs.add(orgStruct);
        orgSaveOutput.addOrg(orgStruct);
        if (OrgType.COMPANY.equals(orgStruct.getOrgType())) {
            companyService.saveCompanyNos(orgStruct.getOrgId(), model.getCompanyNos(), isOverwrite);
        }
        Set<OrgExtension> extensions = orgExtensionService.batchSave(orgStruct.getOrgId(), model.getExtensions(), isOverwrite);
        if (CollectionUtils.isNotEmpty(extensions) && orgStruct.getExtensions() == null) {
            Set<OrgExtensionDto> extensionDtos = extensions.stream().filter(Objects::nonNull).collect(toSet());
            orgStruct.setExtensions(extensionDtos);
        }
        return orgSaveOutput;
    }

    /**
     * 保存组织
     *
     * @param tenantId 租户id
     * @param model    OrgModel.Request.Save
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public OrgSaveOutput update(long tenantId, Save model, boolean isOverwrite) {
        if (!tenantDao.existsById(tenantId)) {
            String message = "不合法的租户id(" + tenantId + ")";
            logger.warn(message);
            throw new IllegalArgumentException(message);
        }
        logger.info("save Org tenantId({})", tenantId);
        OrgSaveInput orgSaveInput = new OrgSaveInput(tenantId, 0, null, new ArrayList<>());
        OrgSaveOutput orgSaveOutput = this.saveOrg(orgSaveInput, model, isOverwrite);
        return orgSaveOutput;
    }

    public List<OrgStruct> findByTenantIdAndCompanyId(long tenantId, long companyId) {
        Query query = new Query();
        query.setTenantId(tenantId);
        query.setCompanyId(companyId);
        return this.list(query, Sort.unsorted());
    }

    public Page<OrgStruct> findRoots(Pageable pageable) {
        Specification<OrgStruct> specification = (Specification<OrgStruct>) (root, criteriaQuery, builder) -> {
            List<Predicate> predicates = new ArrayList<>();
            predicates.add(builder.or(builder.isNull(root.<Long>get("parentId")), builder.equal(root.<Long>get("parentId"), 0)));
            predicates.add(builder.equal(root.<Integer>get("status"), 1));
            if (!predicates.isEmpty()) {
                criteriaQuery.where(predicates.stream().toArray(Predicate[]::new));
            }
            return criteriaQuery.getRestriction();
        };
        return orgStructDao.findAll(specification, pageable);
    }

    public Page<OrgStruct> findRoots(long tenantId, Pageable pageable) {
        Specification<OrgStruct> specification = this.findRootsSpecification(tenantId);
        return orgStructDao.findAll(specification, pageable);
    }

    public List<OrgStruct> findAllRoots(long tenantId, Sort sort) {
        Specification<OrgStruct> specification = this.findRootsSpecification(tenantId);
        return orgStructDao.findAll(specification, sort);
    }

    private Specification<OrgStruct> findRootsSpecification(long tenantId) {
        Specification<OrgStruct> specification = (Specification<OrgStruct>) (root, criteriaQuery, builder) -> {
            List<Predicate> predicates = new ArrayList<>();
            predicates.add(builder.equal(root.<Long>get("tenantId"), tenantId));
            predicates.add(builder.equal(root.<Integer>get("status"), 1));
            predicates.add(builder.or(builder.isNull(root.<Long>get("parentId")), builder.equal(root.<Long>get("parentId"), 0)));
            if (!predicates.isEmpty()) {
                criteriaQuery.where(predicates.stream().toArray(Predicate[]::new));
            }
            return criteriaQuery.getRestriction();
        };
        return specification;
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateStatus(long orgId, int status) {
        int result;
        if (status > 0) {
            result = orgStructDao.enable(orgId);
        } else {
            List<OrgVirtual> orgVirtuals = orgVirtualDao.findByOrgStructId(orgId);
            if (CollectionUtils.isNotEmpty(orgVirtuals)) {
                throw new IllegalArgumentException("该组织(" + orgId + ")状态不能停用，已关联虚拟组织");
            }
            this.beforeDelOrDisableOrgValidation(orgId);
            result = orgStructDao.disable(orgId);
        }
        if (result == 0) {
            throw new IllegalArgumentException("未找到组织实体(" + orgId + ")");
        }

        OrgStruct orgStruct = orgStructDao.findById(orgId).get();
        orgPubService.sendOrgMsg(SAVE, orgStruct);
    }

    public List<OrgStruct> findTreeByTenantId(TreeQuery treeQuery) {
        Query query = new Query();
        query.setTenantId(treeQuery.getTenantId());
        query.setStatus(treeQuery.getStatus());
        query.setModules(treeQuery.getModules());
        query.setOrgNameLike(treeQuery.getOrgName());
        query.setUserId(treeQuery.getUserId());
        query.setWithUserBoundFlag(treeQuery.getWithUserBound());
        query.setWithChildrenCount(treeQuery.getWithChildrenCount());
        Set<String> treeAttributes = Stream.of("orgId", "orgName", "parentId", "orgType", "parentIds").collect(Collectors.toSet());
        if (treeQuery.getAttributes() != null && !treeQuery.getAttributes().isEmpty()) {
            treeAttributes.addAll(treeQuery.getAttributes());
        }
        query.setAttributes(treeAttributes);

        query.setTree(true);
        String filterParentIds = null;
        if (treeQuery.getRootId() != null && treeQuery.getRootId() > 0) {
            filterParentIds = orgStructDao.findCommittedParentIdsByOrgId(treeQuery.getRootId());
            if (filterParentIds != null) {
                query.setFilterParentIds(Stream.of(filterParentIds).collect(toSet()));
            }
        }
        List<OrgStruct> orgs = this.list(query, Sort.unsorted());
        if (StringUtils.isNotBlank(treeQuery.getOrgName()) && !orgs.isEmpty()) {
            Set<Long> orgIdSet = orgs.stream().map(o -> OrgUtils.findOrgIdInParentIds(o.getParentIds())).flatMap(Collection::stream).collect(toSet());
            if (!orgIdSet.isEmpty()) {
                Set<Long> selectedOrgIds = orgs.stream().map(OrgStruct::getOrgId).collect(toSet());
                Set<Long> selectingOrgIds = new HashSet<>();
                Set<OrgStruct> orgSet = new HashSet<>();
                for (Long orgId : orgIdSet) {
                    boolean anyMatch = false;
                    for (OrgStruct selectedOrg : orgs) {
                        if (selectedOrg.getOrgId().equals(orgId)) {
                            anyMatch = true;
                            orgSet.add(selectedOrg);
                            break;
                        }
                    }
                    if (!anyMatch) {
                        selectingOrgIds.add(orgId);
                    }
                }
                if (selectingOrgIds != null && !selectingOrgIds.isEmpty()) {
                    query = new Query();
                    query.setTenantId(treeQuery.getTenantId());
                    query.setStatus(treeQuery.getStatus());
                    query.setModules(treeQuery.getModules());
                    query.setOrgIds(selectingOrgIds);
                    query.setWithUserBoundFlag(treeQuery.getWithUserBound());
                    query.setWithChildrenCount(treeQuery.getWithChildrenCount());
                    if (treeQuery.getAttributes() != null && !treeQuery.getAttributes().isEmpty()) {
                        treeAttributes.addAll(treeQuery.getAttributes());
                    }
                    query.setAttributes(treeAttributes);

                    query.setTree(true);
                    if (filterParentIds != null) {
                        query.setParentIds(filterParentIds);
                    }
                    List<OrgStruct> list = this.list(query, Sort.unsorted());
                    if (!list.isEmpty()) {
                        orgSet.addAll(list);
                    }
                }
                if (filterParentIds != null) {
                    final String finalParentIds = filterParentIds;
                    orgSet = orgSet.stream().filter(o -> StringUtils.startsWith(o.getParentIds(), finalParentIds)).collect(toSet());
                }
                orgs = new ArrayList<>(orgSet);
            }
        }

        return TreeUtils.buildTree(orgs);
    }

    public List<OrgStruct> findByTenantIdAndOrgCode(long tenantId, String orgCode) {
        Query query = new Query();
        query.setTenantId(tenantId);
        query.setOrgCode(orgCode);
        query.setStatus(1);
        Specification<OrgStruct> specification = OrgQueryHelper.querySpecification(query);
        List<OrgStruct> orgs = orgStructDao.findAll(specification);
        return orgs;
    }


    public List<OrgStruct> listByTenantIdAndUserId(long tenantId, long userId, String modules, Set<String> attributes) {
        Query orgQuery = new Query();
        orgQuery.setTenantId(tenantId);
        orgQuery.setUserId(userId);
        orgQuery.setStatus(1);
        orgQuery.setWithIsHost(Boolean.TRUE);
        orgQuery.setModules(modules);
        orgQuery.setAttributes(attributes);
        List<OrgStruct> orgs = this.list(orgQuery, Sort.unsorted());
        return orgs;
    }

    public List<OrgStruct> listByTenantIdAndUserId(long tenantId, long userId, String modules) {
        return this.listByTenantIdAndUserId(tenantId, userId, modules, null);
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateStatus(long tenantId, long orgId, int status) {
        int result;
        if (status > 0) {
            result = orgStructDao.enable(tenantId, orgId);
        } else {
            List<OrgVirtual> orgVirtuals = orgVirtualDao.findByOrgStructId(orgId);
            if (CollectionUtils.isNotEmpty(orgVirtuals)) {
                throw new IllegalArgumentException("该组织(" + orgId + ")状态不能停用，已关联虚拟组织");
            }
            this.beforeDelOrDisableOrgValidation(orgId);
            result = orgStructDao.disable(tenantId, orgId);
        }
        if (result == 0) {
            throw new IllegalArgumentException("未找到组织实体(" + orgId + ")");
        }
        OrgStruct orgStruct = orgStructDao.findById(orgId).get();
        orgPubService.sendOrgMsg(SAVE, orgStruct);
    }

    public Long findIdByTenantIdAndCode(long tenantId, String orgCode) {
        return orgStructDao.findIdByTenantIdAndCode(tenantId, orgCode);
    }

    public boolean checkName(Long tenantId, String orgName) {
        Query query = new Query();
        query.setTenantId(tenantId);
        query.setOrgName(orgName);
        Specification<OrgStruct> specification = OrgQueryHelper.queryOneSpecification(query);
        return orgStructDao.count(specification) == 0;
    }

    public ExcelModel findParentOrgInModels(ExcelModel org, List<ExcelModel> excelModels) {

        List<ExcelModel> parents = excelModels.stream().filter(Objects::nonNull).filter(excelModel -> excelModel.getOrgCode() != null && excelModel.getOrgCode().equals(org.getParentOrgCode()) && StringUtils.isNotBlank(org.getParentOrgCode())).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(parents) || parents.size() > 1) {
            return null;
        }
        return parents.get(0);
    }


    @Transactional(rollbackFor = Exception.class)
    public void saveSingle(Long tenantId, Map<String, OrgStruct> currentSaveOrgs, ExcelModel org, OrgStruct
            parentOrg) throws ImportException {
        //校验非空字段
        ExcelUtils.checkField(org, validator);
        if (org.getOrgType() != null && OrgType.COMPANY.value() == org.getOrgType()) {
            //校验公司类型的组织公司参数
            this.checkCompany(org);
        }

        //检查当前组织和父组织关系
        this.checkCurrentOrgs(parentOrg, org);

        List<OrgStruct> oldOrgs = orgStructDao.findByTenantIdAndOrgCode(tenantId, org.getOrgCode());
        if (oldOrgs.size() > 1) {
            throw new ImportException(Stream.of("组织code存在多个" + org.getOrgCode()).collect(toList()));
        }
        //如果组织不存在，直接增加组织
        if (CollectionUtils.isEmpty(oldOrgs)) {
            OrgStruct save = this.saveOrg(tenantId, org, parentOrg);

            orgPubService.sendOrgMsg(SAVE, save);

            if (save != null && save.getOrgCode() != null) {
                currentSaveOrgs.put(save.getOrgCode(), save);
            }
        } else {
            //组织存在判断当前组织和原来组织是否一致，判断要素，组织代码，组织名称，上级组织id，组织类型,如果组织是公司，判断公司税号
            //和公司名称是不是一致，不一致直接报错，一致直接更新组织
            OrgStruct oldOrg = oldOrgs.get(0);
            Company company = oldOrg.getCompany();

            //新老公司不一样报错

            if (null != company && (!company.getCompanyName().equals(org.getCompanyName()) || !company.getTaxNum().equals(org.getTaxNum()))) {
                throw new ImportException(Stream.of("新老组织名称或者税号不一致" + org.getOrgCode()).collect(toList()));
            }
            //新老组织类型不一样报错
            if (oldOrg.getOrgType().value() != org.getOrgType()) {
                throw new ImportException(Stream.of("新老组织类型不相同" + org.getOrgCode()).collect(toList()));
            }
            if (!checkCurrentDataCompareOrigin(oldOrg, org, parentOrg.getOrgId())) {
                //此处只会更新组织，不会更新公司
                OrgStruct updater = this.updateOrg(oldOrg, org, parentOrg.getOrgId());
                if (StringUtils.isNotBlank(org.getCompanyNo())) {
                    Set<String> companyNos = Stream.of(StringUtils.split(org.getCompanyNo(), "/")).filter(Objects::nonNull).collect(toSet());
                    if (!companyNos.isEmpty()) {
                        companyService.saveCompanyNos(oldOrg.getOrgId(), companyNos, true);
                    }
                }
                currentSaveOrgs.put(org.getOrgCode(), updater);
            } else {
                currentSaveOrgs.put(org.getOrgCode(), oldOrgs.get(0));
            }

        }
    }


    private boolean checkCurrentDataCompareOrigin(OrgStruct orgStruct, ExcelModel org, Long orgId) {
        return orgStruct.getOrgCode().equals(org.getOrgCode()) &&
                orgStruct.getOrgName().equals(org.getOrgName()) && orgId.equals(orgStruct.getOrgId());
    }

    private OrgStruct updateOrg(OrgStruct orgStruct, ExcelModel org, Long orgId) {
        if (null != orgStruct.getOrgId() && Constant.STATUS_DISABLE == org.getStatus()) {
            List<OrgVirtual> orgVirtuals = orgVirtualDao.findByOrgStructId(orgId);
            if (CollectionUtils.isNotEmpty(orgVirtuals)) {
                throw new IllegalArgumentException("该组织(" + orgId + ")状态不能停用，已关联虚拟组织");
            }
        }
        orgStruct.setOrgName(org.getOrgName());
        orgStruct.setStatus(org.getStatus());
        orgStruct.setOrgCode(org.getOrgCode());
        orgStruct.setParentId(orgId);
        this.saveOrgStructEntity(orgStruct);

        return orgStruct;
    }

    private OrgStruct saveOrg(Long tenantId, ExcelModel org, OrgStruct parentOrg) {
        Save testSave = new Save();
        CompanyModel.Request.Save company = new CompanyModel.Request.Save();
        BeanUtils.copyProperties(org, testSave);
        testSave.setParentId(parentOrg.getOrgId());
        testSave.setTenantId(tenantId);
        testSave.setStatus(org.getStatus());
        testSave.setWithApplication(false);
        if (StringUtils.isNotBlank(org.getCompanyNo())) {
            testSave.setCompanyNos(Stream.of(org.getCompanyNo().split(BACKSLASH)).collect(toSet()));
        }
        OrgType enumOrgType = OrgType.COMPANY;
        if (org.getOrgType() == OrgType.GROUP.value()) {
            enumOrgType = OrgType.GROUP;
        } else if (org.getOrgType() == OrgType.NORMAL.value()) {
            enumOrgType = OrgType.NORMAL;
        }
        testSave.setOrgType(enumOrgType);
        testSave.setParentCode(org.getParentOrgCode());
        if (org.getOrgType().equals(OrgType.COMPANY.value())) {
            BeanUtils.copyProperties(org, company);
            company.setStatus(1);
            testSave.setCompany(company);
        }
        //查询公司是否存在
        OrgSaveInput orgSaveInput = new OrgSaveInput(tenantId, parentOrg.getOrgId(), parentOrg, new ArrayList<>());
        OrgSaveOutput orgSaveOutput = this.saveOrg(orgSaveInput, testSave, true);
        if (null == orgSaveOutput) {
            return null;
        }
        return orgSaveOutput.findFirst();

    }


    private void checkCurrentOrgs(OrgStruct parentOrg, ExcelModel org) throws ImportException {
        if (OrgType.GROUP == parentOrg.getOrgType() && (!(org.getOrgType() != 0 || org.getOrgType() != 1))) {
            throw new ImportException(Stream.of("父组织是母公司，子组织必须是公司或者母公司" + org.getOrgCode()).collect(toList()));
        }
        //组织的父公司为待审核下面组织一定待审核
        logger.info("组织信息id{},{}状态{}", parentOrg.getOrgId(), parentOrg.getOrgType(), parentOrg.getAuditStatus());
        if (parentOrg.getOrgType().equals(OrgType.COMPANY) && (null == parentOrg.getAuditStatus() || parentOrg.getAuditStatus() == 0)) {
            org.setAuditStatus(0);
            return;
        }
        org.setAuditStatus(1);
        //org.setStatus(0);

    }


    public OrgStruct findParentOrgType(Long tenantId, ExcelModel org) {
        Set<String> orgCodes = Stream.of(org.getParentOrgCode()).filter(Objects::nonNull).collect(toSet());
        List<OrgStruct> parentOrgs = orgStructDao.findByTenantIdAndOrgCodes(tenantId, orgCodes);
        if (CollectionUtils.isNotEmpty(parentOrgs) && StringUtils.isNotBlank(parentOrgs.get(0).getOrgCode())) {
            return parentOrgs.get(0);
        }
        return null;
    }

    private void checkCompany(ExcelModel org) throws ImportException {
        List<String> errMsg = new ArrayList<>();

        if (StringUtils.isBlank(org.getCompanyName())) {
            errMsg.add("公司名称不能为空:" + org.getOrgCode());
        }
        if (StringUtils.isBlank(org.getTaxNum())) {
            errMsg.add("公司税号不能为空:" + org.getOrgCode());
        }
        if (CollectionUtils.isNotEmpty(errMsg)) {
            throw new ImportException(errMsg);
        }

    }

    @Transactional(rollbackFor = Exception.class)
    public void updateParentIds(long orgId, String parentIds) {
        Optional<OrgStruct> optionalOrgStruct = orgStructDao.findById(orgId);
        if (!optionalOrgStruct.isPresent()) {
            logger.warn("修改parentIds失败，orgId:{}", orgId);
            return;
        }
        OrgStruct org = optionalOrgStruct.get();
        org.setParentIds(parentIds);
        orgStructDao.saveAndFlush(org);
    }

    /**
     * 批量保存
     *
     * @param tenantId
     * @param models
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public List<OrgStruct> saveWithResult(long tenantId, List<Save> models) {

        if (CollectionUtils.isEmpty(models)) {
            return new ArrayList<>(0);
        }
        if (!tenantDao.existsById(tenantId)) {
            throw new IllegalArgumentException("不合法的租户id(" + tenantId + ")");
        }

        List<OrgStruct> parents = orgStructDao.findRootsByTenantId(tenantId);
        if (CollectionUtils.isEmpty(parents)) {
            throw new IllegalArgumentException("根组织不能为空(tenantId: " + tenantId + ")");
        }
        OrgStruct parent = parents.get(0);
        OrgSaveInput orgSaveInput = new OrgSaveInput(tenantId, parent.getOrgId(), parent, new ArrayList<>());

        models.forEach(model -> this.create(model));
        List<OrgStruct> res = orgSaveInput.orgs.stream().map(item -> {
            OrgStruct org = new OrgStruct();
            org.setCompanyId(item.getCompanyId());
            org.setOrgId(item.getOrgId());
            org.setOrgCode(item.getOrgCode());
            org.setTenantId(item.getTenantId());
            return org;
        }).collect(Collectors.toList());
        return res;
    }


    /**
     * @param tenantId 租户ID
     * @param model    保存数据
     * @return Company
     */
    @Transactional(rollbackFor = Exception.class)
    public <C extends CompanyModel.Request.Save> Company saveCompanyAsync(long tenantId, Long rootOrgId, C model) {
        Save orgModel = new Save();
        orgModel.setTenantId(tenantId);
        orgModel.setOrgName(model.getCompanyName());
        orgModel.setOrgDesc(model.getCompanyName());
        orgModel.setOrgType(OrgType.COMPANY);
        orgModel.setCompany(model);
        orgModel.setIsStrict(false);
        OrgStruct org = this.createAsync(orgModel);
        return org.getCompany();
    }

    /**
     * @param tenantId
     * @param model
     * @return
     */
    @Transactional
    public <C extends CompanyModel.Request.Save> Company saveCompany(long tenantId, Long rootOrgId, C model) {
        Save orgModel = new Save();
        orgModel.setTenantId(tenantId);
        orgModel.setOrgName(model.getCompanyName());
        orgModel.setOrgDesc(model.getCompanyName());
        orgModel.setParentId(rootOrgId);
        orgModel.setOrgType(OrgType.COMPANY);
        orgModel.setCompany(model);
        OrgStruct org = this.create(orgModel);
        return org.getCompany();
    }

    /**
     * @param tenantId
     * @param model
     * @return
     */
    @Transactional
    public <C extends CompanyModel.Request.Save> Company saveCompany(long tenantId, C model) {
        return this.saveCompany(tenantId, null, model);
    }

    /**
     * 根据税号查询
     *
     * @param queryBo OrgCompanyQueryBo
     * @return List<OrgCompanyDTO>
     */
    public List<OrgCompanyDTO> findOrgCompanyByTaxNum(@Valid OrgCompanyQueryBo queryBo) {
        List<OrgCompanyDTO> companyDTOList = orgStructDao.findOrgCompanyByTaxNum(queryBo);
        List<Long> orgIds = companyDTOList.stream().map(e -> e.getOrgId()).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(orgIds)) {
            return companyDTOList;
        }
        List<OrgCompanyNoDTO> companyNoDTOList = orgStructDao.findCompanyNoByOrgIds(orgIds);
        //分组汇总数据
        Map<Long, List<String>> companyNoMap = companyNoDTOList.stream()
                .filter(e -> StringUtils.isNotBlank(e.getCompanyNo()))
                .collect(groupingBy(OrgCompanyNoDTO::getOrgId, mapping(OrgCompanyNoDTO::getCompanyNo, toList())));
        //转
        companyDTOList.stream()
                .map(e -> {
                    List<String> companyNos = companyNoMap.get(e.getOrgId());
                    if (CollectionUtils.isEmpty(companyNos)) {
                        companyNos = Collections.emptyList();
                    }
                    e.setCompanyNos(companyNos);
                    return e;
                })
                .collect(Collectors.toList());
        return companyDTOList;
    }

    public Optional<OrgCompanyRel> findOrgComNum(OrgCompanyNumberDTO orgCompanyNumberDTO) {

        Specification<OrgCompanyRel> specification = new Specification<OrgCompanyRel>() {
            @Override
            public Predicate toPredicate(Root<OrgCompanyRel> root, CriteriaQuery<?> query, CriteriaBuilder cb) {
                List<Predicate> predicateList = new ArrayList<>();
                //条件1：orgId
                if (orgCompanyNumberDTO.getOrgId() != null) {
                    predicateList.add(cb.equal(root.get("orgStructId").as(Long.class), orgCompanyNumberDTO.getOrgId()));
                }

                //条件2：companyNo
                if (StringUtils.isNotBlank(orgCompanyNumberDTO.getCompanyNumber())) {
                    predicateList.add(cb.equal(root.get("companyNo").as(String.class), orgCompanyNumberDTO.getCompanyNumber()));
                }
                Predicate[] pre = new Predicate[predicateList.size()];
                pre = predicateList.toArray(pre);
                return query.where(pre).getRestriction();
            }
        };

        return orgCompanynoDao.findOne(specification);
    }

    public List<OrgStruct> listByOrgKeys(Long tenantId, TreeModel.OrgScope scope, boolean byId, Set<String> keySet) {

        Query query = new Query();
        query.setTenantId(tenantId);
        if (byId) {
            Set<Long> idSet = keySet.stream().map(s -> Long.parseLong(s.trim())).collect(Collectors.toSet());
            query.setOrgIds(idSet);
        } else {
            query.setOrgCodes(keySet);
        }
        List<OrgStruct> models = orgStructDao.findAll(OrgQueryHelper.querySpecification(query), EntityGraphs.named(OrgStruct.NAMED_ENTITY_GRAPH_TENANT));
        if (TreeModel.OrgScope.SELF.equals(scope)) {
            return models;
        }
        for (OrgStruct orgStruct : models) {
            List<OrgStruct> orgs = null;
            switch (scope) {
                case ANCESTOR:
                    query = new Query();
                    query.setIds(OrgUtils.findOrgIdInParentIds(orgStruct.getParentIds()));
                    orgs = orgStructDao.findAll(OrgQueryHelper.querySpecification(query), EntityGraphs.named(OrgStruct.NAMED_ENTITY_GRAPH_DEFAULT));
                    orgs = orgs.stream().filter(o -> !o.getOrgId().equals(orgStruct.getOrgId())).collect(toList());
                    List<OrgStruct> ancestors = new ArrayList<>(orgs.size());
                    for (OrgStruct o : orgs) {
                        OrgStruct o2 = new OrgStruct();
                        BeanUtils.copyProperties(o, o2, Stream.of("ancestors").toArray(String[]::new));
                        ancestors.add(o2);
                    }
                    orgStruct.setAncestors(ancestors);
                    break;
                case DESCENDANT:
                    query = new Query();
                    query.setParentIds(orgStruct.getParentIds());
                    orgs = orgStructDao.findAll(OrgQueryHelper.querySpecification(query), EntityGraphs.named(OrgStruct.NAMED_ENTITY_GRAPH_DEFAULT));
                    orgs = orgs.stream().filter(o -> !o.getOrgId().equals(orgStruct.getOrgId())).collect(toList());
                    List<OrgStruct> descendants = new ArrayList<>(orgs.size());
                    for (OrgStruct o : orgs) {
                        OrgStruct o2 = new OrgStruct();
                        BeanUtils.copyProperties(o, o2, Stream.of("descendants").toArray(String[]::new));
                        descendants.add(o2);
                    }
                    orgStruct.setDescendants(descendants);
                    break;
                default:
                    break;
            }
        }
        return models;
    }

    public boolean existOrgId(long orgId) {
        return orgStructDao.existsById(orgId);
    }

    public boolean existByTenantIdAndOrgId(long tenantId, long orgId) {
        return orgStructDao.existsByTenantIdAndOrgId(tenantId, orgId);
    }

    @Setter
    @Getter
    @ToString
    @AllArgsConstructor
    @NoArgsConstructor
    static class OrgSaveInput {
        private long tenantId;
        private long parentId;
        private OrgStruct parent;
        private List<OrgStruct> orgs;
    }

    @Setter
    @Getter
    @ToString
    @AllArgsConstructor
    @NoArgsConstructor
    static class OrgSaveOutput {

        private List<OrgStruct> orgs;

        public void addOrg(OrgStruct org) {
            if (this.orgs == null) {
                this.orgs = new ArrayList<>();
            }
            this.orgs.add(org);
        }

        public OrgStruct findFirst() {
            return this.orgs.stream().filter(Objects::nonNull).findFirst().orElse(null);
        }

        public Company findFirstCompany() {
            OrgStruct firstOrg = this.findFirst();
            if (firstOrg != null) {
                return firstOrg.getCompany();
            } else {
                return null;
            }
        }

        public Long findFirstCompanyId() {
            OrgStruct firstOrg = this.findFirst();
            if (firstOrg != null) {
                return firstOrg.getCompanyId();
            } else {
                return null;
            }
        }
    }

    public OrgStruct saveOrgStructEntity(OrgStruct orgStruct) {
        Long existDefaultRoleId = null;
        Long newDefaultRoleId = orgStruct.getDefaultOrgRoleId();
        Tenant tenant = orgStruct.getTenant();
        if (orgStruct.getOrgId() != null && orgStruct.getOrgId() > 0) {
            Optional<OrgStruct> existOptional = orgStructDao.findById(orgStruct.getOrgId());
            if (existOptional.isPresent()) {
                existDefaultRoleId = existOptional.get().getDefaultOrgRoleId();
            }
        }
        this.setGradingRoleIdIfParentOrgRelated(orgStruct);
        this.relOrgUserWithNewOrgRoleId(orgStruct);
        orgStruct = orgStructDao.save(orgStruct);
        if ((tenant == null || tenant.getCreateTime() == null) && orgStruct.getTenantId() != null) {
            Optional<Tenant> optionalTenant = tenantDao.findById(orgStruct.getTenantId());
            if (optionalTenant.isPresent()) {
                tenant = optionalTenant.get();
            }
        }
        if (tenant != null) {
            orgStruct.setTenant(tenant);
        }
        Company company = orgStruct.getCompany();
        if ((company == null || company.getTaxNum() == null) && orgStruct.getCompanyId() != null) {
            Optional<Company> optionalCompany = companyDao.findById(orgStruct.getCompanyId());
            if (optionalCompany.isPresent()) {
                company = optionalCompany.get();
            }
        }
        if (company != null) {
            orgStruct.setCompany(company);
        }
        orgPubService.sendOrgMsg(SAVE, orgStruct);
        orgVirtualOrgStructService.asyncRelateParentOrgVirtual(orgStruct);

        if (existDefaultRoleId != null && !existDefaultRoleId.equals(newDefaultRoleId)) {
            if (newDefaultRoleId == null) {
                roleUserRelDao.deleteByRoleIdAndOrgId(existDefaultRoleId, orgStruct.getOrgId());
            } else {
                roleUserRelDao.updateRoleId(existDefaultRoleId, newDefaultRoleId);
            }
        }
        orgStruct.postLoad();
        return orgStruct;
    }

    /**
     * 如果父级组织关联了分级角色id，则本组织也需要关联
     *
     * @param orgStruct
     */
    private void setGradingRoleIdIfParentOrgRelated(OrgStruct orgStruct) {
        Long parentOrgId = orgStruct.getParentId();
        if (parentOrgId != null && parentOrgId > 0) {
            OrgStruct parentOrg = orgStructDao.findById(parentOrgId).orElseThrow(() -> new IllegalArgumentException("组织id 【" + parentOrgId + "】不存在"));
            if (parentOrg.getGradingRoleId() != null && parentOrg.getGradingRoleId() > 0) {
                orgStruct.setGradingRoleId(parentOrg.getGradingRoleId());
            }
        }
    }

    /**
     * 保存组织时只要默认组织角色存在，就需要把这个组织关联的所有用户也关联上这个组织角色
     *
     * @param orgStruct
     */
    private void relOrgUserWithNewOrgRoleId(OrgStruct orgStruct) {
        if (null != orgStruct.getDefaultOrgRoleId() && orgStruct.getDefaultOrgRoleId() > 0) {
            List<Long> userIds = orgUserRelDao.findUserIdsNotBindOrgRole(orgStruct.getOrgId(), orgStruct.getDefaultOrgRoleId());
            if (!CollectionUtils.isEmpty(userIds)) {
                List<RoleUserRel> list = new ArrayList<>();
                for (Long userId : userIds) {
                    RoleUserRel rel = new RoleUserRel();
                    rel.setOrgId(orgStruct.getOrgId());
                    rel.setUserId(userId);
                    rel.setRelType(2);
                    rel.setRoleId(orgStruct.getDefaultOrgRoleId());
                    rel.setTenantId(orgStruct.getTenantId());
                    list.add(rel);
                }
                roleUserRelDao.saveAllAndFlush(list);

            }
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void bindUsers(long tenantId, long orgId, BindUsers bindUsers) {
        if (bindUsers == null) {
            return;
        }
        this.bindUsers(tenantId, orgId, bindUsers.getUserIds(), bindUsers.isOverwrite());
    }

    @Transactional(rollbackFor = Exception.class)
    public void bindUsers(long orgId, BindUsers bindUsers) {
        if (bindUsers == null) {
            return;
        }
        this.bindUsers(null, orgId, bindUsers.getUserIds(), bindUsers.isOverwrite());
    }

    @Transactional(rollbackFor = Exception.class)
    public void bindUsers(Long tenantId, long orgId, Collection<Long> userIds, boolean isOverwrite) {
        Query query = new Query();
        query.setTenantId(tenantId);
        query.setOrgId(orgId);
        Optional<OrgStruct> existsOrgOptional = orgStructDao.findOne(OrgQueryHelper.queryOneSpecification(query));
        if (!existsOrgOptional.isPresent()) {
            throw new IllegalArgumentException("未找到组织实体(" + orgId + ")");
        }
        OrgStruct existsOrg = existsOrgOptional.get();
        Long defaultOrgRoleId = existsOrg.getDefaultOrgRoleId();
        if (defaultOrgRoleId != null && defaultOrgRoleId > 0) {
            roleService.bindUsers(defaultOrgRoleId, userIds, false);
        }
        List<OrgUserRel> existRels = orgUserRelDao.findByOrgId(orgId);
        Set<OrgUserRel> insertingRels = userIds.stream().filter(Objects::nonNull).filter(userId -> existRels.stream().map(OrgUserRel::getUserId).noneMatch(relUserId -> relUserId.equals(userId))).map(userId -> {
            Optional<User> userOptional = userDao.findById(userId);
            if (userOptional.isPresent()) {
                OrgUserRel orgUserRel = new OrgUserRel();
                if (tenantId != null && tenantId > 0) {
                    orgUserRel.setTenantId(tenantId);
                } else {
                    orgUserRel.setTenantId(userOptional.get().getTenantId());
                }
                orgUserRel.setOrgStructId(orgId);
                orgUserRel.setUserId(userId);
                return orgUserRel;
            } else {
                return null;
            }
        }).filter(Objects::nonNull).collect(Collectors.toSet());
        if (!insertingRels.isEmpty()) {
            orgUserRelDao.saveAllAndFlush(insertingRels);
        }
        if (isOverwrite) {
            Set<Long> relIds = existRels.stream().filter(rel -> userIds.stream().noneMatch(userId -> userId.equals(rel.getUserId()))).map(OrgUserRel::getId).collect(toSet());
            if (relIds != null && !relIds.isEmpty()) {
                orgUserRelDao.deleteAllByIdInBatch(relIds);
            }
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void unbindUsers(long tenantId, long orgId, UnbindUsers unbindUsers) {
        Set<Long> userIds = unbindUsers.getUserIds();
        if (CollectionUtils.isNotEmpty(userIds)) {
            orgUserRelDao.deleteByTenantIdAndOrgIdAndUserIds(tenantId, orgId, userIds);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void changeRootName(long tenantId, String orgName) {
        List<OrgStruct> orgStructList = orgStructDao.findRootsByTenantId(tenantId);
        //应该只有一个，取第一个
        if (CollectionUtils.isNotEmpty(orgStructList)) {
            OrgStruct orgStruct = orgStructList.get(0);
            orgStruct.setOrgName(orgName);
            this.saveOrgStructEntity(orgStruct);
        }
    }
}
