package com.xforceplus.business.tenant.service;

import com.xforceplus.api.model.user.OrgUserModel;
import com.xforceplus.bo.user.OrgUserBO;
import com.xforceplus.bo.user.OrgUserComplexBO;
import com.xforceplus.dao.OrgStructDao;
import com.xforceplus.dao.OrgUserRelDao;
import com.xforceplus.dao.user.OrgUserExtendDao;
import com.xforceplus.dto.user.OrgUserDTO;
import com.xforceplus.entity.OrgUserRel;
import com.xforceplus.utils.OrgUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.HashSet;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;


/**
 * 保存 组织-用户关联表
 *
 * @author geewit
 */
@Service
public class OrgUserService {

    private static final Logger logger = LoggerFactory.getLogger(OrgUserService.class);

    private final OrgUserRelDao orgUserRelDao;

    private final OrgStructDao orgStructDao;

    private final OrgUserExtendDao orgUserExtendDao;

    public OrgUserService(OrgUserRelDao orgUserRelDao, OrgStructDao orgStructDao, OrgUserExtendDao orgUserExtendDao) {
        this.orgUserRelDao = orgUserRelDao;
        this.orgStructDao = orgStructDao;
        this.orgUserExtendDao = orgUserExtendDao;
    }


    /**
     * 根据父组织绑定用户
     *
     * @param tenantId 租户Id
     * @param orgId    组织Id
     */
    public void autoBindUsers(Long tenantId, Long orgId, Long startTime) {
        //region 传入的组织id对应的parentIds
        String parentIds = orgStructDao.findCommittedParentIdsByOrgId(orgId);
        if (StringUtils.isBlank(parentIds)) {
            logger.warn("传的orgId({})不存在或对应的parentIds为空, parentIds={}", orgId, parentIds);
            return;
        }
        //endregion

        //region 传入的组织id对应的所有父组织id集合
        Set<Long> parentOrgIds = OrgUtils.findOrgIdInParentIds(parentIds);
        if (CollectionUtils.isEmpty(parentOrgIds)) {
            logger.warn("传的orgId({})不存在或对应的parentIds为空", orgId);
            return;
        } else {
            parentOrgIds.remove(orgId);
        }
        if (CollectionUtils.isEmpty(parentOrgIds)) {
            logger.warn("传的orgId({})不存在或对应的parentIds为空", orgId);
            return;
        }
        //endregion

        //region 已经绑定的用户组织关系集合
        Set<OrgUserRel> boundUserOrgRels = orgUserRelDao.findRelsByTenantIdAndOrgIds(tenantId, parentOrgIds);
        if (CollectionUtils.isEmpty(boundUserOrgRels)) {
            logger.warn("在父组织中没有找到已绑定的用户,orgId={},parentOrgIds={}", orgId, parentIds);
            return;
        }
        logger.info("boundUserOrgRels.size = {}, orgId = {}", boundUserOrgRels.size(), orgId);
        //endregion

        //region 应该绑定的用户id集合
        Set<Long> shouldBoundUserIds = boundUserOrgRels.stream().map(OrgUserRel::getUserId).collect(Collectors.toSet());
        if (CollectionUtils.isEmpty(shouldBoundUserIds)) {
            logger.warn("在父组织中没有找到已绑定的用户");
            return;
        }
        logger.info("shouldBoundUserIds.size = {}, orgId = {}", shouldBoundUserIds.size(), orgId);
        //endregion

        //region 循环应该绑定的用户id集合, 并保存存入的组织对应的关系
        Set<Long> boundUserIds = orgUserRelDao.findUserIdByTenantIdAndOrgId(tenantId, orgId);
        Set<OrgUserRel> shouldBoundRels = new HashSet<>();
        shouldBoundUserIds.removeAll(boundUserIds);
        shouldBoundUserIds.stream()
                .filter(Objects::nonNull)
//                .filter(shouldBoundUserId ->
//                        boundUserOrgRels.stream().noneMatch(rel -> rel.getOrgStructId().equals(orgId) && rel.getUserId().equals(shouldBoundUserId))
//                )
                .forEach(shouldBoundUserId -> {
                    OrgUserRel rel = new OrgUserRel();
                    rel.setOrgStructId(orgId);
                    rel.setTenantId(tenantId);
                    rel.setUserId(shouldBoundUserId);
                    rel.setFullSelectedFlag(true);
                    shouldBoundRels.add(rel);
                });
        //endregion

        //分批次处理
        if (!shouldBoundRels.isEmpty()) {
            orgUserRelDao.saveAllAndFlush(shouldBoundRels);
        }
        logger.info("auto_bind_user_end, orgId={},count:{}, caused:{}", orgId, shouldBoundRels.size(), (System.currentTimeMillis() - startTime));

    }

    /**
     * 查询当前组织节点的用户列表
     *
     * @param pageable  {@link Pageable}
     * @param orgUserBO OrgUserBO
     * @return Page<OrgUserDTO>
     */
    @Transactional(readOnly = true, rollbackFor = RuntimeException.class)
    public Page<OrgUserDTO> pageByCurrentOrg(Pageable pageable, OrgUserBO orgUserBO) {
        return this.orgUserExtendDao.pageByCurrentOrg(pageable, orgUserBO);
    }


    /**
     * 查询当前节点的用户及以下用户
     *
     * @param pageable  {@link Pageable}
     * @param orgUserBO 查询条件
     * @return Page<OrgUserDTO>
     */
    @Transactional(readOnly = true, rollbackFor = RuntimeException.class)
    public Page<OrgUserDTO> pageByBelongOrg(Pageable pageable, OrgUserBO orgUserBO) {
        //查询当前节点并获取当前节点parentIds
        String parentIds = orgStructDao.findCommittedParentIdsByOrgId(orgUserBO.getOrgId());
        if (StringUtils.isEmpty(parentIds)) {
            throw new IllegalArgumentException("没有查询到组织节点");
        }
        orgUserBO.setParentIds(parentIds);
        return orgUserExtendDao.pageByBelongOrg(pageable, orgUserBO);
    }

    /**
     * 复杂条件查询当前组织节点的用户列表
     *
     * @param pageable  {@link Pageable}
     * @param orgUserBO OrgUserComplexBO
     * @return Page<OrgUserDTO>
     */
    public Page<OrgUserDTO> pageByCurrentOrgQueryType(Pageable pageable, OrgUserComplexBO orgUserBO) {
        return this.orgUserExtendDao.pageByCurrentOrgQueryType(pageable, orgUserBO);
    }

    /**
     * 复杂条件查询当前节点的用户及以下用户
     *
     * @param pageable  {@link Pageable}
     * @param orgUserBO 查询条件
     * @return Page<OrgUserDTO>
     */
    public Page<OrgUserDTO> pageByBelongOrgQueryType(Pageable pageable, OrgUserComplexBO orgUserBO) {
        String parentIds = null;
        //查询parentIds
        if (orgUserBO.getQueryType() == OrgUserModel.OrgQueryType.CODE) {
            Set<String> parentIdsSet = orgStructDao.findParentIdsByOrgCode(orgUserBO.getOrgCode());
            if (!CollectionUtils.isEmpty(parentIdsSet)) {
                parentIds = parentIdsSet.stream().findFirst().orElse(null);
            }
        } else {
            //查询当前节点并获取当前节点parentIds
            parentIds = orgStructDao.findCommittedParentIdsByOrgId(orgUserBO.getOrgId());
        }

        if (StringUtils.isEmpty(parentIds)) {
            throw new IllegalArgumentException("没有查询到组织节点");
        }
        orgUserBO.setParentIds(parentIds);
        return orgUserExtendDao.pageByBelongOrgQueryType(pageable, orgUserBO);
    }


}