package com.xforceplus.business.tenant.service;

import com.google.common.collect.Lists;
import com.xforceplus.api.model.RoleModel;
import com.xforceplus.business.file.service.FileService;
import com.xforceplus.business.resource.service.ResourceService;
import com.xforceplus.business.resource.service.ResourcesetService;
import com.xforceplus.constants.RoleTypeEnum;
import com.xforceplus.dao.*;
import com.xforceplus.domain.tenant.PreRoleDTO;
import com.xforceplus.entity.Role;
import com.xforceplus.entity.RoleServicePackage;
import com.xforceplus.entity.RoleUserRel;
import com.xforceplus.utils.ObjectCheckAndExcuteUtils;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 预置角色service
 */
@Service
public class PreRoleService {
    private final static Logger logger = LoggerFactory.getLogger(PreRoleService.class);

    private final RoleDao roleDao;

    private final UserDao userDao;

    private final RoleUserRelDao roleUserRelDao;

    private final RoleResourcesetRelDao roleResourcesetRelDao;

    private final ResourcesetDao resourcesetDao;

    private final TenantDao tenantDao;

    private final FileService fileService;

    private final ResourcesetService resourcesetService;

    private final ResourceService resourceService;

    private final RoleServicePackageRelDao roleServicePackageRelDao;

    private final ServiceResourcesetRelDao serviceResourcesetRelDao;

    private final RoleService roleService;

    public   static final Long PER_TENANT_ID = -1L;

    public PreRoleService(RoleDao roleDao, UserDao userDao, RoleUserRelDao roleUserRelDao, RoleResourcesetRelDao roleResourcesetRelDao
            , ResourcesetDao resourcesetDao, TenantDao tenantDao, FileService fileService, ResourcesetService resourcesetService
            , ResourceService resourceService, RoleServicePackageRelDao roleServicePackageRelDao, ServiceResourcesetRelDao serviceResourcesetRelDao, RoleService roleService) {
        this.roleDao = roleDao;
        this.userDao = userDao;
        this.roleUserRelDao = roleUserRelDao;
        this.roleResourcesetRelDao = roleResourcesetRelDao;
        this.resourcesetDao = resourcesetDao;
        this.tenantDao = tenantDao;
        this.fileService = fileService;
        this.resourcesetService = resourcesetService;
        this.resourceService = resourceService;
        this.roleServicePackageRelDao = roleServicePackageRelDao;
        this.serviceResourcesetRelDao = serviceResourcesetRelDao;
        this.roleService = roleService;
    }

    /**
     * 判断预置角色 是否可见
     * @param tanantId
     * @param roleId
     * @return
     */
    public int isVisiable(Long tanantId,Long roleId) {
       return roleDao.queryCountPreRoleByTenantIdAndRoleIdOrRoleCode(tanantId, roleId,null, RoleTypeEnum.PRE.getType());
    }

    public Integer queryCountPreRoleByTenantIdAndRoleIdOrRoleCode(Long tenantId,Long roleId,String roleCode,Integer roleTye) {
       return roleDao.queryCountPreRoleByTenantIdAndRoleIdOrRoleCode(tenantId, roleId, roleCode, roleTye);
    }
    /**
     * 查询租户下可见的预置角色列表
     * @param tanantId
     * @return
     */
    public Page<PreRoleDTO> queryPreRoleListByTenantId(Long tanantId,Long roleId,String roleCode,String roleName,Pageable pageable) {
        return   roleDao.queryPreRoleByTenantId(tanantId, roleId, roleName, roleCode,RoleTypeEnum.PRE.getType(),  pageable);

     }

    /**
     * 根据产品服务包查询 预置角色列表 分页
     * @param query
     * @param pageable
     * @return
     */
    public Page<PreRoleDTO> queryPreRoleListByAdmin(RoleModel.Request.PreRoleQuery query, Pageable pageable) {
        if (Objects.isNull(query)) {
            throw new IllegalArgumentException("无效查询参数");
        }
        return   roleDao.queryPreRoleByAdmin( RoleTypeEnum.PRE.getType(),query.getAppId(),query.getServicePackageId(), query.getRoleName(), query.getRoleCode(), query.getId(), pageable);
    }

    /**
     * 绑定角色与服务包关系
     * @param roleId
     * @param servicePackageId
     * @param userId
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public Role bindServicePackageRole(RoleModel.Request.PreCreate model,  Long userId) {
        if (Objects.isNull(model.getBindResourceSets()) || CollectionUtils.isEmpty(model.getBindResourceSets().getResourcesetIds())) {
            throw new IllegalArgumentException("所选择的功能集不能为空");
        }
        checkServicePackageAndSetId(model.getServicePackageId(), model.getBindResourceSets().getResourcesetIds());
        model.setType(RoleTypeEnum.PRE.getType());
        RoleModel.Request.Create create = new RoleModel.Request.Create();
        BeanUtils.copyProperties(model, create);
        /**
         *默认使用 -1 表示预置角色
         */
        create.setTenantId(PER_TENANT_ID);
        Role role = roleService.create(create);
        RoleServicePackage roleServicePackage = new RoleServicePackage();
        roleServicePackage.setServicePackageId(model.getServicePackageId());
        roleServicePackage.setRoleId(role.getId());
        roleServicePackage.setCreateTime(new Date());
        roleServicePackage.setStatus(1);
        roleServicePackage.setUserId(userId);
        roleServicePackageRelDao.save(roleServicePackage);
        //查询功能集是否属于此功能包
        return role;
    }

    /**
     * 比对 产品包下的 功能集
     * @param serviceId
     * @param setId
     * @return
     */
    private boolean checkServicePackageAndSetId(Long serviceId,List<Long> setId) {
        List<Long> res = serviceResourcesetRelDao.findResourcesetIdByServicePackageId(serviceId);

        if (CollectionUtils.isEmpty(res)) {
            throw new IllegalArgumentException("该资源包下功能集为空");
        }
        if (CollectionUtils.isEmpty(setId)) {
            throw new IllegalArgumentException("功能集不能为空");
        }
        return CollectionUtils.isEqualCollection(res, setId);
     }
    @Transactional(rollbackFor = Exception.class)
    public void delete(Long roleId)   {
        //管理员角色
        Role role = roleService.findById(roleId);
        ObjectCheckAndExcuteUtils.docheckAndExcute(role, x->{return Objects.nonNull(x) && x.getType().compareTo(RoleTypeEnum.PRE.getType()) == 0;}, x -> x, new IllegalArgumentException("只能删除预置角色"));
        long userCount = roleUserRelDao.countByRoleId(roleId);
        if (userCount > 0) {
            throw new IllegalArgumentException("已经关联用户的角色(" + roleId + ")不能删除");
        }
        roleService.commonDeleteRole(roleId) ;
        roleServicePackageRelDao.deleteByRoleId(roleId);
    }


    /**
     * 用户授权时 需要判断用户所在租户是否 具备预置角色的授权条件
     * @param userId
     * @param roleId
     * @return
     */
    public List<Long> checkUser(List<Long> userId, Long tenantId) {

        return userDao.checkUser(userId, tenantId);
    }

    public void updatePreRoleStatus(Long roleId, Integer status, Function<Role, Boolean> function,Throwable throwable) throws Throwable {
        Role role = roleService.findById(roleId);
        Boolean res = function.apply(role);
        if (!res) {
            throw throwable;
        }
        roleService.updateStatus(roleId, status);
    }

    /**
     * 查询用户绑定或未绑定的预置角色列表
     * @param tenantId
     * @param binded
     * @return
     */
    public List<Long> queryUserPreRole(Long tenantId,Long userId, Boolean binded) {
        binded = Objects.isNull(binded) ? Boolean.FALSE : binded;
        /**
         * 如果用户为空 则不获取用户的预置角色信息
         */
        if(Objects.isNull(userId) || Objects.isNull(tenantId)){
            return Collections.EMPTY_LIST;
        }
        List<RoleUserRel> roleUserRels = roleUserRelDao.findByUserId(userId);
        List<Long> roleIds = Lists.newArrayList(0L);
        if (CollectionUtils.isNotEmpty(roleUserRels)) {
            roleIds = roleUserRels.stream().map(roleUserRel -> {
                return roleUserRel.getRoleId();
            }).collect(Collectors.toList());
        }
        return roleDao.queryUserPreRole(tenantId,   binded,roleIds);
    }
 }
