package com.xforceplus.business.tenant.service;

import com.alibaba.excel.event.AnalysisEventListener;
import com.alibaba.fastjson.JSON;
import com.xforceplus.business.excel.BusinessType;
import com.xforceplus.business.excel.ExcelSheet;
import com.xforceplus.business.excel.reader.Context;
import com.xforceplus.business.excel.reader.MessageRow;
import com.xforceplus.business.excel.reader.SimpleDataReadListener;
import com.xforceplus.business.excel.writer.ExcelConfigBusinessType;
import com.xforceplus.business.service.ExcelReaderService;
import com.xforceplus.business.tenant.dto.RoleAccountImportDto;
import com.xforceplus.utils.BatchUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.ExecutionException;

/**
 * <p>
 * Title:
 * </p>
 * <p>
 * Description:
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author dhy
 * <b>Creation Time:</b> 2020-09-24 14-03-25
 * @since V1.0
 */
@Service
public class RoleAccountImportServiceImpl implements ExcelReaderService {
    /**
     * 账号角色绑定
     */
    public static final String ACTION_BIND = "绑定";
    /**
     * 账号角色解绑
     */
    public static final String ACTION_UNBIND = "解绑";
    /**
     * 操作列表
     */
    public static final List<String> ACTION_LISTS = Arrays.asList(ACTION_BIND, ACTION_UNBIND);
    /**
     * 日志
     */
    private static final Logger logger = LoggerFactory.getLogger(RoleAccountImportServiceImpl.class);
    @Resource
    private UserService userService;

    private static final HashMap<String, Integer> HEADER_ROW_NUMBER = new HashMap<>(1);

    static {
        HEADER_ROW_NUMBER.put(RoleConfigExcel.ROLE_ACCOUNT_IMPORT_SHEET, 2);
    }

    /**
     * 获取导入类型，用于Event事件调整导入方法
     * @return ImportBusinessType
     */
    @Override
    public BusinessType getBusinessType() {
        return ExcelConfigBusinessType.ROLE_BIND_ACCOUNT_IMPORT;
    }

    /**
     * 异步导出处理
     *
     * @param context 上下文
     * @return Context
     */
    @Override
    public Context importExcel(Context context) {
        List<ExcelSheet> sheets = context.getExcelBook().getExcelSheets();

        for (ExcelSheet sheet : sheets) {
            MessageRow messageRows = new MessageRow(sheet.getSheetName());
            Integer headerRow = getSheetHeaderNumber().get(sheet.getSheetName());
            if (headerRow == null) {
                headerRow = 2;
            }
            //构建监听器
            AnalysisEventListener<RoleAccountImportDto> excelDataListener = SimpleDataReadListener.listener(context, (rows) -> {
                this.saveData(messageRows, context, rows);
            });
            //开始处理
            context.getSimpleExcelReader().read(RoleAccountImportDto.class, excelDataListener,headerRow);
        }
        return context;
    }

    /**
     * 写入数据
     *
     * @param list
     */
    public void saveData(MessageRow messageRows, Context context, List<RoleAccountImportDto> list) {
        logger.info("list:{}", JSON.toJSONString(list));
       try {
           BatchUtils.doBatchAsync(list,(sublist -> this.batchAsyncSaveData(messageRows,context,sublist))).get();
       }catch (InterruptedException | ExecutionException exception){
           logger.error("error async batch save data {} ",exception.getMessage());
           throw new IllegalArgumentException(exception.getMessage());
       }
        context.messageRow(messageRows.getSheetName(), messageRows);
    }

    private void batchAsyncSaveData(MessageRow messageRows,Context context,List<RoleAccountImportDto> list){
        for (RoleAccountImportDto e : list) {
            logger.info("e:{}", JSON.toJSONString(e));
            if (!e.getValidatedStatus()) {
                messageRows.fail(e.getRowIndex(), e.getValidatedMessage());
                continue;
            }
            String msg = userService.bindRoleAccountRel(e, context.getFileDTO().getTenantId());
            if (StringUtils.isNoneBlank(msg)) {
                messageRows.fail(e.getRowIndex(), msg);
            } else {
                messageRows.success(e.getRowIndex());
            }
        }
    }

    /**
     * 设置表头从第几行开始读
     *
     * @return
     */
    @Override
    public HashMap<String, Integer> getSheetHeaderNumber() {
        return HEADER_ROW_NUMBER;
    }
}
