package com.xforceplus.business.tenant.service;

import com.google.common.collect.Lists;
import com.xforceplus.api.common.response.ResponseBoolenEntity;
import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.model.ResourcesetModel;
import com.xforceplus.api.model.RoleModel.Request.*;
import com.xforceplus.business.file.service.FileService;
import com.xforceplus.business.resource.service.ResourceService;
import com.xforceplus.business.resource.service.ResourcesetService;
import com.xforceplus.dao.ResourcesetDao;
import com.xforceplus.dao.RoleDao;
import com.xforceplus.dao.RoleResourcesetRelDao;
import com.xforceplus.dao.RoleUserRelDao;
import com.xforceplus.dao.TenantDao;
import com.xforceplus.dao.UserDao;
import com.xforceplus.domain.resource.ResourceDto;
import com.xforceplus.domain.resource.ResourcesetDto;
import com.xforceplus.domain.tenant.OrgRoleCntDTO;
import com.xforceplus.domain.tenant.RoleBindDTO;
import com.xforceplus.domain.tenant.RoleExportDto;
import com.xforceplus.domain.tenant.RoleRelAccountExportDto;
import com.xforceplus.domain.tenant.RoleResourcesetRelDto;
import com.xforceplus.domain.tenant.RoleUserCntDTO;
import com.xforceplus.entity.*;
import com.xforceplus.query.ResourcesetQueryHelper;
import com.xforceplus.query.RoleQueryHelper;
import com.xforceplus.utils.excel.ExcelUtils;
import io.geewit.core.utils.reflection.BeanUtils;
import io.geewit.data.jpa.essential.domain.EntityGraphs;
import io.geewit.utils.uuid.UUID;
import io.geewit.web.utils.JsonUtils;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.web.multipart.MultipartFile;

import javax.persistence.Tuple;

import static java.util.stream.Collectors.toSet;


@Service
public class RoleService {
    private final static Logger logger = LoggerFactory.getLogger(RoleService.class);

    private final RoleDao roleDao;

    private final UserDao userDao;

    private final RoleUserRelDao roleUserRelDao;

    private final RoleResourcesetRelDao roleResourcesetRelDao;

    private final ResourcesetDao resourcesetDao;

    private final TenantDao tenantDao;

    private final FileService fileService;

    private final ResourcesetService resourcesetService;

    private final ResourceService resourceService;

    public RoleService(RoleDao roleDao, UserDao userDao, RoleUserRelDao roleUserRelDao,
                       RoleResourcesetRelDao roleResourcesetRelDao, ResourcesetDao resourcesetDao,
                       TenantDao tenantDao, FileService fileService,
                       ResourcesetService resourcesetService, ResourceService resourceService) {
        this.roleDao = roleDao;
        this.userDao = userDao;
        this.roleUserRelDao = roleUserRelDao;
        this.roleResourcesetRelDao = roleResourcesetRelDao;
        this.resourcesetDao = resourcesetDao;
        this.tenantDao = tenantDao;
        this.fileService = fileService;
        this.resourcesetService = resourcesetService;
        this.resourceService = resourceService;
    }

    @SuppressWarnings("AlibabaLowerCamelCaseVariableNaming")
    public Page<Role> page(Query query, Pageable pageable) {
        Page<Role> page = this.rawPage(query, pageable);;
        if(!page.isEmpty()) {
            Set<Long> roleIds = page.stream().map(Role::getId).collect(Collectors.toSet());
            List<RoleUserCntDTO> roleUserCntDTOS = roleDao.getRoleUserCntByRoleIds(query.getTenantId(), roleIds);
            Map<Long, Long> map = new HashMap<>();
            if(!CollectionUtils.isEmpty(roleUserCntDTOS)) {
                for (RoleUserCntDTO roleUserCntDTO : roleUserCntDTOS) {
                    map.put(roleUserCntDTO.getRoleId(), roleUserCntDTO.getRoleUserCnt());
                }

                for (Role role : page) {
                    role.setUserCnt(map.getOrDefault(role.getId(), 0L));
                }
            }
        }

        return page;
    }

    public Page<Role> preRolePage(Query query, Pageable pageable) {
        query.setType(3);
        query.setAttributes(Stream.of("role", "appName", "servicePackageName").collect(Collectors.toSet()));
        return this.rawPage(query, pageable);
    }

    private Page<Role> rawPage(Query query, Pageable pageable) {
        Page<Role> page;
        if (query.isTupleSelection()) {
            Page<Tuple> tuples = roleDao.findTuples(query, pageable);
            List<Role> contents = tuples.getContent().stream().map(RoleQueryHelper.tupleMapper(query)).collect(Collectors.toList());
            page = new PageImpl<>(contents, pageable, tuples.getTotalElements());
        } else {
            Specification<Role> specification = RoleQueryHelper.querySpecification(query);
            page = roleDao.findAll(specification, pageable, EntityGraphs.named(OrgStruct.NAMED_ENTITY_GRAPH_DEFAULT));
        }
        if (page == null) {
            return Page.empty(pageable);
        }
        return page;
    }

    public Page<Role> page(Specification<Role> specification, Pageable pageable) {
        return roleDao.findAll(specification, pageable);
    }

    public List<Role> list(Query query, Sort sort) {
        Specification<Role> specification = RoleQueryHelper.querySpecification(query);
        List<Role> list = roleDao.findAll(specification, sort);
        return list;
    }

    public List<Role> listByUserId(long userId) {
        Query roleQuery = new Query();
        roleQuery.setUserId(userId);
        roleQuery.setStatus(1);
        roleQuery.setFilterExpired(true);
        return this.list(roleQuery, Sort.unsorted());
    }

    public List<Role> list(Specification<Role> specification, Sort sort) {
        return roleDao.findAll(specification, sort);
    }

    public Role findOne(Specification<Role> specification) {
        return roleDao.findOne(specification).orElseThrow(() -> new IllegalArgumentException("未找到角色实体"));
    }

    @Transactional(rollbackFor = Exception.class)
    public Role create(Create model) {
        if(StringUtils.isBlank(model.getRoleCode()) || StringUtils.isBlank(model.getRoleName())) {
            throw new IllegalArgumentException("角色代码或者名称不能为空!");
        }
        Role entity = null;
        //新角色时，在租户下角色Code已存在会覆盖原角色信息
        if(Objects.isNull(model.getTenantId())){
            throw new IllegalArgumentException("角色租户不能为空!");
        }
        List<Role> list = roleDao.findByListTenantIdAndCode(model.getTenantId(), model.getRoleCode());
        if (!CollectionUtils.isEmpty(list)) {
            throw new IllegalArgumentException("角色代码已存在!");
        }
        List<Role> ll = roleDao.findByListTenantIdAndName(model.getTenantId(), model.getRoleName());
        if (!CollectionUtils.isEmpty(ll)) {
            throw new IllegalArgumentException("角色名称已存在!");
        }
        if (entity == null) {
            entity = new Role();
            entity.setCode(model.getRoleCode());
        }
        BeanUtils.copyProperties(model, entity);
        entity.setName(model.getRoleName());
        entity = roleDao.saveAndFlush(entity);
        if (model.getBindResourceSets() != null) {
            this.bindResourceSets(model.getTenantId(), entity, model.getBindResourceSets());
        }
        if (null != model.getBindResources()) {
            this.bindResources(entity.getTenantId(), entity, model.getBindResources());
        }
        return entity;
    }

    @Transactional(rollbackFor = Exception.class)
    public Role save(Save model) {
        Role entity = null;
        if (model.getTenantId() != null && model.getTenantId() > 0) {
            entity = roleDao.findByTenantIdAndCode(model.getTenantId(), model.getRoleCode());
        }
        if (entity == null) {
            entity = new Role();
            entity.setCode(model.getRoleCode());
        }
        BeanUtils.copyProperties(model, entity);
        if (StringUtils.isNotBlank(model.getRoleCode())) {
            entity.setCode(model.getRoleCode());
        }
        if (StringUtils.isNotBlank(model.getRoleName())) {
            entity.setName(model.getRoleName());
        }
        entity = roleDao.saveAndFlush(entity);
        this.bindResourceSets(entity, model.getBindResourceSets());
        return entity;
    }

    /**
     * 更新角色时的检查
     * @param roleId
     * @param model
     * @param function
     */
    public void updateByCheck(long roleId, Save model, Function<Role, Boolean> function) {
        Role existEntity = this.findById(roleId);
        function.apply(existEntity);
    }
    @Transactional(rollbackFor = Exception.class)
    public Role update(long roleId, Save model) {
        Role existEntity = this.findById(roleId);
        BeanUtils.copyProperties(model, existEntity);
        existEntity = roleDao.saveAndFlush(existEntity);
        if (model.getBindResourceSets() != null) {
            this.bindResourceSets(existEntity, model.getBindResourceSets());
        }
        return existEntity;
    }

    public Role findById(long roleId) {
        Role role = roleDao.findById(roleId).orElseThrow(() -> new IllegalArgumentException("未找到角色实体(" + roleId + ")"));
        List<Resourceset> resourcesets = resourcesetService.listByRoleId(roleId);
        Set<ResourcesetDto> resourcesetSet = resourcesets.stream().filter(Objects::nonNull).collect(Collectors.toSet());
        role.setResourcesets(resourcesetSet);
        return role;
    }

    public Role findByTenantId(long tenantId, long roleId) {
        Query query = new Query();
        query.setTenantId(tenantId);
        query.setId(roleId);
        Specification<Role> specification = RoleQueryHelper.querySpecification(query);
        Role role = this.findOne(specification);
        List<Resourceset> resourcesets = resourcesetService.listByRoleId(role.getId());
        Set<ResourcesetDto> resourcesetSet = resourcesets.stream().filter(Objects::nonNull).collect(Collectors.toSet());
        ResourcesetModel.Request.Query resourcesetQuery = new ResourcesetModel.Request.Query();
        resourcesetQuery.setRoleId(roleId);
        Optional<Resourceset> customResourceset = resourcesetService.findOne(resourcesetQuery);
        if (customResourceset.isPresent()) {
            Resourceset resourceset = customResourceset.get();
            List<Resource> resources = resourceService.listByResourcesetId(resourceset.getResourcesetId(), 1);
            Set<ResourceDto> resourceSet = resources.stream().filter(Objects::nonNull).collect(Collectors.toSet());
            role.setResources(resourceSet);
        }
        role.setResourcesets(resourcesetSet);
        return role;
    }

    /**
     * 查询角色信息
     *
     * @param tenantId 租户ID
     * @param roleId   角色Id
     * @return Role
     */
    public Role findByRoleId(Long tenantId, Long roleId) {
        Query query = new Query();
        query.setTenantId(tenantId);
        query.setId(roleId);
        Specification<Role> specification = RoleQueryHelper.querySpecification(query);
        return this.findOne(specification);
    }

    /**
     * 查询角色本身的信息且查询绑定功能集、资源码
     *
     * @param tenantId 租户ID
     * @param roleId   角色ID
     * @return
     */
    public RoleBindDTO findBindedById(Long tenantId, Long roleId) {
        Role role = this.findByRoleId(tenantId, roleId);
        RoleBindDTO roleBindDTO = new RoleBindDTO();
        BeanUtils.copyProperties(role, roleBindDTO);
        return roleBindDTO;
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteById(long roleId) {
        Role role = this.findById(roleId);
        if (role.getTenantId() == null || role.getTenantId() <= 0) {
            throw new IllegalArgumentException("系统角色不能删除");
        }
        commonDeleteRole(roleId);
    }

    public void commonDeleteRole(Long roleId) {
        long userCount = roleUserRelDao.countByRoleId(roleId);
        if (userCount > 0) {
            throw new IllegalArgumentException("已经关联用户的角色(" + roleId + ")不能删除");
        }
        roleDao.deleteById(roleId);
        roleResourcesetRelDao.deleteByRoleId(roleId);
        roleUserRelDao.deleteByRoleId(roleId);
     }

    @Transactional(rollbackFor = Exception.class)
    public Role updateByTenantId(long tenantId, long roleId, Save model) {
        Role existEntity = this.findByTenantId(tenantId, roleId);
        BeanUtils.copyProperties(model, existEntity, Stream.of("roleResourcesetRels", "roleUserRels", "tags").toArray(String[]::new));
        if (StringUtils.isNotBlank(model.getRoleName())) {
            existEntity.setName(model.getRoleName());
        }
        existEntity = roleDao.saveAndFlush(existEntity);
        if (model.getBindResourceSets() != null) {
            this.bindResourceSets(tenantId, existEntity, model.getBindResourceSets());
        }

        if (null != model.getBindResources()) {
            this.bindResources(tenantId, existEntity, model.getBindResources());
        }
        return existEntity;
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteByTenantIdAndId(long tenantId, long roleId) {
        Role existEntity = this.findByTenantId(tenantId, roleId);
        long userCount = roleUserRelDao.countByRoleId(roleId);
        if (userCount > 0) {
            throw new IllegalArgumentException("已经关联用户的角色(" + roleId + ")不能删除");
        }
        roleDao.deleteById(roleId);
        roleResourcesetRelDao.deleteByRoleId(roleId);
        roleUserRelDao.deleteByRoleId(roleId);
    }

    @Transactional(rollbackFor = Exception.class)
    public void bindUsers(long roleId, BindUsers bindUsers) {
        if (bindUsers == null) {
            return;
        }
        this.bindUsers(roleId, bindUsers.getUserIds(), bindUsers.isOverwrite());
    }

    @Transactional(rollbackFor = Exception.class)
    public void bindUsers(long roleId, Collection<Long> userIds, boolean isOverwrite) {

        Role role = roleDao.findById(roleId).orElseThrow(() -> new IllegalArgumentException("未找到角色实体(" + roleId + ")"));

        List<RoleUserRel> existRels = roleUserRelDao.findByRoleId(roleId);
        Set<RoleUserRel> insertingRels = userIds.stream().filter(Objects::nonNull).filter(userId -> existRels.stream().map(RoleUserRel::getUserId).noneMatch(relUserId -> relUserId.equals(userId))).map(userId -> {
            Optional<User> userOptional = userDao.findById(userId);
            if (userOptional.isPresent()) {
                RoleUserRel roleUserRel = new RoleUserRel();
                roleUserRel.setTenantId(userOptional.get().getTenantId());
                roleUserRel.setRelType(role.getType());
                roleUserRel.setRoleId(roleId);
                roleUserRel.setUserId(userId);
                return roleUserRel;
            } else {
                return null;
            }
        }).filter(Objects::nonNull).collect(Collectors.toSet());
        if (!insertingRels.isEmpty()) {
            roleUserRelDao.saveAllAndFlush(insertingRels);
        }
        if (isOverwrite) {
            Set<Long> relIds = existRels.stream().filter(rel -> userIds.stream().noneMatch(userId -> userId.equals(rel.getUserId()))).map(RoleUserRel::getId).collect(toSet());
            if (relIds != null && !relIds.isEmpty()) {
                roleUserRelDao.deleteAllByIdInBatch(relIds);
            }
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateStatus(long roleId, int status) {
        this.updateStatus(0, roleId, status);
    }

    @Transactional(rollbackFor = Exception.class)
    public void enable(long roleId) {
        this.updateStatus(roleId, 1);
    }

    @Transactional(rollbackFor = Exception.class)
    public void disable(long roleId) {
        this.updateStatus(roleId, 0);
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateStatus(long tenantId, long roleId, int status) {
        Query query = new Query();
        if (tenantId > 0) {
            query.setTenantId(tenantId);
        }
        query.setId(roleId);
        Specification<Role> specification = RoleQueryHelper.queryOneSpecification(query);
        roleDao.findOne(specification).orElseThrow(() -> new IllegalArgumentException("找不到角色实体(" + roleId + ")"));
        roleDao.updateStatus(roleId, status);
    }

    @Transactional(rollbackFor = Exception.class)
    public void bindResourceSets(long roleId, BindResourceSets bindResourceSets) {
        Role role = this.findById(roleId);
        this.bindResourceSets(role, bindResourceSets);
    }

    @Transactional(rollbackFor = Exception.class)
    public void bindResourceSets(Long tenantId, Role role, BindResourceSets bindResourceSets) {
        if (bindResourceSets == null || bindResourceSets.getResourcesetIds() == null) {
            return;
        }

        //列表数据
        List<Resourceset> resourceSetList = this.resourcesetDao.findByResourcesetIds(bindResourceSets.getResourcesetIds());
        List<RoleResourcesetRel> existRels = roleResourcesetRelDao.findByRoleId(role.getId());
        //如果为空
        if (CollectionUtils.isEmpty(resourceSetList)) {
            //是否重写,如是取消所的功能，则删除所有数据
            this.bindResourceSetsOverwrite(existRels, bindResourceSets);
            return;
        }

        Set<Long> bindedResourceSetIds = existRels.stream()
                .map(e -> e.getResousesetId())
                .collect(Collectors.toSet());
        //管理
        Set<RoleResourcesetRel> insertingRels = resourceSetList
                .stream()
                .filter(e -> !bindedResourceSetIds.contains(e.getResourcesetId())).map(
                        e -> {
                            RoleResourcesetRel insertingRel = new RoleResourcesetRel();
                            insertingRel.setTenantId(tenantId);
                            insertingRel.setRoleId(role.getId());
                            insertingRel.setResousesetId(e.getResourcesetId());
                            return insertingRel;
                        }
                ).filter(Objects::nonNull).collect(Collectors.toSet());

        //删除数据
        if (!CollectionUtils.isEmpty(insertingRels)) {
            roleResourcesetRelDao.saveAllAndFlush(insertingRels);
        }
        //是否重写，重写则删除已经绑定外所有的功能集资源;
        this.bindResourceSetsOverwrite(existRels, bindResourceSets);
    }

    /**
     * 是否重复写数据
     *
     * @param existRels        删除数据
     * @param bindResourceSets 判断数据
     */
    private void bindResourceSetsOverwrite(List<RoleResourcesetRel> existRels, BindResourceSets bindResourceSets) {
        //是否重写
        if (bindResourceSets.isOverwrite()) {
            existRels.stream().filter(rel -> bindResourceSets.getResourcesetIds().stream().noneMatch(resourcesetId -> resourcesetId.equals(rel.getResousesetId()))).forEach(rel -> {
                logger.info("deleting Role-Resourceset-Rel record, {}", rel);
                try {
                    roleResourcesetRelDao.deleteById(rel.getId());
                } catch (Exception e) {
                    logger.warn(e.getMessage(), e);
                }
            });
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void bindResourceSets(Role role, BindResourceSets bindResourceSets) {
        this.bindResourceSets(role.getTenantId(), role, bindResourceSets);
    }

    @Transactional(rollbackFor = Exception.class)
    public ResponseBoolenEntity<Boolean, BindResourceSet> bindResourceSets(List<BindResourceSet> bindResourceSets) {
        if (CollectionUtils.isEmpty((bindResourceSets))) {
            return ResponseBoolenEntity.fail("0", "入参列表不能为空");
        }
        List<BindResourceSet> resultBindResourceSet = new ArrayList<>(bindResourceSets.size());
        List<BindResourceSet> successResult = new ArrayList<>();
        Set<RoleResourcesetRel> roleResourcesetRels = new HashSet<>();
        for (BindResourceSet bindResourceSet : bindResourceSets) {

            List<Role> rolses = roleDao.findByTenantIdAndRoleCodes(bindResourceSet.getTenantId(), Stream.of(bindResourceSet.getRoleCode()).collect(Collectors.toList()));
            if (CollectionUtils.isEmpty(rolses)) {
                bindResourceSet.setMsg("没有返现角色");
                bindResourceSet.setResult(false);
                resultBindResourceSet.add(bindResourceSet);
                continue;
            }
            if (rolses.size() > 1) {
                bindResourceSet.setMsg("发现多个角色");
                bindResourceSet.setResult(false);
                resultBindResourceSet.add(bindResourceSet);
                continue;
            }
            successResult.add(bindResourceSet);
            List<RoleResourcesetRel> rels = rolses.stream().map(role -> {
                RoleResourcesetRel roleResourcesetRel = new RoleResourcesetRel();
                roleResourcesetRel.setResousesetId(bindResourceSet.getResourceSetId());
                roleResourcesetRel.setRoleId(role.getId());
                roleResourcesetRel.setTenantId(bindResourceSet.getTenantId());
                return roleResourcesetRel;
            }).collect(Collectors.toList());
            roleResourcesetRels.addAll(rels);
        }
        try {
            roleResourcesetRelDao.saveAllAndFlush(roleResourcesetRels);
            successResult.forEach(se -> se.setResult(true));
        } catch (Exception e) {
            logger.warn(e.getMessage(), e);
            for (BindResourceSet se : successResult) {
                se.setResult(false);
                se.setMsg(e.getMessage());
            }
        }
        successResult.addAll(resultBindResourceSet);
        return ResponseBoolenEntity.ok(true, successResult);
    }

    public void bindUsers(long tenantId, long roleId, List<Long> userIds, boolean isOverwrite, Integer relType) {
        Query query = new Query();
        query.setTenantId(tenantId);
        query.setId(roleId);
        Specification<Role> specification = RoleQueryHelper.queryOneSpecification(query);
        Role existsRole = roleDao.findOne(specification).orElseThrow(() -> new IllegalArgumentException("未找到角色实体(" + roleId + ")"));
        List<RoleUserRel> existRels = roleUserRelDao.findByRoleId(existsRole.getId());
        if (userIds == null) {
            throw new IllegalArgumentException("参数缺少用户id集合");
        }
        Set<RoleUserRel> insertingRels = userIds.stream().filter(Objects::nonNull).filter(userId -> existRels.stream().map(RoleUserRel::getUserId).noneMatch(relUserId -> relUserId.equals(userId))).map(userId -> {
            Optional<User> userOptional = userDao.findById(userId);
            if (userOptional.isPresent()) {
                RoleUserRel roleUserRel = new RoleUserRel();
                roleUserRel.setTenantId(userOptional.get().getTenantId());
                roleUserRel.setRoleId(roleId);
                roleUserRel.setUserId(userId);
                roleUserRel.setRelType(relType);
                return roleUserRel;
            } else {
                return null;
            }
        }).filter(Objects::nonNull).collect(Collectors.toSet());
        if (!insertingRels.isEmpty()) {
            roleUserRelDao.saveAllAndFlush(insertingRels);
        }
        if (isOverwrite) {
            existRels.stream().filter(rel -> userIds.stream().noneMatch(userId -> userId.equals(rel.getUserId()))).forEach(rel -> {
                logger.info("deleting Role-User-Rel record, {}", rel);
                try {
                    roleUserRelDao.deleteById(rel.getId());
                } catch (Exception e) {
                    logger.warn(e.getMessage(), e);
                }
            });
        }

    }

    @Transactional(rollbackFor = Exception.class)
    public void bindUsers(long tenantId, long roleId, BindUsers bindUsers, Integer relType) {
        if (bindUsers == null) {
            return;
        }
        this.bindUsers(tenantId, roleId, bindUsers.getUserIds(), bindUsers.isOverwrite(), relType);
    }

    @Transactional(rollbackFor = Exception.class)
    public void sysSave(SysCreate model) {
        // 创建系统角色，检查绑定资源集和租户IDS是否为空
        this.checkResourceSetIdsAndTenantIds(model.getResourcesetIds(), model.getTenantIds());
        model.setResourcesetIds(model.getResourcesetIds().stream().filter(Objects::nonNull).collect(Collectors.toSet()));
        model.setTenantIds(model.getTenantIds().stream().filter(Objects::nonNull).collect(Collectors.toSet()));

        String roleCode = UUID.randomUUID().toString();
        Save save = new Save();
        BeanUtils.copyProperties(model, save);
        save.setRoleCode(roleCode);
        save.setTenantId(0L);

        Role role = this.save(save);
        this.doSysRole(role, model, true);

    }

    /**
     * 创建系统角色，检查绑定资源集是否为空 如空则throw new  IllegalArgumentException 分别是：绑定资源集为空，绑定租户信息为空
     *
     * @param resourceSetIds 功能集Ids
     * @param tenantIds      租户Ids
     */
    private void checkResourceSetIdsAndTenantIds(Set<Long> resourceSetIds, Set<Long> tenantIds) {
        //判断资源代码是否为空
        if (CollectionUtils.isEmpty(resourceSetIds)) {
            throw new IllegalArgumentException("绑定资源集为空");
        }
        if (CollectionUtils.isEmpty(tenantIds)) {
            throw new IllegalArgumentException("绑定租户信息为空");
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void sysUpdate(Long roleId, SysCreate model) {
        // 创建系统角色，检查绑定资源集和租户IDS是否为空
        this.checkResourceSetIdsAndTenantIds(model.getResourcesetIds(), model.getTenantIds());
        model.setResourcesetIds(model.getResourcesetIds().stream().filter(Objects::nonNull).collect(Collectors.toSet()));
        model.setTenantIds(model.getTenantIds().stream().filter(Objects::nonNull).collect(Collectors.toSet()));
        Role role = this.findById(roleId);
        if (null == role) {
            throw new IllegalArgumentException("未找到角色(" + roleId + ")");
        }
        //检查角色是否已经存在，但不于当前ID throw new  "已存在的角色名称(" +name+ ")"
        role.setName(model.getRoleName());
        role.setStatus(model.getStatus());
        role.setRoleDesc(model.getRoleDesc());

        this.doSysRole(role, model, true);
    }


    private void doSysRole(Role role, SysCreate model, boolean isOverride) {

        if (role.getId() == null) {
            logger.error("新建系统角色获取roleId失败,{}", JsonUtils.toJson(role));
            throw new IllegalArgumentException("新建系统角色失败");
        }

        //删除原有角色、角色对应功能集
        if (isOverride) {
            Query query = new Query();
            query.setFromRoleId(role.getId());
            List<Role> delRoles = this.list(query, Sort.unsorted());

            if (null != delRoles) {
                roleDao.deleteAll(delRoles);

                delRoles.add(role);
                delRoles.forEach(r -> {
                    List<RoleResourcesetRel> existRels = roleResourcesetRelDao.findByRoleId(r.getId());
                    existRels.forEach(rel -> roleResourcesetRelDao.deleteById(rel.getId()));
                });
            }
        }

        List<Role> saveRoles = Stream.of(role).collect(Collectors.toList());
        model.getTenantIds().forEach(tenantId -> {
            Optional<Tenant> optionalTenant = tenantDao.findById(tenantId);
            if (optionalTenant.isPresent()) {
                List<Role> existRoles = roleDao.findByTenantIdAndFromRoleId(tenantId, role.getId());
                if (CollectionUtils.isEmpty(existRoles)) {
                    Role r = new Role();
                    r.setCode(role.getCode());
                    r.setName(model.getRoleName());
                    r.setRoleDesc(model.getRoleDesc());
                    r.setStatus(model.getStatus());
                    r.setTenantId(tenantId);
                    r.setFromRoleId(role.getId());

                    Role saveRole = roleDao.saveAndFlush(r);
                    saveRoles.add(saveRole);
                } else {
                    saveRoles.addAll(existRoles);
                }
            }
        });


        List<RoleResourcesetRel> insertingRels = new ArrayList<>();
        saveRoles.forEach(r -> model.getResourcesetIds().forEach(setId -> {
            Optional<Resourceset> optionalResourceset = resourcesetDao.findById(setId);
            if (optionalResourceset.isPresent()) {
                RoleResourcesetRel existRel = roleResourcesetRelDao.findByRoleIdAndResourcesetIdAndTenantId(r.getId(), setId, r.getTenantId());
                if (null == existRel) {
                    RoleResourcesetRel rel = new RoleResourcesetRel();
                    rel.setResousesetId(setId);
                    rel.setRoleId(r.getId());
                    rel.setTenantId(r.getTenantId());
                    insertingRels.add(rel);
                }
            }

        }));

        if (!CollectionUtils.isEmpty(insertingRels)) {
            insertingRels.forEach(roleResourcesetRelDao::saveAndFlush);
        }
    }

    public Role sysInfo(long roleId) {
        Role role = this.findById(roleId);
        if (null == role) {
            throw new IllegalArgumentException("未找到角色(" + roleId + ")");
        }
        Query query = new Query();
        query.setFromRoleId(roleId);
        query.setTenantId(0L);
        List<Role> roles = roleDao.findAll(RoleQueryHelper.querySpecification(query));
        Set<Long> tenantIds = roles.stream().map(Role::getTenantId).collect(Collectors.toSet());
        role.setTenantIds(tenantIds);
        ResourcesetModel.Request.Query resourcesetQuery = new ResourcesetModel.Request.Query();
        resourcesetQuery.setRoleId(roleId);
        resourcesetQuery.setStatus(1);
        List<Resourceset> resourcesets = resourcesetDao.findAll(ResourcesetQueryHelper.querySpecification(resourcesetQuery));
        role.setResourcesets(new HashSet<>(resourcesets));
        Set<Long> resourcesetIds = resourcesets.stream().map(Resourceset::getResourcesetId).collect(Collectors.toSet());
        role.setResourcesetIds(resourcesetIds);
        return role;
    }

    public Long findIdByTenantIdAndCode(long tenantId, String roleCode) {
        return roleDao.findIdByTenantIdAndCode(tenantId, roleCode);
    }

    public ResponseEntity<Long> importSysData(long roleId, MultipartFile file) {
        final String separator = ",";
        Role role = this.findById(roleId);
        if (null == role) {
            throw new IllegalArgumentException("未找到角色(" + roleId + ")");
        }

        List<String> errorMsg = new ArrayList<>();
        String template = "sys_role_rel.json";

        List<RoleResourcesetRelDto> list = ExcelUtils.list(file, template, RoleResourcesetRelDto.class);
        Set<Long> tenantIds = new HashSet<>();
        Set<Long> resourcesetIds = new HashSet<>();
        for (int i = 0; i < list.size(); i++) {
            RoleResourcesetRelDto r = list.get(i);
            Long tenantId;

            String msg;

            if (StringUtils.isBlank(r.getTenantCode()) && StringUtils.isNotBlank(r.getResourcesetCodes())) {
                msg = "导入第 " + (i + 1) + " 行失败，租户编号不能为空！";
                errorMsg.add(msg);
            }
            if (StringUtils.isNotBlank(r.getTenantCode()) && StringUtils.isBlank(r.getResourcesetCodes())) {
                msg = "导入第 " + (i + 1) + " 行失败，功能集编号不能为空！";
                errorMsg.add(msg);
            }

            tenantId = tenantDao.findTenantIdByTenantCode(r.getTenantCode());
            if (tenantId == null) {
                msg = ("导入第 " + (i + 1) + " 行失败，租户编号【" + r.getTenantCode() + "】不存在！");
                errorMsg.add(msg);
                continue;
            }
            tenantIds.add(tenantId);
            if (StringUtils.isNotBlank(r.getResourcesetCodes())) {
                for (String setCode : r.getResourcesetCodes().split(separator)) {
                    Long resourcesetId = resourcesetDao.findByResourcesetCode(setCode);
                    if (null == resourcesetId) {
                        msg = "导入第 " + (i + 1) + " 行失败，功能集编号【" + setCode + "】不存在！";
                        errorMsg.add(msg);
                    } else {
                        resourcesetIds.add(resourcesetId);
                    }
                }
            }
        }

        SysCreate sysCreate = new SysCreate();
        sysCreate.setResourcesetIds(resourcesetIds);
        sysCreate.setTenantIds(tenantIds);
        sysCreate.setRoleDesc(role.getRoleDesc());
        sysCreate.setRoleName(role.getName());
        sysCreate.setStatus(role.getStatus());

        this.doSysRole(role, sysCreate, false);

        //上传错误信息到oss
        Long fileId = fileService.uploadMsgExcel(errorMsg);

        ResponseEntity<Long> restResponse = new ResponseEntity<>();
        restResponse.setCode(null == fileId ? "1" : "0");
        restResponse.setResult(fileId);
        restResponse.setMessage("导入完成，合计" + errorMsg.size() + "个错误," + errorMsg);
        return restResponse;
    }

    @Transactional(rollbackFor = Exception.class)
    public void unbindUsers(long tenantId, long roleId, UnbindUsers unbindUsers) {
        Set<Long> userIds = unbindUsers.getUserIds();
        if (!CollectionUtils.isEmpty(userIds)) {
            roleUserRelDao.deleteByTenantIdAndRoleIdAndUserIds(tenantId, roleId, userIds);
            userIds.stream().filter(Objects::nonNull).forEach(userId -> roleUserRelDao.deleteByTenantIdAndUserIdAndRoleId(tenantId, userId, roleId));
        }
    }

    public Set<Role> copyRoleTempates(long tenantId, Collection<Role> roleTemplates) {
        Set<Role> copiedRoles = new HashSet<>(roleTemplates.size());
        Set<RoleResourcesetRel> allCopiedRoleResourcesetRels = new HashSet<>();
        if (!CollectionUtils.isEmpty(roleTemplates)) {
            for (Role roleTemplate : roleTemplates) {
                Role copiedRole = new Role();
                BeanUtils.copyProperties(roleTemplate, copiedRole, Stream.of("id", "roleResourcesetRels", "roleUserRels").toArray(String[]::new));
                copiedRole.setTenantId(tenantId);
                copiedRole = roleDao.saveAndFlush(copiedRole);
                List<RoleResourcesetRel> roleResourcesetRels = roleTemplate.getRoleResourcesetRels();
                if (!CollectionUtils.isEmpty(roleResourcesetRels)) {
                    for (RoleResourcesetRel roleResourcesetRel : roleResourcesetRels) {
                        RoleResourcesetRel copiedRoleResourcesetRel = new RoleResourcesetRel();
                        BeanUtils.copyProperties(roleResourcesetRel, copiedRoleResourcesetRel, Stream.of("id").toArray(String[]::new));
                        copiedRoleResourcesetRel.setTenantId(tenantId);
                        copiedRoleResourcesetRel.setRoleId(copiedRole.getId());
                        allCopiedRoleResourcesetRels.add(copiedRoleResourcesetRel);
                    }
                }
                copiedRoles.add(copiedRole);
            }
        }
        if (!CollectionUtils.isEmpty(allCopiedRoleResourcesetRels)) {
            roleResourcesetRelDao.saveAllAndFlush(allCopiedRoleResourcesetRels);
        }
        return copiedRoles;
    }

    public List<RoleExportDto> findRolesResourcesetRel(Long tenantId, Export query) {
        return roleDao.findRolesResourcesetRel(tenantId, query, null);
    }

    public List<RoleRelAccountExportDto> findRolesRelAccount(Long tenantId, Export query) {
        return roleDao.findRolesRelAccount(tenantId, query, null);
    }

    /**
     * 查询租户角色.
     *
     * @param tenantId
     * @param roleId
     * @param userId
     * @return
     */
    public List<RoleUserRel> listRoleUserRel(Long tenantId, Long roleId, Long userId) {
        Objects.requireNonNull(tenantId, "tenantId is not allow null");
        Specification<RoleUserRel> specification = RoleQueryHelper.queryRoleUserRelSpecification(tenantId, roleId, userId);
        List<RoleUserRel> list = roleUserRelDao.findAll(specification);
        return list;
    }

    @Transactional(rollbackFor = Exception.class)
    public void bindResources(Long tenantId, Role role, ResourcesetModel.Request.BindResources bindResources) {

        ResourcesetModel.Request.Query query = new ResourcesetModel.Request.Query();
        query.setRoleId(role.getId());
        Optional<Resourceset> optionalResourceset = resourcesetService.findOne(query);
        Resourceset resourceset = null;
        List<Long> resourcesetIds = null;
        if (optionalResourceset.isPresent()) {
            resourceset = optionalResourceset.get();
            //没有选择资源码，已有资源码和自定义功能集关系解绑，删除自定义功能集
            resourcesetService.bindResources(resourceset, bindResources);
            resourcesetIds = Lists.newArrayList(resourceset.getResourcesetId());

        } else {
            ResourcesetModel.Request.Create create = new ResourcesetModel.Request.Create();
            create.setResourcesetName(role.getName() + "_自定义功能集");
            create.setResourcesetCode(UUID.randomUUID().toString());
            create.setResourcesetDesc("自定义功能集");
            create.setStatus(1);
            create.setAppId(1L);
            create.setValidatedAppId(false);
            create.setBindResources(bindResources);
            create.setRoleId(role.getId());
            resourceset = resourcesetService.create(create);
            resourcesetIds = Lists.newArrayList(resourceset.getResourcesetId());
        }
        BindResourceSets bindResourceSets = new BindResourceSets();
        bindResourceSets.setOverwrite(false);
        bindResourceSets.setResourcesetIds(resourcesetIds);
        this.bindResourceSets(role, bindResourceSets);
    }

    public List<OrgRoleCntDTO> getOrgRoleCntByGradingRoleIds(Long tenantId, Set<Long> roleIds) {
        return roleDao.getOrgRoleCntByGradingRoleIds(tenantId, roleIds);
    }

    public void disableOrgRolesByGradingRoleId(Long roleId) {
        roleDao.disableOrgRolesByGradingRoleId(roleId);
    }

    public List<Role> findByTenantIdAndFromRoleId(long tenantId, Long roleId) {
        return roleDao.findByTenantIdAndFromRoleId(tenantId, roleId);
    }

    public void deleteByTenantIdAndIds(long tenantId, List<Long> ids) {
        roleDao.deleteByTenantIdAndIds(tenantId, ids);
    }
}
