/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: TenentExcelExportService.java   2020-10-27 09-28-20
 * Author: Evan
 */
package com.xforceplus.business.tenant.service;

import com.xforceplus.api.model.TenantModel.Request.Query;
import com.xforceplus.business.excel.BusinessType;
import com.xforceplus.business.excel.SimpleExcelWriter;
import com.xforceplus.business.excel.reader.Context;
import com.xforceplus.business.service.ExcelWriteService;
import com.xforceplus.business.tenant.dto.TenantExportDTO;
import com.xforceplus.entity.Tenant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Service;

import java.util.Arrays;
import java.util.List;
import java.util.Objects;

import static com.xforceplus.business.excel.ExcelFile.*;
import static com.xforceplus.business.excel.writer.ExcelConfigBusinessType.TENANT_EXCEL_FILE;
import static com.xforceplus.business.tenant.service.TenantExcel.SORT_FIELD;
import static org.springframework.data.domain.Sort.Direction.DESC;

/**
 * <p>
 * Title: 租户Excel处理
 * </p>
 * <p>
 * Description: 租户Excel处理
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-10-27 09-28-20
 * @since V1.0
 */
@Service
public class TenantExcelExportServiceImpl implements ExcelWriteService {

    /**
     * 日志
     */
    private static final Logger logger = LoggerFactory.getLogger(TenantExcelExportServiceImpl.class);

    @Autowired
    private TenantService tenantService;

    /**
     * 获取导入类型，用于Event事件调整导入方法
     * @return ImportBusinessType
     */
    @Override
    public BusinessType getBusinessType() {
        return TENANT_EXCEL_FILE;
    }

    /**
     * 导出公共方法
     *
     * @param context 上下文
     * @return Context
     */
    @Override
    public void write(Context context) {
        Query query = context.getParam(PARAMS_QUERY, Query.class);
        SimpleExcelWriter simpleExcelWriter = context.getSimpleExcelWriter();
        List<Tenant> tenantList;
        Integer maxPageSize = context.getMaxPageSize();
        logger.info("maxPageSize:{}", maxPageSize);
        //fix bug: check empty array.
        if (Objects.nonNull(query.getTenantIds()) && query.getTenantIds().length > 0) {
            logger.info("tenant export:{}", query.getTenantIds().length);
            tenantList = tenantService.findAllById(Arrays.asList(query.getTenantIds()));
            List<TenantExportDTO> exportVOList = tenantService.convert2ExportDto(tenantList);
            simpleExcelWriter.fill(query.getSheets()[0], exportVOList);
        } else {
            Page<Tenant> page;
            Pageable pageable = PageRequest.of(START_PAGE, PAGE_SIZE, Sort.by(DESC, SORT_FIELD));
            do {

                page = this.tenantService.page(query, pageable);
                List<TenantExportDTO> exportVOList = tenantService.convert2ExportDto(page.toList());
                simpleExcelWriter.fill(query.getSheets()[0], exportVOList);
                pageable = pageable.next();
            } while (page.hasNext() && page.getNumber() < maxPageSize);
            //关闭写入数据
            logger.info("tenant page export, total page:{}", page);
        }
    }
}
