package com.xforceplus.business.tenant.service;

import com.xforceplus.api.model.TenantExtensionModel;
import com.xforceplus.api.model.TenantExtensionModel.Request.Query;
import com.xforceplus.api.model.TenantModel;
import com.xforceplus.constants.ExtensionStatus;
import com.xforceplus.dao.TenantExtensionDao;
import com.xforceplus.domain.tenant.TenantExtensionDto;
import com.xforceplus.entity.TenantExtension;
import io.geewit.core.utils.reflection.BeanUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.validation.annotation.Validated;

import javax.annotation.Resource;
import javax.persistence.criteria.Predicate;
import java.util.*;
import java.util.stream.Collectors;

@Validated
@Service
@Slf4j
public class TenantExtensionService {
    private final static Logger logger = LoggerFactory.getLogger(TenantExtensionService.class);

    @Resource
    private TenantExtensionDao tenantExtensionDao;


    private Specification<TenantExtension> querySpecification(Query query, Integer status) {
        return (Specification<TenantExtension>) (root, criteriaQuery, builder) -> {
            List<Predicate> predicates = new ArrayList<>();
            if (query.getTenantId() != null && query.getTenantId() > 0) {
                predicates.add(builder.equal(root.get("tenantId"), query.getTenantId()));
            }
            if (StringUtils.isNotBlank(query.getExtensionKey())) {
                predicates.add(builder.equal(root.get("extensionKey"), query.getExtensionKey()));
            }
            if (StringUtils.isNotBlank(query.getExtensionValue())) {
                predicates.add(builder.equal(root.get("extensionValue"), query.getExtensionValue()));
            }
            if (status != null && status > 0) {
                predicates.add(builder.equal(root.get("status"), status));
            }
            if (!predicates.isEmpty()) {
                criteriaQuery.where(predicates.stream().toArray(Predicate[]::new));
            }
            return criteriaQuery.getRestriction();
        };
    }

    public Page<TenantExtension> page(Query query, Pageable pageable) {
        Specification<TenantExtension> specification = querySpecification(query, ExtensionStatus.ENABLE);
        return tenantExtensionDao.findAll(specification, pageable);
    }

    public List<TenantExtension> list(Query query, Sort sort) {
        Specification<TenantExtension> specification = querySpecification(query, ExtensionStatus.ENABLE);
        return tenantExtensionDao.findAll(specification, sort);
    }

    @Transactional(rollbackFor = Exception.class)
    public TenantExtension save(TenantExtensionDto tenantExtensionDto) {
        List<TenantExtension> list = getList(tenantExtensionDto);
        if (CollectionUtils.isEmpty(list)) {
            TenantExtension entity = new TenantExtension();
            BeanUtils.copyProperties(tenantExtensionDto, entity);
            entity.setStatus(ExtensionStatus.ENABLE);
            entity.setCreateTime(new Date());
            entity.setUpdateTime(new Date());
            return tenantExtensionDao.saveAndFlush(entity);
        } else {
            TenantExtension entity = list.get(0);
            entity.setExtensionValue(tenantExtensionDto.getExtensionValue());
            entity.setStatus(ExtensionStatus.ENABLE);
            entity.setUpdateTime(new Date());
            return tenantExtensionDao.saveAndFlush(entity);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public Set<TenantExtension> batchSave(long tenantId, List<TenantModel.Request.Extension> extensions, boolean isOverwrite) {
        if (tenantId <= 0 || CollectionUtils.isEmpty(extensions)) {
            return null;
        }
        List<TenantExtension> existExtensions = this.findByTenantId(tenantId);
        Set<TenantExtension> savingExtensions = extensions.stream().filter(Objects::nonNull).map(extension -> {
            List<TenantExtension> exists = tenantExtensionDao.findByTenantIdAndKey(tenantId, extension.getExtensionKey());
            TenantExtension savingExtension;
            if (CollectionUtils.isEmpty(exists)) {
                savingExtension = new TenantExtension();
                savingExtension.setTenantId(tenantId);
                savingExtension.setExtensionKey(extension.getExtensionKey());
            } else {
                savingExtension = exists.get(0);
            }
            savingExtension.setExtensionValue(extension.getExtensionValue());
            return savingExtension;
        }).collect(Collectors.toSet());
        if (!savingExtensions.isEmpty()) {
            tenantExtensionDao.saveAllAndFlush(savingExtensions);
        }
        Set<TenantExtension> result = new HashSet<>(savingExtensions);
        result.addAll(existExtensions);
        if (isOverwrite) {
            existExtensions.stream().filter(existExtension -> extensions.stream().noneMatch(extension -> extension.getExtensionKey().equals(existExtension.getExtensionKey()))).forEach(extension -> {
                logger.info("deleting CompanyExtension record, {}", extension);
                try {
                    tenantExtensionDao.deleteById(extension.getTenantExtensionId());
                } catch (Exception e) {
                    logger.warn(e.getMessage(), e);
                }
            });
        } else {
            result.addAll(existExtensions);
        }
        return result;
    }

    public List<TenantExtension> findByTenantId(long tenantId) {
        TenantExtensionModel.Request.Query query = new TenantExtensionModel.Request.Query();
        query.setTenantId(tenantId);
        Specification<TenantExtension> specification = this.querySpecification(query, null);
        return tenantExtensionDao.findAll(specification, Sort.unsorted());
    }

    private List<TenantExtension> getList(TenantExtensionDto tenantExtensionWithIdDto) {
        Query query = new Query();
        query.setTenantId(tenantExtensionWithIdDto.getTenantId());
        query.setExtensionKey(tenantExtensionWithIdDto.getExtensionKey());
        Specification<TenantExtension> specification = querySpecification(query, null);
        return tenantExtensionDao.findAll(specification, Sort.unsorted());
    }


    public List<TenantExtension> getListByTenantId(Long tenantId) {
        TenantExtensionModel.Request.Query query = new TenantExtensionModel.Request.Query();
        query.setTenantId(tenantId);
        Specification<TenantExtension> specification = querySpecification(query, null);
        return tenantExtensionDao.findAll(specification, Sort.unsorted());
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteByTenantId(long tenantId) {
        tenantExtensionDao.deleteByTenantId(tenantId);
    }
}
