package com.xforceplus.business.tenant.service;

import com.xforceplus.api.model.TenantModel;
import com.xforceplus.api.model.TenantPolicyModel;
import com.xforceplus.dao.TenantDao;
import com.xforceplus.dao.TenantPolicyDao;
import com.xforceplus.entity.Tenant;
import com.xforceplus.entity.TenantPolicy;
import com.xforceplus.query.TenantPolicyQueryHelper;

import java.security.InvalidParameterException;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.xforceplus.security.strategy.event.RefreshingStrategyCacheEvent;
import io.geewit.data.jpa.essential.domain.EntityGraphs;
import io.geewit.utils.uuid.UUID;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.ApplicationEventPublisherAware;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

@Service
public class TenantPolicyService implements ApplicationEventPublisherAware {
    private final static Logger logger = LoggerFactory.getLogger(TenantPolicyService.class);

    public static final String TENANT_GRADING_MANAGEMENT_ENABLED = "tenantGradingMgrEnabled";

    public TenantPolicyService(TenantPolicyDao tenantPolicyDao, TenantDao tenantDao) {
        this.tenantPolicyDao = tenantPolicyDao;
        this.tenantDao = tenantDao;
    }

    private final TenantPolicyDao tenantPolicyDao;
    private final TenantDao tenantDao;
    private ApplicationEventPublisher publisher;

    @Override
    public void setApplicationEventPublisher(ApplicationEventPublisher publisher) {
        this.publisher = publisher;
    }


    /**
     * 分页查询.
     *
     * @param query
     * @param page
     * @return
     */
    public Page<TenantPolicy> page(TenantPolicyModel.Request.Query query, Pageable page) {
        Specification<TenantPolicy> specification = TenantPolicyQueryHelper.querySpecification(query);
        Page<TenantPolicy> policies = tenantPolicyDao.findAll(specification, page, EntityGraphs.named(TenantPolicy.NAMED_ENTITY_GRAPH_DEFAULT));
        return policies;
    }

    public Map<String, TenantPolicy> policies(long tenantId) {
        List<TenantPolicy> policies = tenantPolicyDao.findByTenantId(tenantId);
        if (policies.isEmpty()) {
            return Stream.of(TenantPolicy.ofDefault(TenantPolicy.CREDENTIAL_LENGTH_NAME, "8")).collect(Collectors.toMap(TenantPolicy::getName, policy -> policy));
        } else {
            return policies.stream().collect(Collectors.toMap(TenantPolicy::getName, policy -> policy));
        }
    }

    /**
     * 获取租户分级管理开关
     * @param tenantId
     * @return
     */
    public boolean tenantGradingManagementEnabled(long tenantId) {
        final String defaultPolicy = "1";
        Map<String, TenantPolicy> map = this.policies(tenantId);
        if(null != map && null != map.get(TENANT_GRADING_MANAGEMENT_ENABLED)) {
            String policy = map.get(TENANT_GRADING_MANAGEMENT_ENABLED).getPolicy();
            if (defaultPolicy.equals(policy)) {
                return true;
            }
        }

        return false;
    }

    @Transactional(rollbackFor = Exception.class)
    public void policies(long tenantId, TenantModel.Request.Policies policies) {
        if (policies == null || policies.getPolicies() == null) {
            logger.info("policies == null, return null");
            return;
        }
        logger.info("isOverwrite = {}", policies.isOverwrite());
        logger.info("policies = " + policies.getPolicies().stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.joining(",")));
        List<TenantPolicy> existPolicies = tenantPolicyDao.findByTenantId(tenantId);
        logger.info("exist tenant.policies.size = " + existPolicies.size());
        //region 往绑定不在数据库中但是报文 policies 中有的策略
        Set<TenantPolicy> savingPolicies = policies.getPolicies().stream().filter(Objects::nonNull).map(policy -> {
            TenantPolicy tenantPolicy = existPolicies.stream().filter(p -> p.getName().equals(policy.getName())).findAny().orElseGet(TenantPolicy::new);
            tenantPolicy.setTenantId(tenantId);
            tenantPolicy.setName(policy.getName());
            tenantPolicy.setPolicy(policy.getPolicy());
            return tenantPolicy;

        }).collect(Collectors.toSet());
        tenantPolicyDao.saveAllAndFlush(savingPolicies);
        //endregion
        if (policies.isOverwrite()) {
            //region 从数据库中解绑不在报文 roleIds 中的用户-角色关系
            existPolicies.stream().filter(Objects::nonNull).filter(existPolicy -> policies.getPolicies().stream().filter(Objects::nonNull).noneMatch(policy -> policy.getName().equals(existPolicy.getName()))).forEach(existPolicy -> {
                try {
                    tenantPolicyDao.deleteById(existPolicy.getId());
                } catch (Exception e) {
                    logger.warn(e.getMessage(), e);
                }
            });
            //endregion
        }
    }

    /**
     * save or update.
     *
     * @param model
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public TenantPolicy save(TenantPolicyModel.Request.Save model) {
        Tenant tenant = tenantDao.findById(model.getTenantId()).orElseThrow(() -> new InvalidParameterException("租户(id: " + model.getTenantId() + ")不存在！"));
        List<TenantPolicy> policies = tenantPolicyDao.findByTenantIdAndName(model.getTenantId(), model.getName());
        if (!CollectionUtils.isEmpty(policies)) {
            throw new InvalidParameterException("租户(id: " + model.getTenantId() + ")策略(name:" + model.getName() + ")已存在！");
        }
        TenantPolicy policy = new TenantPolicy();
        policy.setName(model.getName());
        policy.setPolicy(model.getPolicy());
        policy.setTenantId(model.getTenantId());
        String refreshId = UUID.randomUUID().toString();
        logger.info("onMessage with redis expire: " + refreshId);
        this.publisher.publishEvent(new RefreshingStrategyCacheEvent(refreshId));
        return tenantPolicyDao.saveAndFlush(policy);
    }

    /**
     * delete policy by id.
     *
     * @param id
     */
    @Transactional(rollbackFor = Exception.class)
    public void deleteById(Long id) {
        tenantPolicyDao.deleteById(id);
        String refreshId = UUID.randomUUID().toString();
        logger.info("onMessage with redis expire: " + refreshId);
        this.publisher.publishEvent(new RefreshingStrategyCacheEvent(refreshId));
    }

    /**
     * find policy by id.
     *
     * @param id
     * @return
     */
    public TenantPolicy findById(long id) {
        return tenantPolicyDao.findById(id, EntityGraphs.named(TenantPolicy.NAMED_ENTITY_GRAPH_DEFAULT)).orElse(null);
    }

    /**
     * udpate policy.
     *
     * @param update
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public TenantPolicy update(TenantPolicyModel.Request.Update update) {
        String refreshId = UUID.randomUUID().toString();
        logger.info("onMessage with redis expire: " + refreshId);
        this.publisher.publishEvent(new RefreshingStrategyCacheEvent(refreshId));
        TenantPolicy policy = tenantPolicyDao.findById(update.getId()).orElseThrow(() -> new InvalidParameterException("策略ID不存在！"));
        policy.setId(update.getId());
        policy.setName(update.getName());
        policy.setPolicy(update.getPolicy());
        return tenantPolicyDao.saveAndFlush(policy);
    }
}
