package com.xforceplus.business.tenant.service;

import com.xforceplus.api.model.*;
import com.xforceplus.api.model.TenantModel.Request.Create;
import com.xforceplus.api.model.TenantModel.Request.Query;
import com.xforceplus.api.model.TenantModel.Request.Save;
import com.xforceplus.api.model.TenantModel.Response.CreateResult;
import com.xforceplus.api.model.UserModel.Request.SaveUserContext;
import com.xforceplus.business.company.service.CompanyService;
import com.xforceplus.business.tenant.dto.TenantExportDTO;
import com.xforceplus.dao.*;
import com.xforceplus.domain.account.AccountDto;
import com.xforceplus.domain.company.CompanyDto;
import com.xforceplus.domain.org.OrgDto;
import com.xforceplus.domain.tenant.RoleDto;
import com.xforceplus.domain.tenant.TenantDto;
import com.xforceplus.domain.tenant.TenantExtensionDto;
import com.xforceplus.domain.tenant.TenantManagerDto;
import com.xforceplus.domain.user.UserDto;
import com.xforceplus.entity.*;
import com.xforceplus.query.TenantQueryHelper;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.tenant.security.core.domain.OrgType;
import io.geewit.core.utils.reflection.BeanUtils;
import io.geewit.data.jpa.envers.domain.ComparedRevision;
import io.geewit.utils.uuid.UUID;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.history.RevisionSort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;


@SuppressWarnings("all")
@Service
public class TenantService {
    private final static Logger logger = LoggerFactory.getLogger(TenantService.class);
    private final static int MAX_EXTNESION_COUNT = 10;
    private final static int MAX_EXTNESION_LENGTH = 64;
    private final TenantDao tenantDao;
    private final TenantSecretDao tenantSecretDao;
    private final UserService userService;
    private final OrgService orgService;
    private final WrapperOrgService wrapperOrgService;
    private final CompanyService companyService;
    private final CompanyServiceRelDao companyServiceRelDao;
    private final ServicePackageDao servicePackageDao;
    private final RoleDao roleDao;
    private final RoleService roleService;
    private final ServiceResourcesetRelDao serviceResourcesetRelDao;
    private final CompanyDao companyDao;
    private final OrgStructDao orgStructDao;
    private final TenantExtensionService tenantExtensionService;
    private final AccountDao accountDao;

    public TenantService(TenantDao tenantDao, TenantSecretDao tenantSecretDao, UserService userService,
                         OrgService orgService, WrapperOrgService wrapperOrgService, CompanyService companyService,
                         CompanyServiceRelDao companyServiceRelDao, ServicePackageDao servicePackageDao,
                         RoleDao roleDao, RoleService roleService, ServiceResourcesetRelDao serviceResourcesetRelDao,
                         CompanyDao companyDao, OrgStructDao orgStructDao,
                         TenantExtensionService tenantExtensionService, AccountDao accountDao) {
        this.tenantDao = tenantDao;
        this.tenantSecretDao = tenantSecretDao;
        this.userService = userService;
        this.orgService = orgService;
        this.wrapperOrgService = wrapperOrgService;
        this.companyService = companyService;
        this.companyServiceRelDao = companyServiceRelDao;
        this.servicePackageDao = servicePackageDao;
        this.roleDao = roleDao;
        this.roleService = roleService;
        this.serviceResourcesetRelDao = serviceResourcesetRelDao;
        this.companyDao = companyDao;
        this.orgStructDao = orgStructDao;
        this.tenantExtensionService = tenantExtensionService;
        this.accountDao = accountDao;
    }

    public Page<Tenant> page(Query query, Pageable pageable) {
        Specification<Tenant> specification = TenantQueryHelper.querySpecification(query);
        Page<Tenant> page = tenantDao.findAll(specification, pageable);
        if (StringUtils.isNotBlank(query.getWithExtendParams())) {
            String[] withExtendParams = StringUtils.split(query.getWithExtendParams(), ",");
            boolean enableCompanyCount = false;
            boolean enableExtensions = false;
            for (String withExtendParam : withExtendParams) {
                if ("companyCount".equalsIgnoreCase(withExtendParam)) {
                    enableCompanyCount = true;
                } else if ("extensions".equalsIgnoreCase(withExtendParam)) {
                    enableExtensions = true;
                }
            }
            //获取租户管理员
            List<TenantManagerDto> managerDtos = userService.findTenantManager(page.stream().map(tenant -> tenant.getTenantId()).collect(Collectors.toList()));
            Map<Long, List<TenantManagerDto>> managerMap = managerDtos.stream().collect(Collectors.groupingBy(TenantManagerDto::getTenantId));

            if (enableCompanyCount || enableExtensions) {
                for (Tenant tenant : page) {
                    if (!CollectionUtils.isEmpty(managerMap)) {
                        List<TenantManagerDto> managerDtoList = managerMap.get(tenant.getTenantId());
                        if (!CollectionUtils.isEmpty(managerDtoList)) {
                            tenant.setTenantManager(managerDtoList.stream().findFirst().orElse(null));
                        }
                    }
                    if (enableCompanyCount) {
                        int companyCount = orgStructDao.countCompaniesByTenantId(tenant.getTenantId());
                        tenant.setCompanyCount(companyCount);
                    }
                    if (enableExtensions) {
                        List<TenantExtension> extensions = tenantExtensionService.getListByTenantId(tenant.getTenantId());
                        if (!CollectionUtils.isEmpty(extensions)) {
                            List<TenantExtensionDto> extensionDtos = extensions.stream().filter(Objects::nonNull).collect(Collectors.toList());
                            tenant.setExtensions(extensionDtos);
                        }
                    }
                }
            }
        }

        return page;
    }

    /**
     * tenant 转换成 export dto.
     *
     * @param tenantList
     * @return
     * @author feihu.wang
     */
    public List<TenantExportDTO> convert2ExportDto(List<Tenant> tenantList) {
        List<TenantExportDTO> exportVOList = tenantList.stream().map(t -> {
            int companyCount = orgStructDao.countCompaniesByTenantId(t.getTenantId());
            TenantExportDTO dto = new TenantExportDTO();
            BeanUtils.copyProperties(t, dto);
            dto.setCompanyCount(companyCount * 1L);
            dto.setTenantId(t.getTenantId().toString());
            return dto;
        }).map(t -> {
            UserModel.Request.Query userQuery = new UserModel.Request.Query();
            userQuery.setTenantId(Long.parseLong(t.getTenantId()));
            List<RoleUserRel> userRelList = roleService.listRoleUserRel(Long.parseLong(t.getTenantId()), 1L, null);
            if (userRelList != null && userRelList.size() > 0) {
                User user = userService.findById(userRelList.get(userRelList.size() - 1).getUserId());
                try {
                    Account account = user.getAccount();
                    if (account != null && account.getCreateTime() != null) {
                        t.setAdmin(account.getEmail());
                        if (StringUtils.isEmpty(t.getAdmin())) {
                            t.setAdmin(account.getTelPhone());
                        }
                    }
                } catch (Exception e) {
                    logger.error("notfound account", e);
                    t.setAdmin("");
                }
            }
            return t;
        }).map(t -> {
            if (StringUtils.isEmpty(t.getAdmin())) {
                t.setAdmin("");
            }
            if (t.getCompanyCount() == null) {
                t.setCompanyCount(0L);
            }
            return t;
        }).collect(Collectors.toList());
        return exportVOList;
    }

    public Page<Tenant> page(Specification<Tenant> specification, Pageable pageable) {
        return tenantDao.findAll(specification, pageable);
    }

    public List<Tenant> list(Query query, Sort sort) {
        Specification<Tenant> specification = TenantQueryHelper.querySpecification(query);
        List<Tenant> list = tenantDao.findAll(specification, sort);
        return list;
    }

    public List<Tenant> list(Specification<Tenant> specification, Sort sort) {
        return tenantDao.findAll(specification, sort);
    }

    public Optional<Tenant> findOne(Query query) {
        Specification<Tenant> specification = TenantQueryHelper.queryOneSpecification(query);
        return tenantDao.findOne(specification);
    }

    @Transactional(rollbackFor = Exception.class)
    public <T extends TenantDto, O extends OrgDto<O>, C extends CompanyDto<O>, U extends UserDto<O, R, A>, R extends RoleDto, A extends AccountDto> CreateResult<T, O, C, U, R, A> create(Create model) {
        this.checkExtensions(model.getExtensions());
        CreateResult<T, O, C, U, R, A> createResult = new CreateResult<>();
        model.setOverwrite(false);
        //region 创建租户核心方法
        this.save(model, createResult);
        //endregion
        if (createResult.getTenant() != null) {
            Set<TenantExtension> extensions = tenantExtensionService.batchSave(createResult.getTenant().getTenantId(), model.getExtensions(), model.isOverwrite());
            List<TenantExtensionModel.Response> responses = this.getExtensions(extensions);
            createResult.setExtensions(responses);
        }
        return createResult;
    }

    private void checkExtensions(List<TenantModel.Request.Extension> extensions) {
        if (CollectionUtils.isEmpty(extensions)) {
            return;
        }
        if (extensions.size() > MAX_EXTNESION_COUNT) {
            throw new IllegalArgumentException("扩展标签个数不能超过【" + MAX_EXTNESION_COUNT + "】");
        }
        boolean checkResult = extensions.stream().anyMatch(p -> StringUtils.isBlank(p.getExtensionKey()) || StringUtils.isBlank(p.getExtensionValue()));
        if (checkResult) {
            throw new IllegalArgumentException("扩展标签字段不能为空");
        }
        checkResult = extensions.stream().anyMatch(p -> p.getExtensionKey().length() > MAX_EXTNESION_LENGTH || p.getExtensionValue().length() > MAX_EXTNESION_LENGTH);
        if (checkResult) {
            throw new IllegalArgumentException("扩展标签最大长度不能超过【" + MAX_EXTNESION_LENGTH + "】");
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public Tenant update(long tenantId, Save model) {

        this.checkExtensions(model.getExtensions());
        model.setTenantId(tenantId);

        Tenant result = this.save(model, null);
        Set<TenantExtension> extensions = tenantExtensionService.batchSave(tenantId, model.getExtensions(), true);
        if (!CollectionUtils.isEmpty(extensions)) {
            result.setExtensions(new ArrayList<>(extensions));
        }
        return result;
    }

    private List<TenantExtensionModel.Response> getExtensions(Collection<TenantExtension> extensions) {
        if (CollectionUtils.isEmpty(extensions)) {
            return null;
        }
        List<TenantExtensionModel.Response> responses = extensions.stream().map(p -> {
            TenantExtensionModel.Response response = new TenantExtensionModel.Response();
            response.setExtensionKey(p.getExtensionKey());
            response.setExtensionValue(p.getExtensionValue());
            response.setTenantId(p.getTenantId());
            return response;
        }).collect(Collectors.toList());
        return responses;
    }

    /**
     * 创建/保存租户
     *
     * @param model
     * @param createResult
     * @param <S>          TenantModel.Request.Save
     * @param <T>          TenantDto
     * @param <O>          OrgDto
     * @param <C>          CompanyDto
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public <S extends Save, T extends TenantDto, O extends OrgDto<O>, C extends CompanyDto<O>, U extends UserDto<O, R, A>, R extends RoleDto, A extends AccountDto> Tenant save(S model, CreateResult<T, O, C, U, R, A> createResult) {
        Tenant tenantEntity;
        final boolean isNew = model instanceof Create;
        if (model.getTenantId() != null && model.getTenantId() > 0) {
            Optional<Tenant> tenantOptional = tenantDao.findById(model.getTenantId());
            if (tenantOptional.isPresent()) {
                tenantEntity = tenantOptional.get();
            } else {
                tenantEntity = new Tenant();
                model.setTenantId(null);
            }
            //更新租户， 租户名称或者code，不能是其他已经存在的
        } else {
            tenantEntity = new Tenant();
            model.setTenantId(null);
        }
        if (StringUtils.isBlank(model.getTenantName())) {
            model.setTenantName(null);
        }
        if (StringUtils.isBlank(model.getTenantCode())) {
            model.setTenantCode(null);
        }
        BeanUtils.copyProperties(model, tenantEntity, Stream.of("extensions").toArray(String[]::new));

        tenantEntity = tenantDao.saveAndFlush(tenantEntity);

        if (isNew) {
            Create tenantCreate = (Create) model;
            //region 保存 User
            UserModel.Request.Create userSave = tenantCreate.getUser();
            if (userSave != null) {  // 创建管理员账号
                userSave.setIsOverwrite(false);
                userSave.setIsMergeAccount(true);
                userSave.setIsStrict(false);
                userSave.addRoleId(1L); // 添加超级管理员角色
                OrgModel.Request.Save rootOrgSave = new OrgModel.Request.Save();
                rootOrgSave.setOrgName(tenantEntity.getTenantName());
                rootOrgSave.setOrgCode(tenantEntity.getTenantCode());
                rootOrgSave.setOrgType(OrgType.GROUP);
                rootOrgSave.setStatus(1);
                OrgService.OrgSaveOutput rootOrgSaveOutput = orgService.update(tenantEntity.getTenantId(), rootOrgSave, false);
                Set<O> orgs = new HashSet<>();
                OrgStruct rootOrg = rootOrgSaveOutput.findFirst();
                orgs.add((O) rootOrg);
                //region 创建根组织
                //region 保存 Company
                OrgService.OrgSaveOutput orgSaveOutput = null;
                OrgStruct orgStruct;
                CompanyModel.Request.Save companySave = tenantCreate.getCompany();
                if (companySave != null) {
                    companySave.setHostTenantId(tenantEntity.getTenantId());
                    OrgModel.Request.Save orgSave = new OrgModel.Request.Save();
                    orgSave.setCompany(companySave);
                    orgSave.setOrgId(companySave.getOrgId());
                    orgSave.setOrgName(companySave.getCompanyName());
                    orgSave.setOrgCode(companySave.getCompanyCode());
                    orgSave.setOrgType(OrgType.COMPANY);
                    orgSave.setParentId(rootOrg.getOrgId());
                    orgSave.setStatus(1);
                    orgSaveOutput = wrapperOrgService.save(tenantEntity.getTenantId(), rootOrg.getOrgId(), rootOrg, orgSave, false);
                    orgStruct = orgSaveOutput.findFirst();
                } else {
                    orgStruct = rootOrgSaveOutput.findFirst();
                }
                //endregion 保存 Company
                //endregion 创建根组织

                //region 保存 User
                Company company = orgStruct.getCompany();
                if (orgStruct.getCompanyId() != null && orgStruct.getCompanyId() > 0 && (company == null || company.getTaxNum() == null)) {
                    company = companyDao.findById(orgStruct.getCompanyId()).orElse(null);
                    if (company == null) {
                        String message = "存在脏数据, org(" + orgStruct.getOrgId() + "), companyId(" + orgStruct.getCompanyId() + ")";
                        logger.warn(message);
                        throw new IllegalArgumentException(message);
                    }
                }
                UserModel.Request.SaveUserOutput<U, O, R, A> saveUserOutput = userService.save(tenantEntity.getTenantId(), orgStruct, userSave, userSave.isRoleOverwrite(), userSave.isOrgOverwrite(), userSave.isTagOverwrite(), userSave.isAppOverwrite(), userSave.isMergeAccount(), userSave.isStrict());
                //endregion 保存 User

                SaveUserContext<U, O, R, A> saveUserContext = null;
                if (!saveUserOutput.getUserResultMap().isEmpty()) {
                    Optional<SaveUserContext<U, O, R, A>> saveUserResultOptional = saveUserOutput.getUserResultMap().values().stream().findFirst();
                    if (saveUserResultOptional.isPresent()) {
                        saveUserContext = saveUserResultOptional.get();
                        createResult.setUser(saveUserContext.getUser());
                        createResult.setAccount(saveUserContext.getAccount());
                    }
                }
                if (createResult != null && orgSaveOutput != null) {
                    OrgStruct org = orgSaveOutput.findFirst();
                    if (org != null) {
                        orgs.add((O) org);
                        createResult.setOrgs(orgs);
                    }
                    if (company != null) {
                        createResult.setCompany((C) company);
                    }
                } else if (createResult != null && orgStruct != null) {
                    createResult.setOrgs(Stream.of((O) orgStruct).collect(Collectors.toSet()));
                }

                if (companySave != null) {
                    //region 关联公司服务包
                    if (!CollectionUtils.isEmpty(tenantCreate.getPackageIds())) {
                        logger.info("tenantCreate.packageIds = " + tenantCreate.getPackageIds().stream().map(Object::toString).collect(Collectors.joining(",")));
                        long tenantId = tenantEntity.getTenantId();
                        long companyId = orgSaveOutput.findFirstCompanyId();
                        this.bindServicePackages(tenantId, companyId, tenantCreate.getPackageIds(), tenantCreate.isDefaultRole(), saveUserContext);
                    }
                    //endregion 关联公司服务包
                    if (saveUserContext != null) {
                        if (!saveUserContext.getRoleIds().isEmpty()) {
                            userService.bindRoles(saveUserContext.getUser(), null, saveUserContext.getRoleIds(), null, null, null, userSave.isRoleOverwrite(), userSave.isStrict(), false);
                        }
                        if (rootOrg != null) {
                            saveUserContext.addOrgId(rootOrg.getOrgId());
                        }
                        if (!saveUserContext.getOrgIds().isEmpty()) {
                            userService.bindOrgs(saveUserContext.getUser(), saveUserContext.getOrgIds(), saveUserContext.getUser().getModules(), userSave.isOrgOverwrite(), userSave.isStrict(), false);
                        }
                    }
                }
            }
            //endregion 保存 User
        }
        if (createResult != null) {
            createResult.setTenant((T) tenantEntity);
        }
        return tenantEntity;
    }

    public Tenant findById(Long tenantId) {
        return tenantDao.findById(tenantId).orElseThrow(() -> new IllegalArgumentException("未找到租户(id:" + tenantId + ")"));
    }

    /**
     * 关联公司服务包
     *
     * @param tenantId
     * @param companyId
     * @param packageIds
     */
    @Transactional(rollbackFor = Exception.class)
    public <U extends UserDto<O, R, A>, O extends OrgDto<O>, R extends RoleDto, A extends AccountDto> void bindServicePackages(long tenantId, long companyId, Collection<Long> packageIds, boolean isDefaultRole, SaveUserContext<U, O, R, A> saveUserContext) {
        logger.info("tenantId = {}, companyId = {}, isDefaultRole = {}", tenantId, companyId, isDefaultRole);
        if (!CollectionUtils.isEmpty(packageIds)) {
            Iterable<ServicePackage> packages = servicePackageDao.findAllById(packageIds);
            if (packages.iterator().hasNext()) {
                logger.info("packages is not empty");
                List<CompanyServiceRel> companyPackages = new ArrayList<>();
                List<Long> roleIds = new ArrayList<>();
                for (ServicePackage servicePackage : packages) {
                    CompanyServiceRel companyServiceRel = new CompanyServiceRel();
                    companyServiceRel.setTenantId(tenantId);
                    companyServiceRel.setCompanyId(companyId);
                    companyServiceRel.setServicePackageId(servicePackage.getServicePackageId());
                    companyServiceRel.setStatus(1);
                    companyServiceRel.setOperateReason("一站式入驻");
                    companyServiceRel.setRemarks("一站式入驻");
                    companyPackages.add(companyServiceRel);

                    if (isDefaultRole) {
                        Role role = new Role();
                        role.setCode(UUID.randomUUID().toString());
                        role.setName(servicePackage.getServicePackageName());
                        role.setRoleDesc(servicePackage.getServicePackageDesc());
                        role.setStatus(1);
                        role.setTenantId(tenantId);
                        role = roleDao.saveAndFlush(role);
                        roleIds.add(role.getId());

                        List<Long> resourcesetIds = serviceResourcesetRelDao.findResourcesetIdByServicePackageId(servicePackage.getServicePackageId());
                        RoleModel.Request.BindResourceSets bindResourceSets = new RoleModel.Request.BindResourceSets();
                        bindResourceSets.setOverwrite(false);
                        bindResourceSets.setResourcesetIds(resourcesetIds);
                        roleService.bindResourceSets(role, bindResourceSets);
                    }
                }
                if (saveUserContext != null && !CollectionUtils.isEmpty(roleIds)) {
                    saveUserContext.addRoleIds(roleIds);
                }

                companyServiceRelDao.saveAllAndFlush(companyPackages);
            }
        }

    }

    /**
     * 租户绑定多个公司
     *
     * @param tenantId
     * @param companyIds
     */
    @Transactional(rollbackFor = Exception.class)
    public boolean bindCompanies(long tenantId, List<Long> companyIds) {
        logger.info("tenantId = {}, companyIds = {}", tenantId, companyIds);
        if (CollectionUtils.isEmpty(companyIds)) {
            throw new IllegalArgumentException("公司id集合不能为空");
        }
        if (!tenantDao.existsById(tenantId)) {
            throw new IllegalArgumentException("不合法的租户id");
        }
        for (Long companyId : companyIds) {
            this.bindCompany(tenantId, companyId);
        }
        return true;
    }

    /**
     * 租户绑定单个公司
     *
     * @param tenantId
     * @param companyId
     */
    @Transactional(rollbackFor = Exception.class)
    public boolean bindCompany(long tenantId, long companyId) {
        logger.info("tenantId = {}, companyId = {}", tenantId, companyId);
        Optional<Company> companyOptional = companyDao.findById(companyId);
        if (!companyOptional.isPresent()) {
            throw new IllegalArgumentException("未找到对应的公司, companyId = " + companyId);
        }
        Company company = companyOptional.get();
        List<OrgStruct> companyOrgs = orgService.findByTenantIdAndCompanyId(tenantId, companyId);
        if (companyOrgs.isEmpty()) { //没有已存在组织, 需要创建新的
            OrgStruct orgStruct = new OrgStruct();
            orgStruct.setTenantId(tenantId);
            orgStruct.setCompanyId(companyId);
            orgStruct.setOrgName(company.getCompanyName());
            orgStruct.setOrgDesc(company.getCompanyName());
            orgStruct.setOrgCode(company.getCompanyCode());
            orgStruct.setOrgType(OrgType.COMPANY);
            List<OrgStruct> roots = orgService.findAllRoots(tenantId, Sort.unsorted());
            if (roots == null || roots.isEmpty()) {
                throw new IllegalArgumentException("未找到对应租户下的根组织, tenantId = " + tenantId);
            }
            Long parentId = roots.stream().map(OrgStruct::getOrgId).filter(Objects::nonNull).findFirst().get();
            logger.info("新绑定的租户在parentId:{}下", parentId);
            orgStruct.setParentId(parentId);
            orgStruct.setStatus(1);
            orgService.saveOrgStructEntity(orgStruct);
        }

        Tenant tenant = tenantDao.findById(tenantId).get();
        companyService.saveTenantCompany(tenant, company);

        return true;
    }

    @Deprecated
    public void fxCompanies(long tenantId) {

        List<Company> companies = orgStructDao.findCompaniesByTenantId(tenantId);
        Tenant tenant = tenantDao.findById(tenantId).get();
        for (Company company : companies) {
            companyService.saveTenantCompany(tenant, company);
        }
    }

    public Long findTenantByCode(String tenantCode) {
        Long tenantId = tenantDao.findTenantIdByTenantCode(tenantCode);
        return tenantId;
    }

    public String findSecretByTenantId(long tenantId) {
        return tenantSecretDao.findSecretByTenantId(tenantId);
    }

    public Tenant findByCode(String tenantCode) {
        Query query = new Query();
        query.setTenantCode(tenantCode);
        Specification<Tenant> specification = TenantQueryHelper.querySpecification(query);
        Optional<Tenant> optionalTenant = tenantDao.findOne(specification);
        return optionalTenant.orElseThrow(() -> new IllegalArgumentException("不合法的租户code"));
    }


    public List<Tenant> currentTenants() {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        long accountId = authorizedUser.getAccountId();
        return tenantDao.listTenantByAccountId(accountId);
    }

    public Tenant currentTenantInfo() {
        IAuthorizedUser authorizedUser = UserInfoHolder.currentUser();
        List<Long> tenantIds = new ArrayList<>();
        tenantIds.add(authorizedUser.getTenantId());
        //获取租户管理员
        List<TenantManagerDto> managerDtos = userService.findTenantManager(tenantIds);
        Tenant tenant = new Tenant();
        tenant.setTenantId(authorizedUser.getTenantId());
        tenant.setTenantCode(authorizedUser.getTenantCode());
        tenant.setTenantName(authorizedUser.getTenantName());
        if (!CollectionUtils.isEmpty(managerDtos)) {
            tenant.setTenantManager(managerDtos.get(0));
        }
        return tenant;
    }


    /**
     * 根据tenantCode或tenantName获取租户信息
     *
     * @param tenantCode 租户代码
     * @param tenantName 租户名称
     * @return
     */
    public Tenant findByTenantCodeOrTenantName(String tenantCode, String tenantName) {

        List<Tenant> tenants = this.findAllByTenantCodeOrTenantName(tenantCode, tenantName);
        if (CollectionUtils.isEmpty(tenants)) {
            String message = "查询租户(tenantCode:" + tenantCode + ",tenantName:" + tenantName + ")不存在";
            logger.warn(message);
            throw new IllegalArgumentException(message);
        } else if (tenants.size() > 1) {
            throw new IllegalArgumentException("查询到多个租户！");
        }
        return tenants.get(0);
    }

    private List<Tenant> findAllByTenantCodeOrTenantName(String tenantCode, String tenantName) {
        Query query = new Query();
        if (StringUtils.isNotEmpty(tenantCode)) {
            query.setTenantCode(tenantCode);
        }
        if (StringUtils.isNotEmpty(tenantName)) {
            query.setTenantName(tenantName);
        }

        Specification<Tenant> specification = TenantQueryHelper.queryOneSpecification(query);
        List<Tenant> tenants = tenantDao.findAll(specification, Sort.unsorted());
        return tenants;
    }

    /**
     * 8
     *
     * @param ids
     * @return
     */
    public List<Tenant> findAllById(List<Long> ids) {
        return this.tenantDao.findAllById(ids);
    }


    public Map<Long, String> allTenantSecrets() {
        List<TenantSecret> tenantSecrets = tenantSecretDao.findAll();
        Map<Long, String> result = tenantSecrets.stream().collect(Collectors.toMap(TenantSecret::getTenantId, TenantSecret::getSecret, (a, b) -> b, () -> new HashMap<>(tenantSecrets.size())));
        return result;
    }


    public Page<ComparedRevision<Tenant, String>> findTenantHistories(long tenantId, Pageable pageable) {
        pageable = PageRequest.of(pageable.getPageNumber(), pageable.getPageSize(), pageable.getSort().and(RevisionSort.desc()));
        Page<ComparedRevision<Tenant, String>> page = tenantDao.findComparedRevisions(tenantId, pageable);
        return page;
    }
}
