/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: UserExcelExportService.java   2020-10-26 19-24-00
 * Author: Evan
 */
package com.xforceplus.business.tenant.service;

import com.xforceplus.api.model.UserModel.Request.Query;
import com.xforceplus.business.excel.BusinessType;
import com.xforceplus.business.excel.SimpleExcelWriter;
import com.xforceplus.business.excel.reader.Context;
import com.xforceplus.business.excel.writer.ExcelConfigBusinessType;
import com.xforceplus.business.service.ExcelWriteService;
import com.xforceplus.entity.User;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Service;

import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import static com.xforceplus.business.excel.ExcelFile.*;
import static com.xforceplus.business.tenant.service.UserExcel.SORT_FIELD;
import static org.springframework.data.domain.Sort.Direction.DESC;

/**
 * <p>
 * Title: 用户导出
 * </p>
 * <p>
 * Description: 用户导出
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-10-26 19-24-00
 * @since V1.0
 */
@Service
public class UserExcelExportServiceImpl implements ExcelWriteService {

    /**
     * 日志
     */
    private static final Logger logger = LoggerFactory.getLogger(UserExcelExportServiceImpl.class);

    /**
     *  用户
     */
    private final UserService userService;


    public UserExcelExportServiceImpl(UserService userService) {
        this.userService = userService;
    }

    /**
     * 获取导入类型，用于Event事件调整导入方法
     * @return ImportBusinessType
     */
    @Override
    public BusinessType getBusinessType() {
        return ExcelConfigBusinessType.USER_IMPORT;
    }

    /**
     * 导出公共方法
     *
     * @param context 上下文
     * @return Context
     */
    @Override
    public void write(Context context) {
        Query query = context.getParam(PARAMS_QUERY, Query.class);
        SimpleExcelWriter simpleExcelWriter = context.getSimpleExcelWriter();
        List<User> userList;
        Integer maxPageSize = context.getMaxPageSize();
        logger.info("maxPageSize:{}", maxPageSize);
        /**
         * fix bug: empty list;
         */
        Set<Long> filterUserIds = new HashSet<>();
        if (query.getUserId() != null && query.getUserId() > 0) {
            filterUserIds.add(query.getUserId());
        }
        if (StringUtils.isNotBlank(query.getUserIds())) {
            Arrays.stream(StringUtils.split(query.getUserIds(), ",")).forEach(id -> {
                try {
                    Long parseLong = Long.parseLong(id);
                    filterUserIds.add(parseLong);
                } catch (NumberFormatException e) {
                    String message = e.getMessage() + ", id: " + id;
                    logger.warn(message);
                }
            });
        }
        if (CollectionUtils.isNotEmpty(filterUserIds)) {
            userList = userService.findAllById(filterUserIds);
            userService.fillExcel(userList, simpleExcelWriter, query.getSheets());
        } else {
            Page<User> page;
            Pageable pageable = PageRequest.of(START_PAGE, PAGE_SIZE, Sort.by(DESC, SORT_FIELD));
            do {
                page = userService.page(query, pageable);
                userService.fillExcel(page.toList(), simpleExcelWriter, query.getSheets());
            } while (page.hasNext()  && page.getNumber() < maxPageSize);
            logger.info("user page export, total page:{}", page.getNumber());
        }
    }
}
