/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: CompanyImportServiceImpl.java   2020-09-24 09-31-40
 * Author: Evan
 */
package com.xforceplus.business.tenant.service;

import com.alibaba.excel.event.AnalysisEventListener;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import com.xforceplus.api.model.AccountModel;
import com.xforceplus.api.model.UserModel;
import com.xforceplus.api.utils.Separator;
import com.xforceplus.bo.org.OrgCompanyQueryBo;
import com.xforceplus.business.account.service.AccountService;
import com.xforceplus.business.enums.SourceTypeEnum;
import com.xforceplus.business.excel.BusinessType;
import com.xforceplus.business.excel.DataRow;
import com.xforceplus.business.excel.ExcelSheet;
import com.xforceplus.business.excel.reader.Context;
import com.xforceplus.business.excel.reader.MessageRow;
import com.xforceplus.business.excel.reader.SimpleDataReadListener;
import com.xforceplus.business.excel.writer.ExcelConfigBusinessType;
import com.xforceplus.business.service.ExcelReaderService;
import com.xforceplus.business.tenant.dto.UserImportDTO;
import com.xforceplus.business.tenant.dto.UserOrgImportDTO;
import com.xforceplus.business.tenant.dto.UserRoleImportDTO;
import com.xforceplus.business.tenant.dto.UserTagImportDTO;
import com.xforceplus.config.ImportExportThreadPool;
import com.xforceplus.constants.RoleTypeEnum;
import com.xforceplus.domain.account.AccountType;
import com.xforceplus.domain.user.view.ExtraInfo;
import com.xforceplus.dto.org.OrgCompanyDTO;
import com.xforceplus.entity.*;
import com.xforceplus.tenant.security.core.domain.OrgType;
import com.xforceplus.utils.ObjectCheckAndExcuteUtils;
import io.geewit.core.utils.enums.BinaryUtils;
import io.geewit.core.utils.reflection.BeanUtils;
import org.apache.commons.collections4.ListUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.function.Consumer;
import java.util.stream.Collectors;

/**
 * user import service.
 * Company/Department: xforceplus
 * </p>
 *
 * @author feihu.wang
 * <b>Creation Time:</b> 2020-09-24 09-31-40
 * @since V1.0
 */
@Service
public class UserImportServiceImpl implements ExcelReaderService {
    /**
     * 日志
     */
    private static final Logger logger = LoggerFactory.getLogger(UserImportServiceImpl.class);

    public static final HashMap<String, Integer> SHEET_HEADER_NUMBER = new HashMap<>(6);

    /**
     * excel默认表头行数，新版本的人员导入模板表头都是占两行，如果不同sheet中表头所占行数不同，重写getSheetHeaderNumber()方法动态设置即可
     * 这个服务必须重写getSheetHeaderNumber()方法，否则回写导入结果会按表头占一行处理，导致表头丢失
     */
    private static final Integer HEADER_ROW_NUMBER = 2;

    static {
        SHEET_HEADER_NUMBER.put(UserExcel.SN_USER, HEADER_ROW_NUMBER);
        SHEET_HEADER_NUMBER.put(UserExcel.SN_USER_ORG, HEADER_ROW_NUMBER);
        SHEET_HEADER_NUMBER.put(UserExcel.SN_USER_ROLE, HEADER_ROW_NUMBER);
        SHEET_HEADER_NUMBER.put(UserExcel.SN_USER_DEVICE, HEADER_ROW_NUMBER);
        SHEET_HEADER_NUMBER.put(UserExcel.SN_USER_TERMINAL, HEADER_ROW_NUMBER);
        SHEET_HEADER_NUMBER.put(UserExcel.SN_USER_INVOICE_TYPE, HEADER_ROW_NUMBER);
    }

    @Autowired
    private UserService userService;
    @Autowired
    private TenantService tenantService;
    @Autowired
    private OrgService orgService;
    @Autowired
    private RoleService roleService;
    @Autowired
    private PreRoleService preRoleService;

    @Autowired
    private AccountService accountService;

    /**
     * 获取导入类型，用于Event事件调整导入方法
     * @return ImportBusinessType
     */
    @Override
    public BusinessType getBusinessType() {
        return ExcelConfigBusinessType.USER_IMPORT;
    }

    /**
     * 导入定义
     *
     * @param context 上下文
     * @return Context
     */
    @Override
    public Context importExcel(Context context) {
        List<ExcelSheet> sheets = context.getExcelBook().getExcelSheets();
        for (ExcelSheet sheet : sheets) {
            MessageRow messageRows = new MessageRow(sheet.getSheetName());
            Integer headerNumber = SHEET_HEADER_NUMBER.get(sheet.getSheetName());
            if (UserExcel.SN_USER.equalsIgnoreCase(sheet.getSheetName())) {
                AnalysisEventListener<UserImportDTO> excelDataListener = null;
                //构建监听器
                excelDataListener = SimpleDataReadListener.listener(context, (rows) -> {
                    this.saveUserData(messageRows, context, rows);
                });
                context.getSimpleExcelReader().read(UserImportDTO.class, excelDataListener, sheet.getSheetName(), headerNumber);
            } else if (UserExcel.SN_USER_ORG.equalsIgnoreCase(sheet.getSheetName())) {
                AnalysisEventListener<UserOrgImportDTO> excelDataListener = null;
                //构建监听器
                excelDataListener = SimpleDataReadListener.listener(context, (rows) -> {
                    this.saveUserOrgData(messageRows, context, rows);
                });
                context.getSimpleExcelReader().read(UserOrgImportDTO.class, excelDataListener, sheet.getSheetName(), headerNumber);
            } else if (UserExcel.SN_USER_ROLE.equalsIgnoreCase(sheet.getSheetName())) {
                AnalysisEventListener<UserRoleImportDTO> excelDataListener = null;
                //构建监听器
                excelDataListener = SimpleDataReadListener.listener(context, (rows) -> {
                    this.saveUserRoleData(messageRows, context, rows);
                });
                context.getSimpleExcelReader().read(UserRoleImportDTO.class, excelDataListener, sheet.getSheetName(), headerNumber);
            } else if (UserExcel.SN_USER_INVOICE_TYPE.equalsIgnoreCase(sheet.getSheetName())) {
                AnalysisEventListener<UserTagImportDTO> excelDataListener = null;
                //构建监听器
                excelDataListener = SimpleDataReadListener.listener(context, (rows) -> {
                    this.saveUserTagData(messageRows, context, rows, "invoiceType");
                });
                context.getSimpleExcelReader().read(UserTagImportDTO.class, excelDataListener, sheet.getSheetName(), headerNumber);
            } else if (UserExcel.SN_USER_DEVICE.equalsIgnoreCase(sheet.getSheetName())) {
                AnalysisEventListener<UserTagImportDTO> excelDataListener = null;
                //构建监听器
                excelDataListener = SimpleDataReadListener.listener(context, (rows) -> {
                    this.saveUserTagData(messageRows, context, rows, "printingEquipment");
                });
                context.getSimpleExcelReader().read(UserTagImportDTO.class, excelDataListener, sheet.getSheetName(), headerNumber);
            } else if (UserExcel.SN_USER_TERMINAL.equalsIgnoreCase(sheet.getSheetName())) {
                AnalysisEventListener<UserTagImportDTO> excelDataListener = null;
                //构建监听器
                excelDataListener = SimpleDataReadListener.listener(context, (rows) -> {
                    this.saveUserTagData(messageRows, context, rows, "ticketOpeningTerminal");
                });
                context.getSimpleExcelReader().read(UserTagImportDTO.class, excelDataListener, sheet.getSheetName(), headerNumber);
            }

        }
        return context;
    }

    /**
     * 保存user主数据
     *
     * @param list
     */
    public void saveUserData(MessageRow messageRows, Context context, List<UserImportDTO> list) {
        Long tenantId = context.getFileDTO().getTenantId();
        Tenant tenant = tenantService.findById(tenantId);
        logger.info("user-data-size:{}", list.size());
//        List<List<UserImportDTO>> partitionList = ListUtils.partition(list,list.size()/ ImportExportThreadPool.CORE_POOL_SIZE);
//        List<CompletableFuture<Void>> futureList = new ArrayList<>();
//        partitionList.forEach(item->futureList.add(CompletableFuture.runAsync(()->this.saveUser(item,messageRows,tenant,tenantId,context),ImportExportThreadPool.get())));
        //block to get Result
        try {
            this.asyncBatchOperation((dataRowList) -> this.saveUser(dataRowList, messageRows, tenant, tenantId, context), list).get();
        } catch (InterruptedException | ExecutionException e) {
            logger.error("save user error:{}", e.getMessage());
            throw new IllegalArgumentException("导入用户失败，当前服务器繁忙，请稍后重试");
        }
        context.messageRow(messageRows.getSheetName(), messageRows);
    }

    private void saveUser(List<UserImportDTO> list, MessageRow messageRows, Tenant tenant, Long tenantId, Context context) {
        logger.info("batch-user-data-size:{}", list.size());
        for (UserImportDTO userImportDTO : list) {
            logger.info("user excel info:{}", JSONObject.toJSONString(userImportDTO));
            if (!userImportDTO.getValidatedStatus()) {
                messageRows.fail(userImportDTO.getRowIndex(), userImportDTO.getValidatedMessage());
                continue;
            }

            if (StringUtils.isEmpty(userImportDTO.getEmail()) && StringUtils.isEmpty(userImportDTO.getTelPhone()) && StringUtils.isEmpty(userImportDTO.getUsername())) {
                messageRows.fail(userImportDTO.getRowIndex(), "账号不能为空");
                continue;
            }

            String username = getAccountName(userImportDTO);
            username = userService.getDomainAccountName(username, tenant.getTenantCode());
            User user = null;

            //fix bug: 如果以前没有邮箱，现在有邮箱，就找不到这个用户，会新建一个用户
            AccountModel.Request.Login query = new AccountModel.Request.Login();
            if (StringUtils.isNotBlank(userImportDTO.getTelPhone())) {
                query.setTelPhone(userImportDTO.getTelPhone());
            }
            if (StringUtils.isNotBlank(userImportDTO.getEmail())) {
                query.setEmail(userImportDTO.getEmail());
            }
            if (StringUtils.isNotBlank(userImportDTO.getUsername())) {
                query.setUsername(username);

            }
            //之前根据导入条件如果查询出多个帐号会默认取第一个，在更新用户信息会造成各种意想不到的问题，
            // 比如导入文件时填写的手机号/域帐号/邮箱刚好对应三个帐号，那么更新操作会把错误的信息更新进来，同时查询的用户也不一定是对的
            Account account = null;
            List<Account> accountList = accountService.findAllByQuery(query);
            //帐号不为空
            if (!CollectionUtils.isEmpty(accountList)) {
                //只查询出一个帐号，说明按条件查询的到帐号是同一个主体
                if (accountList.size() == 1) {
                    account = accountList.get(0);
                    //处理帐号添加邮箱，手机或者域帐号的情况
                    this.handleUserAccountUpdate(account, userImportDTO, tenant.getTenantCode());
                } else {
                    //根据条件查出多个不同的帐号，认为无法进行新增或者修改操作,直接结束流程
                    messageRows.fail(userImportDTO.getRowIndex(), "平台已存在多个对应账号，导入冲突");
                    context.messageRow(messageRows.getSheetName(), messageRows);
                    return;
                }
            }

            //TODO: 这里还有一个隐藏问题： 历史账号，可能没有添加租户code，这一步可能就找不到这个账号。
            // 但是如果找到了， 又可能有风险，担心找错了账号。   这里暂时不做处理！

            if (account != null) {
                user = userService.findUserByTenantIdAndAccountIdWithoutStatus(tenantId, account.getAccountId());
            }
            //获取用户的来源类型
             if (("新建").equalsIgnoreCase(userImportDTO.getAction())) {
                if (user != null) {
                    messageRows.fail(userImportDTO.getRowIndex(), "用户已经存在");
                    continue;
                } else {
                    //check password, new user, need pwd
                    if (StringUtils.isEmpty(userImportDTO.getPassword())) {
                        messageRows.fail(userImportDTO.getRowIndex(), "密码不能为空");
                        continue;
                    }
                    //用户类型 默认为内部用户
                    if (Objects.isNull(userImportDTO.getSourceType())) {
                         userImportDTO.setSourceType(SourceTypeEnum.INTERNAL.getSourceTypeDesc());
                    }

                    UserModel.Request.Create userModel = new UserModel.Request.Create();
                    userModel.setType(AccountType.OTHER);
                    BeanUtils.copyProperties(userImportDTO, userModel);

                    AccountModel.Request.Create accountModel = new AccountModel.Request.Create();
                    accountModel.setTelPhone(userImportDTO.getTelPhone());
                    accountModel.setEmail(userImportDTO.getEmail());
                    accountModel.setUsername(userImportDTO.getUsername());

                    if (StringUtils.isNotBlank(userImportDTO.getUsername())) {
                        String tenantUserName = userService.getDomainAccountName(userImportDTO.getUsername(), tenant.getTenantCode());
                        accountModel.setUsername(tenantUserName);
                    }
                    if (userImportDTO.getChangePasswordFlag() != null) {
                        accountModel.setChangePasswordFlag(userImportDTO.getChangePasswordFlag() > 0 ? true : false);
                    }
                    accountModel.setType(AccountType.OTHER);

                    if (!StringUtils.isEmpty(userImportDTO.getPassword())) {
                        accountModel.setPassword(userImportDTO.getPassword());
                        accountModel.setRandomPassword(false);
                    }
                    if (!StringUtils.isEmpty(accountModel.getEmail()) || !StringUtils.isEmpty(accountModel.getTelPhone())) {
                        userModel.setType(AccountType.PHONE_EMAIL);
                        accountModel.setType(AccountType.PHONE_EMAIL);
                    }
                    //判断Sex不能判断
                    if (StringUtils.isNotBlank(userImportDTO.getUserSex())) {
                        if (userImportDTO.getUserSex().contains("男")) {
                            userModel.setUserSex(0);
                        } else {
                            userModel.setUserSex(1);
                        }
                    }

                    if (StringUtils.isNotBlank(userImportDTO.getSourceType())) {
                        if (userImportDTO.getSourceType().contains(SourceTypeEnum.INTERNAL.getSourceTypeDesc())) {
                            userModel.setSourceType(SourceTypeEnum.INTERNAL.getSourceType());
                        } else {
                            userModel.setSourceType(SourceTypeEnum.EXTERNAL.getSourceType());
                        }
                    }
                    userModel.setTenantId(tenantId);
                    userModel.setContactAddr(userImportDTO.getContactAddr());
                    userModel.setAccount(accountModel);
                    try {
                        userService.saveUserPure(tenant, userModel, accountModel, account);
                        messageRows.success(userImportDTO.getRowIndex());
                    } catch (Exception e) {
                        logger.error("修改异常", e);
                        messageRows.fail(userImportDTO.getRowIndex(), e.getMessage());
                    }
                }
            } else if (("修改").equalsIgnoreCase(userImportDTO.getAction())) {
                if (user != null) {
                    if (!StringUtils.isEmpty(userImportDTO.getEmail())) {
                        user.setEmail(userImportDTO.getEmail());
                    }
                    if (!StringUtils.isEmpty(userImportDTO.getUserPhone())) {
                        user.setUserPhone(userImportDTO.getUserPhone());
                    }
                    if (!StringUtils.isEmpty(userImportDTO.getUserName())) {
                        user.setUserName(userImportDTO.getUserName());
                    }
                    if (!StringUtils.isEmpty(userImportDTO.getUserIdCard())) {
                        user.setUserIdCard(userImportDTO.getUserIdCard());
                    }
                    if (!StringUtils.isEmpty(userImportDTO.getContactAddr())) {
                        user.setContactAddr(userImportDTO.getContactAddr());
                    }
                    if (!StringUtils.isEmpty(userImportDTO.getUserWorkTel())) {
                        user.setUserWorkTel(userImportDTO.getUserWorkTel());
                    }
                    if (!StringUtils.isEmpty(userImportDTO.getUserSex())) {
                        if (userImportDTO.getUserSex().contains("男")) {
                            user.setUserSex(0);
                        } else {
                            user.setUserSex(1);
                        }
                    }

                    if (userImportDTO.getStatus() != null && userImportDTO.getStatus() > 0) {
                        user.setStatus(1);
                        user.setActiveStatus(1);
                    } else {
                        user.setStatus(0);
                        user.setActiveStatus(0);
                    }

                    if (!StringUtils.isEmpty(userImportDTO.getUserNumber())) {
                        user.setUserNumber(userImportDTO.getUserNumber());
                    }

                    if (!StringUtils.isEmpty(userImportDTO.getUserCode())) {
                        user.setUserCode(userImportDTO.getUserCode());
                    }
                    if (!StringUtils.isEmpty(userImportDTO.getContactAddr())) {
                        user.setContactAddr(userImportDTO.getContactAddr());
                    }
                    if (userImportDTO.getChangePasswordFlag() != null && userImportDTO.getChangePasswordFlag() > 0) {
                        account.setChangePasswordFlag(true);
                    } else {
                        account.setChangePasswordFlag(false);
                    }
                    if (userImportDTO.getExpiredDate() != null) {
                        user.setExpiredDate(userImportDTO.getExpiredDate());
                    }
                     try {
                        userService.updateUserPure(user, account);
                        messageRows.success(userImportDTO.getRowIndex());
                    } catch (Exception e) {
                        logger.error("修改异常", e);
                        messageRows.fail(userImportDTO.getRowIndex(), e.getMessage());
                    }
                } else {
                    messageRows.fail(userImportDTO.getRowIndex(), "该用户不存在");
                }
            }
        }
    }

    /**
     * 保存user org数据
     *
     * @param list
     */
    public void saveUserOrgData(MessageRow messageRows, Context context, List<UserOrgImportDTO> list) {
        Long tenantId = context.getFileDTO().getTenantId();
        Tenant tenant = tenantService.findById(tenantId);
        try {
            this.asyncBatchOperation((dataList) -> this.saveUserOrg(dataList, messageRows, tenant, tenantId, context), list).get();
        } catch (InterruptedException | ExecutionException e) {
            logger.error("save user-org error:{}", e.getMessage());
            throw new IllegalArgumentException("导入用户关联组织失败，当前服务器繁忙，请稍后重试");
        }
        context.messageRow(messageRows.getSheetName(), messageRows);
    }

    private void saveUserOrg(List<UserOrgImportDTO> list, MessageRow messageRows, Tenant tenant, Long tenantId, Context context) {
        for (UserOrgImportDTO orgImportDTO : list) {
            logger.info("user org:{}", JSONObject.toJSONString(orgImportDTO));
            if (!orgImportDTO.getValidatedStatus()) {
                messageRows.fail(orgImportDTO.getRowIndex(), orgImportDTO.getValidatedMessage());
                continue;
            }
            String username = orgImportDTO.getUsername();
            String tenantUsername = userService.getDomainAccountName(username, tenant.getTenantCode());
            Optional<User> userOptional = userService.findByTenantIdAndUsername(tenantId, tenantUsername);
            User user = null;
            if (!userOptional.isPresent()) {
                //fix bug: 旧的账号可能没有添加tenantCode,使用用户名查找
                userOptional = userService.findByTenantIdAndUsername(tenantId, username);
                if (!userOptional.isPresent()) {
                    messageRows.fail(orgImportDTO.getRowIndex(), "该用户不存在");
                    continue;
                }
            }

            user = userOptional.get();
            user = userService.findById(user.getId(), BinaryUtils.toBinary(ExtraInfo.currentOrgs));
            String orgCode = orgImportDTO.getOrgCode();
            String taxNum = orgImportDTO.getTaxNum();
            if (("绑定").equalsIgnoreCase(orgImportDTO.getAction())) {
                boolean isBind = false;
                if (!CollectionUtils.isEmpty(user.getOrgs())) {
                    for (OrgStruct org : user.getOrgs()) {
                        if (org.getOrgCode().equalsIgnoreCase(orgImportDTO.getOrgCode())) {
                            messageRows.fail(orgImportDTO.getRowIndex(), "该组织已经绑定");
                            isBind = true;
                            break;
                        }
                    }
                }
                if (!isBind) {
                    // bind
                    try {
                        //新模板添加了税号，先判断组织代码和税号不能同时为空，然后再根据组织代码或者税号来绑定，如果组织代码和税号都填了，校验组织代码跟税号是否一致
                        if (StringUtils.isBlank(orgCode) && StringUtils.isBlank(taxNum)) {
                            messageRows.fail(orgImportDTO.getRowIndex(), "组织代码和公司税号不能同时为空");
                            context.messageRow(messageRows.getSheetName(), messageRows);
                            return;
                        }
                        //优先按组织代码来绑定
                        if (StringUtils.isNotBlank(orgCode)) {
                            List<OrgStruct> orgStructList = orgService.findByTenantIdAndOrgCode(tenantId, orgImportDTO.getOrgCode());
                            if (!CollectionUtils.isEmpty(orgStructList)) {
                                Optional<OrgStruct> optionalOrgStruct;
                                if (StringUtils.isNotBlank(taxNum)) {
                                    optionalOrgStruct = orgStructList.stream().filter(item -> taxNum.equals(item.getTaxNum())).findFirst();
                                } else {
                                    optionalOrgStruct = orgStructList.stream().findFirst();
                                }
                                if (optionalOrgStruct.isPresent()) {
                                    //判断绑定的不是母公司
                                    OrgStruct orgStruct = optionalOrgStruct.get();
                                    if (orgStruct.getOrgType().equals(OrgType.GROUP)) {
                                        messageRows.fail(orgImportDTO.getRowIndex(), "不能绑定母公司");
                                    } else {
                                        userService.bindOrgs(user, Lists.newArrayList(optionalOrgStruct.get().getOrgId()), null, false, false, false);
                                        messageRows.success(orgImportDTO.getRowIndex());
                                    }
                                } else {
                                    messageRows.fail(orgImportDTO.getRowIndex(), "组织代码和公司税号不一致或不存在");
                                }
                            } else {
                                messageRows.fail(orgImportDTO.getRowIndex(), "组织code未找到！");
                            }
                        } else {
                            //按税号来查询组织并绑定用户和组织
                            OrgCompanyQueryBo orgCompanyQueryBo = new OrgCompanyQueryBo();
                            orgCompanyQueryBo.setTaxNum(taxNum);
                            orgCompanyQueryBo.setTenantId(tenantId);
                            List<OrgCompanyDTO> orgCompanyDTOList = orgService.findOrgCompanyByTaxNum(orgCompanyQueryBo);
                            if (!CollectionUtils.isEmpty(orgCompanyDTOList)) {
                                userService.bindOrgs(user, Lists.newArrayList(orgCompanyDTOList.get(0).getOrgId()), null, false, false, false);
                                messageRows.success(orgImportDTO.getRowIndex());
                            } else {
                                messageRows.fail(orgImportDTO.getRowIndex(), "公司税号未找到！");
                            }
                        }

                    } catch (Exception e) {
                        logger.error("绑定组织异常", e);
                        messageRows.fail(orgImportDTO.getRowIndex(), e.getMessage());
                    }
                }
            } else if (("解绑").equalsIgnoreCase(orgImportDTO.getAction())) {
                boolean isUnBind = false;
                if (StringUtils.isBlank(orgCode) && StringUtils.isBlank(taxNum)) {
                    messageRows.fail(orgImportDTO.getRowIndex(), "组织代码和公司税号不能同时为空");
                    context.messageRow(messageRows.getSheetName(), messageRows);
                    return;
                }
                if (!CollectionUtils.isEmpty(user.getCurrentOrgs())) {
                    for (OrgStruct org : user.getCurrentOrgs()) {
                        //解绑的时候校验组织代码和公司税号
                        //优先使用组织代码
                        if (StringUtils.isNotBlank(orgCode)) {
                            if (StringUtils.isNotBlank(taxNum)) {
                                if (orgCode.equalsIgnoreCase(org.getOrgCode()) && taxNum.equalsIgnoreCase(org.getTaxNum())) {
                                    userService.unbindOrgs(tenantId, user.getId(), null, Lists.newArrayList(org.getOrgId()));
                                    isUnBind = true;
                                    messageRows.success(orgImportDTO.getRowIndex());
                                    break;
                                }
                            } else {
                                if (orgCode.equalsIgnoreCase(org.getOrgCode())) {
                                    userService.unbindOrgs(tenantId, user.getId(), null, Lists.newArrayList(org.getOrgId()));
                                    isUnBind = true;
                                    messageRows.success(orgImportDTO.getRowIndex());
                                    break;
                                }
                            }
                        } else {
                            if (taxNum.equalsIgnoreCase(org.getTaxNum())) {
                                userService.unbindOrgs(tenantId, user.getId(), null, Lists.newArrayList(org.getOrgId()));
                                isUnBind = true;
                                messageRows.success(orgImportDTO.getRowIndex());
                                break;
                            }
                        }
                    }
                }

                if (!isUnBind) {
                    messageRows.fail(orgImportDTO.getRowIndex(), "该组织未绑定");
                }
            }
        }
    }

    /**
     * 保存user tag数据
     *
     * @param list
     */
    public void saveUserTagData(MessageRow messageRows, Context context, List<UserTagImportDTO> list, String tagName) {
        Long tenantId = context.getFileDTO().getTenantId();
        Tenant tenant = tenantService.findById(tenantId);
        logger.info("import-tag-data:{},{}", tenant.getTenantCode(), tagName);
        for (UserTagImportDTO tag : list) {
            if (!tag.getValidatedStatus()) {
                messageRows.fail(tag.getRowIndex(), tag.getValidatedMessage());
                continue;
            }
        }
        logger.info("user-tag-size:{}", list.size());
        Map<String, List<UserTagImportDTO>> tagMap = list.stream().filter(tag -> StringUtils.isNotEmpty(tag.getUsername())).collect(Collectors.groupingBy(UserTagImportDTO::getUsername));
        //转为list，然后调用通用处理方法
        List<Map.Entry<String, List<UserTagImportDTO>>> dataList = new ArrayList<>(tagMap.entrySet());
        try {
            this.asyncBatchOperation((data) -> this.saveUserTag(data, messageRows, tenant, tenantId, context, tagName), dataList).get();
        } catch (InterruptedException | ExecutionException e) {
            logger.error("save user-tag error:{}", e.getMessage());
            throw new IllegalArgumentException("导入用户标签失败，当前服务器繁忙，请稍后重试");
        }
        context.messageRow(messageRows.getSheetName(), messageRows);
    }

    private void saveUserTag(List<Map.Entry<String, List<UserTagImportDTO>>> list, MessageRow messageRows, Tenant tenant, Long tenantId, Context context, String tagName) {
        Set<Map.Entry<String, List<UserTagImportDTO>>> set = new HashSet(list);
        for (Map.Entry<String, List<UserTagImportDTO>> tagList : set) {
            String username = tagList.getKey();
            String tenantUsername = userService.getDomainAccountName(username, tenant.getTenantCode());
            Optional<User> userOptional = userService.findByTenantIdAndUsername(tenantId, tenantUsername);
            User user = null;
            if (!userOptional.isPresent()) {
                userOptional = userService.findByTenantIdAndUsername(tenantId, username);
                if (!userOptional.isPresent()) {
                    tagList.getValue().forEach(tag -> messageRows.fail(tag.getRowIndex(), "该用户不存在"));
                    continue;
                }
            }
            user = userOptional.get();

            String currentTagVal = "";
            UserTag currentUserTag = null;
            List<UserTag> userTagsList = user.getUserTags();
            if (!CollectionUtils.isEmpty(userTagsList)) {
                for (UserTag userTag : userTagsList) {
                    if (tagName.equalsIgnoreCase(userTag.getTagName())) {
                        currentTagVal = userTag.getTagValue();
                        currentUserTag = userTag;
                    }
                }
            }

            Map<String, List<UserTagImportDTO>> tagMapByAction = tagList.getValue().stream().collect(Collectors.groupingBy(UserTagImportDTO::getAction));
            List<UserTagImportDTO> bindList = tagMapByAction.get("绑定");
            List<UserTagImportDTO> unbindList = tagMapByAction.get("解绑");
            Set<String> tagSet = Arrays.stream(currentTagVal.split(",")).collect(Collectors.toSet());
            if ("invoiceType".equalsIgnoreCase(tagName)) {
                if (!CollectionUtils.isEmpty(bindList)) {
                    currentTagVal = currentTagVal + "," + bindList.stream().map(tag -> tag.getInvoiceType()).distinct().collect(Collectors.joining(","));
                    tagSet.addAll(bindList.stream().filter(tag -> StringUtils.isNotEmpty(tag.getInvoiceType())).map(tag -> tag.getInvoiceType().trim()).distinct().collect(Collectors.toList()));
                }
                if (!CollectionUtils.isEmpty(unbindList)) {
                    for (UserTagImportDTO userTagImportDTO : unbindList) {
                        if (userTagImportDTO.getInvoiceType() != null) {
                            tagSet.remove(userTagImportDTO.getInvoiceType().trim());
                        }
                    }
                }
            } else if ("printingEquipment".equalsIgnoreCase(tagName)) {
                if (!CollectionUtils.isEmpty(bindList)) {
                    currentTagVal = currentTagVal + "," + bindList.stream().map(tag -> tag.getPrintingEquipment()).distinct().collect(Collectors.joining(","));
                    tagSet.addAll(bindList.stream().filter(tag -> StringUtils.isNotEmpty(tag.getPrintingEquipment())).map(tag -> tag.getPrintingEquipment().trim()).distinct().collect(Collectors.toList()));
                }
                if (!CollectionUtils.isEmpty(unbindList)) {
                    for (UserTagImportDTO userTagImportDTO : unbindList) {
                        if (userTagImportDTO.getPrintingEquipment() != null) {
                            tagSet.remove(userTagImportDTO.getPrintingEquipment().trim());
                        }
                    }
                }
            } else if ("ticketOpeningTerminal".equalsIgnoreCase(tagName)) {
                if (!CollectionUtils.isEmpty(bindList)) {
                    currentTagVal = currentTagVal + "," + bindList.stream().map(tag -> tag.getTicketOpeningTerminal()).distinct().collect(Collectors.joining(","));
                    tagSet.addAll(bindList.stream().filter(tag -> StringUtils.isNotEmpty(tag.getTicketOpeningTerminal())).map(tag -> tag.getTicketOpeningTerminal().trim()).distinct().collect(Collectors.toList()));
                }
                if (!CollectionUtils.isEmpty(unbindList)) {
                    for (UserTagImportDTO userTagImportDTO : unbindList) {
                        if (userTagImportDTO.getTicketOpeningTerminal() != null) {
                            tagSet.remove(userTagImportDTO.getTicketOpeningTerminal().trim());
                        }
                    }
                }
            }
            if (currentUserTag == null) {
                currentUserTag = new UserTag();
                currentUserTag.setTagName(tagName);
                currentUserTag.setUserId(user.getId());
            }
            currentUserTag.setTagValue(tagSet.stream().filter(t -> StringUtils.isNotBlank(t)).collect(Collectors.joining(",")));

            try {
                //fix bug: 4702
                if (StringUtils.isBlank(currentUserTag.getTagValue()) && currentUserTag.getId() != null) {
                    userService.deleteUserTagById(currentUserTag.getId());
                } else {
                    // 删除第一个字符是","
                    if (currentUserTag.getTagValue().startsWith(Separator.COMMA)) {
                        currentUserTag.setTagValue(currentUserTag.getTagValue().replaceFirst(Separator.COMMA, StringUtils.EMPTY));
                    }
                    userService.saveUserTag(currentUserTag);
                }
                if (!CollectionUtils.isEmpty(bindList)) {
                    for (UserTagImportDTO tag : bindList) {
                        logger.info("sheetName{},bindListSize:{}", tagName, bindList.size());
                        messageRows.success(tag.getRowIndex());
                    }
                }
                if (!CollectionUtils.isEmpty(unbindList)) {
                    for (UserTagImportDTO tag : unbindList) {
                        messageRows.success(tag.getRowIndex());
                    }
                }

            } catch (Exception e) {
                logger.error("更新usertag失败", e);
                if (!CollectionUtils.isEmpty(bindList)) {
                    for (UserTagImportDTO tag : bindList) {
                        messageRows.fail(tag.getRowIndex(), "更新失败：" + e.getMessage());
                    }
                }
                if (!CollectionUtils.isEmpty(unbindList)) {
                    for (UserTagImportDTO tag : unbindList) {
                        messageRows.fail(tag.getRowIndex(), "更新失败：" + e.getMessage());
                    }
                }
            }
        }
    }

    /**
     * 保存user role数据
     *
     * @param list
     */
    public void saveUserRoleData(MessageRow messageRows, Context context, List<UserRoleImportDTO> list) {
        Long tenantId = context.getFileDTO().getTenantId();
        Tenant tenant = tenantService.findById(tenantId);
        try {
            this.asyncBatchOperation((data) -> this.saveUserRole(data, messageRows, tenant, tenantId, context), list).get();
        } catch (InterruptedException | ExecutionException e) {
            logger.error("save user-role error:{}", e.getMessage());
            throw new IllegalArgumentException("导入用户角色失败，当前服务器繁忙，请稍后重试");
        }
        context.messageRow(messageRows.getSheetName(), messageRows);
    }

    private void saveUserRole(List<UserRoleImportDTO> list, MessageRow messageRows, Tenant tenant, Long tenantId, Context context) {
        for (UserRoleImportDTO roleImportDTO : list) {
            logger.info("user role:{}", JSONObject.toJSONString(roleImportDTO));
            if (!roleImportDTO.getValidatedStatus()) {
                messageRows.fail(roleImportDTO.getRowIndex(), roleImportDTO.getValidatedMessage());
                continue;
            }
            String username = roleImportDTO.getUsername();
            String tenantUsername = userService.getDomainAccountName(username, tenant.getTenantCode());
            Optional<User> userOptional = userService.findByTenantIdAndUsername(tenantId, tenantUsername);
            User user = null;
            if (!userOptional.isPresent()) {
                //fix bug: 旧的账号可能没有添加tenantCode,使用用户名查找
                userOptional = userService.findByTenantIdAndUsername(tenantId, username);
                if (!userOptional.isPresent()) {
                    messageRows.fail(roleImportDTO.getRowIndex(), "该用户不存在");
                    continue;
                }
            }
            user = userOptional.get();

            if (("绑定").equalsIgnoreCase(roleImportDTO.getAction())) {
                boolean isBind = false;
                List<Role> userRoles = roleService.listByUserId(user.getId());
                if (!CollectionUtils.isEmpty(userRoles)) {
                    for (Role userRole : userRoles) {
                        if (userRole.getCode().equalsIgnoreCase(roleImportDTO.getRoleCode())) {
                            messageRows.fail(roleImportDTO.getRowIndex(), "该角色已经绑定");
                            isBind = true;
                            break;
                        }
                    }
                }

                if (!isBind) {
                    try {
                        Long roleId = roleService.findIdByTenantIdAndCode(tenantId, roleImportDTO.getRoleCode());
                        roleId =  ObjectCheckAndExcuteUtils.docheckAndExcute(roleId, x->{ return Objects.isNull(x)
                                &&preRoleService.queryCountPreRoleByTenantIdAndRoleIdOrRoleCode(tenantId, null,roleImportDTO.getRoleCode(), RoleTypeEnum.PRE.getType())>0 ;}, x -> {
                            return roleService.findIdByTenantIdAndCode(PreRoleService.PER_TENANT_ID, roleImportDTO.getRoleCode());
                        }, x->{return x;});
                        if (roleId != null) {
                            userService.bindRoles(tenantId, null, user.getId(), Lists.newArrayList(roleId), null, null, null, false, false, false);
                            messageRows.success(roleImportDTO.getRowIndex());
                        } else {
                            messageRows.fail(roleImportDTO.getRowIndex(), "角色code不存在");
                        }
                    } catch (Exception e) {
                        logger.error("role bind error", e);
                        messageRows.fail(roleImportDTO.getRowIndex(), e.getMessage());
                    }
                }
            } else if (("解绑").equalsIgnoreCase(roleImportDTO.getAction())) {
                boolean isUnBind = false;
                List<Role> userRoles = roleService.listByUserId(user.getId());
                if (!CollectionUtils.isEmpty(userRoles)) {
                    for (Role userRole : userRoles) {
                        if (userRole.getCode().equalsIgnoreCase(roleImportDTO.getRoleCode())) {
                            userService.unbindRoles(tenantId, user.getId(), Lists.newArrayList(userRole.getId()));
                            isUnBind = true;
                            messageRows.success(roleImportDTO.getRowIndex());
                            break;
                        }
                    }
                }
                if (!isUnBind) {
                    messageRows.fail(roleImportDTO.getRowIndex(), "该角色code未绑定");
                }
            }

        }
    }


    private String getAccountName(UserImportDTO userImportDTO) {
        String accountName = userImportDTO.getEmail();
        if (StringUtils.isEmpty(accountName)) {
            accountName = userImportDTO.getTelPhone();
        }
        if (StringUtils.isEmpty(accountName)) {
            accountName = userImportDTO.getUsername();
        }
        return accountName;
    }

    @Override
    public HashMap<String, Integer> getSheetHeaderNumber() {
        return SHEET_HEADER_NUMBER;
    }

    private CompletableFuture<Void> asyncBatchOperation(Consumer<List> consumer, List dataList) {
        int partitionSize = dataList.size() > ImportExportThreadPool.CORE_POOL_SIZE ? dataList.size() / ImportExportThreadPool.CORE_POOL_SIZE : 1;
        List<List<DataRow>> partitionList = ListUtils.partition(dataList, partitionSize);
        List<CompletableFuture<Void>> futureList = new ArrayList<>();
        partitionList.forEach(item -> futureList.add(CompletableFuture.runAsync(() -> consumer.accept(item), ImportExportThreadPool.get())));
        return CompletableFuture.allOf(futureList.toArray(new CompletableFuture[]{}));
    }

    /**
     * 更新时补充用户帐号
     *
     * @param account    查询的用户帐号
     * @param importDTO  excel中的数据
     * @param tenantCode
     */
    private void handleUserAccountUpdate(Account account, UserImportDTO importDTO, String tenantCode) {
        if (StringUtils.isNotBlank(importDTO.getEmail())) {
            if (StringUtils.isBlank(account.getEmail())) {
                account.setEmail(importDTO.getEmail());
            }
        }
        if (StringUtils.isNotBlank(importDTO.getTelPhone())) {
            if (StringUtils.isBlank(account.getTelPhone())) {
                account.setTelPhone(importDTO.getTelPhone());
            }
        }
        if (StringUtils.isNotBlank(importDTO.getUsername())) {
            //没有域帐号字段才处理
            if (StringUtils.isBlank(account.getUsername())) {
                String tenantUsername = userService.getDomainAccountName(importDTO.getUsername(), tenantCode);
                account.setUsername(tenantUsername);
            }
        }
    }
}
