/*
 * Copyright (c)  2015~2021, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: UserRedisService.java   2021-01-08 15-07-15
 * Author: Evan
 */
package com.xforceplus.business.tenant.service;

import com.xforceplus.entity.User;
import io.geewit.core.utils.reflection.BeanUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.cache.Cache;
import org.springframework.cache.CacheManager;
import org.springframework.stereotype.Service;

import java.util.Optional;
import java.util.stream.Stream;

/**
 * <p>
 * Title:
 * </p>
 * <p>
 * Description:
 * </p>
 * <p>
 * Copyright: 2015~2021
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2021-01-08 15-07-15
 * @since V1.0
 */
@Slf4j
@Service
public class UserRedisCacheService {
    /**
     * Redis
     */
    public static final String LOGIN_USER_KEY = "userId:%s:loginId:%s";
    /**
     * CacheName
     */
    public static final String USER_CACHE_NAME = "User";

    private final CacheManager cacheManager;

    public UserRedisCacheService(CacheManager cacheManager) {
        this.cacheManager = cacheManager;
    }

    /**
     * 根据用户Id和登录ID获取租户信息
     *
     * @param userId  用户ID
     * @param loginId 登录
     * @return Optional<User>
     */
    public Optional<User> getUserByUserIdAndLoginId(Long userId, String loginId) {
        Cache cache = cacheManager.getCache(USER_CACHE_NAME);
        if (cache == null) {
            return Optional.empty();
        }
        //如登录ID为空，则返回空数据
        if (StringUtils.isEmpty(loginId)) {
            return Optional.empty();
        }
        String cacheKey = String.format(LOGIN_USER_KEY, userId, loginId);
        User user;
        try {
            user = cache.get(cacheKey, User.class);
        } catch (Exception e) {
            log.warn(e.getMessage());
            user = null;
        }
        log.info("cached user in redis: {}", user);
        return Optional.ofNullable(user);
    }

    /**
     * 设置缓存数据
     *
     * @param user    用户信息
     * @param userId  用户ID
     * @param loginId 登录ID
     */
    public void pubUserByUserIdAndLoginId(User user, Long userId, String loginId) {
        //如登录ID为空，则返回空数据
        if (StringUtils.isEmpty(loginId)) {
            return;
        }
        Cache cache = cacheManager.getCache(USER_CACHE_NAME);
        if (cache == null) {
            return;
        }
        String cacheKey = String.format(LOGIN_USER_KEY, userId, loginId);
        User cachingUser = new User();
        BeanUtils.copyProperties(user, cachingUser, Stream.of("taxNums", "roleUserRels", "orgUserRels", "orgVirtualNodeUserRels", "userApps").toArray(String[]::new));
        cache.put(cacheKey, cachingUser);
    }

    public void cleanUserByUserIdAndLoginId(long userId, String loginId) {
        Cache cache = cacheManager.getCache(USER_CACHE_NAME);
        if (cache == null) {
            return;
        }
        String cacheKey = String.format(LOGIN_USER_KEY, userId, loginId);
        cache.evictIfPresent(cacheKey);
    }
}
