package com.xforceplus.business.tenant.service;

import com.google.common.collect.Lists;
import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.model.AccountModel;
import com.xforceplus.api.model.OrgModel;
import com.xforceplus.api.model.RoleModel;
import com.xforceplus.api.model.UserModel;
import com.xforceplus.api.model.UserModel.Request.*;
import com.xforceplus.api.utils.Separator;
import com.xforceplus.business.account.service.AccountService;
import com.xforceplus.business.enums.BaseEnum;
import com.xforceplus.business.enums.SourceTypeEnum;
import com.xforceplus.business.excel.SimpleExcelWriter;
import com.xforceplus.business.externalservice.terminal.TerminalApiServiceImpl;
import com.xforceplus.business.externalservice.terminal.model.MsDeviceInfo;
import com.xforceplus.business.externalservice.terminal.model.MsTerminalQueryResponseInfo;
import com.xforceplus.business.messagebus.UserPubService;
import com.xforceplus.business.resource.service.ResourceService;
import com.xforceplus.business.resource.service.ServiceApiService;
import com.xforceplus.business.resource.service.ServicePackageService;
import com.xforceplus.business.tenant.dto.*;
import com.xforceplus.config.ImportExportThreadPool;
import com.xforceplus.constants.EnvProfile;
import com.xforceplus.constants.RoleTypeEnum;
import com.xforceplus.dao.*;
import com.xforceplus.domain.account.AccountDto;
import com.xforceplus.domain.account.AccountType;
import com.xforceplus.domain.org.OrgDto;
import com.xforceplus.domain.resource.RequestUri;
import com.xforceplus.domain.resource.RequestUriAuthz;
import com.xforceplus.domain.tenant.RoleDto;
import com.xforceplus.domain.tenant.TenantManagerDto;
import com.xforceplus.domain.user.UserDto;
import com.xforceplus.domain.user.UserExportDto;
import com.xforceplus.domain.user.view.ExtraInfo;
import com.xforceplus.entity.*;
import com.xforceplus.query.OrgQueryHelper;
import com.xforceplus.query.UserQueryHelper;
import com.xforceplus.redis.lock.RedisLock;
import com.xforceplus.tenant.core.exception.UnknownException;
import com.xforceplus.tenant.security.core.context.UserInfoHolder;
import com.xforceplus.tenant.security.core.domain.IAuthorizedUser;
import com.xforceplus.tenant.security.core.domain.OrgType;
import com.xforceplus.utils.*;
import io.geewit.core.jackson.view.View;
import io.geewit.core.utils.enums.BinaryUtils;
import io.geewit.core.utils.reflection.BeanUtils;
import io.geewit.core.utils.tree.TreeUtils;
import io.geewit.data.jpa.essential.domain.EntityGraph;
import io.geewit.data.jpa.essential.domain.EntityGraphs;
import io.geewit.data.jpa.essential.search.DynamicSpecifications;
import io.geewit.data.jpa.essential.search.Operator;
import io.geewit.data.jpa.essential.search.SearchFilter;
import io.geewit.web.utils.JsonUtils;
import lombok.Builder;
import org.apache.commons.collections4.ListUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.lang3.tuple.Triple;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cache.CacheManager;
import org.springframework.cache.annotation.CacheConfig;
import org.springframework.data.domain.*;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import javax.persistence.criteria.Predicate;
import javax.validation.ConstraintViolation;
import javax.validation.Validator;
import javax.validation.groups.Default;
import java.security.InvalidParameterException;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.xforceplus.business.tenant.service.RoleAccountImportServiceImpl.*;
import static com.xforceplus.business.tenant.service.UserExcel.*;
import static java.util.stream.Collectors.toSet;

@CacheConfig(cacheNames = {"AuthorizedUser", "User"})
@Service
public class UserService implements DisposableBean {

    private final static Logger logger = LoggerFactory.getLogger(UserService.class);

    private static final String EVENT_USERS_STATUS = "EVENT_USERS_STATUS";

    private static final String PRINTING_EQUIPMENT = "printingEquipment";
    private static final String TICKET_TERMINAL = "ticketOpeningTerminal";

    private final AppDao appDao;

    private final AccountDao accountDao;

    private final OrgService orgService;

    private final RoleService roleService;

    private final UserDao userDao;

    private final UserAppDao userAppDao;

    private final UserTagDao userTagDao;

    private final RoleDao roleDao;

    private final OrgStructDao orgStructDao;

    private final OrgUserRelDao orgUserRelDao;

    private final RoleUserRelDao roleUserRelDao;

    private final TenantDao tenantDao;

    private final AccountService accountService;

    private final ServicePackageService packageService;

    private final ServiceApiService serviceApiService;

    private final CacheManager cacheManager;

    private final RedisTemplate redisTemplate;
    private final UserRedisCacheService userRedisCacheService;
    private TerminalApiServiceImpl terminalApiService;

    private final Validator validator;

    private final ResourceService resourceService;

    private final CompanyTenantRelDao companyTenantRelDao;

    private final TenantPolicyService tenantPolicyService;

    private final UserPubService userPubService;

    @Value("${xforce.global.grading-managment.enable:true}")
    private boolean globalGradingEnabled;

    public UserService(AppDao appDao, AccountDao accountDao, UserDao userDao, UserAppDao userAppDao, UserTagDao userTagDao,
                       RoleDao roleDao, OrgStructDao orgStructDao, OrgUserRelDao orgUserRelDao, RoleUserRelDao roleUserRelDao,
                       TenantDao tenantDao, AccountService accountService, OrgService orgService, RoleService roleService,
                       ServicePackageService packageService, ServiceApiService serviceApiService, CacheManager cacheManager,
                       RedisTemplate redisTemplate, UserRedisCacheService userRedisCacheService,
                       TerminalApiServiceImpl terminalApiService, Validator validator, ResourceService resourceService, CompanyTenantRelDao companyTenantRelDao, TenantPolicyService tenantPolicyService, UserPubService userPubService) {
        this.appDao = appDao;
        this.accountDao = accountDao;
        this.userDao = userDao;
        this.userAppDao = userAppDao;
        this.userTagDao = userTagDao;
        this.roleDao = roleDao;
        this.orgStructDao = orgStructDao;
        this.orgUserRelDao = orgUserRelDao;
        this.roleUserRelDao = roleUserRelDao;
        this.tenantDao = tenantDao;
        this.accountService = accountService;
        this.orgService = orgService;
        this.roleService = roleService;
        this.packageService = packageService;
        this.serviceApiService = serviceApiService;
        this.cacheManager = cacheManager;
        this.redisTemplate = redisTemplate;
        this.userRedisCacheService = userRedisCacheService;
        this.terminalApiService = terminalApiService;
        this.validator = validator;
        this.resourceService = resourceService;
        this.companyTenantRelDao = companyTenantRelDao;
        this.tenantPolicyService = tenantPolicyService;
        this.userPubService = userPubService;
    }

    @Override
    public void destroy() {
        //region 关闭线程
//        this.threadPoolExecutor.shutdown();
        //endregion
    }

    public Page<User> page(Query query, Pageable pageable) {
        Page<User> page = this.page(query, pageable, EntityGraphs.named(User.NAMED_ENTITY_GRAPH_DEFAULT));
        return page;
    }

    public Page<User> page(Query query, Pageable pageable, EntityGraph entityGraph) {
        IAuthorizedUser authorizedUser = UserInfoHolder.get();
        this.buildCurrentQuery(query, authorizedUser);
        Specification<User> specification = UserQueryHelper.querySpecification(query);
        Page<User> result;
        if (entityGraph == null) {
            result = userDao.findAll(specification, pageable);
        } else {
            result = userDao.findAll(specification, pageable, entityGraph);
        }
        if (query.isDetail()) {
            for (User user : result) {
                this.fulfill(user, 0);
                this.fillOrgRoles(user, query.getOrgId());
            }
        }
        return result;
    }

    public Page<User> page(Specification<User> specification, Pageable pageable) {
        return userDao.findAll(specification, pageable);
    }

    public List<User> list(Query query, Sort sort) {
        IAuthorizedUser authorizedUser = UserInfoHolder.get();
        this.buildCurrentQuery(query, authorizedUser);
        Specification<User> specification = UserQueryHelper.querySpecification(query);
        List<User> list = userDao.findAll(specification, sort, EntityGraphs.named(User.NAMED_ENTITY_GRAPH_DEFAULT));
        return list;
    }

    public List<User> list(Specification<User> specification, Sort sort) {
        return userDao.findAll(specification, sort);
    }

    public Optional<User> findOne(Query query) {
        Specification<User> specification = UserQueryHelper.queryOneSpecification(query);
        return userDao.findOne(specification);
    }

    public long count(Query query) {
        Specification<User> specification = UserQueryHelper.querySpecification(query);
        return userDao.count(specification);
    }

    @Transactional(rollbackFor = Exception.class)
    public <S extends Save, U extends UserDto<O, R, A>, O extends OrgDto<O>, R extends RoleDto, A extends AccountDto> SaveUserOutput<U, O, R, A> save(S model, boolean isRoleOverwrite, boolean isOrgOverwrite, boolean isTagOverwrite, boolean isAppOverwrite, boolean isMergeAccount, boolean isStrict) {
        User entity = new User();
        BeanUtils.copyProperties(model, entity, Stream.of("account").toArray(String[]::new));
        long tenantId = model.getTenantId() != null ? model.getTenantId() : 0L;
        return this.save(tenantId, 0, Stream.of(model).collect(Collectors.toList()), isRoleOverwrite, isOrgOverwrite, isTagOverwrite, isAppOverwrite, isMergeAccount, isStrict);
    }

    @Transactional(rollbackFor = Exception.class)
    public <S extends Save, U extends UserDto<O, R, A>, O extends OrgDto<O>, R extends RoleDto, A extends AccountDto> SaveUserOutput<U, O, R, A> register(S model, String openId) {
        Object bindInfo = redisTemplate.opsForValue().get("BIND_WX_MP_" + openId);
        Long tenantId = 0L;
        long roleId;
        if (bindInfo != null) {
            String[] binds = bindInfo.toString().split("#");
            if (binds.length == 2) {
                tenantId = Long.parseLong(binds[0]);
                roleId = Long.parseLong(binds[1]);
                Set<Long> roles = new HashSet<>();
                roles.add(roleId);
                model.setRoleIds(roles);
            }

        } else {
            return null;
        }
        return this.save(tenantId, 0, Stream.of(model).collect(Collectors.toList()), false, false, false, false, false, false);
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateStatus(long userId, int status) {
        User existEntity = this.findById(userId);
        existEntity.setStatus(status);
        this.saveAndFlush(existEntity);

        userPubService.sendUserToPubsub(existEntity, existEntity.getTenantId());
    }

    /**
     * 构造查询Query对象
     *
     * @param query
     * @param authorizedUser
     */
    public void buildCurrentQuery(Query query, IAuthorizedUser authorizedUser) {
        if (authorizedUser == null) {
            return;
        }
        boolean tenantGradingEnabled = tenantPolicyService.tenantGradingManagementEnabled(authorizedUser.getTenantId());
        if (globalGradingEnabled && tenantGradingEnabled) {
            Triple<Boolean, Set<Long>, Set<Long>> adminAndGradingRoles = RoleUtils.calcAdminAndGradingRoles(authorizedUser);
            boolean isAdmin = adminAndGradingRoles.getLeft();
            Set<Long> gradingRoleIds = adminAndGradingRoles.getMiddle();
            Set<Long> orgRoleIds = adminAndGradingRoles.getRight();
            Set<String> filterParentIds = new HashSet<>();
            if (orgRoleIds != null && !orgRoleIds.isEmpty()) {
                Set<Long> orgIds = roleUserRelDao.findOrgIdsByTenantIdAndRoleIdsAndType(authorizedUser.getTenantId(), orgRoleIds);
                orgIds = orgIds.stream().filter(Objects::nonNull).collect(toSet());
                if (!orgIds.isEmpty()) {
                    filterParentIds = orgStructDao.findParentIdsByTenantIdAndOrgIds(authorizedUser.getTenantId(), orgIds);
                }
            }
            if (gradingRoleIds != null && !gradingRoleIds.isEmpty()) {
                List<String> gradingOrgParentIdsList = orgStructDao.listParentIdsByGradingRoleIds(gradingRoleIds);
                filterParentIds.addAll(gradingOrgParentIdsList);
            }
            if (filterParentIds != null && !filterParentIds.isEmpty()) {
                query.setFilterOrgParentIds(filterParentIds);
            } else if (!isAdmin) {
                throw new InvalidParameterException("非管理员身份, 返回空列表");
            }
        } else {
            query.setIncludeIndependents(false);
        }
    }


    @RedisLock(lockPrefix = "userCreate", key = "userName")
    public User saveWithLock(Save model) {
        Optional<User> entityOptional = userDao.findByTenantIdAndUserCode(model.getTenantId(), model.getUserCode());
        if (entityOptional.isPresent()) {
            logger.info("查询到用户直接返回");
            return entityOptional.get();
        }
        User entity = new User();
        BeanUtils.copyProperties(model, entity, Stream.of("account").toArray(String[]::new));
        if (entity.getUserPeriodTime() == null) {
            entity.setUserPeriodTime(new Date());
        }
        entity = this.saveAndFlush(entity);
        return entity;
    }

    @Transactional(rollbackFor = Exception.class)
    public ResponseEntity<List<BindOrg>> bindUserOrg(List<BindOrg> bindOrgs) {

        if (CollectionUtils.isEmpty(bindOrgs)) {
            return ResponseEntity.fail("0", "输入数据为空");
        }
        List<BindOrg> resultOrgs = new ArrayList<>(bindOrgs.size());
        for (BindOrg bindOrd : bindOrgs) {
            try {
                this.saveUserOrg(bindOrd);
                cachedRoleOrgUserRelsThreadLocal.remove();
                bindOrd.setResult(true);
            } catch (Exception e) {
                logger.warn(e.getMessage(), e);
                bindOrd.setMsg(e.getMessage());
                bindOrd.setResult(false);
            }

            resultOrgs.add(bindOrd);
        }

        return ResponseEntity.ok(resultOrgs);

    }

    private void saveUserOrg(BindOrg bindOrg) {
        String account = bindOrg.getAccount();
        AccountModel.Request.Login query = new AccountModel.Request.Login();
        if (account.contains(Separator.AT)) {
            query.setEmail(account);
        } else if (RegExUtil.checkMobile(account)) {
            query.setTelPhone(account);
        } else {
            query.setUsername(account);
        }

        Account accountDto = accountService.findOneByQuery(query);
        if (null == accountDto || accountDto.getAccountId() < 1) {
            throw new IllegalArgumentException("账号不存在");
        }
        Query userQuery = new Query();
        userQuery.setAccountId(accountDto.getAccountId());
        if (bindOrg.getTenantId() != null && bindOrg.getTenantId() > 0) {
            userQuery.setTenantId(bindOrg.getTenantId());
        }
        List<OrgStruct> orgs = orgStructDao.findByTenantIdAndOrgCode(bindOrg.getTenantId(), bindOrg.getOrgCode());
        OrgStruct org;
        if (orgs.isEmpty()) {
            //Evan 未判断，如果为空，则throw   机构不存在
            throw new IllegalArgumentException("组织不存在");
        }
        org = orgs.get(0);

        List<User> users = this.list(userQuery, Sort.unsorted());
        if (CollectionUtils.isEmpty(users)) {
            this.createUserAndBindOrg(accountDto, org);
            return;
        }
        for (User user : users) {
            if (!user.getTenantId().equals(org.getTenantId())) {
                String message = "用户(" + user.getId() + ")所在租户(" + user.getTenantId() + ")和组织(" + org.getOrgId() + ")所在租户(" + org.getTenantId() + ")不一致";
                throw new IllegalArgumentException(message);
            }
            this.bindUserIdAndOrgId(user, org.getOrgId());
        }

    }

    private void createUserAndBindOrg(Account accountDto, OrgStruct org) {
        User entity = new User();

        entity.setAccountId(accountDto.getAccountId());
        entity.setUserNumber(accountDto.getTelPhone());
        entity.setUserName(accountDto.getEmail());
        entity.setStatus(1);
        User user = this.saveAndFlush(entity);
        if (!user.getTenantId().equals(org.getTenantId())) {
            String message = "用户(" + user.getId() + ")所在租户(" + user.getTenantId() + ")和组织(" + org.getOrgId() + ")所在租户(" + org.getTenantId() + ")不一致";
            throw new IllegalArgumentException(message);
        }
        this.bindUserIdAndOrgId(user, org.getOrgId());

    }

    private void bindUserIdAndOrgId(User user, Long orgId) {

        OrgUserRel orgUserRel = new OrgUserRel();
        orgUserRel.setFullSelectedFlag(true);
        orgUserRel.setOrgStructId(orgId);
        orgUserRel.setUserId(user.getId());
        orgUserRel.setTenantId(user.getTenantId());
//        List<OrgUserRel> orgUserRels = orgUserRelDao.findByUserIdAndOrgIdAndOrgStructId(orgUserRel.getUserId(),orgUserRel.getOrgStructId(),orgUserRel.getTenantId());
//        if(CollectionUtils.isEmpty(orgUserRels)){
        //2022-04-11 cached改为false，否则绑定组织不走持久化流程
        this.bindUserOrgs(Stream.of(orgUserRel).collect(Collectors.toList()), false);
//        }
    }

    @Transactional(rollbackFor = Exception.class)
    public ResponseEntity<List<BindRole>> bindUsersAndRoles(List<BindRole> bindRoles) {

        List<BindRole> resultObj = new ArrayList<>(bindRoles.size());
        for (BindRole bindRole : bindRoles) {
            try {
                bindRoleAndAccount(bindRole.getAccount(), bindRole.getTenantId(), bindRole.getRoleCode());
                bindRole.setResult(true);
                resultObj.add(bindRole);
            } catch (Exception e) {
                logger.warn(e.getMessage(), e);
                bindRole.setResult(false);
                bindRole.setMsg(e.getMessage());
                resultObj.add(bindRole);
            }
        }
        return ResponseEntity.ok(resultObj);
    }

    private void bindRoleAndAccount(String account, Long tenantId, String roleCode) {
        AccountModel.Request.Login query = new AccountModel.Request.Login();
        if (account.contains(Separator.AT)) {
            query.setEmail(account);
        } else {
            query.setTelPhone(account);
        }
        Account accountDto = accountService.findOneByQuery(query);
        if (null == accountDto || accountDto.getAccountId() < 1) {
            throw new IllegalArgumentException("账号不存在");
        }
        Query userQuery = new Query();
        userQuery.setAccountId(accountDto.getAccountId());
        List<User> users = this.list(userQuery, Sort.unsorted());

        List<Role> roles = roleDao.findByTenantIdAndRoleCodes(tenantId, Stream.of(roleCode).collect(Collectors.toList()));

        if (CollectionUtils.isEmpty(roles) || roles.size() > 1) {
            throw new IllegalArgumentException("角色不存在或者为多个");
        }
        List<RoleUserRel> roleUserRels = new ArrayList<>(users.size());
        for (User user : users) {
            RoleUserRel roleUserRel = new RoleUserRel();
            roleUserRel.setTenantId(tenantId);
            roleUserRel.setRole(roles.get(0));
            roleUserRel.setUserId(user.getId());
            roleUserRel.setRoleId(roles.get(0).getId());
            Example<RoleUserRel> example = Example.of(roleUserRel);
            if (roleUserRelDao.exists(example)) {
                continue;
            }
            roleUserRels.add(roleUserRel);
        }
        if (!CollectionUtils.isEmpty(roleUserRels)) {
            roleUserRels.forEach(roleUserRelDao::saveAndFlush);
        }

    }

    @Transactional(rollbackFor = Exception.class)
    public List<BatchSaveVo> batchCreate(List<BatchSave> models) {

        List<BatchSaveVo> batchSaveVo = new ArrayList<>(models.size());
        for (BatchSave save : models) {
            try {
                User user = this.createOneUser(save);
                batchSaveVo.add(new BatchSaveVo(user.getId(), true, StringUtils.EMPTY, save.getUserNumber()));
            } catch (Exception e) {
                logger.warn(e.getMessage(), e);
                batchSaveVo.add(new BatchSaveVo(0L, false, e.getMessage(), save.getUserNumber()));
            }

        }
        return batchSaveVo;

    }

    private User createOneUser(BatchSave save) {
        AccountModel.Request.Login query = new AccountModel.Request.Login();
        query.setTelPhone(save.getUserNumber());
        Account account = accountService.findOneByQuery(query);
        if (account == null) {
            AccountModel.Request.Create create = new AccountModel.Request.Create();
            create.setPassword(save.getPassword());
            create.setStatus(1);
            create.setTelPhone(save.getUserNumber());
            create.setEmail(save.getUserEmailAddr());
            account = accountService.saveOriginPassword(create);
        }
        if (account == null) {
            throw new IllegalArgumentException("创建账号失败");
        }
        save.setAccountId(account.getAccountId());
        return saveWithLock(save);
    }


    /**
     * 批量保存
     *
     * @param saveUserInput
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public <S extends Save, U extends UserDto<O, R, A>, O extends OrgDto<O>, R extends RoleDto, A extends AccountDto> SaveUserOutput<U, O, R, A> save(SaveUserInput<S, O, U, R, A> saveUserInput) {
        Tenant tenant = null;
        if (saveUserInput.getTenantId() > 0) {
            tenant = tenantDao.findById(saveUserInput.getTenantId()).orElseThrow(() -> new IllegalArgumentException("非法的租户id(" + saveUserInput.getTenantId() + ")"));
        }
        SaveUserOutput<U, O, R, A> saveUserOutput = new SaveUserOutput<>(saveUserInput.getTenantId(), new HashMap<>());
        //如果为空则返回
        if (CollectionUtils.isEmpty(saveUserInput.getUsers())) {
            return saveUserOutput;
        }
        for (S userModel : saveUserInput.getUsers()) {
            //region 保存account
            AccountModel.Request.Save accountRequest;
            AccountType accountType = null;
            Account account = null;
            long accountId = 0;
            long userId = 0;
            User user = null;
            boolean isNew = false;
            if (userModel instanceof Create) {
                accountRequest = ((Create) userModel).getAccount();
                accountType = ((Create) userModel).getType();

            } else if (userModel instanceof Update) {
                accountRequest = ((Update) userModel).getAccount();
                userId = userModel.getUserId();
            } else {
                accountRequest = null;
            }
            if (accountRequest == null) {
                //region  直接从user里获取 userPhone, userEmailAddr 创建account
                logger.info("accountRequest == null");
                if (userModel.getAccountId() != null && userModel.getAccountId() > 0) {
                    accountId = userModel.getAccountId();
                }
                if (accountId == 0 && userId > 0) {
                    Optional<User> userOptional = userDao.findById(userId);
                    if (userOptional.isPresent()) {
                        user = userOptional.get();
                        if (saveUserInput.getTenantId() > 0) {
                            if (user.getTenantId() == null || saveUserInput.getTenantId() != user.getTenantId()) {
                                String message = "用户所在的租户(" + user.getTenantId() + ")和传入参数的租户(" + saveUserInput.getTenantId() + ")不一致";
                                logger.warn(message);
                                throw new IllegalArgumentException(message);
                            }
                        }
                        accountId = user.getAccountId();
                    } else {
                        String message = "不存在的userId(" + userId + ")";
                        throw new IllegalArgumentException(message);
                    }
                }
                if (user == null && userModel.getTenantId() != null && userModel.getTenantId() > 0 && StringUtils.isNotBlank(userModel.getUserCode())) {
                    //检查保存时，输入的租户Id和用户tenantId是否相等
                    this.checkSaveUserTenantIdEquals(saveUserInput.getTenantId(), userModel.getTenantId());
                    //这里User只部分信息
                    Optional<User> userOptional = userDao.findByTenantIdAndUserCode(userModel.getTenantId(), userModel.getUserCode());
                    if (userOptional.isPresent()) {
                        if (saveUserInput.isMergeAccount()) {
                            user = userOptional.get();
                            accountId = user.getAccountId();
                            if (tenant == null) {
                                tenant = user.getTenant();
                            }
                            isNew = false;
                        } else {
                            throw new IllegalArgumentException("重复的用户code(" + userModel.getUserCode() + ")");
                        }
                    }
                }
                if (accountId > 0) {
                    String message = "非法的账户id(" + accountId + ")";
                    account = accountDao.findById(accountId).orElseThrow(() -> new IllegalArgumentException(message));
                }
                if (account == null) {
                    //region 根据userModel中的信息创建account, 不推荐
                    account = accountService.create(tenant,
                            userModel.getUserPhone(),
                            userModel.getUserEmailAddr(),
                            userModel.getUserName(),
                            null,
                            userModel.getStatus(),
                            accountType,
                            true,
                            false,
                            saveUserInput.isMergeAccount(),
                            false,
                            null);
                    //endregion
                }

                if (account == null) {
                    String message = "无法关联帐号userPhone({" + userModel.getUserPhone() + "}), email({" + userModel.getUserEmailAddr() + "}), userName({" + userModel.getUserName() + "})";
                    logger.warn(message);
                    throw new IllegalArgumentException(message);
                } else {
                    accountId = account.getAccountId();
                }
                //endregion
            } else {
                //region 请求对象包含AccountRequest对象
                logger.info("accountRequest != null");
                if (accountRequest instanceof AccountModel.Request.Update) {
                    if (((AccountModel.Request.Update) accountRequest).getAccountId() != null && ((AccountModel.Request.Update) accountRequest).getAccountId() != 0) {
                        accountId = ((AccountModel.Request.Update) accountRequest).getAccountId();
                    }
                } else if (accountRequest instanceof AccountModel.Request.Create) {
                    //region 根据telphone或email查找是否有存在的account
                    if (accountId == 0) {
                        if (accountType == null) {
                            accountType = accountRequest.getType();
                        }
                        account = accountService.create(tenant,
                                accountRequest.getTelPhone(),
                                accountRequest.getEmail(),
                                accountRequest.getUsername(),
                                ((AccountModel.Request.Create) accountRequest).getPassword(),
                                userModel.getStatus(),
                                accountType,
                                accountRequest.isEnableSendMsg(),
                                ((AccountModel.Request.Create) accountRequest).isRandomPassword(),
                                saveUserInput.isMergeAccount(),
                                accountRequest.isChangePasswordFlag(),
                                accountRequest.getUpdateIgnoreProperties());
                        if (account == null) {
                            String message = "user: " + accountRequest.toString() + "无法关联帐号";
                            logger.warn(message);
                            throw new IllegalArgumentException(message);
                        }
                        accountId = account.getAccountId();
                    }
                    //endregion
                }
                //endregion
            }
            SaveUserContext<U, O, R, A> saveUserContext = new SaveUserContext<>();
            if (account != null) {
                saveUserContext.setAccount((A) account);
            }
            if (accountId > 0) {
                userModel.setAccountId(accountId);
            }
            //endregion 保存account

            //region 保存User
            if (user == null && userModel.getUserId() != null && userModel.getUserId() > 0) { // 如果已经存在userId则获取该user对象
                Optional<User> optionalUser = this.findUserById(userModel.getUserId());
                if (optionalUser.isPresent()) {
                    user = optionalUser.get();
                } else {
                    user = null;
                    userModel.setUserId(null);
                }
            }
            if (user == null && accountId > 0) {
                List<User> users = userDao.findByTenantIdAndAccountId(saveUserInput.getTenantId(), accountId);
                user = users.stream().findFirst().orElse(null);
            }

            if (user == null && StringUtils.isNotBlank(userModel.getUserCode())) {
                // 如果已经存在userCode则获取该user对象
                // 如果tenantId和userCode都存在, 去数据库查是否已经存在
                Optional<User> userOptional = userDao.findByTenantIdAndUserCode(saveUserInput.getTenantId(), userModel.getUserCode());
                if (userOptional.isPresent()) {
                    user = userOptional.get();
                }
            }

            if (user != null) {
                //region 如果存在user对象则更新
                if (StringUtils.isBlank(userModel.getUserCode())) {
                    userModel.setUserCode(null);
                } else {
                    //region 检查用户代码是否已经存在
                    this.validExistsByTenantIdAndUserCode(saveUserInput.getTenantId(), user.getId(), userModel.getUserCode());
                    //endregion
                }
                //删除userCode检查代码是否存在
                BeanUtils.copyProperties(userModel, user, Stream.of("account").toArray(String[]::new));
                //endregion
            } else {
                user = new User();
                isNew = true;
                BeanUtils.copyProperties(userModel, user, Stream.of("account").toArray(String[]::new));
                user.setTenantId(saveUserInput.getTenantId());
            }
            if (accountId > 0) {
                user.setAccountId(accountId);
            }
            if (userModel.getStatus() != null) {
                user.setStatus(userModel.getStatus());
                user.setActiveStatus(userModel.getStatus());
            }
            if (StringUtils.isBlank(user.getUserEmailAddr())) {
                if (account != null && StringUtils.isNotBlank(account.getEmail())) {
                    user.setUserEmailAddr(account.getEmail());
                }
            }
            if (StringUtils.isBlank(user.getUserPhone())) {
                if (account != null && StringUtils.isNotBlank(account.getTelPhone())) {
                    user.setUserPhone(account.getTelPhone());
                }
            }
            if (tenant == null) {
                tenant = user.getTenant();
            }
            user = this.saveAndFlush(user);
            if (tenant != null) {
                user.setTenant(tenant);
            }
            if (account != null) {
                user.setAccount(account);
                saveUserContext.setAccount((A) account);
            }
            this.saveUserTags(user.getId(), userModel, saveUserInput.isTagOverwrite());
            //endregion

            //region 保存 OrgUserRel
            if ((saveUserInput.getOrgStruct() != null && saveUserInput.getOrgStruct().getOrgId() > 0) || userModel.getOrgIds() != null || userModel.getOrgCodes() != null) {
                Set<Long> filteredOrgIds = new HashSet<>();
                Set<Long> tempOrgIds = new HashSet<>();
                if (saveUserInput.getOrgStruct() != null && saveUserInput.getOrgStruct().getOrgId() > 0) { // 如果orgId > 0 直接绑定用户和组织
                    tempOrgIds.add(saveUserInput.getOrgStruct().getOrgId());
                }
                if (!CollectionUtils.isEmpty(userModel.getOrgIds())) {
                    tempOrgIds.addAll(userModel.getOrgIds());
                }
                if (!CollectionUtils.isEmpty(tempOrgIds)) {
                    List<Long> orgIds = orgStructDao.findOrgIdsByTenantIdAndOrgIds(saveUserInput.getTenantId(), tempOrgIds);
                    filteredOrgIds.addAll(orgIds);
                }
                if (!CollectionUtils.isEmpty(userModel.getOrgCodes())) {
                    // 如果orgId == 0 直接根据orgCode绑定用户和组织
                    Set<String> orgCodes = userModel.getOrgCodes();
                    List<OrgStruct> orgs = new ArrayList<>();
                    if (CollectionUtils.isEmpty(orgCodes)) {
                        //region 如果orgCodes为空且是新建 则 创建根组织且绑定到该组织节点
                        if (isNew) {
                            // 是新建用户
                            orgs = orgStructDao.findRootsByTenantId(saveUserInput.getTenantId());
                        }
                        //endregion
                    } else {
                        orgs = orgStructDao.findByTenantIdAndOrgCodes(saveUserInput.getTenantId(), orgCodes);
                        if (orgCodes.size() != orgs.size()) {
                            List<OrgStruct> finalOrgs = orgs;
                            Set<String> diffOrgCodes = orgCodes.stream().filter(orgCode -> finalOrgs.stream().map(OrgStruct::getOrgCode).noneMatch(finalOrgCode -> finalOrgCode.equals(orgCode))).collect(Collectors.toSet());
                            String message = "User(" + user.getUserCode() + ")包含含有数据库中不存在的组织代码(" + String.join(",", diffOrgCodes) + ")";
                            logger.warn(message);
                            if (saveUserInput.isStrict()) {
                                throw new IllegalArgumentException(message);
                            }
                        }
                    }
                    if (!orgs.isEmpty()) {
                        Set<Long> orgIdSet = orgs.stream().map(OrgStruct::getOrgId).collect(Collectors.toSet());
                        if (logger.isInfoEnabled()) {
                            logger.info("orgIdSet = " + orgIdSet.stream().map(Object::toString).collect(Collectors.joining(",")));
                        }
                        filteredOrgIds.addAll(orgIdSet);
                    }
                }
                saveUserContext.addOrgIds(filteredOrgIds);
            }
            // endregion 保存 OrgUserRel

            //region 保存 RoleUserRel
            if (userModel.getRoleIds() != null || userModel.getRoleCodes() != null) {
                Set<Long> roleIds = new HashSet<>();
                List<Role> roles;
                if (!CollectionUtils.isEmpty(userModel.getRoleIds())) {
                    Set<Long> roleIdSet = userModel.getRoleIds().stream().filter(roleId -> roleId != null && roleId > 0).collect(Collectors.toSet());
                    if (logger.isInfoEnabled()) {
                        logger.info("roleIdSet = " + roleIdSet.stream().map(Object::toString).collect(Collectors.joining(",")));
                    }

                    roles = roleDao.findAllById(roleIdSet);
                    //region 过滤不合法的角色id
                    roleIdSet = roles.stream().filter(role -> (role.getTenantId().equals(saveUserInput.getTenantId()) || role.getTenantId() == -1)).map(Role::getId).collect(Collectors.toSet());
                    if (logger.isDebugEnabled()) {
                        logger.debug("roleIdSet = " + roleIdSet.stream().map(Object::toString).collect(Collectors.joining(",")));
                    }
                    roleIds.addAll(roleIdSet);
                    //endregion
                    Set<Role> copingRoleTemplates = roles.stream().filter(role -> role.getTenantId() == 0).collect(Collectors.toSet());
                    Set<Role> copiedRoles = roleService.copyRoleTempates(saveUserInput.getTenantId(), copingRoleTemplates);

                    if (!CollectionUtils.isEmpty(copiedRoles)) {
                        Set<Long> copiedRoleIds = copiedRoles.stream().map(Role::getId).collect(Collectors.toSet());
                        roleIds.addAll(copiedRoleIds);
                    }
                }
                Set<String> roleCodes = userModel.getRoleCodes();
                if (!CollectionUtils.isEmpty(roleCodes)) {
                    roles = roleDao.findByTenantIdAndRoleCodes(saveUserInput.getTenantId(), roleCodes);
                    if (roleCodes.size() != roles.size()) {
                        String message = "User(" + user.getUserCode() + ")包含含有数据库中不存在的角色代码";
                        logger.warn(message);
                        if (saveUserInput.isStrict()) {
                            throw new IllegalArgumentException(message);
                        }
                    }
                    if (!roles.isEmpty()) {
                        Set<Long> roleIdSet = roles.stream().map(Role::getId).collect(Collectors.toSet());
                        if (logger.isDebugEnabled()) {
                            logger.debug("roleIdSet = " + roleIdSet.stream().map(Object::toString).collect(Collectors.joining(",")));
                        }
                        roleIds.addAll(roleIdSet);
                    }
                }
                saveUserContext.addRoleIds(roleIds);
            }
            //region 如果用户类型是管理员, 则增加角色id = 1的角色
            if (null != userModel.getUserType() && userModel.getUserType() == 0) {
                saveUserContext.addRoleId(1L);
            }
            //endregion
            //endregion 保存 RoleUserRel

            //region 保存 UserApp
            if (userModel.getAppIds() != null) {
                Set<Long> appIds = new HashSet<>();
                List<App> apps;
                if (!userModel.getAppIds().isEmpty()) {
                    Set<Long> appIdSet = userModel.getAppIds().stream().filter(appId -> appId != null && appId > 0).collect(Collectors.toSet());
                    if (logger.isInfoEnabled()) {
                        logger.info("appIdSet = " + appIdSet.stream().map(Object::toString).collect(Collectors.joining(",")));
                    }

                    apps = appDao.findAllById(appIdSet);
                    appIdSet = apps.stream().map(App::getAppId).collect(Collectors.toSet());
                    if (logger.isInfoEnabled()) {
                        logger.info("appIdSet = " + appIdSet.stream().map(Object::toString).collect(Collectors.joining(",")));
                    }
                    if (!appIdSet.isEmpty()) {
                        appIds.addAll(appIdSet);
                    }
                }
                saveUserContext.addAppIds(appIds);
            }
            // endregion

            //region User 绑定 Org
            this.bindOrgs(user, saveUserContext.getOrgIds(), saveUserInput.getModules(), saveUserInput.isOrgOverwrite(), saveUserInput.isStrict(), true);
            //endregion User 绑定 Org

            //region User 绑定 Role
            this.bindRoles(user, null, saveUserContext.getRoleIds(), null, null, null, saveUserInput.isRoleOverwrite(), saveUserInput.isStrict(), true);
            //endregion User 绑定 Role

            //region User 绑定 App
            this.bindApps(user, saveUserContext.getAppIds(), saveUserInput.isAppOverwrite(), saveUserInput.isStrict());
            //endregion User 绑定 App

            this.commitBindUserOrgs();
            this.commitBindUserRoles();

            if (tenant != null) {
                user.setTenant(tenant);
            }
            saveUserContext.setUser((U) user);
            saveUserOutput.putSaveUserContext(user.getId(), saveUserContext);
        }

        return saveUserOutput;
    }

    /**
     * 检查保存用户时租户是否相等
     *
     * @param sourceTenantId 传入参数的租户(源)
     * @param targetTenantId 用户所在的租户(目标)
     */
    private void checkSaveUserTenantIdEquals(Long sourceTenantId, Long targetTenantId) {
        if (sourceTenantId > 0 && (targetTenantId == null || !sourceTenantId.equals(targetTenantId))) {
            String message = "用户所在的租户(" + targetTenantId + ")和传入参数的租户(" + sourceTenantId + ")不一致";
            logger.warn(message);
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * 查询ID
     *
     * @param tenantId 租户Id
     * @param userCode 用户Code
     * @param marge    是否合并
     * @return Optional<User> user
     */
    public Optional<User> findUserByTenantIdAndUserCode(Long tenantId, String userCode, Boolean marge, EntityGraph entityGraph) {
        Query query = new Query();
        query.setTenantId(tenantId);
        query.setUserCode(userCode);
        List<User> users;
        if (entityGraph == null) {
            users = userDao.findAll(UserQueryHelper.querySpecification(query));
        } else {
            users = userDao.findAll(UserQueryHelper.querySpecification(query), entityGraph);
        }

        if (CollectionUtils.isEmpty(users)) {
            return Optional.empty();
        }
        //判断如果合并存在两条以上则报错
        if (marge && users.size() > 1) {
            logger.error("合并的用户存在两条以上记录tenantId:{},userCode:{},marge:{}:size:{}", tenantId, userCode, marge);
            throw new IllegalStateException("合并的用户存在两条以上记录 userCode:" + userCode);
        }
        User user = this.findOneValidUser(users);
        return Optional.of(user);
    }


    public User findUserByTenantIdAndAccountId(Long tenantId, Long accountId) {
        Query query = new Query();
        query.setAccountId(accountId);
        query.setTenantId(tenantId);
        query.setStatus(1);

        List<User> userList = userDao.findAll(UserQueryHelper.querySpecification(query), EntityGraphs.named(User.NAMED_ENTITY_GRAPH_DEFAULT));
        if (!CollectionUtils.isEmpty(userList)) {
            return userList.get(0);
        }
        return null;
    }

    /**
     * 查询所有，不传status
     *
     * @param tenantId
     * @param accountId
     * @return
     */

    public User findUserByTenantIdAndAccountIdWithoutStatus(Long tenantId, Long accountId) {
        Query query = new Query();
        query.setAccountId(accountId);
        query.setTenantId(tenantId);
        List<User> userList = userDao.findAll(UserQueryHelper.querySpecification(query), EntityGraphs.named(User.NAMED_ENTITY_GRAPH_DEFAULT));
        if (!CollectionUtils.isEmpty(userList)) {
            return userList.get(0);
        }
        return null;
    }

    /**
     * 检查是否已经存在
     *
     * @param tenantId 租户Id
     * @param userId   用户Id
     * @param userCode 用户代码
     */

    public void validExistsByTenantIdAndUserCode(long tenantId, Long userId, String userCode) {
        logger.info("validExistsByTenantIdAndUserCode-tenantId:{},userCode:{}", tenantId, userCode);
        //用户代码不能为空
        Assert.hasText(userCode, "用户代码不能为空");
        List<Long> ids = userDao.selectIdsByTenantIdAndUserCode(tenantId, userCode);
        if (!CollectionUtils.isEmpty(ids)) {
            if (userId == null || !ids.contains(userId)) {
                String message = "用户代码(" + userCode + ")冲突";
                throw new IllegalArgumentException(message);
            }
        }
    }

    /**
     * 批量保存
     *
     * @param tenantId
     * @param orgStruct
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public <S extends Save, U extends UserDto<O, R, A>, O extends OrgDto<O>, R extends RoleDto, A extends AccountDto> SaveUserOutput<U, O, R, A> save(long tenantId, OrgStruct orgStruct, S userSaveModel, boolean isRoleOverwrite, boolean isOrgOverwrite, boolean isTagOverwrite, boolean isAppOverwrite, boolean isMergeAccount, boolean isStrict) {
        if (!tenantDao.existsById(tenantId)) {
            throw new IllegalArgumentException("不合法的租户id(" + tenantId + ")");
        }
        List<S> models = Stream.of(userSaveModel).collect(Collectors.toList());
        List<OrgUserRel> orgUserRels = new ArrayList<>();
        List<RoleUserRel> roleUserRels = new ArrayList<>();
        SaveUserInput<S, O, U, R, A> saveUserInput = new SaveUserInput(tenantId, orgStruct, models, orgUserRels, roleUserRels, userSaveModel.getModules(), isRoleOverwrite, isOrgOverwrite, isAppOverwrite, isTagOverwrite, isMergeAccount, isStrict);
        //region 保存 User 核心方法
        SaveUserOutput<U, O, R, A> saveUserOutput = this.save(saveUserInput);
        //endregion 保存 User

        return saveUserOutput;
    }

    /**
     * 批量保存
     *
     * @param tenantId
     * @param orgId
     * @param models
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public <S extends Save, U extends UserDto<O, R, A>, O extends OrgDto<O>, R extends RoleDto, A extends AccountDto> SaveUserOutput<U, O, R, A> save(long tenantId, long orgId, List<S> models, boolean isRoleOverwrite, boolean isOrgOverwrite, boolean isTagOverwrite, boolean isAppOverwrite, boolean isMergeAccount, boolean isStrict) {
        if (CollectionUtils.isEmpty(models)) {
            throw new IllegalArgumentException("没有有效报文");
        }
        if (tenantId > 0 && !tenantDao.existsById(tenantId)) {
            throw new IllegalArgumentException("不合法的租户id(" + tenantId + ")");
        }

        Optional<OrgStruct> orgStructOptional;
        OrgStruct orgStruct = null;
        if (orgId > 0) {
            if (tenantId > 0) {
                OrgModel.Request.Query query = new OrgModel.Request.Query();
                query.setTenantId(tenantId);
                query.setOrgId(orgId);
                orgStructOptional = orgStructDao.findOne(OrgQueryHelper.querySpecification(query), EntityGraphs.named(OrgStruct.NAMED_ENTITY_GRAPH_DEFAULT));
            } else {
                orgStructOptional = orgStructDao.findById(orgId, EntityGraphs.named(OrgStruct.NAMED_ENTITY_GRAPH_DEFAULT));
            }
            if (orgStructOptional.isPresent()) {
                orgStruct = orgStructOptional.get();
            } else {
                String message = "不合法的组织id(" + orgId + ")和租户id(" + tenantId + ")";
                logger.warn(message);
                throw new IllegalArgumentException(message);
            }
        }
        List<OrgUserRel> orgUserRels = new ArrayList<>();
        List<RoleUserRel> roleUserRels = new ArrayList<>();
        SaveUserInput<S, O, U, R, A> saveUserInput = new SaveUserInput(tenantId, orgStruct, models, orgUserRels, roleUserRels, null, isRoleOverwrite, isOrgOverwrite, isTagOverwrite, isAppOverwrite, isMergeAccount, isStrict);
        SaveUserOutput<U, O, R, A> saveUserOutput = this.save(saveUserInput);
        return saveUserOutput;
    }


    @Transactional(rollbackFor = Exception.class)
    public User update(long userId, Save model) {
        User existEntity = this.findById(userId);
        if (StringUtils.isBlank(model.getUserCode())) {
            model.setUserCode(null);
        }
        BeanUtils.copyProperties(model, existEntity, Stream.of("account").toArray(String[]::new));
        User user = this.saveAndFlush(existEntity);
        this.saveUserTags(user.getId(), model, true);
        this.bindApps(user, model.getAppIds(), true, false);
        this.bindOrgs(user.getTenantId(), user.getId(), model.getOrgIds(), model.getModules(), true, false, false);
        return user;
    }

    public void saveUserTags(long userId, Save model, boolean isOverwrite) {
        this.saveUserTags(userId, model.getInvoiceType(), model.getPrintingEquipment(), model.getTicketOpeningTerminal(), model.getBusinessExtensionAttribute(), isOverwrite);
    }

    public void saveUserTags(long userId, String invoiceType, String printingEquipment, String ticketOpeningTerminal, Object businessExtensionAttribute, boolean isOverwrite) {
        UserTag userTagTicketOpeningTerminal = null;
        UserTag userTagPrintingEquipment = null;
        UserTag userTagInvoiceType = null;
        UserTag userTagBusinessExtensionAttribute = null;
        List<UserTag> userTags = userTagDao.findByUserId(userId);
        for (UserTag userTag : userTags) {
            userTag.setUser(null);
            if ("ticketOpeningTerminal".equals(userTag.getTagName())) {
                userTagTicketOpeningTerminal = userTag;
                continue;
            }
            if ("printingEquipment".equals(userTag.getTagName())) {
                userTagPrintingEquipment = userTag;
                continue;
            }
            if ("invoiceType".equals(userTag.getTagName())) {
                userTagInvoiceType = userTag;
                continue;
            }
            if ("businessExtensionAttribute".equals(userTag.getTagName())) {
                userTagBusinessExtensionAttribute = userTag;
            }
        }
        if (userTagTicketOpeningTerminal == null) {
            userTagTicketOpeningTerminal = new UserTag();
            userTagTicketOpeningTerminal.setUserId(userId);
            userTagTicketOpeningTerminal.setTagName("ticketOpeningTerminal");
        }
        if (userTagPrintingEquipment == null) {
            userTagPrintingEquipment = new UserTag();
            userTagPrintingEquipment.setUserId(userId);
            userTagPrintingEquipment.setTagName("printingEquipment");
        }
        if (userTagInvoiceType == null) {
            userTagInvoiceType = new UserTag();
            userTagInvoiceType.setUserId(userId);
            userTagInvoiceType.setTagName("invoiceType");
        }
        if (userTagBusinessExtensionAttribute == null) {
            userTagBusinessExtensionAttribute = new UserTag();
            userTagBusinessExtensionAttribute.setUserId(userId);
            userTagBusinessExtensionAttribute.setTagName("businessExtensionAttribute");
        }
        List<UserTag> saveTags = new ArrayList<>();
        //String ticketOpeningTerminal = model.getTicketOpeningTerminal();
        if (StringUtils.isBlank(ticketOpeningTerminal) || "null".equals(ticketOpeningTerminal) || "[]".equals(ticketOpeningTerminal) || "{}".equals(ticketOpeningTerminal)) {
            if (isOverwrite) {
                userTagTicketOpeningTerminal.setTagValue(StringUtils.EMPTY);
                saveTags.add(userTagTicketOpeningTerminal);
            }
        } else {
            userTagTicketOpeningTerminal.setTagValue(ticketOpeningTerminal);
            saveTags.add(userTagTicketOpeningTerminal);
        }


        //String printingEquipment = model.getPrintingEquipment();
        if (StringUtils.isBlank(printingEquipment) || "null".equals(printingEquipment) || "[]".equals(printingEquipment) || "{}".equals(printingEquipment) || "\"\"".equals(printingEquipment)) {
            if (isOverwrite) {
                userTagPrintingEquipment.setTagValue(StringUtils.EMPTY);
                saveTags.add(userTagPrintingEquipment);
            }
        } else {
            userTagPrintingEquipment.setTagValue(printingEquipment);
            saveTags.add(userTagPrintingEquipment);
        }

        //String invoiceType = model.getInvoiceType();
        if (StringUtils.isBlank(invoiceType) || "null".equals(invoiceType) || "[]".equals(invoiceType) || "{}".equals(invoiceType) || "\"\"".equals(invoiceType)) {
            if (isOverwrite) {
                userTagInvoiceType.setTagValue(StringUtils.EMPTY);
                saveTags.add(userTagInvoiceType);
            }
        } else {
            userTagInvoiceType.setTagValue(invoiceType);
            saveTags.add(userTagInvoiceType);
        }

        //Object businessExtensionAttribute = businessExt;
        try {
            String businessExtensionAttributeJson = JsonUtils.toJson(businessExtensionAttribute);
            if (StringUtils.isBlank(businessExtensionAttributeJson) || "null".equals(businessExtensionAttributeJson) || "[]".equals(businessExtensionAttributeJson) || "{}".equals(businessExtensionAttributeJson) || "\"\"".equals(businessExtensionAttributeJson)) {
                if (isOverwrite) {
                    userTagBusinessExtensionAttribute.setTagValue(StringUtils.EMPTY);
                    saveTags.add(userTagBusinessExtensionAttribute);
                }
            } else {
                userTagBusinessExtensionAttribute.setTagValue(businessExtensionAttributeJson);
                saveTags.add(userTagBusinessExtensionAttribute);
            }
        } catch (Exception e) {
            logger.warn(e.getMessage(), e);
            if (isOverwrite) {
                userTagBusinessExtensionAttribute.setTagValue(StringUtils.EMPTY);
                saveTags.add(userTagBusinessExtensionAttribute);
            }
        }
        saveTags.forEach(userTagDao::saveAndFlush);
    }

    public User findById(Long userId) {
        return this.findById(userId, 0);
    }

    public User findById(Long userId, int extraInfoDimension) {
        String message = "未找到用户实体(" + userId + ")";
        User user = userDao.findById(userId, EntityGraphs.named(User.NAMED_ENTITY_GRAPH_DEFAULT)).orElseThrow(() -> new IllegalArgumentException(message));
        this.fulfill(user, extraInfoDimension);
        this.setUserTags(user);
        return user;
    }

    public User findByLoginId(long userId, String loginId, String modules, int extraInfoDimension) {
        logger.info("findByLoginId,userId:{},loginId:{},extraInfoDimension:{}", userId, loginId, extraInfoDimension);
        return this.findByTenantIdAndUserIdAndLoginId(null, userId, loginId, modules, extraInfoDimension);
    }

    public User findByTenantIdAndUserId(long tenantId, long userId, String modules, Integer extraInfoDimension) {
        return this.findByTenantIdAndUserIdAndLoginId(tenantId, userId, null, modules, extraInfoDimension);
    }

    public User findByTenantIdAndUserIdAndLoginId(Long tenantId, long userId, String loginId, String modules, Integer extraInfoDimension) {
        logger.info("findByTenantIdAndUserIdAndLoginId(tenantId:{},userId:{},loginId:{},modules:{},extraInfoDimension:{})", tenantId, userId, loginId, modules, extraInfoDimension);
        Optional<User> optionalUser = userRedisCacheService.getUserByUserIdAndLoginId(userId, loginId);
        User user = null;
        //如果缓存中没有数据，则直接查询数据
        boolean caching = false;
        if (optionalUser.isPresent()) {
            user = optionalUser.get();
            if (tenantId != null && tenantId > 0 && !tenantId.equals(user.getTenantId())) {
                throw new IllegalArgumentException("未找到用户实体(tenantId:" + tenantId + ", userId:" + userId + ")");
            }
            //为重复利用原查询数据的资源
            if (extraInfoDimension != null && extraInfoDimension > 0 && ((user.getCachedExtraInfoDimension() & extraInfoDimension) != extraInfoDimension)) {
                extraInfoDimension = extraInfoDimension | user.getCachedExtraInfoDimension();
                caching = true;
            }
        } else {
            if (tenantId != null && tenantId > 0) {
                String message = "未找到用户实体(tenantId:" + tenantId + ", userId:" + userId + ")";
                Query query = new Query();
                query.setUserId(userId);
                query.setTenantId(tenantId);
                user = userDao.findOne(UserQueryHelper.querySpecification(query), EntityGraphs.named(User.NAMED_ENTITY_GRAPH_DEFAULT))
                        .orElseThrow(() -> new IllegalArgumentException(message));
            } else {
                String message = "未找到用户实体(userId:" + userId + ")";
                user = userDao.findById(userId, EntityGraphs.named(User.NAMED_ENTITY_GRAPH_DEFAULT)).orElseThrow(() -> new IllegalArgumentException(message));
            }
            caching = true;
        }
        if (caching) {
            user.setModules(modules);
            user.setCachedExtraInfoDimension(extraInfoDimension);
            this.fulfill(user, extraInfoDimension);
            //再次放入到缓存中
            this.userRedisCacheService.pubUserByUserIdAndLoginId(user, userId, loginId);
        }
        //userTags 需要实时生效，目前因为修改后没有缓存管理逻辑
        this.setUserTags(user);
        return user;
    }

    public User login(AccountModel.Request.Login login, int extraInfoDimension) {
        Account account = accountService.findOneByLogin(login);
        accountService.validPassword(login, account);
        List<User> users = userDao.findByAccountId(account.getAccountId());
        //判断用户是否为空，则throw 异常信息
        if (users.isEmpty()) {
            String message = "未找到用户实体(accountId:" + account.getAccountId() + ")";
            throw new IllegalArgumentException(message);
        }
        //先过滤注销状态用户，再判断注销后的数据是否为空
        users = users.stream()
                .filter(e -> (e.getStatus() != 0 && (e.getExpiredDate() == null || DateUtils.afterNow(e.getExpiredDate(), true))))
                .collect(Collectors.toList());
        //如未找到有效状态的用户实体，则throw  IllegalArgumentException
        if (CollectionUtils.isEmpty(users)) {
            String message = "未找到有效状态的用户实体(accountId:" + account.getAccountId() + ")";
            throw new IllegalArgumentException(message);
        }
        Set<Long> tenantIds = users.stream().map(User::getTenantId).collect(toSet());

        Boolean changePwd = this.needModifyPassword(account.getPwdLastUpdateTime());
        //数据库值为1 || 时间pwd_last_update_time到了 就要改密码
        account.setChangePasswordFlag(changePwd || account.getChangePasswordFlag());

        User user = users.get(0);
        user.setTenantIds(tenantIds);
        user.setAccount(account);
        this.fulfill(user, extraInfoDimension);
        user.setUsername(account.getUsername());
        user.setEmail(account.getEmail());
        user.setMobile(account.getTelPhone());
        return user;
    }

    public User findByUsername(String username, int extraInfoDimension) {
        Account account = accountService.findOneByUsername(username);
        Query query = new Query();
        query.setAccountId(account.getAccountId());
        Set<Long> tenantIds = null;
        List<User> users = this.list(query, Sort.unsorted());
        if (users.isEmpty()) {
            String message = "未找到用户实体(accountId:" + account.getAccountId() + ")";
            throw new IllegalArgumentException(message);
        } else {
            tenantIds = users.stream().map(User::getTenantId).collect(toSet());
        }
        Optional<User> optionalUser = users.stream().filter(u -> 1 == u.getStatus()).findFirst();
        User user = users.get(0);
        if (optionalUser.isPresent()) {
            user = optionalUser.get();
            user.setTenantIds(tenantIds);
        }
        user.setAccount(account);
        this.fulfill(user, extraInfoDimension);
        this.setUserTags(user);
        return user;
    }

    public User findByTenantCodeAndUsername(String tenantCode, String username, int extraInfoDimension) {
        Long tenantId = tenantDao.findTenantIdByTenantCode(tenantCode);
        if (tenantId == null) {
            String message = "未找到租户实体(" + tenantCode + ")";
            throw new IllegalArgumentException(message);
        }
        List<Account> accounts = accountService.findByTenantCodeAndUsername(tenantCode, username);
        Set<Long> accountIds = accounts.stream().map(Account::getAccountId).collect(Collectors.toSet());
        Query query = new Query();
        query.setAccountIds(accountIds);
        query.setTenantId(tenantId);
        List<User> users = this.list(query, Sort.unsorted());
        if (users.isEmpty()) {
            String message = "未找到用户实体(tenantCode: " + tenantCode + ", username:" + username + ")";
            throw new IllegalArgumentException(message);
        }

        User user = this.findOneValidUser(users);

        this.fillAccount(user, accounts);
        this.fulfill(user, extraInfoDimension);
        this.setUserTags(user);
        return user;
    }

    private void fillAccount(User user, Collection<Account> accounts) {
        if (user == null || user.getAccountId() == null || accounts == null || accounts.isEmpty()) {
            return;
        }
        if (user.getAccount() != null && user.getAccount().getCreateTime() != null) {
            return;
        }
        Account account = accounts.stream()
                .filter(Objects::nonNull)
                .filter(a -> user.getAccountId().equals(a.getAccountId()))
                .findFirst().orElse(null);
        user.setAccount(account);
    }

    /**
     * 补充User对象的扩展信息
     *
     * @param user               user对象
     * @param extraInfoDimension 扩展信息纬度
     * @return
     */
    private User fulfill(User user, Integer extraInfoDimension) {
        if ((user.getAccount() == null || user.getAccount().getCreateTime() == null) && user.getAccountId() != null) {
            Optional<Account> accountOptional = accountDao.findById(user.getAccountId());
            if (accountOptional.isPresent()) {
                user.setAccount(accountOptional.get());
            } else {
                throw new IllegalArgumentException("不存在id(" + user.getAccountId() + ")的Account");
            }
        }
        user.setEmail(user.getAccount().getEmail());
        user.setMobile(user.getAccount().getTelPhone());
        if (user.getTenantId() != null && user.getTenantId() > 0 && (user.getTenant() == null || user.getTenant().getCreateTime() == null)) {
            Optional<Tenant> tenantOptional = tenantDao.findById(user.getTenantId());
            if (tenantOptional.isPresent()) {
                Tenant tenant = tenantOptional.get();
                user.setTenant(tenant);
                if (StringUtils.isBlank(user.getTenantName()) || StringUtils.isBlank(user.getTenantCode())) {
                    user.setTenantName(tenant.getTenantName());
                    user.setTenantCode(tenant.getTenantCode());
                }
            }

        }
        if (extraInfoDimension == null) {
            return user;
        }
        //region 获取角色
        Set<Role> roles;
        if (user.getRoles() == null) {
            roles = new HashSet<>(roleService.listByUserId(user.getId()));
            user.setRoles(roles);
        } else {
            roles = user.getRoles();
        }

        if (extraInfoDimension <= 0) {
            return user;
        }
        //endregion

        //region 获取关联公司集合
        if (BinaryUtils.is(ExtraInfo.relatedCompanies, extraInfoDimension)) {
            List<OrgStruct> relatedCompanyOrgs = orgStructDao.findRelatedCompaniesByUserId(user.getId());
            user.setRelatedCompanies(relatedCompanyOrgs.stream().collect(Collectors.toSet()));
        }
        //endregion

        if (BinaryUtils.hasAny(Stream.of(ExtraInfo.currentOrgs, ExtraInfo.orgs, ExtraInfo.companies, ExtraInfo.resources, ExtraInfo.packages, ExtraInfo.parentOrgs, ExtraInfo.parentCompanies, ExtraInfo.resourceDetail).collect(Collectors.toList()), extraInfoDimension)) {
            final List<OrgStruct> currentOrgs = orgService.listByTenantIdAndUserId(user.getTenantId(), user.getId(), user.getModules(), Stream.of("orgId", "companyId", "gradingRoleId", "defaultOrgRoleId", "orgCode", "orgName", "parentId", "parentIds", "orgType", "orgBizType", "status", "company").collect(toSet()));

            if (BinaryUtils.hasAny(Stream.of(ExtraInfo.currentOrgs, ExtraInfo.orgs).collect(Collectors.toList()), extraInfoDimension)) {
                user.setCurrentOrgs(new HashSet<>(currentOrgs));
            }

            Set<OrgStruct> parentOrgs = new HashSet<>();
            if (BinaryUtils.hasAny(Stream.of(ExtraInfo.resources, ExtraInfo.packages, ExtraInfo.parentOrgs, ExtraInfo.parentCompanies, ExtraInfo.companies, ExtraInfo.resourceDetail).collect(Collectors.toList()), extraInfoDimension)) {
                Set<Long> parentOrgIds = new HashSet<>();
                for (OrgStruct currentOrg : currentOrgs) {
                    parentOrgs.add(currentOrg);
                    Set<Long> currentParentOrgIds = OrgUtils.findOrgIdInParentIds(currentOrg.getParentIds());
//                    parentOrgIds = parentOrgIds.stream().filter(id -> !id.equals(currentOrg.getOrgId())).collect(toSet());
                    parentOrgIds.addAll(currentParentOrgIds);
                }
                parentOrgIds = parentOrgIds.stream().filter(Objects::nonNull).filter(orgId -> currentOrgs
                        .stream().map(OrgStruct::getOrgId).noneMatch(orgId::equals)).collect(Collectors.toSet());
                //region 根据parentOrgId构造parentOrg
                if (!parentOrgIds.isEmpty()) {
                    Set<Long> orgIds = new HashSet<>();
                    for (Long parentOrgId : parentOrgIds) {
                        OrgStruct parentOrg = parentOrgs.stream().filter(o -> o.getOrgId().equals(parentOrgId)).findAny().orElse(null);
                        if (parentOrg != null) {
                            logger.debug("this org exist in parentOrgs");
                            continue;
                        }
                        orgIds.add(parentOrgId);
                        Optional<OrgStruct> parentOrgOptional = orgStructDao.findById(parentOrgId, EntityGraphs.named(OrgStruct.NAMED_ENTITY_GRAPH_DEFAULT));
                        if (parentOrgOptional.isPresent()) {
                            parentOrgs.add(parentOrgOptional.get());
                        }
                    }
                    if (!orgIds.isEmpty()) {
                        OrgModel.Request.Query query = new OrgModel.Request.Query();
                        query.setIds(orgIds);
                        query.setAttributes(Stream.of("orgId", "companyId", "gradingRoleId", "defaultOrgRoleId", "orgCode", "orgName", "orgType", "company").collect(toSet()));
                        List<OrgStruct> orgs = orgStructDao.findAttributes(query, Sort.unsorted());
                        if (!orgs.isEmpty()) {
                            parentOrgs.addAll(orgs);
                        }
                    }
                }
                if (BinaryUtils.hasAny(Stream.of(ExtraInfo.parentOrgs, ExtraInfo.parentCompanies).collect(Collectors.toList()), extraInfoDimension)) {
                    user.setParentOrgs(parentOrgs);
                }
                //endregion
                if (!CollectionUtils.isEmpty(currentOrgs)) {
                    if (BinaryUtils.is(ExtraInfo.companies, extraInfoDimension)) {
                        Set<OrgStruct> companies = currentOrgs.stream().filter(org -> OrgType.COMPANY.equals(org.getOrgType())).collect(Collectors.toSet());
                        user.setCompanies(companies);
                    }
                }
            }
            if (BinaryUtils.hasAny(Stream.of(ExtraInfo.resources, ExtraInfo.packages, ExtraInfo.resourceDetail).collect(Collectors.toList()), extraInfoDimension)) {
                Set<Resource> roleResources = new HashSet<>();
                if (!CollectionUtils.isEmpty(user.getRoles())) {
                    Set<Long> roleIds = user.getRoles().stream()
                            .map(Role::getId).filter(Objects::nonNull)
                            .collect(Collectors.toSet());
                    List<Resource> resources = resourceService.listByRoleIds(roleIds, Stream.of("resourceId", "appId", "resourceCode", "isServicePackage").collect(toSet()));
                    roleResources.addAll(resources);
                }

                roleResources = roleResources.stream().collect(Collectors.toSet());

                if (BinaryUtils.is(ExtraInfo.resourceDetail, extraInfoDimension)) {
                    user.setRoleResources(roleResources);
                }

                Set<ServicePackage> packages = new HashSet<>();
                Set<Resource> packageResources = new HashSet<>();
                //region 加载公司服务包
                Set<OrgStruct> parentCompanies = parentOrgs.stream().filter(o -> OrgType.COMPANY.equals(o.getOrgType())).collect(Collectors.toSet());
                if (!parentCompanies.isEmpty()) {
                    Map<Long, List<ServicePackage>> companyPackagesMap = packageService.listByTenantId(user.getTenantId());
                    Set<Long> packageIds = new HashSet<>();
                    for (OrgStruct org : parentCompanies) {
                        org.setTenantId(user.getTenantId());
                        if (org.getCompanyId() != null && org.getCompanyId() > 0) {
                            List<ServicePackage> companyPackages = companyPackagesMap.get(org.getCompanyId());
                            if (companyPackages != null && !companyPackages.isEmpty()) {
                                if (BinaryUtils.is(ExtraInfo.packages, extraInfoDimension)) {
                                    packages.addAll(companyPackages);
                                }
                                packageIds.addAll(companyPackages.stream().filter(Objects::nonNull).map(ServicePackage::getServicePackageId).collect(Collectors.toSet()));
                            }
                        }
                    }
                    if (!packageIds.isEmpty()) {
                        List<Resource> resources = resourceService.listByPackageIds(packageIds, Stream.of("resourceId", "appId", "resourceCode", "isServicePackage").collect(toSet()));
                        packageResources.addAll(resources);
                    }
                }
                //endregion
                if (BinaryUtils.is(ExtraInfo.packages, extraInfoDimension)) {
                    user.setPackages(packages);
                }
                if (BinaryUtils.is(ExtraInfo.resourceDetail, extraInfoDimension)) {
                    user.setPackageResources(packageResources);
                }

                //region 求角色资源码和公司服务包资源码交集
                if (BinaryUtils.hasAny(Stream.of(ExtraInfo.resources, ExtraInfo.resourceDetail).collect(Collectors.toList()), extraInfoDimension)) {
                    Map<Long, Set<String>> totalAppResourceCodes = new HashMap<>();
                    Set<String> resourceCodes = new HashSet<>();
                    for (Resource roleResource : roleResources) {
                        if (!roleResource.getIsServicePackage() || CollectionUtils.contains(packageResources.iterator(), roleResource)) {
                            if (StringUtils.isNotBlank(roleResource.getResourceCode())) {
                                if (roleResource.getAppId() != null) {
                                    resourceCodes.add(roleResource.getResourceCode());
                                    if (BinaryUtils.is(ExtraInfo.resourceDetail, extraInfoDimension)) {
                                        Set<String> appResourceCodes = totalAppResourceCodes.getOrDefault(roleResource.getAppId(), new HashSet<>());
                                        appResourceCodes.add(roleResource.getResourceCode());
                                        totalAppResourceCodes.put(roleResource.getAppId(), appResourceCodes);
                                    }
                                }
                            }
                        }
                    }

                    if (BinaryUtils.is(ExtraInfo.resourceDetail, extraInfoDimension)) {
                        user.setAppResources(totalAppResourceCodes);
                    }
                    //
                    user.setResourceCodes(resourceCodes);
                }
                //endregion
            }
        }
        return user;
    }

    private void setUserTags(User user) {
        //region if(BinaryUtils.is(ExtraInfo.userTags, extraInfoDimension))
        List<UserTag> userTags = userTagDao.findByUserId(user.getId());
        for (UserTag userTag : userTags) {
            userTag.setUser(null);
            if (userTag.getTagName() == null) {
                continue;
            }
            switch (userTag.getTagName()) {
                case "invoiceType": {
                    user.setInvoiceType(userTag.getTagValue());
                    break;
                }
                case "printingEquipment": {
                    user.setPrintingEquipment(userTag.getTagValue());
                    break;
                }
                case "ticketOpeningTerminal": {
                    user.setTicketOpeningTerminal(userTag.getTagValue());
                    break;
                }
                case "businessExtensionAttribute": {
                    user.setBusinessExtensionAttribute(userTag.getTagValue());
                    break;
                }
                default: {
                    break;
                }
            }
        }
        //endregion
    }

    private void fillOrgRoles(User user, Long orgId) {
        if (orgId == null || orgId == 0) {
            return;
        }
        if (user.getRoles() == null || user.getRoles().isEmpty()) {
            return;
        }
        Set<Long> filterOrgIds = Stream.of(orgId).collect(toSet());
        if (filterOrgIds != null && !filterOrgIds.isEmpty()) {
            Set<Role> allRoles = user.getRoles();
            Set<Long> roleIds = allRoles.stream()
                    .filter(r -> r.getTenantId() != null && r.getTenantId() > 0 && r.getType() != null && r.getType() == 2)
                    .map(Role::getId).filter(id -> id != null && id > 0).collect(toSet());
            RoleModel.Request.Query query = new RoleModel.Request.Query();
            query.setOrgIds(filterOrgIds);
            query.setIds(roleIds);
            query.setUserId(user.getId());
            query.setType(2);
            query.setTenantId(user.getTenantId());
            if (query.getStatus() == null) {
                query.setStatus(1);
            }
            List<Role> filteredRoles = roleService.list(query, Sort.unsorted());
            Set<Role> orgRoles = filteredRoles.stream().collect(toSet());
            user.setOrgRoles(orgRoles);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteById(long userId) {
        User existUser = this.findById(userId);
        userDao.deleteById(existUser.getId());
        orgUserRelDao.deleteByUserId(userId);
        userTagDao.deleteByUserId(userId);
        roleUserRelDao.deleteByUserId(userId);
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteByTenantIdAndUserId(long tenantId, long userId) {
        Query query = new Query();
        query.setTenantId(tenantId);
        query.setUserId(userId);
        User user = this.findOne(query).orElseThrow(() -> new IllegalArgumentException("未找到用户实体"));
        userDao.deleteById(userId);
        orgUserRelDao.deleteByUserId(userId);
        userTagDao.deleteByUserId(userId);
        roleUserRelDao.deleteByUserId(userId);
    }

    private static final ThreadLocal<CachedRoleOrgUserRels> cachedRoleOrgUserRelsThreadLocal = new ThreadLocal<>();

    @Builder
    public static class CachedRoleOrgUserRels {
        Set<RoleUserRel> insertingRoleUserRels;
        Set<RoleUserRel> deletingRoleUserRels;
        Set<OrgUserRel> insertingOrgUserRels;
        Set<OrgUserRel> deletingOrgUserRels;
    }

    private void bindUserRoles(Collection<RoleUserRel> userRels, boolean cached) {
        if (userRels == null || userRels.isEmpty()) {
            return;
        }
        Set<RoleUserRel> rels;
        if (userRels instanceof Set) {
            rels = (Set<RoleUserRel>) userRels;
        } else {
            rels = new HashSet<>(userRels);
        }

        CachedRoleOrgUserRels cachedAllRels = cachedRoleOrgUserRelsThreadLocal.get();
        if (cachedAllRels == null) {
            cachedAllRels = CachedRoleOrgUserRels.builder().insertingRoleUserRels(rels).build();
        } else {
            if (cachedAllRels.insertingRoleUserRels == null) {
                cachedAllRels.insertingRoleUserRels = rels;
            } else {
                cachedAllRels.insertingRoleUserRels.addAll(rels);
            }
        }
        if (cached) {
            cachedRoleOrgUserRelsThreadLocal.set(cachedAllRels);
        } else {
            roleUserRelDao.saveAllAndFlush(cachedAllRels.insertingRoleUserRels);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void commitBindUserRoles() {
        CachedRoleOrgUserRels cachedAllRels = cachedRoleOrgUserRelsThreadLocal.get();
        if (cachedAllRels != null && cachedAllRels.insertingRoleUserRels != null && !cachedAllRels.insertingRoleUserRels.isEmpty()) {
            roleUserRelDao.saveAllAndFlush(cachedAllRels.insertingRoleUserRels);
        }
    }

    private void bindUserOrgs(Collection<OrgUserRel> userRels, boolean cached) {
        if (userRels == null || userRels.isEmpty()) {
            return;
        }
        Set<OrgUserRel> rels;
        if (userRels instanceof Set) {
            rels = (Set<OrgUserRel>) userRels;
        } else {
            rels = new HashSet<>(userRels);
        }
        CachedRoleOrgUserRels cachedAllRels = cachedRoleOrgUserRelsThreadLocal.get();
        if (cachedAllRels == null) {
            cachedAllRels = CachedRoleOrgUserRels.builder().insertingOrgUserRels(rels).build();
        } else {
            if (cachedAllRels.insertingOrgUserRels == null) {
                cachedAllRels.insertingOrgUserRels = rels;
            } else {
                cachedAllRels.insertingOrgUserRels.addAll(rels);
            }
        }
        if (cached) {
            cachedRoleOrgUserRelsThreadLocal.set(cachedAllRels);
        } else {
            orgUserRelDao.saveAllAndFlush(cachedAllRels.insertingOrgUserRels);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void commitBindUserOrgs() {
        CachedRoleOrgUserRels cachedAllRels = cachedRoleOrgUserRelsThreadLocal.get();
        if (cachedAllRels != null && cachedAllRels.insertingOrgUserRels != null && !cachedAllRels.insertingOrgUserRels.isEmpty()) {
            orgUserRelDao.saveAllAndFlush(cachedAllRels.insertingOrgUserRels);
        }
    }

    public static void removeCachedRoleOrgUserRelsThreadLocal() {
        if (cachedRoleOrgUserRelsThreadLocal != null) {
            cachedRoleOrgUserRelsThreadLocal.remove();
        }
    }

    /**
     * 用户绑定角色列表(* 此方法会解绑 roleIds 中不存在但是数据库中有的用户-角色关系)
     *
     * @param userId
     * @param roleIds
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public void bindRoles(Long tenantId, Long orgId, long userId, Collection<Long> allRoleIds, Collection<Long> globalRoleIds, Collection<Long> gradingRoleIds, Collection<Long> orgRoleIds, boolean isOverwrite, boolean isStrict, boolean needCache) {
        User existEntity;
        if (tenantId != null && tenantId > 0) {
            Query query = new Query();
            query.setUserId(userId);
            query.setTenantId(tenantId);
            existEntity = this.findOne(query).orElseThrow(() -> new IllegalArgumentException("未找到用户实体(tenantId:" + tenantId + ", userId:" + userId + ")"));
        } else {
            existEntity = this.findUserById(userId).orElseThrow(() -> new IllegalArgumentException("未找到用户实体(userId:" + userId + ")"));
        }
        this.bindRoles(existEntity, orgId, allRoleIds, globalRoleIds, gradingRoleIds, orgRoleIds, isOverwrite, isStrict, needCache);
    }

    /**
     * 用户绑定角色列表(* 此方法会解绑 roleIds 中不存在但是数据库中有的用户-角色关系)
     *
     * @param user
     * @param roleIds
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public <U extends UserDto<O, R, A>, O extends OrgDto<O>, R extends RoleDto, A extends AccountDto> void bindRoles(U user, Long orgId, Collection<Long> allRoleIds, Collection<Long> globalRoleIds, Collection<Long> gradingRoleIds, Collection<Long> orgRoleIds, boolean isOverwrite, boolean isStrict, boolean needCache) {
        logger.info("tenantId:{},isOverwrite = {}, isStrict = {}", user.getTenantId(), isOverwrite, isStrict);
        if (allRoleIds == null && globalRoleIds == null && gradingRoleIds == null && orgRoleIds == null) {
            logger.info("roleIds == null, return");
            if (!needCache) {
                this.commitBindUserRoles();
            }
            return;
        }
        if (logger.isInfoEnabled()) {
            if (allRoleIds != null) {
                logger.info("allRoleIds = " + allRoleIds.stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.joining(",")));
            }
            if (globalRoleIds != null) {
                logger.info("globalRoleIds = " + globalRoleIds.stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.joining(",")));
            }
            if (gradingRoleIds != null) {
                logger.info("gradingRoleIds = " + gradingRoleIds.stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.joining(",")));
            }
            if (orgRoleIds != null) {
                logger.info("orgRoleIds = " + orgRoleIds.stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.joining(",")));
            }
        }

        List<RoleUserRel> existRels = roleUserRelDao.findByUserId(user.getId());
        logger.info("exist role-user-rels.size = " + existRels.size());
        //region 往绑定不在数据库中但是报文 roleIds 中有的用户-角色关系
        Triple<Set<RoleUserRel>, Set<RoleUserRel>, Set<Long>> allRelsTriple = this.processRoleUserRels(user, null, allRoleIds, existRels, null, isOverwrite, isStrict);
        Triple<Set<RoleUserRel>, Set<RoleUserRel>, Set<Long>> globalRelsTriple = this.processRoleUserRels(user, null, globalRoleIds, existRels, RoleTypeEnum.GLOBAL.getType(), isOverwrite, isStrict);
        Triple<Set<RoleUserRel>, Set<RoleUserRel>, Set<Long>> gradingRelsTriple = this.processRoleUserRels(user, null, gradingRoleIds, existRels, RoleTypeEnum.HIERARCHY.getType(), isOverwrite, isStrict);
        Triple<Set<RoleUserRel>, Set<RoleUserRel>, Set<Long>> orgRelsTriple = this.processRoleUserRels(user, orgId, orgRoleIds, existRels, RoleTypeEnum.ORG.getType(), isOverwrite, isStrict);

        //process prerole
        Triple<Set<RoleUserRel>, Set<RoleUserRel>, Set<Long>> preRoleRelsTriple = this.processRoleUserRels(user, null, globalRoleIds, existRels, RoleTypeEnum.PRE.getType(), isOverwrite, isStrict);

        Set<RoleUserRel> exisAllRels = null;
        Set<RoleUserRel> insertingAllRels = null;
        Set<Long> deletingAllRoleIds = null;
        if (allRelsTriple != null) {
            exisAllRels = allRelsTriple.getLeft();
            insertingAllRels = allRelsTriple.getMiddle();
            deletingAllRoleIds = allRelsTriple.getRight();
        }

        Set<RoleUserRel> existGlobalRels = null;
        Set<RoleUserRel> insertingGlobalRels = null;
        Set<Long> deletingGlobalRoleIds = null;
        if (globalRelsTriple != null) {
            existGlobalRels = globalRelsTriple.getLeft();
            insertingGlobalRels = globalRelsTriple.getMiddle();
            deletingGlobalRoleIds = globalRelsTriple.getRight();
        }

        Set<RoleUserRel> existGradingRels = null;
        Set<RoleUserRel> insertingGradingRels = null;
        Set<Long> deletingGradingRoleIds = null;
        if (gradingRelsTriple != null) {
            existGradingRels = gradingRelsTriple.getLeft();
            insertingGradingRels = gradingRelsTriple.getMiddle();
            deletingGradingRoleIds = gradingRelsTriple.getRight();
        }

        Set<RoleUserRel> existOrgRels = null;
        Set<RoleUserRel> insertingOrgRels = null;
        Set<Long> deletingOrgRoleIds = null;
        if (orgRelsTriple != null) {
            existOrgRels = orgRelsTriple.getLeft();
            insertingOrgRels = orgRelsTriple.getMiddle();
            deletingOrgRoleIds = orgRelsTriple.getRight();
            if (orgId != null && orgId > 0) {
                RoleModel.Request.Query query = new RoleModel.Request.Query();
                query.setTenantId(user.getTenantId());
                if (query.getStatus() == null) {
                    query.setStatus(1);
                }
                query.setType(2);
                query.setOrgId(orgId);
                List<Role> roles = roleService.list(query, Sort.unsorted());
                if (roles != null && !roles.isEmpty()) {
                    if (existOrgRels != null && !existOrgRels.isEmpty()) {
                        existOrgRels = existOrgRels.stream().filter(rel -> roles.stream().anyMatch(r -> r.getId().equals(rel.getRoleId()))).collect(Collectors.toSet());
                    }
                    if (insertingOrgRels != null && !insertingOrgRels.isEmpty()) {
                        insertingOrgRels = insertingOrgRels.stream().filter(rel -> roles.stream().anyMatch(r -> r.getId().equals(rel.getRoleId()))).collect(Collectors.toSet());
                    }
                    if (deletingOrgRoleIds != null && !deletingOrgRoleIds.isEmpty()) {
                        deletingOrgRoleIds = deletingOrgRoleIds.stream().filter(id -> roles.stream().anyMatch(r -> r.getId().equals(id))).collect(Collectors.toSet());
                    }
                }
            }
        }


        Set<RoleUserRel> existPreRoleRels = null;
        Set<RoleUserRel> insertingPreRoleRels = null;
        Set<Long> deletingPreRoleIds = null;
        if (preRoleRelsTriple != null) {
            existPreRoleRels = preRoleRelsTriple.getLeft();
            insertingPreRoleRels = preRoleRelsTriple.getMiddle();
            deletingPreRoleIds = preRoleRelsTriple.getRight();
        }


        Set<RoleUserRel> insertingRels = new HashSet<>();
        if (insertingAllRels != null) {
            insertingRels.addAll(insertingAllRels);
        }
        if (insertingGlobalRels != null) {
            insertingRels.addAll(insertingGlobalRels);
        }
        if (insertingGradingRels != null) {
            insertingRels.addAll(insertingGradingRels);
        }
        if (insertingOrgRels != null) {
            insertingRels.addAll(insertingOrgRels);
        }

        if (insertingPreRoleRels != null) {
            insertingRels.addAll(insertingPreRoleRels);
        }

        this.bindUserRoles(insertingRels, needCache);
        //endregion
        if (isOverwrite) {
            //region 从数据库中解绑不在报文 roleIds 中的用户-角色关系
            Set<Long> deleltingRoleIds = new HashSet<>();
            if (deletingAllRoleIds != null && !deletingAllRoleIds.isEmpty()) {
                deleltingRoleIds.addAll(deletingAllRoleIds);
            }
            if (deletingGlobalRoleIds != null && !deletingGlobalRoleIds.isEmpty()) {
                deleltingRoleIds.addAll(deletingGlobalRoleIds);
            }
            if (deletingGradingRoleIds != null && !deletingGradingRoleIds.isEmpty()) {
                deleltingRoleIds.addAll(deletingGradingRoleIds);
            }
            if (deletingOrgRoleIds != null && !deletingOrgRoleIds.isEmpty()) {
                deleltingRoleIds.addAll(deletingOrgRoleIds);
            }

            if (deletingPreRoleIds != null && !deletingPreRoleIds.isEmpty()) {
                deleltingRoleIds.addAll(deletingPreRoleIds);
            }


            if (deleltingRoleIds != null && !deleltingRoleIds.isEmpty()) {
                this.unbindRoles(user.getTenantId(), user.getId(), deleltingRoleIds);
            }
            //endregion
        }
    }

    private <U extends UserDto<O, R, A>, O extends OrgDto<O>, R extends RoleDto, A extends AccountDto> Triple<Set<RoleUserRel>, Set<RoleUserRel>, Set<Long>> processRoleUserRels(U user, Long orgId, Collection<Long> roleIds, Collection<RoleUserRel> existTotalRels, Integer roleType, boolean isOverwrite, boolean isStrict) {
        if (roleIds == null || roleIds.isEmpty()) {
            return null;
        }
        Set<RoleUserRel> existRels = existTotalRels.stream().filter(rel -> roleType == null || rel.getRelType() == roleType).collect(toSet());
        Set<RoleUserRel> insertingRels = roleIds.stream()
                .filter(roleId -> existRels.stream().map(RoleUserRel::getRoleId).noneMatch(existRoleId -> existRoleId.equals(roleId)))
                .filter(Objects::nonNull)
                .map(roleId -> {
                    Specification<Role> specification = (Specification<Role>) (root, criteriaQuery, builder) -> {
                        List<Predicate> predicates = new ArrayList<>();
                        if (roleId != null && roleId > 0) {
                            predicates.add(builder.equal(root.<Long>get("id"), roleId));
                        }
                        if (!predicates.isEmpty()) {
                            criteriaQuery.where(predicates.stream().toArray(Predicate[]::new));
                        }
                        return criteriaQuery.getRestriction();
                    };
                    Optional<Role> existsRole = roleDao.findOne(specification);
                    Integer relType;
                    if (existsRole.isPresent()) {
                        Role role = existsRole.get();
                        if (role.getTenantId() != null && role.getTenantId() > 0 && !role.getTenantId().equals(user.getTenantId())) {
                            if (isStrict) {
                                String message = "用户(" + user.getId() + ")所在租户(" + user.getTenantId() + ")和角色(" + role.getId() + ")所在租户(" + role.getTenantId() + ")不一致";
                                throw new IllegalArgumentException(message);
                            } else {
                                return null;
                            }
                        }
                        relType = role.getType();
                        if (roleType != null && !relType.equals(roleType)) {
                            if (isStrict) {
                                String message = "角色(id:" + role.getId() + "type:" + relType + ")不是type = " + roleType + "的角色";
                                throw new IllegalArgumentException(message);
                            } else {
                                return null;
                            }
                        }
                    } else {
                        if (isStrict) {
                            String message = "数据库中不存在角色(id:" + roleId + ")";
                            throw new IllegalArgumentException(message);
                        } else {
                            return null;
                        }
                    }
                    logger.info("RoleUserRel.relType = {}", relType);
                    RoleUserRel roleUserRel = new RoleUserRel();
                    roleUserRel.setTenantId(user.getTenantId());
                    roleUserRel.setRoleId(roleId);
                    roleUserRel.setUserId(user.getId());
                    if (relType == 2) {
                        roleUserRel.setOrgId(orgId);
                    }
                    roleUserRel.setRelType(relType);
                    return roleUserRel;
                })
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());

        Set<Long> deletingRoleIds;
        if (isOverwrite) {
            //region 从数据库中解绑不在报文 roleIds 中的用户-角色关系
            deletingRoleIds = existRels.stream()
                    .filter(Objects::nonNull)
                    .filter(rel -> roleIds.stream().filter(Objects::nonNull).noneMatch(roleId -> roleId.equals(rel.getRoleId())))
                    .filter(rel -> rel.getRoleId() > 1).map(RoleUserRel::getRoleId)
                    .filter(Objects::nonNull).collect(toSet());
            //endregion
        } else {
            deletingRoleIds = null;
        }
        return Triple.of(existRels, insertingRels, deletingRoleIds);
    }

    /**
     * 用户绑定组织列表(* 此方法会解绑 orgIds 中不存在但是数据库中有的用户-组织关系)
     *
     * @param userId
     * @param orgIds
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public void bindOrgs(Long tenantId, long userId, Collection<Long> orgIds, String modules, boolean isOverwrite, boolean isStrict, boolean needCache) {
        User existEntity;
        Query query = new Query();
        query.setUserId(userId);
        if (tenantId != null && tenantId > 0) {
            query.setTenantId(tenantId);
            existEntity = this.findOne(query).orElseThrow(() -> new IllegalArgumentException("未找到用户实体(tenantId:" + tenantId + ", userId:" + userId + ")"));
        } else {
            existEntity = this.findOne(query).orElseThrow(() -> new IllegalArgumentException("未找到用户实体(userId:" + userId + ")"));
        }

        this.bindOrgs(existEntity, orgIds, modules, isOverwrite, isStrict, needCache);
    }

    /**
     * 用户绑定多个角色(* 此方法会解绑 orgIds 中不存在但是数据库中有的用户-组织关系)
     *
     * @param user
     * @param orgIds
     */
    @Transactional(rollbackFor = Exception.class)
    public <U extends UserDto<O, R, A>, O extends OrgDto<O>, R extends RoleDto, A extends AccountDto> void bindOrgs(U user, Collection<Long> orgIds, String modules, boolean isOverwrite, boolean isStrict, boolean needCache) {
        logger.info("isOverwrite = {}, isStrict = {}", isOverwrite, isStrict);
        Set<Long> orgIdSet;
        if (orgIds == null) {
            logger.info("orgIds == null, return");
            if (!needCache) {
                this.commitBindUserOrgs();
            }
            return;
        } else {
            if (orgIds.isEmpty() && !isOverwrite) {
                logger.info("orgIds isEmpty and isOverwrite = false, return");
                if (!needCache) {
                    this.commitBindUserOrgs();
                }
                return;
            }
            if (orgIds instanceof Set) {
                orgIdSet = (Set<Long>) orgIds;
            } else {
                orgIdSet = new HashSet<>(orgIds);
            }
            orgIdSet = orgIdSet.stream().filter(Objects::nonNull).collect(toSet());
        }
        if (logger.isInfoEnabled()) {
            logger.info("orgIds = " + orgIdSet.stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.joining(",")));
        }
        OrgModel.Request.Query query = new OrgModel.Request.Query();
        query.setTenantId(user.getTenantId());
        query.setStatus(1);
        query.setAttributes(Stream.of("orgId", "orgName", "parentId", "orgType").collect(toSet()));
        List<OrgStruct> tenantOrgs = orgStructDao.findAttributes(query, Sort.unsorted());
        Set<String> filterParentIdsSet = this.filterParentIds(user.getTenantId(), modules, tenantOrgs);
        Set<String> parentIdsSet;
        if (orgIdSet.isEmpty()) {
            parentIdsSet = Collections.emptySet();
        } else {
            final Set<Long> orgIdSets = orgIdSet;
            parentIdsSet = tenantOrgs.stream().filter(Objects::nonNull)
                    .filter(o -> orgIdSets.stream().anyMatch(id -> id.equals(o.getOrgId())))
                    .map(OrgStruct::getParentIds)
                    .filter(Objects::nonNull).collect(toSet());
            //如果是按模块隔离的， 就只保留 该模块内的组织
            if (StringUtils.isNotEmpty(modules)) {
                logger.info("update_user_filter,userId:{}, parentIdsSet:{}, filterParentIdsSet:{}", user.getId(), parentIdsSet, filterParentIdsSet);
                parentIdsSet = parentIdsSet.stream().filter(p -> filterParentIdsSet.stream().anyMatch(f -> p.startsWith(f))).collect(toSet());
            }
        }

        //region 压缩parentIds集合减少sql次数
        parentIdsSet = OrgUtils.compressParentIdsCollection(parentIdsSet);

        //region 递归找子组织
        Pair<List<OrgStruct>, Set<Long>> treeAllOrgIdsPair = TreeUtils.buildTreeAndCascadeCheckKeys(tenantOrgs, orgIdSet);
        List<OrgStruct> orgTree = treeAllOrgIdsPair.getLeft();
        Set<Long> allOrgIds = treeAllOrgIdsPair.getRight();

        //endregion
        Set<Long> existOrgIds = orgUserRelDao.findOrgIdsByUserId(user.getId());
        logger.info("exist org-user({})-rels.size = {}", user.getId(), existOrgIds.size());
        Set<OrgUserRel> insertingRels;
        //region 处理上级组织是否选中
        if (!allOrgIds.isEmpty()) {


            //region 绑定不在数据库中但是报文 orgIds 中有的用户-组织关系
            insertingRels = allOrgIds.stream()
                    .filter(orgId -> !existOrgIds.contains(orgId))
                    .filter(Objects::nonNull)
                    .map(orgId -> {
                        Optional<OrgStruct> existsOrg = tenantOrgs.stream().filter(o -> o.getId().equals(orgId)).findFirst();
                        if (existsOrg.isPresent()) {
                            OrgStruct org = existsOrg.get();
                        } else {
                            if (isStrict) {
                                String message = "数据库中不存在组织(id:" + orgId + ", tenantId:" + user.getTenantId() + ")";
                                logger.warn(message);
                                throw new IllegalArgumentException(message);
                            } else {
                                return null;
                            }
                        }
                        OrgUserRel orgUserRel = new OrgUserRel();
                        orgUserRel.setTenantId(user.getTenantId());
                        orgUserRel.setOrgStructId(orgId);
                        orgUserRel.setUserId(user.getId());
                        orgUserRel.setFullSelectedFlag(true);
                        return orgUserRel;
                    })
                    .filter(Objects::nonNull)
                    .collect(Collectors.toSet());
            //endregion
            List<Long> rootIds = tenantOrgs.stream().filter(o -> o.getParentId() == null || o.getParentId() == 0).map(OrgStruct::getOrgId).collect(Collectors.toList());
            if (rootIds == null) {
                String message = "无效数据，租户(" + user.getTenantId() + ")无根组织";
                logger.warn(message);
                throw new IllegalArgumentException(message);
            }
            //region 关联用户和组织默认角色
            Set<RoleUserRel> insertingDefaultRoleRels = tenantOrgs.stream()
                    .filter(o -> insertingRels.stream().anyMatch(rel -> rel.getOrgStructId().equals(o.getOrgId())))
                    .map(o -> {
                        if (o.getDefaultOrgRoleId() != null) {
                            RoleUserRel roleUserRel = new RoleUserRel();
                            roleUserRel.setRoleId(o.getDefaultOrgRoleId());
                            roleUserRel.setTenantId(o.getTenantId());
                            roleUserRel.setUserId(user.getId());
                            roleUserRel.setRelType(2);
                            return roleUserRel;
                        } else {
                            return null;
                        }
                    }).filter(Objects::nonNull).collect(toSet());
            this.bindUserRoles(insertingDefaultRoleRels, true);
            //endregion 关联用户和组织默认角色
        } else {
            insertingRels = Collections.emptySet();
        }
        //endregion 处理根组织是否选中

        this.bindUserOrgs(insertingRels, needCache);

        if (isOverwrite) {
            //region 从数据库中解绑不在报文 orgIds 中的用户-组织关系
            Set<Long> insertedOrgIds = insertingRels.stream()
                    .filter(Objects::nonNull).map(OrgUserRel::getOrgStructId).collect(Collectors.toSet());
            if (!insertedOrgIds.isEmpty()) {
                allOrgIds.addAll(insertedOrgIds);
            }
            Set<Long> deletingOrgIds = existOrgIds.stream()
                    .filter(Objects::nonNull)
                    .filter(existOrgId -> allOrgIds.stream()
                            .filter(Objects::nonNull)
                            .noneMatch(insertedOrgId -> insertedOrgId.equals(existOrgId))
                    ).filter(Objects::nonNull).collect(toSet());

            this.unbindOrgsWithFilterParentIdsSet(user.getTenantId(), user.getId(), filterParentIdsSet, deletingOrgIds, false);
            //endregion
        }

    }

    @Transactional(rollbackFor = Exception.class)
    public void bindApps(long userId, Collection<Long> appIds, boolean isOverwrite, boolean isStrict) {
        User existEntity = this.findUserById(userId).orElseThrow(() -> new IllegalArgumentException("未找到用户(" + userId + ")实体"));
        this.bindApps(existEntity, appIds, isOverwrite, isStrict);
    }

    /**
     * 用户绑定角色列表(* 此方法会解绑 roleIds 中不存在但是数据库中有的用户-角色关系)
     *
     * @param user
     * @param appIds
     * @return
     */
    public <U extends UserDto<O, R, A>, O extends OrgDto<O>, R extends RoleDto, A extends AccountDto> void bindApps(U user, Collection<Long> appIds, boolean isOverwrite, boolean isStrict) {
        logger.info("isOverwrite = {}, isStrict = {}", isOverwrite, isStrict);
        if (appIds == null) {
            return;
        }
        if (appIds.isEmpty()) {
            if (isOverwrite) {
                logger.info("appIds == null, isOverwrite = true, clean all");
                userAppDao.deleteByUserId(user.getId());
            } else {
                logger.info("appIds == null, isOverwrite = false, return");
            }
        }
        logger.info("appIds = " + appIds.stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.joining(",")));
        List<UserApp> existRels = userAppDao.findByUserId(user.getId());
        logger.info("exist user-app-rels.size = " + existRels.size());
        //region 往绑定不在数据库中但是报文 appIds 中有的用户-应用关系
        Set<UserApp> insertingRels = appIds.stream()
                .filter(appId -> existRels.stream().map(UserApp::getAppId).noneMatch(newAppId -> newAppId.equals(appId)))
                .filter(Objects::nonNull)
                .map(appId -> {
                    boolean exists = appDao.existsById(appId);
                    if (!exists) {
                        if (isStrict) {
                            String message = "数据库中不存在应用(id:" + appId + ")";
                            throw new IllegalArgumentException(message);
                        } else {
                            return null;
                        }
                    }
                    UserApp userApp = new UserApp();
                    userApp.setAppId(appId);
                    userApp.setUserId(user.getId());
                    return userApp;
                })
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());
        this.bindApps(insertingRels);
        //endregion
        if (isOverwrite) {
            //region 从数据库中解绑不在报文 roleIds 中的用户-角色关系
            existRels.stream()
                    .filter(rel -> appIds.stream().filter(Objects::nonNull).noneMatch(appId -> appId.equals(rel.getAppId())))
                    .forEach(rel -> {
                        try {
                            userAppDao.deleteById(rel.getId());
                        } catch (Exception e) {
                            logger.warn(e.getMessage(), e);
                        }
                    });
            //endregion
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void bindApps(Collection<UserApp> rels) {
        if (rels != null && !rels.isEmpty()) {
            userAppDao.saveAllAndFlush(rels);
        }
    }

    /**
     * @param checkUriAuthzQuery
     * @return
     */
    public boolean checkUriAuthz(CheckUriAuthzQuery checkUriAuthzQuery) {
        Long userId = checkUriAuthzQuery.getUserId();
        logger.info("userId = {}", userId);

        Long routeId = checkUriAuthzQuery.getRouteId();
        Map<RequestUri, RequestUriAuthz> requestUriAuthzMap = serviceApiService.getRequestUriAndResourceCodesMapByRouteId(routeId);
        RequestUri requestUri = new RequestUri(checkUriAuthzQuery.getRequestUri(), checkUriAuthzQuery.getRequestMethod());
        RequestUriAuthz requestUriAuthz = RequestMappingHelper.lookupRequestUri(requestUri, requestUriAuthzMap, false);
        if (requestUriAuthz == null) {
            logger.info("requestUriAuthz == null, skip, return true");
            return true;
        }
        if (requestUriAuthz.getSkipAuthentication() != null && requestUriAuthz.getSkipAuthentication()) {
            logger.info("skipAuthentication, skip, return true");
            return true;
        }
        if (requestUriAuthz.getSkipAuthorization() != null && requestUriAuthz.getSkipAuthorization()) {
            logger.info("skipAuthorization, skip, return true");
            return true;
        }
        Set<String> resourceCodes = requestUriAuthz.getResourceCodes();
        if (CollectionUtils.isEmpty(resourceCodes)) {
            logger.info("resourceCodes.isEmpty, skip, return true");
            return true;
        }
        User user = this.findById(userId, BinaryUtils.toBinary(ExtraInfo.resources));
        Set<String> userResourceCodes = user.getResourceCodes();
        boolean result = userResourceCodes.parallelStream().anyMatch(resourceCodes::contains);
        return result;
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateStatus(long tenantId, long userId, int status) {
        User existEntity = this.findByTenantIdAndUserId(tenantId, userId, null, 0);
        if (status == 0 && existEntity.isAdmin()) {
            throw new IllegalArgumentException("不能禁用租户管理员");
        }
        existEntity.setStatus(status);
        this.saveAndFlush(existEntity);
        userPubService.sendUserToPubsub(existEntity, tenantId);
    }

    public Long findIdByTenantIdAndCode(long tenantId, String userCode) {
        List<Long> userIds = userDao.selectIdsByTenantIdAndUserCode(tenantId, userCode);
        if (userIds == null || userIds.isEmpty()) {
            return null;
        }
        return userIds.stream().findFirst().get();
    }

    @Transactional(rollbackFor = Exception.class)
    public void unbindRoles(long tenantId, long userId, Collection<Long> roleIds) {
        if (roleIds == null || roleIds.isEmpty()) {
            return;
        }
        Set<Long> deletingRoleIds;
        if (roleIds instanceof Set) {
            deletingRoleIds = (Set<Long>) roleIds;
        } else {
            deletingRoleIds = new HashSet<>(roleIds);
        }
        roleUserRelDao.deleteByTenantIdAndUserIdAndRoleIds(tenantId, userId, deletingRoleIds);
    }

    @Transactional(rollbackFor = Exception.class)
    public void unbindOrgs(long tenantId, long userId, String modules, Collection<Long> orgIds) {
        if (orgIds == null || orgIds.isEmpty()) {
            return;
        }
        Set<String> filterParentIdsSet = this.filterParentIds(tenantId, modules);
        this.unbindOrgsWithFilterParentIdsSet(tenantId, userId, filterParentIdsSet, orgIds, true);
    }

    private void unbindOrgsWithFilterParentIdsSet(long tenantId, long userId, final Set<String> filterParentIdsSet, Collection<Long> orgIds, boolean cascade) {
        if (orgIds == null || orgIds.isEmpty()) {
            return;
        }
        //region 转换orgIds类型Collection到Set
        final Set<Long> deleltingOrgIds;
        if (orgIds instanceof Set) {
            deleltingOrgIds = (Set<Long>) orgIds;
        } else {
            deleltingOrgIds = new HashSet<>(orgIds);
        }
        //endregion

        Set<String> attributes = Stream.of("orgId", "parentIds", "defaultOrgRoleId").collect(toSet());

        //region 根据过滤orgId集合获取组织集合
        OrgModel.Request.Query query = new OrgModel.Request.Query();
        query.setTenantId(tenantId);
        query.setOrgIds(deleltingOrgIds);
        query.setAttributes(attributes);
        List<OrgStruct> orgs = orgStructDao.findAttributes(query, Sort.unsorted());
        //endregion

        //region if cascade == true then 根据parentIds集合查询子组织集合
        if (cascade) {
            Set<String> deletingParentIdsSet = orgs.stream().map(org -> org.getParentIds()).collect(toSet());
            deletingParentIdsSet = OrgUtils.compressParentIdsCollection(deletingParentIdsSet);
            query = new OrgModel.Request.Query();
            query.setTenantId(tenantId);
            query.setFilterParentIds(deletingParentIdsSet);
            query.setAttributes(attributes);
            orgs = orgStructDao.findAttributes(query, Sort.unsorted());
        }
        //endregion

        //region 根据组织集合获取对应的parentIds集合
        Set<Long> parentOrgIdSet = orgs.stream()
                .map(org -> OrgUtils.findOrgIdInParentIds(org.getParentIds()))
                .flatMap(Collection::stream)
                .collect(toSet());
        //endregion

        //region 补充parentIds找到的递归上级组织到orgs和deleltingOrgIds里
        Set<Long> additionOrgIds = parentOrgIdSet.stream()
                .filter(parentOrgId -> deleltingOrgIds.stream()
                        .noneMatch(orgId -> orgId.equals(parentOrgId))
                ).collect(toSet());
        if (!additionOrgIds.isEmpty()) {
            query = new OrgModel.Request.Query();
            query.setTenantId(tenantId);
            query.setOrgIds(additionOrgIds);
            query.setAttributes(attributes);
            List<OrgStruct> additionOrgs = orgStructDao.findAttributes(query, Sort.unsorted());
            orgs.addAll(additionOrgs);
            deleltingOrgIds.addAll(additionOrgIds);
        }
        //endregion

        //region 根据权限返回过滤 deleltingOrgIds
        Set<Long> deletingDefaultRoleIds = new HashSet<>();
        orgs.stream().filter(o -> filterParentIdsSet.isEmpty() || filterParentIdsSet.stream().anyMatch(p -> o.getParentIds().startsWith(p))).forEach(o -> {
            Long orgId = o.getOrgId();
            deleltingOrgIds.add(orgId);
            if (o.getDefaultOrgRoleId() != null) {
                deletingDefaultRoleIds.add(o.getDefaultOrgRoleId());
            }
        });
        //endregion

        if (deleltingOrgIds != null && !deleltingOrgIds.isEmpty()) {
            orgUserRelDao.deleteByTenantIdAndUserIdAndOrgIds(tenantId, userId, deleltingOrgIds);
        }

        //region 解绑人员-组织默认角色关联
        if (!deletingDefaultRoleIds.isEmpty()) {
            this.unbindRoles(tenantId, userId, deletingDefaultRoleIds);
        }
        //endregion
    }

    private Set<String> filterParentIds(long tenantId, String modules) {
        String[] orgCodeArray = StringUtils.split(modules, ",");
        if (ArrayUtils.isEmpty(orgCodeArray)) {
            return Collections.emptySet();
        }
        Set<String> orgCodes = Stream.of(orgCodeArray).filter(Objects::nonNull).collect(toSet());
        Set<String> filterParentIdsSet = new HashSet<>();
        List<String> filteringParentIdsList;
        if (orgCodes != null && !orgCodes.isEmpty()) {
            filteringParentIdsList = orgStructDao.findParentIdsByTenantIdAndOrgCodes(tenantId, orgCodes);
        } else {
            filteringParentIdsList = Collections.emptyList();
        }

        return this.filterParentIds(filteringParentIdsList);
    }

    private Set<String> filterParentIds(long tenantId, String modules, List<OrgStruct> tenantOrgs) {
        String[] orgCodeArray = StringUtils.split(modules, ",");
        if (ArrayUtils.isEmpty(orgCodeArray)) {
            return Collections.emptySet();
        }
        Set<String> orgCodes = Stream.of(orgCodeArray).filter(Objects::nonNull).collect(toSet());
        Set<String> filterParentIdsSet = new HashSet<>();
        List<String> filteringParentIdsList = tenantOrgs.stream()
                .filter(o -> o.getTenantId() != null && tenantId == o.getTenantId() && orgCodes.stream()
                        .anyMatch(code -> code.equals(o.getOrgCode()))
                )
                .map(OrgStruct::getParentIds)
                .collect(Collectors.toList());
        return this.filterParentIds(filteringParentIdsList);
    }

    private Set<String> filterParentIds(List<String> filteringParentIdsList) {
        if (filteringParentIdsList == null || filteringParentIdsList.isEmpty()) {
            return Collections.emptySet();
        }
        Collections.sort(filteringParentIdsList);
        Set<String> filterParentIdsSet = new HashSet<>();
        filteringParentIdsList.stream()
                .filter(filteringParentIds -> filterParentIdsSet.stream()
                        .noneMatch(filterParentIds -> StringUtils.startsWith(filteringParentIds, filterParentIds))
                )
                .forEach(filterParentIdsSet::add);
        return filterParentIdsSet;
    }

    public void filterUser(List<Create> models) {
        for (Create create : models) {
            if (null == create) {
                continue;
            }
            Long accountId;
            Long userId = 0L;
            Account account = new Account();
            AccountModel.Request.Create createAccount = create.getAccount();
            //如果用户id大于0直接退出
            if (null != create.getUserId() && create.getUserId() > 0) {
                continue;
            }
            if (createAccount != null) {
                if (StringUtils.isNotBlank(createAccount.getTelPhone())) {
                    account = accountDao.findByTelPhone(createAccount.getTelPhone());
                }
                if (StringUtils.isNotBlank(createAccount.getEmail()) && null == account) {
                    account = accountDao.findByEmail(createAccount.getEmail());
                }
                if (StringUtils.isNotBlank(create.getUserCode()) && null == account) {
                    account = accountDao.findByTelPhone(create.getUserCode());
                }
            }
            //如果账号不存在直接退出循环
            if (null == account || null == account.getAccountId() || account.getAccountId() == 0) {
                continue;
            }
            accountId = account.getAccountId();
            if (create.getTenantId() == null || create.getTenantId() == 0) {
                continue;
            }
            List<User> users = userDao.findByTenantIdAndAccountId(create.getTenantId(), accountId);
            User user = users.stream().findFirst().orElse(null);
            if (null == user && StringUtils.isNotBlank(create.getUserCode())) {
                userId = this.findIdByTenantIdAndCode(create.getTenantId(), create.getUserCode());
            }
            //查处userId防止重复创建
            if (null != user) {
                create.setUserId(user.getId());
            } else if (null != userId && userId > 0) {
                create.setUserId(userId);
            }
        }
    }

    public void clearCache(long userId) {
        // TODO clear userId resource cache
    }

    public List<UserExportDto> getExportData(List<Long> userIds) {

        SearchFilter searchFilter = SearchFilter.build("id", Operator.IN, userIds.toArray());
        Specification<User> specification = DynamicSpecifications.bySearchFilter(Stream.of(searchFilter).collect(Collectors.toList()));

        List<User> users = userDao.findAll(specification, Sort.unsorted(), EntityGraphs.named(User.NAMED_ENTITY_GRAPH_ACCOUNT));
        List<UserExportDto> userExportDtos = new ArrayList<>();
        users.forEach(user -> {
            List<OrgStruct> orgs = orgService.listByTenantIdAndUserId(user.getTenantId(), user.getId(), null);
            List<Role> roles = roleService.listByUserId(user.getId());
            Account account = user.getAccount();

            UserExportDto dto = new UserExportDto();
            dto.setUserName(user.getUserName());
            dto.setUserCode(user.getUserCode());
            dto.setUserNumber(user.getUserNumber());
            dto.setUserId(user.getId().toString());
            if (1 == user.getStatus()) {
                dto.setStatus("启用");
            } else if (0 == user.getStatus()) {
                dto.setStatus("未启用");
            } else if (2 == user.getStatus()) {
                dto.setStatus("停用");
            }
            dto.setEmail(account.getEmail());
            dto.setTelPhone(account.getTelPhone());
            dto.setAccountName(account.getUsername());

            List<UserTag> tags = user.getUserTags();
            this.setUserExportDto(tags, dto);
            if (!CollectionUtils.isEmpty(orgs)) {
                dto.setOrgNames(orgs.stream().map(OrgStruct::getOrgName).collect(Collectors.joining("/")));
                dto.setOrgCodes(orgs.stream().map(OrgStruct::getOrgCode).collect(Collectors.joining("/")));
            }

            if (!CollectionUtils.isEmpty(roles)) {
                dto.setRoleCodes(roles.stream().map(Role::getCode).collect(Collectors.joining("/")));
                dto.setRoleNames(roles.stream().map(Role::getName).collect(Collectors.joining("/")));
            }
            userExportDtos.add(dto);
        });

        return userExportDtos;
    }

    /**
     * 设置UserExportDTO的属性
     *
     * @param tags Tags
     * @param dto  DTO
     */
    private void setUserExportDto(List<UserTag> tags, UserExportDto dto) {
        if (CollectionUtils.isEmpty(tags)) {
            return;
        }
        for (UserTag tag : tags) {
            this.setUserExportDtoTags(dto, tag);
        }

    }

    /**
     * 根据TagName
     *
     * @param dto UserExportDto
     * @param tag
     */
    private void setUserExportDtoTags(UserExportDto dto, UserTag tag) {
        switch (tag.getTagName()) {
            case "invoiceType":
                String invoiceType = StringUtils.trimToEmpty(tag.getTagValue());
                invoiceType = invoiceType.replaceAll(",", "/");
                dto.setInvoices(invoiceType);
                break;
            case "printingEquipment":
                String printingEquipment = StringUtils.trimToEmpty(tag.getTagValue());
                printingEquipment = printingEquipment.replaceAll(",", "/");
                dto.setPrintingEquipment(printingEquipment);
                break;
            case "ticketOpeningTerminal":
                String ticketOpeningTerminal = StringUtils.trimToEmpty(tag.getTagValue());
                ticketOpeningTerminal = ticketOpeningTerminal.replaceAll(",", "/");
                dto.setTicketOpeningTerminal(ticketOpeningTerminal);
                break;
            default:
                logger.warn("tagName:{},tagValue:{}", tag.getTagName(), tag.getTagValue());
                break;
        }
    }

    @Transactional
    public void fixUserOrg(User user) {
        Set<Long> orgIds = orgUserRelDao.findOrgIdsByUserId(user.getId());
        Set<String> parentIdsSet;
        if (orgIds.isEmpty()) {
            parentIdsSet = new HashSet<>();
        } else {
            parentIdsSet = orgStructDao.findParentIdsByOrgIds(orgIds);
        }
        Set<Long> descendantOrgIds = new HashSet<>();
        for (String parentIds : parentIdsSet) {
            Set<Long> childrenOrgIds = orgStructDao.findOrgIdsByParentIdsLike(parentIds + "%");
            descendantOrgIds.addAll(childrenOrgIds);
        }
        this.bindOrgs(user, descendantOrgIds, user.getModules(), true, false, false);
    }

    /**
     * @param userList
     * @return
     * @author feihu.wang
     */
    private List<UserExportDTO> buildUserForExport(List<User> userList) {
        return userList.stream().map(user -> {
            UserExportDTO userDTO = new UserExportDTO();
            BeanUtils.copyProperties(user.getAccount(), userDTO);
            BeanUtils.copyProperties(user, userDTO);
            userDTO.setUserName(user.getUserName());
            userDTO.setChangePasswordFlag(user.getAccount().getChangePasswordFlag() == true ? 1 : 0);
            if (user.getUserSex() != null) {
                userDTO.setUserSex(user.getUserSex() == 0 ? "男" : "女");
            }
            SourceTypeEnum sourceTypeEnum = BaseEnum.getEnum(SourceTypeEnum.class, user.getSourceType());
            String sourceType = Objects.isNull(sourceTypeEnum) ? "未知" : sourceTypeEnum.getSourceTypeDesc();
            userDTO.setSourceType(sourceType);
            userDTO.setStatus(user.getStatus().toString());
            userDTO.setExpiredDate(user.getExpiredDate() != null ? user.getExpiredDate().toString() : null);
            return userDTO;
        }).collect(Collectors.toList());
    }

    /**
     * @param userList
     * @return
     * @author feihu.wang
     */
    private List<UserOrgExportDTO> buildUserOrgForExport(List<User> userList) {
        List<UserOrgExportDTO> all = Lists.newArrayList();
        for (User user : userList) {
            List<UserOrgExportDTO> userOrgExportDTOList = user.getCurrentOrgs().stream().map(org -> {
                UserOrgExportDTO userOrgExportDTO = new UserOrgExportDTO();
                BeanUtils.copyProperties(user.getAccount(), userOrgExportDTO);
                userOrgExportDTO.setOrgCode(org.getOrgCode());
                userOrgExportDTO.setOrgName(org.getOrgName());
                userOrgExportDTO.setTaxNum(org.getTaxNum());
                if (!CollectionUtils.isEmpty(org.getCompanyNos())) {
                    userOrgExportDTO.setCompanyNos(org.getCompanyNos().stream().collect(Collectors.joining(",")));
                }
                return userOrgExportDTO;
            }).collect(Collectors.toList());
            all.addAll(userOrgExportDTOList);
        }
        return all;
    }

    /**
     * export user role.
     *
     * @param userList
     * @return
     * @author feihu.wang
     */
    private List<UserRoleExportDTO> buildUserRoleForExport(List<User> userList) {
        List<UserRoleExportDTO> all = Lists.newArrayList();
        for (User user : userList) {
            List<UserRoleExportDTO> userRoleExportDTOList = user.getRoles().stream().map(role -> {
                UserRoleExportDTO userRoleDto = new UserRoleExportDTO();
                BeanUtils.copyProperties(user.getAccount(), userRoleDto);
                userRoleDto.setRoleCode(role.getCode());
                userRoleDto.setRoleName(role.getName());
                return userRoleDto;
            }).collect(Collectors.toList());
            all.addAll(userRoleExportDTOList);
        }
        return all;
    }

    /**
     * export user role.
     *
     * @param userList
     * @return
     * @author feihu.wang
     */
    private List<UserTagDTO> buildUserInvoiceTypeExport(List<User> userList) {
        List<UserTagDTO> all = Lists.newArrayList();
        all = userList.stream().map(user -> {
            List<UserTagDTO> invoiceList = Lists.newArrayList();
            for (UserTag userTag : user.getUserTags()) {
                if ("invoiceType".equalsIgnoreCase(userTag.getTagName())) {
                    String[] invoices = userTag.getTagValue().split(",");
                    Arrays.stream(invoices).forEach(inv -> {
                        UserTagDTO tag = new UserTagDTO();
                        tag.setUsername(user.getAccount().getUsername());
                        tag.setTelPhone(user.getAccount().getTelPhone());
                        tag.setEmail(user.getAccount().getEmail());
                        tag.setInvoiceType(inv);
                        invoiceList.add(tag);
                    });
                }
            }
            return invoiceList;
        }).flatMap(list -> list.stream()).distinct().collect(Collectors.toList());
        return all;
    }

    /**
     * export terminal.
     *
     * @param userList
     * @return
     * @author feihu.wang
     */
    private List<UserTerminalDTO> buildUserTerminalExport(List<User> userList) {
        return userList.stream().map(user -> {
            List<UserTerminalDTO> terminalList = Lists.newArrayList();
            for (UserTag userTag : user.getUserTags()) {
                if (("ticketOpeningTerminal").equalsIgnoreCase(userTag.getTagName())) {
                    if (StringUtils.isNotEmpty(userTag.getTagValue())) {
                        String[] terminals = userTag.getTagValue().split(",");

                        Map<String, MsTerminalQueryResponseInfo> resultMap = terminalApiService.getTerminalMap(Arrays.asList(terminals));

                        for (String terminal : terminals) {
                            UserTerminalDTO dto = new UserTerminalDTO();
                            BeanUtils.copyProperties(user.getAccount(), dto);
                            dto.setTicketOpeningTerminal(terminal);
                            MsTerminalQueryResponseInfo terminalInfo = resultMap.get(terminal);
                            if (terminalInfo != null) {
                                if (terminalInfo.getCompanyInfo() != null) {
                                    dto.setCompanyName(terminalInfo.getCompanyInfo().getCompanyName());
                                    dto.setTaxNum(terminalInfo.getCompanyInfo().getTaxNo());
                                }
                                dto.setDevNum(terminalInfo.getTerminalNo());
                            }
                            terminalList.add(dto);
                        }
                    }
                }
            }
            return terminalList;
        }).flatMap(list -> list.stream()).distinct().collect(Collectors.toList());
    }

    /**
     * equipment.
     *
     * @param userList
     * @return
     */
    private List<UserTerminalDTO> buildUserEquipmentExport(List<User> userList) {
        return userList.stream().map(user -> {
            List<UserTerminalDTO> equipmentList = Lists.newArrayList();
            for (UserTag userTag : user.getUserTags()) {
                if (("printingEquipment").equalsIgnoreCase(userTag.getTagName())) {
                    if (StringUtils.isNotEmpty(userTag.getTagValue())) {
                        String[] strings = userTag.getTagValue().split(",");
                        Map<String, MsDeviceInfo> resultMap = terminalApiService.getDeviceMap(Arrays.asList(strings));

                        for (String equipment : strings) {
                            UserTerminalDTO dto = new UserTerminalDTO();
                            BeanUtils.copyProperties(user.getAccount(), dto);
                            dto.setPrintingEquipment(equipment);

                            MsDeviceInfo terminalInfo = resultMap.get(equipment);
                            if (terminalInfo != null) {
                                if (terminalInfo.getCompanyInfo() != null) {
                                    dto.setCompanyName(terminalInfo.getCompanyInfo().getCompanyName());
                                    dto.setTaxNum(terminalInfo.getCompanyInfo().getTaxNo());
                                }
                                dto.setDevNum(terminalInfo.getDeviceNo());
                            }

                            equipmentList.add(dto);
                        }
                    }
                }
            }
            return equipmentList;
        }).flatMap(list -> list.stream()).distinct().collect(Collectors.toList());
    }

    /**
     * 业务扩展字段.
     *
     * @param userList
     * @return
     */
    private List<UserTagDTO> buildUserExtExport(List<User> userList) {
        return userList.stream().map(user -> {
            UserTagDTO dto = new UserTagDTO();
            for (UserTag userTag : user.getUserTags()) {
                if (("businessExtensionAttribute").equalsIgnoreCase(userTag.getTagName())) {
                    BeanUtils.copyProperties(user.getAccount(), dto);
                    dto.setBusinessExtensionAttribute(userTag.getTagValue());
                    dto.setTagType("businessExtensionAttribute");
                }
            }
            return dto;
        }).collect(Collectors.toList());
    }


    private String getAccountName(Account account) {
        String accountName = account.getEmail();
        if (StringUtils.isEmpty(accountName)) {
            accountName = account.getTelPhone();
        }
        if (StringUtils.isEmpty(accountName)) {
            accountName = account.getUsername();
        }
        return accountName;
    }

    /**
     * 填充数据到excel.
     *
     * @param userList
     * @param simpleExcelWriter
     * @param sheets
     */
    public void fillExcel(List<User> userList, SimpleExcelWriter simpleExcelWriter, Collection<String> sheets) {
        if (CollectionUtils.isEmpty(userList)) {
            return;
        }
        int partitionSize = userList.size() > ImportExportThreadPool.CORE_POOL_SIZE ? userList.size() / ImportExportThreadPool.CORE_POOL_SIZE : 1;
        List<List<User>> partitionList = ListUtils.partition(userList, partitionSize);
        List<CompletableFuture<Void>> futureList = new ArrayList<>();

        partitionList.forEach(list -> futureList.add(CompletableFuture.runAsync(() ->
                        list.forEach(item -> this.fulfill(item, BinaryUtils.toBinary(ExtraInfo.currentOrgs))),
                ImportExportThreadPool.get())));
        try {
            CompletableFuture.allOf(futureList.toArray(new CompletableFuture[]{})).get();
        } catch (InterruptedException e) {
            e.printStackTrace();
        } catch (ExecutionException e) {
            e.printStackTrace();
        }
        List<User> userListWithFullInfo = new ArrayList<>();
        partitionList.forEach(item -> userListWithFullInfo.addAll(item));

        for (String sheet : sheets) {
            switch (sheet) {
                case SN_USER:
                    List<UserExportDTO> userExportDTOList = this.buildUserForExport(userListWithFullInfo);
                    simpleExcelWriter.fill(SN_USER, userExportDTOList);
                    break;
                case SN_USER_INVOICE_TYPE:
                    List<UserTagDTO> userTypeDTOList = this.buildUserInvoiceTypeExport(userListWithFullInfo);
                    simpleExcelWriter.fill(SN_USER_INVOICE_TYPE, userTypeDTOList);
                    break;
                case SN_USER_DEVICE:
                    List<UserTerminalDTO> userEquipmentDTOList = this.buildUserEquipmentExport(userListWithFullInfo);
                    simpleExcelWriter.fill(SN_USER_DEVICE, userEquipmentDTOList);
                    break;
                case SN_USER_TERMINAL:
                    List<UserTerminalDTO> userTerminalDTOList = this.buildUserTerminalExport(userListWithFullInfo);
                    simpleExcelWriter.fill(SN_USER_TERMINAL, userTerminalDTOList);
                    break;
                case SN_USER_ORG:
                    List<UserOrgExportDTO> userOrgExportDTOList = this.buildUserOrgForExport(userListWithFullInfo);
                    simpleExcelWriter.fill(SN_USER_ORG, userOrgExportDTOList);
                    break;
                case SN_USER_ROLE:
                    List<UserRoleExportDTO> userRoleExportDTOList = this.buildUserRoleForExport(userListWithFullInfo);
                    simpleExcelWriter.fill(SN_USER_ROLE, userRoleExportDTOList);
                    break;
                case SN_USER_EXT:
                    List<UserTagDTO> userExtExportDTOList = this.buildUserExtExport(userListWithFullInfo);
                    simpleExcelWriter.fill(SN_USER_EXT, userExtExportDTOList);
                    break;
                default:
                    break;
            }
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public String bindRoleAccountRel(RoleAccountImportDto dto, Long tenantId) {
        List<String> errMsg = new ArrayList<>();

        if (StringUtils.isBlank(dto.getAction())) {
            errMsg.add("操作不能为空");
        }
        if (StringUtils.isBlank(dto.getAccount())) {
            errMsg.add("账号不能为空");
        }
        if (StringUtils.isBlank(dto.getRoleCode())) {
            errMsg.add("角色代码不能为空");
        }
        if (!ACTION_LISTS.contains(dto.getAction())) {
            return "操作类型不正确，操作类型应该为:{" + ACTION_BIND + "," + ACTION_UNBIND + "}";
        }

        Account account = null;
        if (StringUtils.isNotBlank(dto.getAccount())) {
            try {
                account = accountService.findOneByUsername(dto.getAccount());
            } catch (IllegalArgumentException e) {
                errMsg.add("账号\"" + dto.getAccount() + "\"不存在");
            }
        }

        Role role = null;
        if (StringUtils.isNotBlank(dto.getRoleCode())) {
            role = roleDao.findByTenantIdAndCode(tenantId, dto.getRoleCode());
            /**
             * 对于预置角色添加判断 如果是当前用户可见的预置角色 且角色可用，也可以进行授权
             */
            role = ObjectCheckAndExcuteUtils.docheckAndExcute(role, x -> {
                return Objects.isNull(x)
                        && roleDao.queryCountPreRoleByTenantIdAndRoleIdOrRoleCode(tenantId, null, dto.getRoleCode(), RoleTypeEnum.PRE.getType()) > 0;
            }, x -> {
                return roleDao.findByTenantIdAndCode(PreRoleService.PER_TENANT_ID, dto.getRoleCode());
            }, x -> {
                return x;
            });
            if (role == null) {
                errMsg.add("角色代码\"" + dto.getRoleCode() + "\"不存在");
            }
        }

        if (!CollectionUtils.isEmpty(errMsg)) {
            return String.join(";", errMsg);
        }

        Query userQuery = new Query();
        userQuery.setAccountId(account.getAccountId());
        userQuery.setTenantId(tenantId);
        List<User> userList = this.list(userQuery, Sort.unsorted());

        List<Long> roleIds = Arrays.asList(role.getId());

        if (ACTION_BIND.equals(dto.getAction())) {

            if (!CollectionUtils.isEmpty(userList)) {
                for (User user : userList) {
                    this.bindRoles(user, null, roleIds, null, null, null, false, false, false);
                }
            }

        } else if (ACTION_UNBIND.equals(dto.getAction())) {

            if (!CollectionUtils.isEmpty(userList)) {
                for (User user : userList) {
                    this.unbindRoles(tenantId, user.getId(), roleIds);
                }
            }
        }
        return null;
    }

    @Transactional(rollbackFor = Exception.class)
    public void saveUserTag(UserTag userTag) {
        userTagDao.saveAndFlush(userTag);
    }

    public Optional<User> findByTenantIdAndUsername(Long tenantId, String useranme) {
        Objects.requireNonNull(tenantId, "tenantId not null");
        Objects.requireNonNull(useranme, "username not null");
        Query query = new Query();
        query.setTenantId(tenantId);
        query.setAccountName(useranme);
        Page<User> userPage = this.page(query, PageRequest.ofSize(1));
        return userPage.get().findFirst();
    }

    /**
     * only  user and account info.
     *
     * @author feihu.wang
     */
    @Transactional(rollbackFor = Exception.class)
    public void saveUserPure(Tenant tenant, Create userModel, AccountModel.Request.Create accountModel, Account account) {

        if (account == null) {
            AccountModel.Request.Create newAccount = new AccountModel.Request.Create();
            newAccount.setTelPhone(accountModel.getTelPhone());
            newAccount.setEmail(accountModel.getEmail());
            newAccount.setUsername(accountModel.getUsername());
            newAccount.setPassword(accountModel.getPassword());
            newAccount.setStatus(1);
            newAccount.setType(accountModel.getType());
            newAccount.setEnableSendMsg(false);
            newAccount.setChangePasswordFlag(accountModel.isChangePasswordFlag());
            account = accountService.save(tenant, newAccount, accountModel.isEnableSendMsg(), accountModel.isRandomPassword(), true, accountModel.getUpdateIgnoreProperties());
        }
        User user = new User();
        BeanUtils.copyProperties(userModel, user, Stream.of("account").toArray(String[]::new));
        user.setAccountId(account.getAccountId());
        user.setTenantId(tenant.getTenantId());
        user.setStatus(userModel.getStatus());
        user.setActiveStatus(userModel.getStatus());
        //添加一个联系地址
        user.setContactAddr(userModel.getContactAddr());
        this.saveAndFlush(user);
    }

    /**
     * only  user and account info.
     *
     * @author feihu.wang
     */
    @Transactional(rollbackFor = Exception.class)
    public void updateUserPure(User user, Account account) {
        this.saveAndFlush(user);
        accountService.saveAndFlush(account);
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteUserTagById(Long userTagId) {
        userTagDao.deleteById(userTagId);
    }


    public Optional<User> findUserById(Long userId) {
        Specification<User> specification = (Specification<User>) (root, criteriaQuery, builder) -> {
            List<Predicate> predicates = new ArrayList<>();
            if (userId != null && userId > 0) {
                predicates.add(builder.equal(root.<Long>get("id"), userId));
            }
            if (!predicates.isEmpty()) {
                criteriaQuery.where(predicates.stream().toArray(Predicate[]::new));
            }
            return criteriaQuery.getRestriction();
        };
        return userDao.findOne(specification, EntityGraphs.named(User.NAMED_ENTITY_GRAPH_ACCOUNT));
    }

    /**
     * 按ID查询
     *
     * @param ids
     * @return List<User>
     */
    public List<User> findAllById(Collection<Long> ids) {
        return this.userDao.findAllById(ids);
    }

    /**
     * 拷贝人员权限。
     *
     * @param tenantId
     * @param userId
     * @param privilege
     */
    public void copyPrivilege(Long tenantId, CopyPrivilege privilege) {
        User sourceUser = findById(privilege.getFromUserId(), 16);
        if (!tenantId.equals(sourceUser.getTenantId())) {
            throw new IllegalArgumentException("移交人只能为当前租户！");
        }
        Set<Long> toUserIds = privilege.getToUserIds().stream().filter(userId -> !sourceUser.getId().equals(userId)).collect(Collectors.toSet());
        List<String> errorList = new ArrayList<>();
        for (Long toUserId : toUserIds) {
            Optional<User> toUserOptional = this.findUserById(toUserId);
            if (toUserOptional.isPresent()) {
                User toUser = toUserOptional.get();
                if (!tenantId.equals(toUser.getTenantId())) {
                    errorList.add("受让人(" + toUserId + ")只能为当前租户");
                }
                if (1 != toUser.getAccount().getStatus()) {
                    errorList.add("受让人账号(" + toUserId + ")已停用");
                }
                if (1 != toUser.getStatus()) {
                    errorList.add("受让人(" + toUserId + ")已停用");
                }
                if (null != toUser.getExpiredDate() && toUser.getExpiredDate().before(new Date())) {
                    errorList.add("受让人(" + toUserId + ")已过期");
                }
            } else {
                errorList.add("未找到用户实体(" + toUserId + ")");
            }
        }
        if (!CollectionUtils.isEmpty(errorList)) {
            throw new IllegalArgumentException(StringUtils.join(errorList.toArray(), ";"));
        }

        for (Long toUserId : toUserIds) {
            if (privilege.isWithRole()) {
                List<Long> roleIds = sourceUser.getRoles().stream().map(r -> r.getId()).collect(Collectors.toList());
                this.bindRoles(tenantId, null, toUserId, roleIds, null, null, null, false, true, false);
            }
            if (privilege.isWithOrg()) {
                List<Long> orgIds = sourceUser.getCurrentOrgs().stream().map(org -> org.getOrgId()).collect(Collectors.toList());
                this.bindOrgs(tenantId, toUserId, orgIds, null, false, true, false);
            }
            if (privilege.isWithBusiness()) {
                User destUser = this.findById(toUserId);
                Set<String> invoiceTypeSet = new HashSet<>();
                Set<String> equipmentSet = new HashSet<>();
                Set<String> terminalSet = new HashSet<>();
                List<UserTag> tags = sourceUser.getUserTags();
                String sourceBusinessTagVal = null;
                for (UserTag tag : tags) {
                    String[] sourceCodes = tag.getTagValue().split(",");
                    if ("invoiceType".equalsIgnoreCase(tag.getTagName())) {
                        invoiceTypeSet.addAll(Arrays.asList(sourceCodes));
                    }
                    if ("printingEquipment".equalsIgnoreCase(tag.getTagName())) {
                        equipmentSet.addAll(Arrays.asList(sourceCodes));
                    }
                    if ("ticketOpeningTerminal".equalsIgnoreCase(tag.getTagName())) {
                        terminalSet.addAll(Arrays.asList(sourceCodes));
                    }
                    if ("businessExtensionAttribute".equalsIgnoreCase(tag.getTagName())) {
                        sourceBusinessTagVal = tag.getTagValue();
                    }
                }
                UserTag invoiceTypeTag = null;
                UserTag equipmentTag = null;
                UserTag terminalTag = null;
                UserTag businessTag = null;
                for (UserTag userTag : destUser.getUserTags()) {
                    String[] destCodes = userTag.getTagValue().split(",");
                    if ("invoiceType".equalsIgnoreCase(userTag.getTagName())) {
                        invoiceTypeSet.addAll(Arrays.asList(destCodes));
                        userTag.setTagValue(invoiceTypeSet.stream().filter(t -> !"".equals(t)).collect(Collectors.joining(",")));
                        invoiceTypeTag = userTag;
                        userTagDao.saveAndFlush(userTag);
                    } else if ("printingEquipment".equalsIgnoreCase(userTag.getTagName())) {
                        equipmentSet.addAll(Arrays.asList(destCodes));
                        userTag.setTagValue(equipmentSet.stream().filter(t -> !"".equals(t)).collect(Collectors.joining(",")));
                        equipmentTag = userTag;
                        userTagDao.saveAndFlush(userTag);
                    } else if ("ticketOpeningTerminal".equalsIgnoreCase(userTag.getTagName())) {
                        terminalSet.addAll(Arrays.asList(destCodes));
                        userTag.setTagValue(terminalSet.stream().filter(t -> !"".equals(t)).collect(Collectors.joining(",")));
                        terminalTag = userTag;
                        userTagDao.saveAndFlush(userTag);
                    } else if ("businessExtensionAttribute".equalsIgnoreCase(userTag.getTagName())) {
                        businessTag = userTag;
                    }
                }
                if (invoiceTypeTag == null) {
                    invoiceTypeTag = new UserTag();
                    invoiceTypeTag.setTagName("invoiceType");
                    invoiceTypeTag.setTagValue(invoiceTypeSet.stream().filter(t -> !"".equals(t)).collect(Collectors.joining(",")));
                    invoiceTypeTag.setUserId(toUserId);
                    userTagDao.saveAndFlush(invoiceTypeTag);
                }
                if (equipmentTag == null) {
                    equipmentTag = new UserTag();
                    equipmentTag.setTagName("printingEquipment");
                    equipmentTag.setTagValue(equipmentSet.stream().filter(t -> !"".equals(t)).collect(Collectors.joining(",")));
                    equipmentTag.setUserId(toUserId);
                    userTagDao.saveAndFlush(equipmentTag);
                }
                if (terminalTag == null) {
                    terminalTag = new UserTag();
                    terminalTag.setTagName("ticketOpeningTerminal");
                    terminalTag.setTagValue(terminalSet.stream().filter(t -> !"".equals(t)).collect(Collectors.joining(",")));
                    terminalTag.setUserId(toUserId);
                    userTagDao.saveAndFlush(terminalTag);
                }
                if (businessTag == null && StringUtils.isNotEmpty(sourceBusinessTagVal)) {
                    businessTag = new UserTag();
                    businessTag.setTagName("businessExtensionAttribute");
                    businessTag.setTagValue(sourceBusinessTagVal);
                    businessTag.setUserId(toUserId);
                    userTagDao.saveAndFlush(businessTag);
                }
            }
        }
        if (privilege.isDisableFromUser()) {
            sourceUser.setStatus(0);
            userDao.saveAndFlush(sourceUser);
        }
    }

    /**
     * @param tenantId
     * @param batchSync
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public List<UserModel.Response.BatchSyncResult> batchCreate(long tenantId, BatchSync batchSync) {
        List<UserModel.Response.BatchSyncResult> resultList = Lists.newArrayList();
        Tenant tenant = null;
        tenant = tenantDao.findById(tenantId).orElseThrow(() -> new IllegalArgumentException("非法的租户id(" + tenantId + ")"));
        //如果为空则返回
        if (CollectionUtils.isEmpty(batchSync.getUsers())) {
            throw new IllegalArgumentException("用户list不能为空");
        }
        for (StandardCreate userModel : batchSync.getUsers()) {
            UserModel.Response.BatchSyncResult result = new UserModel.Response.BatchSyncResult<>();
            Account account = null;
            User user = null;
            long accountId = 0;
            long userId = 0;
            try {
                Set<ConstraintViolation<StandardCreate>> constraintViolations = validator.validate(userModel, Default.class);
                if (!CollectionUtils.isEmpty(constraintViolations)) {
                    String messages = constraintViolations.stream().map(ConstraintViolation::getMessage).collect(Collectors.joining(","));
                    result.setCode("-1");
                    result.setMessage(messages);
                    resultList.add(result);
                    continue;
                }
                if (userModel.getAccountType().equals(AccountType.PHONE_EMAIL) && StringUtils.isBlank(userModel.getPhone()) && StringUtils.isBlank(userModel.getEmail())) {
                    result.setCode("-1");
                    result.setMessage("Phone Or Email 不能为空!");
                    resultList.add(result);
                    continue;
                }
                if (userModel.getAccountType().equals(AccountType.OTHER) && StringUtils.isBlank(userModel.getUserCode())) {
                    result.setCode("-1");
                    result.setMessage("创建域账号要求 UserCode 不能为空!");
                    resultList.add(result);
                    continue;
                }

                if (userModel.getAccountId() != null && userModel.getAccountId() > 0) {
                    accountId = userModel.getAccountId();
                    if (accountId > 0) {
                        final long finalAccountId = accountId;
                        account = accountDao.findById(accountId).orElseThrow(() -> new IllegalArgumentException("非法的账户id(" + finalAccountId + ")"));
                    }
                }
                if (userModel.getUserId() != null) {
                    user = userDao.findById(userModel.getUserId(), EntityGraphs.named(User.NAMED_ENTITY_GRAPH_ACCOUNT)).orElseThrow(() -> new IllegalArgumentException("不存在的userId(" + userModel.getUserId() + ")"));
                    if (user.getTenantId() == null || tenantId != user.getTenantId()) {
                        String message = "用户所在的租户(" + user.getTenantId() + ")和传入参数的租户(" + tenantId + ")不一致";
                        logger.warn(message);
                        throw new IllegalArgumentException(message);
                    }
                    userId = user.getId();
                }

                if (StringUtils.isNotBlank(userModel.getUserCode())) {

                    Optional<User> userOptional = this.findUserByTenantIdAndUserCode(tenantId, userModel.getUserCode(), batchSync.isMergeAccount(), EntityGraphs.named(User.NAMED_ENTITY_GRAPH_ACCOUNT));
                    if (userOptional.isPresent()) {
                        if (batchSync.isMergeAccount()) {
                            user = userOptional.get();
                            account = user.getAccount();
                            accountId = user.getAccountId();
                        } else {
                            throw new IllegalArgumentException("重复的用户code(" + userModel.getUserCode() + ")");
                        }
                    }
                }
                //保存account
                if (account == null) {
                    account = accountService.create(tenant,
                            userModel.getPhone(),
                            userModel.getEmail(),
                            userModel.getUserCode(),
                            userModel.getPassword(),
                            userModel.getStatus(),
                            userModel.getAccountType(),
                            batchSync.isEnableSendMsg(),
                            batchSync.isRandomPassword(),
                            batchSync.isMergeAccount(),
                            batchSync.isChangePasswordFlag(),
                            null);
                    accountId = account.getAccountId();
                }

                //region 保存User
                boolean isNew = false;

                if (user == null && accountId > 0) {
                    List<User> users = userDao.findByTenantIdAndAccountId(tenantId, accountId);
                    user = users.stream().findFirst().orElse(null);
                }
                // 如果已经存在userCode则获取该user对象
                if (user == null && StringUtils.isNotBlank(userModel.getUserCode())) {
                    Optional<User> userOptional = userDao.findByTenantIdAndUserCode(tenantId, userModel.getUserCode());
                    if (userOptional.isPresent()) {
                        user = userOptional.get();
                    }
                }

                if (user != null) {
                    //如果存在user对象则更新
                    if (StringUtils.isBlank(userModel.getUserCode())) {
                        userModel.setUserCode(null);
                    } else {
                        //检查用户代码是否已经存在
                        this.validExistsByTenantIdAndUserCode(tenantId, user.getId(), userModel.getUserCode());
                    }
                    BeanUtils.copyProperties(userModel, user, Stream.of("account").toArray(String[]::new));
                } else {
                    user = new User();
                    isNew = true;
                    BeanUtils.copyProperties(userModel, user, Stream.of("account").toArray(String[]::new));
                    user.setTenantId(tenantId);
                    user.setAccountId(accountId);
                }
                user.setUserPhone(userModel.getPhone());
                user.setUserEmailAddr(userModel.getEmail());
                if (userModel.getStatus() != null) {
                    user.setStatus(userModel.getStatus());
                    user.setActiveStatus(userModel.getStatus());
                }

                user = this.saveAndFlush(user);
                // save user tags
                this.saveUserTags(user.getId(), userModel.getInvoiceType(), userModel.getPrintingEquipment(), userModel.getTicketOpeningTerminal(), userModel.getBusinessExtensionAttribute(), batchSync.isTagOverwrite());
                if (logger.isInfoEnabled()) {
                    logger.debug("user = " + JsonUtils.toJson(user));
                }
                //region check OrgUserRel
                Set<Long> orgIds = new HashSet<>();
                if ((userModel.getOrgIds() != null || userModel.getOrgCodes() != null)) {
                    if (!CollectionUtils.isEmpty(userModel.getOrgIds())) {
                        List<Long> filteredOrgIds = orgStructDao.findOrgIdsByTenantIdAndOrgIds(tenantId, userModel.getOrgIds());
                        orgIds.addAll(filteredOrgIds);
                    }
                    if (!CollectionUtils.isEmpty(userModel.getOrgCodes())) {
                        // 如果orgId == 0 直接根据orgCode绑定用户和组织
                        Set<String> orgCodes = userModel.getOrgCodes();
                        List<OrgStruct> orgs = new ArrayList<>();
                        if (CollectionUtils.isEmpty(orgCodes)) { // 如果orgCodes为空且是新建 则 创建根组织且绑定到该组织节点
                            if (isNew) { // 是新建用户
                                orgs = orgStructDao.findRootsByTenantId(tenantId);
                            }
                        } else {
                            orgs = orgStructDao.findByTenantIdAndOrgCodes(tenantId, orgCodes);
                            if (orgCodes.size() != orgs.size()) {
                                List<OrgStruct> finalOrgs = orgs;
                                Set<String> diffOrgCodes = orgCodes.stream().filter(orgCode -> finalOrgs.stream().map(OrgStruct::getOrgCode).noneMatch(finalOrgCode -> finalOrgCode.equals(orgCode))).collect(Collectors.toSet());
                                String message = "User(" + user.getUserCode() + ")包含含有数据库中不存在的组织代码(" + String.join(",", diffOrgCodes) + ")";
                                logger.warn(message);
                                if (batchSync.isStrict()) {
                                    throw new IllegalArgumentException(message);
                                }
                            }
                        }
                        if (!orgs.isEmpty()) {
                            Set<Long> orgIdSet = orgs.stream().map(OrgStruct::getOrgId).collect(Collectors.toSet());
                            logger.info("orgIdSet = " + orgIdSet.stream().map(Object::toString).collect(Collectors.joining(",")));
                            orgIds.addAll(orgIdSet);
                        }
                    }

                }
                // endregion 保存 OrgUserRel

                //region check RoleUserRel
                Set<Long> roleIds = new HashSet<>();
                if (userModel.getRoleIds() != null || userModel.getRoleCodes() != null) {
                    List<Role> roles;
                    if (!CollectionUtils.isEmpty(userModel.getRoleIds())) {
                        Set<Long> roleIdSet = userModel.getRoleIds().stream().filter(roleId -> roleId != null && roleId > 0).collect(Collectors.toSet());
                        logger.info("roleIdSet = " + roleIdSet.stream().map(Object::toString).collect(Collectors.joining(",")));

                        roles = roleDao.findAllById(roleIdSet);
                        //region 过滤不合法的角色id
                        roleIdSet = roles.stream().filter(role -> (role.getTenantId().equals(tenantId) || role.getTenantId() == -1)).map(Role::getId).collect(Collectors.toSet());
                        logger.debug("roleIdSet = " + roleIdSet.stream().map(Object::toString).collect(Collectors.joining(",")));
                        roleIds.addAll(roleIdSet);
                        //endregion
                        Set<Role> copingRoleTemplates = roles.stream().filter(role -> role.getTenantId() == 0).collect(Collectors.toSet());
                        Set<Role> copiedRoles = roleService.copyRoleTempates(tenantId, copingRoleTemplates);

                        if (!CollectionUtils.isEmpty(copiedRoles)) {
                            Set<Long> copiedRoleIds = copiedRoles.stream().map(Role::getId).collect(Collectors.toSet());
                            roleIds.addAll(copiedRoleIds);
                        }
                    }
                    Set<String> roleCodes = userModel.getRoleCodes();
                    if (!CollectionUtils.isEmpty(roleCodes)) {
                        roles = roleDao.findByTenantIdAndRoleCodes(tenantId, roleCodes);
                        if (roleCodes.size() != roles.size()) {
                            String message = "User(" + user.getUserCode() + ")包含含有数据库中不存在的角色代码";
                            logger.warn(message);
                            if (batchSync.isStrict()) {
                                throw new IllegalArgumentException(message);
                            }
                        }
                        if (!roles.isEmpty()) {
                            Set<Long> roleIdSet = roles.stream().map(Role::getId).collect(Collectors.toSet());
                            logger.debug("roleIdSet = " + roleIdSet.stream().map(Object::toString).collect(Collectors.joining(",")));
                            roleIds.addAll(roleIdSet);
                        }
                    }
                }
                //endregion 保存 RoleUserRel

                //region User 绑定 Org
                this.bindOrgs(user, orgIds, null, batchSync.isOrgOverwrite(), batchSync.isStrict(), false);
                //endregion User 绑定 Org

                //region User 绑定 Role
                this.bindRoles(user, null, roleIds, null, null, null, batchSync.isRoleOverwrite(), batchSync.isStrict(), false);
                //endregion User 绑定 Role

                user.setPrintingEquipment("");
                user.setTicketOpeningTerminal("");
                user.setBusinessExtensionAttribute("");
                result.setUserInfo(JsonUtils.toJson(user, View.Info.class));
            } catch (IllegalArgumentException e) {
                logger.error("用户同步失败", e);
                result.setCode("-1");
                result.setMessage(e.getMessage());
            } catch (Exception e) {
                logger.error("用户同步异常", e);
                result.setCode("-2");
                result.setMessage(e.getMessage() + ":" + getAccountName(userModel));
            }
            resultList.add(result);
        }
        return resultList;
    }

    private String getAccountName(StandardCreate userModel) {
        if (StringUtils.isNotBlank(userModel.getEmail())) {
            return userModel.getEmail();
        }
        if (StringUtils.isNotBlank(userModel.getPhone())) {
            return userModel.getPhone();
        }
        if (StringUtils.isNotBlank(userModel.getUserCode())) {
            return userModel.getUserCode();
        }
        return "";
    }

    @Transactional(rollbackFor = Exception.class)
    public User changeTenantByAccount(long accountId, long tenantId) {
        Query query = new Query();
        query.setAccountId(accountId);
        query.setTenantId(tenantId);
        query.setStatus(1);
        List<User> users = userDao.findAll(UserQueryHelper.querySpecification(query), EntityGraphs.named(User.NAMED_ENTITY_GRAPH_DEFAULT));
        User user = users.stream().findAny().orElseThrow(() -> new IllegalArgumentException("不存在该用户(tenantId:" + tenantId + ",accountId:" + accountId + ")"));
        user = this.fulfill(user, 0);
        Account account = user.getAccount();
        user.setUsername(account.getUsername());
        user.setEmail(account.getEmail());
        user.setMobile(account.getTelPhone());
        return user;
    }

    @Transactional(rollbackFor = Exception.class)
    public User changeTenant(Long userId, Long tenantId) {
        User existUser = this.findById(userId);
        Query query = new Query();
        query.setAccountId(existUser.getAccountId());
        query.setTenantId(tenantId);
        query.setStatus(1);
        List<User> users = userDao.findAll(UserQueryHelper.querySpecification(query));
        return users.stream().findAny().orElseThrow(() -> new IllegalArgumentException("不存在该用户(tenantId:" + tenantId + ",accountId:" + existUser.getAccountId() + ")"));
    }

    @Transactional(rollbackFor = Exception.class)
    public void bindPrintingEquipment(User user, List<String> equipmentList) {
        this.bindTag(user, equipmentList, PRINTING_EQUIPMENT);
    }

    private void bindTag(User user, List<String> equipmentList, String tagName) {
        Assert.notNull(user, "用戶不能為空");
        List<UserTag> userTags = userTagDao.findByUserId(user.getId());
        UserTag userTag1 = null;

        for (UserTag userTag : userTags) {
            if (tagName.equals(userTag.getTagName())) {
                userTag1 = userTag;
                break;
            }
        }
        // 1. 不存在tag key
        if (userTag1 == null) {
            userTag1 = new UserTag();
            userTag1.setUserId(user.getId());
            userTag1.setTagName(tagName);
            String value = StringUtils.join(equipmentList, Separator.DEFAULT_FILED_SEPARATOR);
            userTag1.setTagValue(value);
            userTagDao.saveAndFlush(userTag1);
            return;
        }

        // 2. tagvalue 為空
        if (StringUtils.isEmpty(userTag1.getTagValue())) {
            String value = StringUtils.join(equipmentList, Separator.DEFAULT_FILED_SEPARATOR);
            userTag1.setTagValue(value);
            userTagDao.saveAndFlush(userTag1);
            return;
        }

        // 3. 已有， 取合集
        String[] existValueArray =
                StringUtils.split(userTag1.getTagValue(),
                        Separator.DEFAULT_FILED_SEPARATOR);
        Set<String> unionValues = new HashSet<>(Arrays.asList(existValueArray));
        unionValues.addAll(equipmentList);
        String newTagValue = StringUtils.join(unionValues, Separator.DEFAULT_FILED_SEPARATOR);

        userTag1.setTagValue(newTagValue);
        userTagDao.saveAndFlush(userTag1);
        return;
    }

    @Transactional(rollbackFor = Exception.class)
    public void unbindPrintingEquipment(User user, List<String> equipmentList) {
        this.unbindTag(user, equipmentList, PRINTING_EQUIPMENT);
    }

    private void unbindTag(User user, List<String> equipmentList, String tagName) {
        Assert.notNull(user, "用戶不能為空");
        List<UserTag> userTags = userTagDao.findByUserId(user.getId());
        UserTag userTag1 = null;

        for (UserTag userTag : userTags) {
            if (tagName.equals(userTag.getTagName())) {
                userTag1 = userTag;
                break;
            }
        }
        // 1. 不存在tag key
        if (userTag1 == null) {
            //初始化空值
            userTag1 = new UserTag();
            userTag1.setUserId(user.getId());
            userTag1.setTagName(tagName);
            userTag1.setTagValue(StringUtils.EMPTY);
            userTagDao.saveAndFlush(userTag1);
            return;
        }

        // 2. tagvalue 為空
        if (StringUtils.isEmpty(userTag1.getTagValue())) {
            //支持冪等操作，直接return
            return;
        }

        // 3. 已有， 取消已有的
        String[] existValueArray =
                StringUtils.split(userTag1.getTagValue(),
                        Separator.DEFAULT_FILED_SEPARATOR);

        List<String> reduce1 = Arrays.asList(existValueArray).stream().filter(item -> !equipmentList.contains(item)).collect(Collectors.toList());

        String newTagValue = StringUtils.join(reduce1, Separator.DEFAULT_FILED_SEPARATOR);

        userTag1.setTagValue(newTagValue);
        userTagDao.saveAndFlush(userTag1);
        return;
    }

    @Transactional(rollbackFor = Exception.class)
    public void bindTicketTerminal(User user, List<String> terminalList) {
        this.bindTag(user, terminalList, TICKET_TERMINAL);
    }

    @Transactional(rollbackFor = Exception.class)
    public void unbindTicketTerminal(User user, List<String> terminalList) {
        this.unbindTag(user, terminalList, TICKET_TERMINAL);
    }

    /**
     * 检查上次密码修改时间距离当前是否超过90天
     *
     * @param pwdLastUpdateTime
     * @return
     */
    private Boolean needModifyPassword(Date pwdLastUpdateTime) {
        LocalDate pwdLastUpdateDate = new DateTime(pwdLastUpdateTime).toLocalDate();
        return pwdLastUpdateDate.plusDays(90).compareTo(LocalDate.now()) <= 0;
    }

    @Transactional(rollbackFor = Exception.class)
    public void changeTenantManager(String username, Long tenantId) {
        Tenant tenant = tenantDao.findById(tenantId).orElseThrow(() -> new IllegalArgumentException(("不存在该租户(" + tenantId + ")")));

        String tenantUsername = this.getDomainAccountName(username, tenant.getTenantCode());

//        User user = this.findByTenantIdAndUsername(tenantId, tenantUsername).orElseThrow(() -> new IllegalArgumentException("租户（" + tenant.getTenantName() + "），不存在用户名为(" + username + ")"));
        Optional<User> userOptional = this.findByTenantIdAndUsername(tenantId, tenantUsername);

        if (!userOptional.isPresent()) {
            //fix bug: 旧的账号可能没有添加tenantCode,使用用户名查找
            userOptional = this.findByTenantIdAndUsername(tenantId, username);
            if (!userOptional.isPresent()) {
                throw new IllegalArgumentException("租户（" + tenant.getTenantName() + "），不存在用户名为(" + username + ")");
            }
        }

        User user = userOptional.get();

        if (!tenantId.equals(user.getTenantId())) {
            throw new IllegalArgumentException("该用户不属于租户(" + tenantId + ")");
        }
        List<RoleUserRel> rels = roleUserRelDao.findByRoleIdAndTenantId(1, tenantId);
        if (!CollectionUtils.isEmpty(rels)) {

            List<RoleUserRel> roleUserRels = rels.stream().filter(rel -> user.getId().equals(rel.getUserId())).collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(roleUserRels)) {
                return;
            }
            RoleUserRel roleUserRel = rels.stream().findFirst().get();
            roleUserRelDao.delete(roleUserRel);
            RoleUserRel saveObj = new RoleUserRel();
            saveObj.setUserId(user.getId());
            saveObj.setTenantId(tenantId);
            saveObj.setRoleId(1L);
            roleUserRelDao.saveAndFlush(saveObj);
        }
    }


    public List<TenantManagerDto> findTenantManager(List<Long> tenantIds) {
        return userDao.findTenantManager(tenantIds);
    }

    public String getDomainAccountName(String username, String tenantCode) {
        if ("wilmar".equalsIgnoreCase(tenantCode)) {
            return username;
        }

        if (EnvProfile.isCrcProfile) {
            return username;
        }
        if (RegExUtil.checkEmail(username)) {
            return username;
        }
        if (RegExUtil.checkMobile(username)) {
            return username;
        }
        if (!username.contains(tenantCode)) {
            username = tenantCode + username;
        }
        return username;

    }

    /**
     * 统一save-update pub
     *
     * @param user
     * @return
     */
    public User saveAndFlush(User user) {
        try {
            user = userDao.saveAndFlush(user);
        } catch (Exception e) {
            logger.warn(e.getMessage());
        }

        final Long accountId = user.getAccountId();
        final Long tenantId = user.getTenantId();
        Account account = accountDao.findById(user.getAccountId()).orElseThrow(() -> new IllegalArgumentException("无法找到该账号(" + accountId + ")"));
        Tenant tenant = tenantDao.findById(user.getTenantId()).orElseThrow(() -> new IllegalArgumentException("无法找到该租户(" + tenantId + ")"));
        userPubService.pub(user, account, tenant);
        if (user.getAccount() == null || user.getAccount().getCreateTime() == null) {
            user.setAccount(account);
        }
        if (user.getTenant() == null || user.getTenant().getCreateTime() == null) {
            user.setTenant(tenant);
        }
        return user;
    }


    public void logoff(long userId, String loginId) {
        userRedisCacheService.cleanUserByUserIdAndLoginId(userId, loginId);
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteByOrgIdAndId(long orgId, long userId) {
        userDao.deleteByOrgIdAndId(orgId, userId);
    }

    private User findOneValidUser(List<User> users) {
        //处理异常情况： 一个租户下有一个禁用，一个启用的用户；需要返回启用的用户 如果都禁用，保持兼容，返回第一个
        User user = null;
        for (User item : users) {
            if (user == null) {
                user = item;
                if (item.getStatus() != null && 1 == item.getStatus()) {
                    break;
                }
            }
        }
        return user;
    }

    /**
     * 校验用户信息
     */
    public void checkUserModel(Save model) {
        if (Objects.isNull(model.getSourceType())) {
            model.setSourceType(SourceTypeEnum.INTERNAL.getSourceType());
        } else {
            SourceTypeEnum sourceTypeEnum = BaseEnum.getEnum(SourceTypeEnum.class, model.getSourceType());
            if (Objects.isNull(sourceTypeEnum)) {
                throw new UnknownException("未知用户类型：sourceType:" + model.getSourceType());
            }
            model.setSourceType(sourceTypeEnum.getSourceType());
        }
    }

    public String getUserTag(Long userId, String tagName) {
        List<UserTag> tagList = userTagDao.findByUserIdAndTagName(userId, tagName);
        if (tagList == null || CollectionUtils.isEmpty(tagList)) {
            return null;
        } else {
            //错误数据兼容，取第一个
            UserTag userTag = tagList.get(0);
            return userTag.getTagValue();
        }
    }

}
