package com.xforceplus.business.tenant.service;

import com.xforceplus.api.model.CompanyModel;
import com.xforceplus.api.model.OrgModel.Request.Save;
import com.xforceplus.dao.OrgStructDao;
import com.xforceplus.dao.TenantDao;
import com.xforceplus.entity.Company;
import com.xforceplus.entity.OrgStruct;
import com.xforceplus.entity.Tenant;
import com.xforceplus.tenant.security.core.domain.OrgType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import javax.validation.Valid;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * @author geewit
 */
@Service
public class WrapperOrgService {
    private static final Logger logger = LoggerFactory.getLogger(WrapperOrgService.class);

    private final OrgService orgService;

    private final AsyncOrgUserService orgUserService;

    private final TenantDao tenantDao;

    private final OrgStructDao orgStructDao;

    public WrapperOrgService(OrgService orgService, AsyncOrgUserService orgUserService, TenantDao tenantDao, OrgStructDao orgStructDao) {
        this.orgService = orgService;
        this.orgUserService = orgUserService;
        this.tenantDao = tenantDao;
        this.orgStructDao = orgStructDao;
    }

    @Transactional(rollbackFor = Exception.class)
    public OrgStruct create(@Valid Save model) {
        OrgStruct orgStruct = orgService.create(model);
        if (model.getIsAutoBindParentOrgUsers()) {
            orgUserService.autoBindParentUsers(orgStruct.getTenantId(), orgStruct.getOrgId());
        }
        return orgStruct;
    }

    @Transactional(rollbackFor = Exception.class)
    public OrgService.OrgSaveOutput save(long tenantId, Save model, boolean isOverwrite) {
        if (!tenantDao.existsById(tenantId)) {
            String message = "不合法的租户id(" + tenantId + ")";
            logger.warn(message);
            throw new IllegalArgumentException(message);
        }
        logger.info("save Org tenantId({})", tenantId);
        OrgService.OrgSaveInput orgSaveInput = new OrgService.OrgSaveInput(tenantId, 0, null, new ArrayList<>());
        OrgService.OrgSaveOutput orgSaveOutput = this.save(orgSaveInput, model, isOverwrite);
        return orgSaveOutput;
    }

    /**
     * 保存组织
     *
     * @param tenantId 租户id
     * @param model    OrgModel.Request.Save
     * @return
     */
    public OrgService.OrgSaveOutput save(long tenantId, long parentId, OrgStruct parent, Save model, boolean isOverwrite) {
        if (!tenantDao.existsById(tenantId)) {
            String message = "不合法的租户id(" + tenantId + ")";
            logger.warn(message);
            throw new IllegalArgumentException(message);
        }
        logger.info("save Org tenantId({})", tenantId);
        OrgService.OrgSaveInput orgSaveInput = new OrgService.OrgSaveInput(tenantId, parentId, parent, new ArrayList<>());
        OrgService.OrgSaveOutput orgSaveOutput = this.save(orgSaveInput, model, isOverwrite);
        return orgSaveOutput;
    }

    /**
     * 批量保存
     *
     * @param tenantId
     * @param models
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public void save(long tenantId, List<Save> models) {
        this.save(tenantId, 0, models);
    }

    /**
     * 批量保存
     *
     * @param tenantId
     * @param models
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public void save(long tenantId, long parentId, List<Save> models) {
        if (CollectionUtils.isEmpty(models)) {
            return;
        }
        if (!tenantDao.existsById(tenantId)) {
            String message = "不合法的租户id(" + tenantId + ")";
            logger.warn(message);
            throw new IllegalArgumentException(message);
        }

        OrgStruct parentOrg = null;
        if (parentId > 0) {
            String message = "不存在的父组织id(" + parentId + ")";
            parentOrg = orgStructDao.findOne((Specification<OrgStruct>) (root, query, criteriaBuilder) -> criteriaBuilder.equal(root.get("orgId"), parentId)).orElseThrow(() -> new IllegalArgumentException(message));
            if (parentOrg == null) {
                logger.warn(message);
                throw new IllegalArgumentException(message);
            }
            if (tenantId != parentOrg.getTenantId()) {
                logger.warn(message);
                throw new IllegalArgumentException(message);
            }
        }

        OrgService.OrgSaveInput orgSaveInput = new OrgService.OrgSaveInput(tenantId, parentId, parentOrg, new ArrayList<>());

        models.forEach(model -> this.save(orgSaveInput, model, model.isOverwrite()));
    }

    /**
     * 批量保存
     *
     * @return
     */
    public OrgService.OrgSaveOutput save(OrgService.OrgSaveInput orgSaveInput, Save model, boolean isOverwrite) {
        OrgService.OrgSaveOutput orgSaveOutput = orgService.saveOrg(orgSaveInput, model, isOverwrite);
        if (model.getIsAutoBindParentOrgUsers()) {
            OrgStruct org = orgSaveOutput.findFirst();
            orgUserService.autoBindParentUsers(org.getTenantId(), org.getOrgId());
        }
        return orgSaveOutput;
    }

    public <C extends CompanyModel.Request.Save> Company saveCompany(Tenant tenant, C model) {
        model.setHostTenantId(tenant.getTenantId());
        Optional<Long> rootOrgIdOptional = Optional.empty();
        List<Long> rootIds = orgStructDao.findRootIdsByTenantId(tenant.getTenantId());
        if (!rootIds.isEmpty()) {
            rootOrgIdOptional = rootIds.stream().filter(Objects::nonNull).findFirst();
        }
        if (!rootOrgIdOptional.isPresent()) {
            Save rootOrgModel = new Save();
            rootOrgModel.setTenantId(tenant.getTenantId());
            rootOrgModel.setOrgName(tenant.getTenantName());
            rootOrgModel.setOrgDesc(tenant.getTenantDesc());
            rootOrgModel.setOrgType(OrgType.GROUP);
            OrgStruct rootOrg = this.create(rootOrgModel);
            if (rootOrg != null) {
                rootOrgIdOptional = Optional.of(rootOrg.getOrgId());
            } else {
                throw new IllegalArgumentException("租户(" + tenant.getTenantId() + ")创建或获取根组织失败");
            }
        }

        return this.saveCompany(tenant.getTenantId(), rootOrgIdOptional.get(), model);
    }

    public <C extends CompanyModel.Request.Save> Company saveCompany(long tenantId, Long rootOrgId, C model) {
        Optional<Long> rootOrgIdOptional = Optional.ofNullable(rootOrgId);
        if (!rootOrgIdOptional.isPresent()) {
            Page<OrgStruct> rootOrgs = orgService.findRoots(tenantId, PageRequest.ofSize(1));
            if (!rootOrgs.isEmpty()) {
                rootOrgIdOptional = rootOrgs.stream().map(OrgStruct::getOrgId).filter(Objects::nonNull).findFirst();
            }
        }
        if (!rootOrgIdOptional.isPresent()) {
            String message = "租户(" + tenantId + ")缺少根组织, 请修复";
            throw new IllegalArgumentException(message);
        }
        Save orgModel = new Save();
        orgModel.setTenantId(tenantId);
        orgModel.setParentId(rootOrgIdOptional.get());
        orgModel.setOrgName(model.getCompanyName());
        orgModel.setOrgDesc(model.getCompanyName());
        orgModel.setOrgType(OrgType.COMPANY);
        model.setHostTenantId(tenantId);
        orgModel.setCompany(model);
        OrgStruct org = this.create(orgModel);
        return org.getCompany();
    }
}
