package com.xforceplus.business.tenant.service;

import com.xforceplus.dao.OrgVirtualNodeDao;
import com.xforceplus.dao.OrgVirtualUserRelDao;
import com.xforceplus.entity.*;
import com.xforceplus.utils.OrgVirtualNodeUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author geewit
 */
@Service
public class WrapperOrgVirtualNodeUserService {
    private static final Logger logger = LoggerFactory.getLogger(WrapperOrgVirtualNodeUserService.class);

    private final OrgVirtualNodeDao orgVirtualNodeDao;
    private final OrgVirtualUserRelDao orgVirtualUserRelDao;

    public WrapperOrgVirtualNodeUserService(OrgVirtualNodeDao orgVirtualNodeDao, OrgVirtualUserRelDao orgVirtualUserRelDao) {
        this.orgVirtualNodeDao = orgVirtualNodeDao;
        this.orgVirtualUserRelDao = orgVirtualUserRelDao;
    }

    /**
     * 根据父虚拟组织树绑定用户
     *
     * @param tenantId 租户Id
     * @param nodeId   虚拟组织树Id
     */
    @Transactional(rollbackFor = Exception.class)
    public void autoBindUsers(Long tenantId, Long nodeId) {
        //region 传入的虚拟组织树id对应的parentIds
        Optional<OrgVirtualNode> optionalOrgVirtualNode = orgVirtualNodeDao.findById(nodeId);
        if (!optionalOrgVirtualNode.isPresent()) {
            logger.warn("传的nodeId({})不存在", nodeId);
            return;
        }
        OrgVirtualNode currentNode = optionalOrgVirtualNode.get();
        if (StringUtils.isBlank(currentNode.getParentIds())) {
            logger.warn("传的nodeId({})不存在或对应的parentIds为空", nodeId);
            return;
        }

        String parentIds = currentNode.getParentIds();

        //region 传入的虚拟组织树id对应的所有父虚拟组织树id集合
        Set<Long> parentnodeIds = OrgVirtualNodeUtils.findNodeIdInParentIds(parentIds);
        parentnodeIds = parentnodeIds.stream().filter(parentId -> !parentId.equals(nodeId)).collect(Collectors.toSet());
        if (CollectionUtils.isEmpty(parentnodeIds)) {
            logger.warn("传的nodeId({})不存在或对应的parentIds为空", nodeId);
            return;
        }
        //endregion

        //region 已经绑定的用户虚拟组织树关系集合
        Set<OrgVirtualNodeUserRel> boundUserOrgVirtualNodeRels = orgVirtualUserRelDao.findByOrgVirtualNodeIdIn(parentnodeIds);
        if (CollectionUtils.isEmpty(boundUserOrgVirtualNodeRels)) {
            logger.warn("在父虚拟虚拟组织树中没有找到已绑定的用户");
            return;
        }
        //endregion

        //region 应该绑定的用户id集合
        Set<Long> shouldBoundUserIds = boundUserOrgVirtualNodeRels.stream().filter(r -> r.getControlFlag() == 1).map(OrgVirtualNodeUserRel::getUserId).collect(Collectors.toSet());
        if (CollectionUtils.isEmpty(shouldBoundUserIds)) {
            logger.warn("在父虚拟虚拟组织树中没有找到已绑定的用户");
            return;
        }
        //endregion

        //region 循环应该绑定的用户id集合, 并保存存入的虚拟组织树对应的关系
        shouldBoundUserIds.stream()
                .filter(shouldBoundUserId ->
                        boundUserOrgVirtualNodeRels.stream().noneMatch(rel -> rel.getOrgVirtualNodeId().equals(nodeId) && rel.getUserId().equals(shouldBoundUserId))
                )
                .filter(Objects::nonNull)
                .forEach(shouldBoundUserId -> {
                    OrgVirtualNodeUserRel rel = new OrgVirtualNodeUserRel();
                    rel.setOrgVirtualNodeId(nodeId);
                    rel.setTenantId(tenantId);
                    rel.setUserId(shouldBoundUserId);
                    rel.setControlFlag(0);
                    orgVirtualUserRelDao.saveAndFlush(rel);
                });
        //endregion
    }
}
