/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: CompanyConfigImportServiceImpl.java   2020-09-24 09-33-25
 * Author: Evan
 */
package com.xforceplus.business.tenant.service.impl;

import com.alibaba.excel.event.AnalysisEventListener;
import com.xforceplus.api.model.OrgModel;
import com.xforceplus.business.company.service.CompanyService;
import com.xforceplus.business.excel.BusinessType;
import com.xforceplus.business.excel.ExcelSheet;
import com.xforceplus.business.excel.company.OrgCompanyNumberDTO;
import com.xforceplus.business.excel.reader.Context;
import com.xforceplus.business.excel.reader.MessageRow;
import com.xforceplus.business.excel.reader.SimpleDataReadListener;
import com.xforceplus.business.excel.writer.ExcelConfigBusinessType;
import com.xforceplus.business.service.ExcelReaderService;
import com.xforceplus.business.tenant.service.OrgService;
import com.xforceplus.entity.OrgCompanyRel;
import com.xforceplus.entity.OrgStruct;
import com.xforceplus.tenant.security.core.domain.OrgType;
import io.geewit.web.utils.JsonUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;

/**
 * Title:
 *
 * <p>Description:
 *
 * <p>Copyright: 2015~2020
 *
 * <p>Company/Department: xforceplus
 *
 * @author Evan <b>Creation Time:</b> 2020-09-24 09-33-25
 * @since V1.0
 */
@Service
public class OrgCompanyNumberImportServiceImpl implements ExcelReaderService {
    /**
     * 新增
     */
    public static final String ACTION_ADD = "新增";
    /**
     * 修改
     */
    public static final String ACTION_DELETE = "删除";
    public static final String SHEET_NAME = "公司编号导入";

    public static final HashMap<String, Integer> SHEET_HEADER_NUMBER = new HashMap<>(1);

    static {
        // 注意，这里的 1， 就是有一行的 头信息；
        SHEET_HEADER_NUMBER.put(SHEET_NAME, 1);
    }

    /**
     * 操作列表
     */
    public static final List<String> ACTION_LISTS = Arrays.asList(ACTION_ADD, ACTION_DELETE);
    /**
     * 日志
     */
    private static final Logger log = LoggerFactory.getLogger(OrgCompanyNumberImportServiceImpl.class);
    @Autowired
    CompanyService companyService;
    @Autowired
    OrgService orgService;

    /**
     * 获取导入类型，用于Event事件调整导入方法
     *
     * @return ImportBusinessType
     */
    @Override
    public BusinessType getBusinessType() {
        return ExcelConfigBusinessType.ORG_COMPANY_NUM_IMPORT;
    }

    @Override
    public HashMap<String, Integer> getSheetHeaderNumber() {
        return SHEET_HEADER_NUMBER;
    }


    /**
     * 异步导出处理
     *
     * @param context 上下文
     * @return Context
     */
    @Override
    public Context importExcel(Context context) {
        List<ExcelSheet> sheets = context.getExcelBook().getExcelSheets();


        for (ExcelSheet sheet : sheets) {
            MessageRow messageRows = new MessageRow(sheet.getSheetName());
            Integer headerNumber = SHEET_HEADER_NUMBER.get(sheet.getSheetName());
            if (headerNumber == null) {
                headerNumber = 1;
            }
            // 构建监听器
            AnalysisEventListener<OrgCompanyNumberDTO> excelDataListener =
                    SimpleDataReadListener.listener(
                            context,
                            (rows) -> {
                                this.saveData(messageRows, context, rows);
                            });
            // 开始处理
            context.getSimpleExcelReader().read(OrgCompanyNumberDTO.class, excelDataListener, headerNumber);
        }
        return context;
    }

    /**
     * 写入数据
     *
     * @param list
     */
    public void saveData(MessageRow messageRow, Context context, List<OrgCompanyNumberDTO> list) {
        if (log.isDebugEnabled()) {
            log.debug("list:{}", JsonUtils.toJson(list));
        }

        for (OrgCompanyNumberDTO dto : list) {
            try {
                if (!dto.getValidatedStatus()) {
                    messageRow.fail(dto.getRowIndex(), dto.getValidatedMessage());
                    continue;
                }
                // 检查操作类型
                if (!ACTION_LISTS.contains(dto.getAction())) {
                    messageRow.fail(
                            dto.getRowIndex(), "操作类型不正确，操作类型应该为:{" + ACTION_ADD + "," + ACTION_DELETE + "}");
                    continue;
                }
                // 1. 通过code 获取公司信息
                OrgModel.Request.Query query = new OrgModel.Request.Query();
                query.setOrgCode(dto.getOrgCode());
                query.setTenantId(context.getFileDTO().getTenantId());

                // fix bug: 公司 code+ 租户id 数据库里面不唯一！！！
                query.setStatus(1);
                List<OrgStruct> orgStructList = orgService.list(query, Sort.unsorted());
                if (CollectionUtils.isEmpty(orgStructList)) {
                    String message = "未找到组织实体(" + dto.getOrgCode() + ")";
                    messageRow.fail(dto.getRowIndex(), message);
                    log.warn(message);
                    continue;
                }
                if (orgStructList.size() > 1) {
                    log.warn("ORG_CODE_DUPLICATE:{}", dto.getOrgCode());
                }
                // 历史数据有问题， 只能随便拿一个！
                OrgStruct orgStruct = orgStructList.get(0);

                // Fix bug: 4752 判断公司类型
                if (!OrgType.COMPANY.equals(orgStruct.getOrgType())) {
                    messageRow.fail(dto.getRowIndex(), "该组织不能绑定编号");
                    continue;
                }

                // 2. 查询绑定关系是否存在
                OrgCompanyNumberDTO orgCompanyNumberDTO = new OrgCompanyNumberDTO();
                orgCompanyNumberDTO.setOrgId(orgStruct.getOrgId());
                orgCompanyNumberDTO.setCompanyNumber(dto.getCompanyNumber());
                Optional<OrgCompanyRel> orgCompanyRel = orgService.findOrgComNum(orgCompanyNumberDTO);

                // 3. 处理新增，或者， 删除
                if (ACTION_ADD.equals(dto.getAction())) {
                    // 新增
                    if (orgCompanyRel.isPresent()) {
                        String message = "编号已存在";
                        messageRow.fail(dto.getRowIndex(), message);
                        log.warn(message);
                        continue;
                    } else {
                        Set<String> orgNumSet = new HashSet<>();
                        orgNumSet.add(dto.getCompanyNumber());
                        companyService.saveCompanyNos(orgStruct.getOrgId(), orgNumSet, false);
                    }
                } else {
                    // 删除
                    if (orgCompanyRel.isPresent()) {
                        companyService.deleteOrgComRelById(orgCompanyRel.get().getId());
                    } else {
                        String message = "编号不存在";
                        messageRow.fail(dto.getRowIndex(), message);
                        log.warn(message);
                        continue;
                    }
                }
                messageRow.success(dto.getRowIndex());
            } catch (Exception e) {
                log.warn("导入公司组织编号出错:{}", dto, e);
                messageRow.fail(dto.getRowIndex(), "处理出错");
            }
        }
        context.messageRow(SHEET_NAME, messageRow);
    }
}
