package com.xforceplus.business.tenant.service.impl;

import com.alibaba.excel.context.AnalysisContext;
import com.alibaba.excel.event.AnalysisEventListener;
import com.alibaba.excel.metadata.CellData;
import com.xforceplus.api.model.CompanyModel;
import com.xforceplus.api.model.OrgModel;
import com.xforceplus.business.company.service.CompanyExtensionService;
import com.xforceplus.business.company.service.CompanyService;
import com.xforceplus.business.excel.BusinessType;
import com.xforceplus.business.excel.DataRow;
import com.xforceplus.business.excel.ExcelSheet;
import com.xforceplus.business.excel.ExcelValidator;
import com.xforceplus.business.excel.file.ExcelFileDTO;
import com.xforceplus.business.excel.reader.Context;
import com.xforceplus.business.excel.reader.MessageRow;
import com.xforceplus.business.excel.reader.SimpleDataReadListener;
import com.xforceplus.business.excel.writer.ExcelConfigBusinessType;
import com.xforceplus.business.service.ExcelReaderService;
import com.xforceplus.business.tenant.excel.OrgCompanyExcelImportData;
import com.xforceplus.business.tenant.excel.OrgExcelImportData;
import com.xforceplus.business.tenant.excel.OrgImportExcel;
import com.xforceplus.business.tenant.service.OrgService;
import com.xforceplus.business.tenant.service.WrapperOrgService;
import com.xforceplus.config.ImportExportThreadPool;
import com.xforceplus.domain.company.CompanyDto;
import com.xforceplus.entity.Company;
import com.xforceplus.entity.CompanyApply;
import com.xforceplus.entity.OrgStruct;
import com.xforceplus.tenant.security.core.domain.OrgType;
import com.xforceplus.utils.BatchUtils;
import io.geewit.core.utils.reflection.BeanUtils;
import lombok.SneakyThrows;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.ListUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.lang3.tuple.Triple;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author geewit
 */
@Service
public class OrgExcelImportServiceImpl implements ExcelReaderService {
    private final static Logger logger = LoggerFactory.getLogger(OrgExcelImportServiceImpl.class);

    @Autowired
    private OrgService orgService;

    @Autowired
    private WrapperOrgService wrapperOrgService;

    @Autowired
    private CompanyService companyService;

    @Autowired
    private CompanyExtensionService companyExtensionService;

    private static final HashMap<String, Integer> HEADER_MAP = new HashMap<>(2);

    static {
        HEADER_MAP.put(OrgImportExcel.SHEET_NAME_1, 2);
        HEADER_MAP.put(OrgImportExcel.SHEET_NAME_2, 2);
    }

    /**
     * 避免报错，提供默认方法，需要各个实现类实现
     *
     * @return
     */
    @Override
    public HashMap<String, Integer> getSheetHeaderNumber() {
        return HEADER_MAP;
    }


    /**
     * 构建S
     *
     * @param context  系统上下文
     * @param consumer 消息处理
     * @return AnalysisEventListener
     */
    public static <T> AnalysisEventListener<T> listener(Context context, Consumer<List<T>> consumer) {
        return new AnalysisEventListener<T>() {
            /**
             * 数据
             */
            private List<T> list = new ArrayList<>();
            /**
             * 默认为3000，
             */
            private int batchSize = context.getFileDTO() != null ? context.getFileDTO().getBatchSize() : SimpleDataReadListener.BATCH_SIZE;

            private List<CompletableFuture<Void>> futureList = new ArrayList<>();


            /**
             * When analysis one row trigger invoke function.
             *
             * @param data    one row value. Is is same as {@link AnalysisContext#readRowHolder()}
             * @param analysisContext
             */
            @Override
            public void invoke(T data, AnalysisContext analysisContext) {

                if (data instanceof DataRow) {
                    Integer rowIndex = analysisContext.readRowHolder().getRowIndex();
                    ((DataRow) data).setRowIndex(rowIndex);
                    if (data instanceof OrgExcelImportData) {
                        ExcelFileDTO importFileDTO = context.getFileDTO();
                        if (importFileDTO != null) {
                            Long tenantId = importFileDTO.getTenantId();
                            if (tenantId != null) {
                                ((OrgExcelImportData) data).setTenantId(tenantId);
                            }
                        }
                    }
                }
                //全局校验
                list.add(data);
                if (list.size() >= batchSize) {
                    this.process(list);
                }
            }

            /**
             * 获取文件头信息
             * @param headMap
             * @param analysisContext
             */
            @Override
            public void invokeHead(Map<Integer, CellData> headMap, AnalysisContext analysisContext) {
                context.setHeadMap(analysisContext.readSheetHolder().getSheetName(), headMap);
                super.invokeHead(headMap, analysisContext);
            }

            @Override
            public void onException(Exception exception, AnalysisContext context) throws Exception {
                if (list.size() > 0) {
                    this.process(list);
                }
                super.onException(exception, context);
            }

            /**
             * if have something to do after all analysis
             *
             * @param context
             */
            @Override
            public void doAfterAllAnalysed(AnalysisContext context) {
                if (list.size() > 0) {
                    this.process(list);
                }
            }

            @SneakyThrows
            private void process(List<T> dataList) {
                //拷贝要处理的一批数据
                List<T> dataToProcess = new ArrayList<>(dataList);
                CompletableFuture<Void> future = BatchUtils.doBatchAsync(dataToProcess, this::batchValidate);
                future.get();
                consumer.accept(dataToProcess);
                list.clear();
            }

            private void batchValidate(List<T> dataList) {
                dataList.forEach(ExcelValidator::validate);
            }
        };
    }

    /**
     * 获取导入类型，用于Event事件调整导入方法
     * @return ImportBusinessType
     */
    @Override
    public BusinessType getBusinessType() {
        return ExcelConfigBusinessType.ORG_IMPORT;
    }

    @Override
    public Context importExcel(Context context) {
        List<ExcelSheet> sheets = context.getExcelBook().getExcelSheets();
        List<OrgExcelImportData> orgDatas = new ArrayList<>();
        List<OrgCompanyExcelImportData> companyDatas = new ArrayList<>();
        this.readSheet(context, sheets, orgDatas, companyDatas);
        this.saveOrgData(context, orgDatas, companyDatas);
        return context;
    }

    private void readSheet(Context context, List<ExcelSheet> sheets, List<OrgExcelImportData> orgDatas, List<OrgCompanyExcelImportData> companyDatas) {

        Optional<ExcelSheet> sheet1Optional = sheets.stream().filter(sheet -> OrgImportExcel.SHEET_NAME_1.equals(sheet.getSheetName())).findAny();
        if (sheet1Optional.isPresent()) {
            this.readSheet1(context, companyDatas);
        }
        Optional<ExcelSheet> sheet2Optional = sheets.stream().filter(sheet -> OrgImportExcel.SHEET_NAME_2.equals(sheet.getSheetName())).findAny();
        if (sheet2Optional.isPresent()) {
            this.readSheet2(context, orgDatas);
        }
    }

    /**
     * 读取 ‘公司’ sheet 到 companyDatas
     *
     * @param context
     * @param companyDatas
     */
    private void readSheet1(Context context, List<OrgCompanyExcelImportData> companyDatas) {
        Integer headerNumber = HEADER_MAP.get(OrgImportExcel.SHEET_NAME_1);
        if (headerNumber == null) {
            headerNumber = 2;
        }
        MessageRow messageRow = new MessageRow(OrgImportExcel.SHEET_NAME_1);
        context.messageRow(OrgImportExcel.SHEET_NAME_1, messageRow);
        //构建监听器
        AnalysisEventListener<OrgCompanyExcelImportData> excelReadListener
                = SimpleDataReadListener.listener(context, rows -> {
            logger.info("consume company data, size:{}", rows.size());
            for (OrgCompanyExcelImportData row : rows) {
                if (!row.getValidatedStatus()) {
                    logger.debug("数据校验不通过, continue");
                    messageRow.fail(row.getRowIndex(), row.getValidatedMessage());
                    continue;
                }
                companyDatas.add(row);
            }
        });
        //开始处理
        context.getSimpleExcelReader().read(OrgCompanyExcelImportData.class, excelReadListener, OrgImportExcel.SHEET_NAME_1, headerNumber);
    }

    /**
     * 读取 ‘组织’ sheet 到 orgDatas
     *
     * @param context
     * @param orgDatas
     */
    private void readSheet2(Context context, List<OrgExcelImportData> orgDatas) {
        Integer headerNumber = HEADER_MAP.get(OrgImportExcel.SHEET_NAME_2);
        if (headerNumber == null) {
            headerNumber = 2;
        }
        MessageRow messageRow = new MessageRow(OrgImportExcel.SHEET_NAME_2);
        context.messageRow(OrgImportExcel.SHEET_NAME_2, messageRow);
        //构建监听器
        AnalysisEventListener<OrgExcelImportData> excelReadListener
                = OrgExcelImportServiceImpl.listener(context, rows -> {
            logger.info("consume org data, size:{}", rows.size());
            for (OrgExcelImportData row : rows) {
                if (!row.getValidatedStatus()) {
                    logger.debug("数据校验不通过, continue");
                    messageRow.fail(row.getRowIndex(), row.getValidatedMessage());
                    continue;
                }
                orgDatas.add(row);
            }
        });
        //开始处理
        context.getSimpleExcelReader().read(OrgExcelImportData.class, excelReadListener, OrgImportExcel.SHEET_NAME_2, headerNumber);
    }

    /**
     * 写入公司配置数据
     */
    public void saveOrgData(Context context, List<OrgExcelImportData> orgDatas, List<OrgCompanyExcelImportData> companyDatas) {
        logger.info("saving org data,company size={},org size={}", companyDatas.size(), orgDatas.size());
        Map<String, CompanyDto> companies = new HashMap<>(companyDatas.size());
        Map<String, Integer> companyRows = new HashMap<>(companyDatas.size());
        Map<String, OrgCompanyExcelImportData> companyRowData = new HashMap<>(companyDatas.size());
        List<Triple<String, Integer, Integer>> tripleList = new ArrayList<>();
        Map<String, OrgExcelImportData> orgExcelImportDataMap = orgDatas.stream()
                .collect(
                        Collectors.toMap(OrgExcelImportData::getTaxNum, Function.identity(), (oldData, newData) -> newData)
                );
        //region 构造保存company或companyApply的hashmap
        for (OrgCompanyExcelImportData companyData : companyDatas) {
            if (!companyData.getValidatedStatus()) {
                logger.info("公司数据校验不通过, continue");
                continue;
            }
            companyRowData.put(companyData.getTaxNum(), companyData);
            //判断是否是新增
            if (companyData.isNew()) {
                CompanyApply companyApply = new CompanyApply();
                BeanUtils.copyProperties(companyData, companyApply, Stream.of("company", "companyApply").toArray(String[]::new));
                companies.put(companyApply.getTaxNum(), companyApply);
                companyRows.put(companyData.getTaxNum(), companyData.getRowIndex());
            } else {
                //判断是否数据库中不存在记录
                if (companyData.getCompany() != null) {
                    Company company = companyData.getCompany();
                    BeanUtils.copyProperties(companyData, company, Stream.of("company", "companyApply").toArray(String[]::new));
                    companies.put(company.getTaxNum(), company);
                    companyRows.put(companyData.getTaxNum(), companyData.getRowIndex());
                } else {
                    CompanyApply companyApply = companyData.getCompanyApply();
                    BeanUtils.copyProperties(companyData, companyApply, Stream.of("company", "companyApply").toArray(String[]::new));
                    companies.put(companyApply.getTaxNum(), companyApply);
                    companyRows.put(companyData.getTaxNum(), companyData.getRowIndex());
                }
            }
            //解决组织在未通过校验的情况，或组织信息没有情况处理公司信息的错误信息
            String taxNum = companyData.getTaxNum();
            if (orgExcelImportDataMap.containsKey(taxNum)) {
                //region 记录 company 成功数
                context.success(OrgImportExcel.SHEET_NAME_1, companyData.getRowIndex());
            } else {
                context.messageRow(OrgImportExcel.SHEET_NAME_1, companyData.getRowIndex(), "公司税号为：(" + taxNum + ")的公司，相应对应组织校验没有通过或没有组织");
            }
            //endregion
        }
        //endregion

        List<Pair<OrgModel.Request.Save, Integer>> orgPairs = new ArrayList<>();
        //region 构造保存保存组织list
        for (OrgExcelImportData orgData : orgDatas) {
            //判断是否校验通过
            if (!orgData.getValidatedStatus()) {
                logger.info("组织数据校验不通过, continue");
                continue;
            }
            OrgModel.Request.Save orgSave = new OrgModel.Request.Save();

            //region 新建 且 税号不为空, 需要税号在公司sheet中存在
            CompanyDto companyDto = null;
            if (StringUtils.isNotBlank(orgData.getTaxNum())) {
                companyDto = companies.get(orgData.getTaxNum());
                if (companyDto == null) {
                    Optional<Company> company = companyService.findOneByTaxNum(orgData.getTaxNum());
                    if (company.isPresent()) {
                        companyDto = company.get();
                        orgSave.setWithApplication(false);
                    }
                }
                if (companyDto == null) {
                    context.messageRow(OrgImportExcel.SHEET_NAME_2, orgData.getRowIndex(), "新增组织税号不存在");
                    continue;
                } else {
                    orgData.setOrgType(OrgType.COMPANY);
                    OrgCompanyExcelImportData orgCompanyExcelImportData = companyRowData.get(orgData.getTaxNum());
                    if (orgCompanyExcelImportData != null && orgCompanyExcelImportData.isNew()) {
                        orgSave.setWithApplication(true);
                    } else {
                        //如果组织已经存在，且数据已进入申请表和修改表，此不在保存数据
                        orgSave.setWithApplication(false);
                    }
                }
                //终端鉴权
                tripleList.add(Triple.of(orgData.getTaxNum(), orgData.getRowIndex(), orgData.getTerminalAuth() == null ? 1 : orgData.getTerminalAuth()));
            }
            //endregion

            //region 修改 且 组织 id 已经在 validator 中获取则直接使用该 id
            if (!orgData.isNew()) {
                OrgStruct org = orgData.getOrg();
                if (org == null) {
                    continue;
                }
                orgSave.setOrgId(org.getOrgId());
                orgSave.setIsAutoBindParentOrgUsers(false);
            }
            //endregion
            CompanyModel.Request.Save companySave = new CompanyModel.Request.Save();

            BeanUtils.copyProperties(orgData, orgSave, Stream.of("company").toArray(String[]::new));
            if (companyDto != null) {
                BeanUtils.copyProperties(companyDto, companySave, Stream.of("company").toArray(String[]::new));
                orgSave.setCompany(companySave);
            }

            orgPairs.add(Pair.of(orgSave, orgData.getRowIndex()));
            //region 记录 org 成功数
            context.success(OrgImportExcel.SHEET_NAME_2, orgData.getRowIndex());
            //endregion
        }
        //endregion

        //region 让新增的 parentCode 依赖 不存在的排队在后面
        int batchSize = orgPairs.size() > ImportExportThreadPool.CORE_POOL_SIZE ? orgPairs.size() / ImportExportThreadPool.CORE_POOL_SIZE : 1;
        List<List<Pair<OrgModel.Request.Save, Integer>>> splistList = ListUtils.partition(orgPairs, batchSize);
        List<CompletableFuture<Set<String>>> futureList = new ArrayList<>();
        List<Set<String>> setList = new ArrayList<>();
        splistList.forEach(item -> futureList.add(CompletableFuture.supplyAsync(() -> item.stream().map(Pair::getLeft)
                .filter(org -> orgService.findByTenantIdAndOrgCode(org.getTenantId(), org.getOrgCode()).isEmpty())
                .map(OrgModel.Request.Save::getOrgCode)
                .collect(Collectors.toSet()), ImportExportThreadPool.get())));
        CompletableFuture<Void> future = CompletableFuture.allOf(futureList.toArray(new CompletableFuture[]{}));
        Set<String> savingOrgCode = new HashSet<>();
        try {
            future.get();
            futureList.forEach(item -> {
                try {
                    Set<String> result = item.get();
                    savingOrgCode.addAll(result);
                } catch (InterruptedException | ExecutionException e) {
                    e.printStackTrace();
                }
            });
        } catch (InterruptedException | ExecutionException e) {
            e.printStackTrace();
        }


        orgPairs = orgPairs.stream().sorted(Comparator.comparing(pair -> {
            OrgModel.Request.Save save = pair.getLeft();
            int comparing = 0;
            if (savingOrgCode.stream().anyMatch(code -> code.equals(save.getParentCode()))) {
                comparing = comparing - 1;
            }
            if (savingOrgCode.stream().anyMatch(code -> code.equals(save.getOrgCode()))) {
                comparing = comparing - 1;
            }
            return comparing;
        })).collect(Collectors.toList());
        //endregion
        if (!CollectionUtils.isEmpty(orgPairs)) {
            this.asyncSave(orgPairs, context, companyRows, batchSize);
        }
        //组织及公司主数据处理完成后，处理终端鉴权
        this.asyncSaveTerminalAuth(tripleList, context.getFileDTO().getTenantId(), context, companyRows, batchSize);

    }

    @SneakyThrows
    private void asyncSave(List<Pair<OrgModel.Request.Save, Integer>> orgPairs, Context context, Map<String, Integer> companyRows, int batchSize) {
        List<List<Pair<OrgModel.Request.Save, Integer>>> pairList = ListUtils.partition(orgPairs, batchSize);
        List<CompletableFuture<Void>> dtFutureList = new ArrayList<>();
        pairList.forEach(item -> dtFutureList.add(CompletableFuture.runAsync(() -> this.doBatch(item, context, companyRows), ImportExportThreadPool.get())));
        CompletableFuture.allOf(dtFutureList.toArray(new CompletableFuture[]{})).get();
    }

    private void doBatch(List<Pair<OrgModel.Request.Save, Integer>> orgPairs, Context context, Map<String, Integer> companyRows) {
        for (Pair<OrgModel.Request.Save, Integer> savingPair : orgPairs) {
            OrgModel.Request.Save save = savingPair.getLeft();
            Integer rowIndex = savingPair.getRight();
            if (save.getOrgId() != null) {
                try {
                    wrapperOrgService.save(save.getTenantId(), save, true);
                } catch (Exception e) {
                    String message = "第" + rowIndex + "行出错: " + e.getMessage();
                    logger.warn(message);
                    context.messageRow(OrgImportExcel.SHEET_NAME_2, rowIndex, e.getMessage());
                    //设置公司数据处理错误信息
                    this.setCompanyErrorMessage(save, companyRows, context);
                }
            } else {
                try {
                    wrapperOrgService.create(save);
                } catch (Exception e) {
                    String message = "第" + rowIndex + "行出错: " + e.getMessage();
                    logger.warn(message);
                    context.messageRow(OrgImportExcel.SHEET_NAME_2, rowIndex, e.getMessage());
                    //设置公司数据处理错误信息
                    this.setCompanyErrorMessage(save, companyRows, context);
                }
            }
        }
    }


    /**
     * 处理例外处理信息
     *
     * @param save    Org保存信息
     * @param params  Map<String, Integer> 税号或行记录
     * @param context 导入导出上下文
     */
    private void setCompanyErrorMessage(OrgModel.Request.Save save, Map<String, Integer> params, Context context) {
        //没有公司信息的情况
        if (save.getCompany() == null) {
            return;
        }
        //税号为空的情况
        String taxNum = save.getCompany().getTaxNum();
        if (StringUtils.isBlank(taxNum)) {
            return;
        }
        //税号不存在的情况
        if (!params.containsKey(taxNum)) {
            return;
        }
        Integer rowIndex = params.get(taxNum);
        context.messageRow(OrgImportExcel.SHEET_NAME_1, rowIndex, "失败");
    }

    /**
     * 异步批量保存终端鉴权
     *
     * @param list        待导入数据
     * @param tenantId    租户ID
     * @param context     上下文
     * @param companyRows 公司信息
     * @param batchSize   批处理大小
     */
    private void asyncSaveTerminalAuth(List<Triple<String, Integer, Integer>> list, Long tenantId, Context context, Map<String, Integer> companyRows, int batchSize) {
        List<CompletableFuture<Void>> completableFutureList = new ArrayList<>();
        List<List<Triple<String, Integer, Integer>>> partitionList = ListUtils.partition(list, batchSize);
        partitionList.forEach(item -> completableFutureList.add(CompletableFuture.runAsync(() -> this.batchSaveTerminalAuth(item, tenantId, context, companyRows), ImportExportThreadPool.get())));
        try {
            CompletableFuture.allOf(completableFutureList.toArray(new CompletableFuture[]{})).get();
        } catch (InterruptedException | ExecutionException e) {
            logger.error("error to async batch save terminal auth {}", e.getMessage());
            throw new IllegalArgumentException("保存终端鉴权失败");
        }
    }

    /**
     * 批量保存终端鉴权
     *
     * @param dataList    待处理数据
     * @param tenantId    租户ID
     * @param context     上下文
     * @param companyRows 公司信息
     */
    private void batchSaveTerminalAuth(List<Triple<String, Integer, Integer>> dataList, Long tenantId, Context context, Map<String, Integer> companyRows) {
        dataList.forEach(item -> {
            try {
                companyExtensionService.setTerminalAuthByTaxNum(item.getLeft(), tenantId, item.getRight());
            } catch (Exception e) {
                logger.error("save terminal auth error,rowIndex={} taxNum={},message={}", item.getMiddle(), item.getLeft(), e.getMessage());
                context.messageRow(OrgImportExcel.SHEET_NAME_2, item.getMiddle(), e.getMessage());
                Integer companyIndex = companyRows.get(item.getLeft());
                if (companyIndex != null) {
                    context.messageRow(OrgImportExcel.SHEET_NAME_1, companyIndex, "终端鉴权导入失败");
                }
            }
        });
    }
}
