/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: CompanyConfigImportServiceImpl.java   2020-09-24 09-33-25
 * Author: Evan
 */
package com.xforceplus.business.tenant.service.impl;

import com.alibaba.excel.event.AnalysisEventListener;
import com.xforceplus.api.model.OrgExtensionModel;
import com.xforceplus.business.company.service.CompanyService;
import com.xforceplus.business.excel.BusinessType;
import com.xforceplus.business.excel.ExcelSheet;
import com.xforceplus.business.excel.org.OrgExtensionImportDTO;
import com.xforceplus.business.excel.reader.Context;
import com.xforceplus.business.excel.reader.MessageRow;
import com.xforceplus.business.excel.reader.SimpleDataReadListener;
import com.xforceplus.business.excel.writer.ExcelConfigBusinessType;
import com.xforceplus.business.service.ExcelReaderService;
import com.xforceplus.business.tenant.service.OrgExtensionService;
import com.xforceplus.business.tenant.service.OrgService;
import com.xforceplus.dao.OrgStructDao;
import com.xforceplus.domain.org.OrgDto;
import com.xforceplus.entity.OrgExtension;
import io.geewit.web.utils.JsonUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.Arrays;
import java.util.List;

/**
 * @Author: HWT @Date: 2020/11/9 16:44
 */
@Service
public class OrgExtensionImportServiceImpl implements ExcelReaderService {
    /**
     * 新增
     */
    public static final String ACTION_ADD = "新建";
    /**
     * 修改
     */
    public static final String ACTION_UPDATE = "修改";
    public static final String SHEET_NAME = "组织扩展字段";
    /**
     * 操作列表
     */
    public static final List<String> ACTION_LISTS = Arrays.asList(ACTION_ADD, ACTION_UPDATE);
    /**
     * 日志
     */
    private static final Logger log = LoggerFactory.getLogger(OrgExtensionImportServiceImpl.class);
    @Autowired
    private CompanyService companyService;
    @Autowired
    private OrgService orgService;
    @Autowired
    private OrgStructDao orgStructDao;
    @Autowired
    private OrgExtensionService orgExtensionService;

    /**
     * 获取导入类型，用于Event事件调整导入方法
     * @return ImportBusinessType
     */
    @Override
    public BusinessType getBusinessType() {
        return ExcelConfigBusinessType.ORG_EXTENSION_IMPORT;
    }

    /**
     * 异步导出处理
     *
     * @param context 上下文
     * @return Context
     */
    @Override
    public Context importExcel(Context context) {
        List<ExcelSheet> sheets = context.getExcelBook().getExcelSheets();
        for (ExcelSheet sheet : sheets) {
            MessageRow messageRows = new MessageRow(sheet.getSheetName());
            // 构建监听器
            AnalysisEventListener<OrgExtensionImportDTO> excelDataListener =
                    SimpleDataReadListener.listener(
                            context,
                            (rows) -> {
                                this.saveData(messageRows, context, rows);
                            });
            // 开始处理
            context.getSimpleExcelReader().read(OrgExtensionImportDTO.class, excelDataListener);
        }
        return context;
    }

    /**
     * 写入数据
     *
     * @param list
     */
    public void saveData(MessageRow messageRow, Context context, List<OrgExtensionImportDTO> list) {
        log.info("list:{}", JsonUtils.toJson(list));

        for (OrgExtensionImportDTO dto : list) {
            try {
                if (!dto.getValidatedStatus()) {
                    messageRow.fail(dto.getRowIndex(), dto.getValidatedMessage());
                    continue;
                }
                // 检查操作类型
                if (!ACTION_LISTS.contains(dto.getAction())) {
                    messageRow.fail(
                            dto.getRowIndex(), "操作类型不正确，操作类型应该为:{" + ACTION_ADD + "," + ACTION_UPDATE + "}");
                    continue;
                }

                // 1. 租户code , 组织code 联合查询公司
                List<OrgDto> orgDtoList =
                        orgStructDao.getOrgByTenantCodeAndOrgCode(dto.getTenantCode(), dto.getOrgCode());
                if (CollectionUtils.isEmpty(orgDtoList)) {
                    messageRow.fail(dto.getRowIndex(), "找不到对应的组织");
                    continue;
                }

                OrgDto orgStruct = orgDtoList.get(0);

                if (orgDtoList.size() > 1) {
                    log.warn(
                            "找到多个组织, 取第一个, tenant_code:{}, org_code:{}, org_struct_id:{}",
                            dto.getTenantCode(),
                            dto.getOrgCode(),
                            orgStruct.getOrgId());
                }

                // 2. 查询绑定关系是否存在
                List<OrgExtension> orgExtensionList = orgExtensionService.findByOrgIdAndKey(orgStruct.getOrgId(), dto.getKey());

                OrgExtension orgExtension = null;
                if (orgExtensionList != null && orgExtensionList.size() > 0) {
                    orgExtension = orgExtensionList.get(0);
                }

                // 3. 处理新增
                if (ACTION_ADD.equals(dto.getAction())) {
                    // 新增
                    if (orgExtension != null) {
                        String message = "key已存在";
                        messageRow.fail(dto.getRowIndex(), message);
                        log.warn(message);
                        continue;
                    } else {
                        OrgExtensionModel.Request.Create orgExtension1 = new OrgExtensionModel.Request.Create();
                        orgExtension1.setOrgId(orgStruct.getOrgId());
                        orgExtension1.setExtensionKey(dto.getKey());
                        orgExtension1.setExtensionValue(dto.getValue());
                        orgExtensionService.create(orgExtension1);
                    }
                } else {
                    // 修改
                    if (orgExtension != null) {

                        OrgExtensionModel.Request.Update update = new OrgExtensionModel.Request.Update();
                        update.setExtensionId(orgExtension.getOrgExtensionId());
                        update.setExtensionValue(dto.getValue());
                        orgExtensionService.update(update);
                    } else {
                        String message = "key不存在";
                        messageRow.fail(dto.getRowIndex(), message);
                        log.warn(message);
                        continue;
                    }
                }
                messageRow.success(dto.getRowIndex());
            } catch (Exception e) {
                log.warn("导入公司扩展信息出错:{}", dto, e);
                messageRow.fail(dto.getRowIndex(), "处理出错");
            }
        }
        context.messageRow(SHEET_NAME, messageRow);
    }
}
