/*
 * Copyright (c)  2015~2020, xforceplus
 * All rights reserved.
 * Project:tenant-service
 * Id: TenantUserServiceImpl.java   2020-12-29 10-09-39
 * Author: Evan
 */
package com.xforceplus.business.tenant.service.impl;

import com.xforceplus.bo.user.TenantUserBo;
import com.xforceplus.business.tenant.service.TenantUserService;
import com.xforceplus.dao.TenantDao;
import com.xforceplus.dao.user.TenantUserExtendDao;
import com.xforceplus.dto.user.TenantAdminUserDTO;
import com.xforceplus.dto.user.TenantUserDTO;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.xforceplus.api.model.user.TenantUserModel.TenantUserQueryType.CODE;
import static com.xforceplus.api.model.user.TenantUserModel.TenantUserQueryType.NAME;
import static com.xforceplus.api.utils.Separator.LIKE;

/**
 * <p>
 * Title: 租户用户服务实现
 * </p>
 * <p>
 * Description:  租户用户服务（主要用户列表查询、租户与用户关系查询）
 * </p>
 * <p>
 * Copyright: 2015~2020
 * </p>
 * <p>
 * Company/Department: xforceplus
 * </p>
 *
 * @author Evan
 * <b>Creation Time:</b> 2020-12-29 10-09-39
 * @since V1.0
 */
@Service
public class TenantUserServiceImpl implements TenantUserService {

    /**
     * 租户管理角色==1
     */
    public static final Long TENANT_ADMIN_ROLE_ID = 1L;
    /**
     * 租户用户Dao
     */
    private final TenantUserExtendDao tenantUserExtendDao;
    /**
     * 租户Dao层
     */
    private final TenantDao tenantDao;

    /**
     * 默认构建函数
     *
     * @param tenantUserExtendDao 租户用户Dao
     */
    public TenantUserServiceImpl(TenantUserExtendDao tenantUserExtendDao, TenantDao tenantDao) {
        this.tenantUserExtendDao = tenantUserExtendDao;
        this.tenantDao = tenantDao;
    }

    /**
     * 根据租户ID、租户Code、租户名称及用户状态、用户类型查询相关信息
     *
     * @param pageable     分页对象
     * @param tenantUserBo 查询条件
     * @return List<TenantAdminUserDTO>
     */
    @Override
    public Page<TenantUserDTO> findTenantUsersByCondition(Pageable pageable, TenantUserBo tenantUserBo) {
        //按租户名称查询:(like 'xx%' )
        if (tenantUserBo.getQueryType().equals(NAME)) {
            String tenantName = tenantUserBo.getTenantName() + LIKE;
            Set<Long> ids = this.tenantDao.findIdsByTenantName(tenantName);
            Assert.notEmpty(ids, "没有找到相应租户信息");
            tenantUserBo.setTenantId(null);
            tenantUserBo.setTenantIds(ids);
        } else if (tenantUserBo.getQueryType().equals(CODE)) {
            //根据Code查询
            Long tenantId = this.tenantDao.findTenantIdByTenantCode(tenantUserBo.getTenantCode());
            Assert.notNull(tenantId, "没有找到相应租户信息");
            tenantUserBo.setTenantId(tenantId);
            tenantUserBo.setTenantIds(null);
        }
        Map<Long, TenantAdminUserDTO> adminUserMap = Collections.emptyMap();
        if (null == tenantUserBo.getUserType()) {
            adminUserMap = this.findTenantAdminByTenantIds(tenantUserBo);
        } else {
            tenantUserBo.setAdminRoleId(TENANT_ADMIN_ROLE_ID);
        }
        Page<TenantUserDTO> page = this.tenantUserExtendDao.findTenantUsersByCondition(pageable, tenantUserBo);

        return this.pageConvert(tenantUserBo, adminUserMap, page);
    }

    /**
     * 分页转换器
     *
     * @param tenantUserBo 租户查询
     * @param adminUsers   租户管理员
     * @param page         分页对象
     * @return Page<TenantUserDTO>
     */
    private Page<TenantUserDTO> pageConvert(final TenantUserBo tenantUserBo, final Map<Long, TenantAdminUserDTO> adminUsers, Page<TenantUserDTO> page) {
        return page.map(e -> {
            if (tenantUserBo.getUserType() != null) {
                e.setUserType(tenantUserBo.getUserType());
                return e;
            }
            Integer tenantAdminType = adminUsers.containsKey(e.getUserId()) ? 0 : 1;
            e.setUserType(tenantAdminType);
            return e;
        });
    }

    /**
     * 查询租户下管理员列表
     *
     * @param tenantUserBo 查询对象
     * @return Map<Long, TenantAdminUserDTO>
     */
    private Map<Long, TenantAdminUserDTO> findTenantAdminByTenantIds(TenantUserBo tenantUserBo) {
        List<TenantAdminUserDTO> adminUserDTOList;
        //等于空的则查询该租户管理员信息
        if (tenantUserBo.getQueryType() != null && NAME == tenantUserBo.getQueryType()) {
            adminUserDTOList = this.findTenantAdminUserByTenantIds(tenantUserBo.getTenantIds(), TENANT_ADMIN_ROLE_ID);
        } else {
            adminUserDTOList = this.findTenantAdminUserByTenantId(tenantUserBo.getTenantId(), TENANT_ADMIN_ROLE_ID);
        }
        if (CollectionUtils.isEmpty(adminUserDTOList)) {
            return Collections.emptyMap();
        }
        return adminUserDTOList.stream()
                .collect(Collectors.toMap(TenantAdminUserDTO::getUserId, Function.identity(), (id1, id2) -> id2));
    }

    /**
     * 按租户ID集合查询租户管理员角色
     *
     * @param tenantIds 租户IDs
     * @param roleId    角色
     * @return List<TenantAdminUserDTO>
     */
    @Override
    public List<TenantAdminUserDTO> findTenantAdminUserByTenantIds(Set<Long> tenantIds, Long roleId) {
        return this.tenantUserExtendDao.findTenantAdminUserByTenantIds(tenantIds, roleId);
    }

    /**
     * 根据租户ID及用户状态、用户类型查询相关信息
     *
     * @param pageable     分页查询
     * @param tenantUserBo 查询条件
     * @return Page<TenantUserDTO>
     */
    @Override
    @Transactional(readOnly = true, rollbackFor = RuntimeException.class)
    public Page<TenantUserDTO> findTenantUsersByTenantId(Pageable pageable, TenantUserBo tenantUserBo) {
        Map<Long, TenantAdminUserDTO> adminUserMap = null;
        //等于空的则查询该租户管理员信息
        if (null == tenantUserBo.getUserType()) {
            adminUserMap = this.findTenantAdminByTenantIds(tenantUserBo);
        } else {
            tenantUserBo.setAdminRoleId(TENANT_ADMIN_ROLE_ID);
        }
        Page<TenantUserDTO> page = this.tenantUserExtendDao.findTenantUsersByTenantId(pageable, tenantUserBo);
        return this.pageConvert(tenantUserBo, adminUserMap, page);
    }

    /**
     * 根据根据租户ID和角色ID查询
     *
     * @param tenantId 租户ID
     * @param roleId   角色ID
     * @return List<TenantAdminUserDTO>
     */
    @Override
    @Transactional(readOnly = true, rollbackFor = RuntimeException.class)
    public List<TenantAdminUserDTO> findTenantAdminUserByTenantId(Long tenantId, Long roleId) {
        return this.tenantUserExtendDao.findTenantAdminUserByTenantId(tenantId, roleId);
    }
}
