package com.xforceplus.business.validator.impl;


import com.xforceplus.api.model.CompanyApplyModel;
import com.xforceplus.api.model.CompanyModel;
import com.xforceplus.business.tenant.excel.ExcelImportData;
import com.xforceplus.business.validator.ValidateCompany;
import com.xforceplus.dao.CompanyApplyDao;
import com.xforceplus.dao.CompanyDao;
import com.xforceplus.entity.Company;
import com.xforceplus.entity.CompanyApply;
import com.xforceplus.query.CompanyApplyQueryHelper;
import com.xforceplus.query.CompanyQueryHelper;
import io.geewit.core.utils.reflection.BeanUtils;
import io.geewit.core.utils.reflection.Reflections;
import io.geewit.web.utils.JsonUtils;
import lombok.Getter;
import lombok.Setter;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.validator.constraintvalidation.HibernateConstraintValidatorContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.validation.ConstraintValidator;
import javax.validation.ConstraintValidatorContext;
import java.util.List;
import java.util.Optional;

/**
 * 公司 validator
 *
 * @author geewit
 */
@Component
public class CompanyValidator implements ConstraintValidator<ValidateCompany, ExcelImportData> {
    private final static Logger logger = LoggerFactory.getLogger(CompanyValidator.class);


    @Autowired
    private CompanyDao companyDao;

    @Autowired
    private CompanyApplyDao companyApplyDao;

    private String[] actions;

    @Override
    public void initialize(ValidateCompany constraintAnnotation) {
        actions = constraintAnnotation.actions();
    }

    @Override
    public boolean isValid(ExcelImportData company, ConstraintValidatorContext constraintValidatorContext) {
        logger.info("company: {}", JsonUtils.toJson(company));
        boolean valid = true;
        boolean isNew;
        boolean isHibernateConstraintValidatorContext = constraintValidatorContext instanceof HibernateConstraintValidatorContext;
        if (StringUtils.isBlank(company.getAction())) {
            if (isHibernateConstraintValidatorContext) {
                String message = "操作类型不能为空";
                constraintValidatorContext.unwrap(HibernateConstraintValidatorContext.class)
                        .addMessageParameter("message", message);
            }
            return false;
        } else {
            switch (company.getAction()) {
                case "新建": {
                    isNew = true;
                    break;
                }
                case "修改": {
                    isNew = false;
                    break;
                }
                default: {
                    if (isHibernateConstraintValidatorContext) {
                        String message = "操作类型应该为(" + String.join(",", this.actions) + ")";
                        constraintValidatorContext.unwrap(HibernateConstraintValidatorContext.class)
                                .addMessageParameter("message", message);
                    }
                    return false;
                }
            }
        }
        //region 设置字段isNew的值
        try {
            Reflections.setFieldValue(company, "isNew", isNew);
        } catch (Exception e) {
            logger.warn(e.getMessage(), e);
        }
        //endregion

        CompanyInfo companyInfo = new CompanyInfo();
        BeanUtils.copyProperties(company, companyInfo);

        if (isNew) {
            if (StringUtils.isNotBlank(companyInfo.taxNum)) {
                CompanyModel.Request.Query query = new CompanyModel.Request.Query();
                query.setTaxNum(companyInfo.taxNum);
                Optional<Company> companyOptional = companyDao.findOne(CompanyQueryHelper.queryOneSpecification(query));
                if (companyOptional.isPresent()) {
                    if (isHibernateConstraintValidatorContext) {
                        String message = "重复的公司税号(" + companyInfo.taxNum + ")";
                        constraintValidatorContext.unwrap(HibernateConstraintValidatorContext.class)
                                .addMessageParameter("message", message);
                    }
                    valid = false;
                }
            }
            if (StringUtils.isNotBlank(companyInfo.companyCode)) {
                CompanyModel.Request.Query query = new CompanyModel.Request.Query();
                query.setCompanyCode(companyInfo.companyCode);
                Optional<Company> companyOptional = companyDao.findOne(CompanyQueryHelper.queryOneSpecification(query));
                if (companyOptional.isPresent()) {
                    if (isHibernateConstraintValidatorContext) {
                        String message = "重复的公司代码(" + companyInfo.companyCode + ")";
                        constraintValidatorContext.unwrap(HibernateConstraintValidatorContext.class)
                                .addMessageParameter("message", message);
                    }
                    valid = false;
                }
            }
            if (StringUtils.isBlank(companyInfo.companyName)) {
                if (isHibernateConstraintValidatorContext) {
                    String message = "公司名称不能为空";
                    constraintValidatorContext.unwrap(HibernateConstraintValidatorContext.class)
                            .addMessageParameter("message", message);
                }
                valid = false;
            }
        } else {
            Company existCompany = null;
            if (StringUtils.isNotBlank(companyInfo.taxNum)) {
                CompanyModel.Request.Query companyQuery = new CompanyModel.Request.Query();
                companyQuery.setTaxNum(companyInfo.taxNum);
                Optional<Company> companyOptional = companyDao.findOne(CompanyQueryHelper.queryOneSpecification(companyQuery));
                if (companyOptional.isPresent()) {
                    existCompany = companyOptional.get();
                    //region 设置字段 company 的值
                    try {
                        Reflections.setFieldValue(company, "company", existCompany);
                    } catch (Exception e) {
                        logger.warn(e.getMessage(), e);
                    }
                    //endregion
                } else {
                    CompanyApplyModel.Request.Query companyApplyQuery = new CompanyApplyModel.Request.Query();
                    companyApplyQuery.setTaxNum(companyInfo.taxNum);
                    List<CompanyApply> companyApplies = companyApplyDao.findAll(CompanyApplyQueryHelper.queryOneSpecification(companyApplyQuery));
                    if (!companyApplies.isEmpty()) {
                        //region 设置字段 company 的值
                        try {
                            Reflections.setFieldValue(company, "companyApply", companyApplies.get(0));
                        } catch (Exception e) {
                            logger.warn(e.getMessage(), e);
                        }
                        //endregion
                    } else {
                        if (isHibernateConstraintValidatorContext) {
                            String message = "不存在的公司税号(" + companyInfo.taxNum + ")";
                            constraintValidatorContext.unwrap(HibernateConstraintValidatorContext.class)
                                    .addMessageParameter("message", message);
                        }
                        return false;
                    }
                }
            }
            if (existCompany != null) {
                //region 修改了和原来不一样的 companyCode
                if (StringUtils.isNotBlank(companyInfo.companyCode) && !companyInfo.companyCode.equals(existCompany.getCompanyCode())) {
                    CompanyModel.Request.Query companyQuery = new CompanyModel.Request.Query();
                    companyQuery.setCompanyCode(companyInfo.companyCode);
                    long count = companyDao.count(CompanyQueryHelper.queryOneSpecification(companyQuery));
                    if (count > 0) {
                        if (isHibernateConstraintValidatorContext) {
                            String message = "重复的公司代码(" + companyInfo.companyCode + ")";
                            constraintValidatorContext.unwrap(HibernateConstraintValidatorContext.class)
                                    .addMessageParameter("message", message);
                        }
                        valid = false;
                    }
                }
                //endregion
                //不允许修改公司名称
                if (StringUtils.isNotBlank(companyInfo.companyName) && !companyInfo.companyName.equals(existCompany.getCompanyName())) {
                    if (isHibernateConstraintValidatorContext) {
                        String message = "导入的公司名称与原公司名称不一致";
                        constraintValidatorContext.unwrap(HibernateConstraintValidatorContext.class)
                                .addMessageParameter("message", message);
                    }
                    valid = false;
                }

            }

        }
        return valid;
    }

    @Setter
    @Getter
    static class CompanyInfo {
        private Long companyId;
        private String companyName;
        private String companyCode;
        private String taxNum;
    }
}
