package com.xforceplus.business.validator.impl;


import com.xforceplus.api.model.OrgModel;
import com.xforceplus.business.tenant.excel.OrgExcelImportData;
import com.xforceplus.business.validator.ValidateOrg;
import com.xforceplus.dao.OrgStructDao;
import com.xforceplus.entity.OrgStruct;
import com.xforceplus.query.OrgQueryHelper;
import com.xforceplus.tenant.security.core.domain.OrgType;
import io.geewit.core.utils.reflection.BeanUtils;
import io.geewit.core.utils.reflection.Reflections;
import io.geewit.web.utils.JsonUtils;
import lombok.Getter;
import lombok.Setter;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.validator.constraintvalidation.HibernateConstraintValidatorContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.validation.ConstraintValidator;
import javax.validation.ConstraintValidatorContext;
import java.util.Optional;

/**
 * 组织 validator
 *
 * @author geewit
 */
@Component
public class OrgValidator implements ConstraintValidator<ValidateOrg, OrgExcelImportData> {
    private final static Logger logger = LoggerFactory.getLogger(OrgValidator.class);

    @Autowired
    private OrgStructDao orgDao;

    private String[] actions;

    @Override
    public void initialize(ValidateOrg constraintAnnotation) {
        actions = constraintAnnotation.actions();
    }

    @Override
    public boolean isValid(OrgExcelImportData org, ConstraintValidatorContext constraintValidatorContext) {
        logger.info("org: {}", JsonUtils.toJson(org));
        boolean valid = true;
        boolean isNew;
        boolean isHibernateConstraintValidatorContext = constraintValidatorContext instanceof HibernateConstraintValidatorContext;
        if (StringUtils.isBlank(org.getAction())) {
            if (isHibernateConstraintValidatorContext) {
                String message = "操作类型不能为空";
                constraintValidatorContext.unwrap(HibernateConstraintValidatorContext.class)
                        .addMessageParameter("message", message);
            }
            return false;
        } else {
            switch (org.getAction()) {
                case "新建": {
                    isNew = true;
                    break;
                }
                case "修改": {
                    isNew = false;
                    break;
                }
                default: {
                    if (isHibernateConstraintValidatorContext) {
                        String message = "操作类型应该为(" + String.join(",", this.actions) + ")";
                        constraintValidatorContext.unwrap(HibernateConstraintValidatorContext.class)
                                .addMessageParameter("message", message);
                    }
                    return false;
                }
            }
        }
        //新模板组织状态改为非必填，如果没填默认设置为1
        if (org.getStatus() == null) {
            org.setStatus(1);
        }
        OrgInfo orgInfo = new OrgInfo();
        BeanUtils.copyProperties(org, orgInfo);
        if (orgInfo.tenantId == null || orgInfo.tenantId <= 0) {
            if (isHibernateConstraintValidatorContext) {
                String message = "非法操作(未获取用户上下文)";
                constraintValidatorContext.unwrap(HibernateConstraintValidatorContext.class)
                        .addMessageParameter("message", message);
            }
            return false;
        }
        if (isNew) {
            //新建组织，组织名称不能为空
            if (StringUtils.isBlank(orgInfo.orgName)) {
                if (isHibernateConstraintValidatorContext) {
                    String message = "组织名称不能为空";
                    constraintValidatorContext.unwrap(HibernateConstraintValidatorContext.class)
                            .addMessageParameter("message", message);
                    return false;
                }
            }
            if (StringUtils.isNotBlank(orgInfo.orgCode)) {
                OrgModel.Request.Query query = new OrgModel.Request.Query();
                query.setTenantId(orgInfo.tenantId);
                query.setOrgCode(orgInfo.orgCode);
                long count = orgDao.count(OrgQueryHelper.queryOneSpecification(query));
                if (count > 0) {
                    if (isHibernateConstraintValidatorContext) {
                        String message = "重复的组织代码(" + orgInfo.orgCode + ")";
                        constraintValidatorContext.unwrap(HibernateConstraintValidatorContext.class)
                                .addMessageParameter("message", message);
                    }
                    valid = false;
                }
            }
            //校验上级组织是否填写以及存在
            valid = this.checkParentCodeAndTaxNumber(orgInfo, isNew, isHibernateConstraintValidatorContext, constraintValidatorContext);
            if (valid) {
                org.setParentCode(orgInfo.getParentCode());
            }
        } else {
            OrgStruct existOrg = null;
            //region 根据 orgCode 查询数据库中查出 OrgStruct
            if (StringUtils.isNotBlank(orgInfo.orgCode)) {
                OrgModel.Request.Query orgQuery = new OrgModel.Request.Query();
                orgQuery.setTenantId(orgInfo.tenantId);
                orgQuery.setOrgCode(orgInfo.orgCode);
                Optional<OrgStruct> orgOptional = orgDao.findOne(OrgQueryHelper.queryOneSpecification(orgQuery));
                if (orgOptional.isPresent()) {
                    existOrg = orgOptional.get();
                    //更新操作，检查上级组织code及上级公司税号
                    valid = this.checkParentCodeAndTaxNumber(orgInfo, isNew, isHibernateConstraintValidatorContext, constraintValidatorContext);
                    //region 设置字段 org 的值
                    try {
                        Reflections.setFieldValue(org, "org", existOrg);
                        if (valid) {
                            if (orgInfo.parentCode != null) {
                                org.setParentCode(orgInfo.getParentCode());
                            }
                        }
                    } catch (Exception e) {
                        logger.warn(e.getMessage(), e);
                    }
                    //endregion
                } else {
                    if (isHibernateConstraintValidatorContext) {
                        String message = "不存在的组织编号(" + orgInfo.orgCode + ")";
                        constraintValidatorContext.unwrap(HibernateConstraintValidatorContext.class)
                                .addMessageParameter("message", message);
                    }
                    return false;
                }
            }
            //endregion
            //region 如果已存在该 orgCode 的组织
            if (existOrg != null) {
                //region 修改了和原来不一样的 orgCode
                if (StringUtils.isNotBlank(orgInfo.orgCode) && !orgInfo.orgCode.equals(existOrg.getOrgCode())) {
                    OrgModel.Request.Query orgQuery = new OrgModel.Request.Query();
                    orgQuery.setTenantId(orgInfo.tenantId);
                    orgQuery.setOrgCode(orgInfo.orgCode);
                    long count = orgDao.count(OrgQueryHelper.queryOneSpecification(orgQuery));
                    if (count > 0) {
                        if (isHibernateConstraintValidatorContext) {
                            String message = "重复的组织编号(" + orgInfo.orgCode + ")";
                            constraintValidatorContext.unwrap(HibernateConstraintValidatorContext.class)
                                    .addMessageParameter("message", message);
                        }
                        valid = false;
                    }
                }
                //endregion
            }
            //endregion

        }
        return valid;
    }

    /**
     * 校验上级组织code是否存在，以及上级组织code与上级公司税号是否一致
     *
     * @param parentCode      上级组织code
     * @param parentTaxNumber 上级公司税号
     * @return 存在且一致ture否则false
     */
    private boolean checkParentCodeAndTaxNumber(OrgInfo orgInfo, Boolean isNew,
                                                boolean isHibernateConstraintValidatorContext, ConstraintValidatorContext constraintValidatorContext) {
        String parentCode = orgInfo.getParentCode();
        String parentTaxNumber = orgInfo.getParentTaxNumber();
        Long tenantId = orgInfo.getTenantId();
        if (isNew) {
            if (StringUtils.isBlank(parentCode) && StringUtils.isBlank(parentTaxNumber)) {
                //上级组织代码及上级公司税号都为空
                if (isHibernateConstraintValidatorContext) {
                    //上级组织code不能同时为空
                    String message = "上级组织code和上级公司税号不能同时为空";
                    constraintValidatorContext.unwrap(HibernateConstraintValidatorContext.class).addMessageParameter("message", message);
                    return false;
                }
            }
        }
        if (StringUtils.isNotBlank(parentCode)) {
            OrgModel.Request.Query orgQuery = new OrgModel.Request.Query();
            orgQuery.setTenantId(tenantId);
            orgQuery.setOrgCode(parentCode);
            orgQuery.setStatus(1);
            Optional<OrgStruct> optionalOrgStruct = orgDao.findOne(OrgQueryHelper.queryOneSpecification(orgQuery));
            if (optionalOrgStruct.isPresent()) {
                if (parentTaxNumber != null) {
                    OrgStruct orgStruct = optionalOrgStruct.get();
                    if (OrgType.COMPANY.equals(orgStruct.getOrgType())) {
                        String taxNumer = orgStruct.getTaxNum();
                        if (!parentTaxNumber.equalsIgnoreCase(taxNumer)) {
                            if (isHibernateConstraintValidatorContext) {
                                //上级组织code与上级公司code不一致
                                String message = "上级组织code和上级公司税号不一致";
                                constraintValidatorContext.unwrap(HibernateConstraintValidatorContext.class).addMessageParameter("message", message);
                                return false;
                            }
                        }
                    } else {
                        if (isHibernateConstraintValidatorContext) {
                            //上级组织不为公司，传入的上级公司税号无效，该行导入失败
                            String message = "上级组织类型不是公司，非法的上级公司税号";
                            constraintValidatorContext.unwrap(HibernateConstraintValidatorContext.class).addMessageParameter("message", message);
                            return false;
                        }
                    }
                } else {
                    orgInfo.setParentCode(optionalOrgStruct.get().getOrgCode());
                    return true;
                }
            } else {
                if (isHibernateConstraintValidatorContext) {
                    //上级组织code不存在
                    String message = "上级组织code不存在";
                    constraintValidatorContext.unwrap(HibernateConstraintValidatorContext.class).addMessageParameter("message", message);
                    return false;
                }
            }
        } else if (parentTaxNumber != null) {
            //只传了上级公司税号
            Optional<OrgStruct> orgStructOptional = orgDao.findOrgStructByTaxNumAndTenantId(parentTaxNumber, tenantId);
            if (orgStructOptional.isPresent()) {
                orgInfo.setParentCode(orgStructOptional.get().getOrgCode());
                return true;
            } else {
                if (isHibernateConstraintValidatorContext) {
                    //上级公司税号不存在
                    String message = "上级公司税号不存在";
                    constraintValidatorContext.unwrap(HibernateConstraintValidatorContext.class).addMessageParameter("message", message);
                    return false;
                }
            }
        } else {
            //更新操作，不传parentCode和ParentTaxNum
            //根据组织code查询上级组织并设置上级组织，如果没找到则在更新的时候自动挂到母公司下
            OrgModel.Request.Query orgQuery = new OrgModel.Request.Query();
            orgQuery.setTenantId(tenantId);
            orgQuery.setOrgCode(orgInfo.getOrgCode());
            orgQuery.setStatus(1);
            Optional<OrgStruct> optionalOrgStruct = orgDao.findOne(OrgQueryHelper.queryOneSpecification(orgQuery));
            if (optionalOrgStruct.isPresent()) {
                Long parentId = optionalOrgStruct.get().getParentId();
                if (parentId != null) {
                    Optional<String> orgCodeOptional = orgDao.findOrgCodeByTenantIdAndOrgId(tenantId, parentId);
                    if (orgCodeOptional.isPresent()) {
                        orgInfo.setParentCode(orgCodeOptional.get());
                    }
                }
            }
            return true;
        }
        return false;
    }

    @Setter
    @Getter
    static class OrgInfo {
        private Long orgId;
        private Long tenantId;
        private String orgName;
        private String orgCode;
        private Long parentId;
        private String parentCode;
        private String parentTaxNumber;
    }
}
