package com.xforceplus.business.wechat.service;

import com.xforceplus.business.account.service.AccessTokenResp;
import com.xforceplus.business.account.service.AccessTokenService;
import com.xforceplus.config.RedisPrefixUtilService;
import com.xforceplus.dao.WechatInfoDao;
import com.xforceplus.entity.WechatInfo;
import io.geewit.web.utils.JsonUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.scheduling.annotation.Scheduled;

import javax.annotation.Resource;
import java.util.Date;
import java.util.concurrent.TimeUnit;


/**
 * 微信dService
 *
 * @author lengmz
 */
public class WechatInfoTask {
    private final static Logger logger = LoggerFactory.getLogger(WechatInfoTask.class);
    private static final int PERIOD = 7200;
    /*
     *提前20分種刷新
     */
    private static final int PRE_REFRESH = 1200;
    private static final String TASK_KEY = "wechat_access_token";
    @Resource
    private WechatInfoDao wechatInfoDao;
    @Resource
    private AccessTokenService accessTokenService;
    @Resource
    private TicketService ticketService;
    @Resource
    private RedisTemplate redisTemplate;
    @Resource
    private RedisPrefixUtilService redisPrefixUtilService;


    @Scheduled(cron = "0 0/10 * * * ?")
    public void updateTask() {
        String key = redisPrefixUtilService.processPrefix(TASK_KEY);
        Boolean result = redisTemplate.opsForValue().setIfAbsent(key, "value", 3, TimeUnit.MINUTES);
        if (result == null || !result) {
            logger.info("沒有獲取到鎖=={}", result == null ? "null" : false);
            return;
        }
        WechatInfo wechatInfo = wechatInfoDao.findFirstByStatusEquals(1);
        boolean needToken = isNeedToken(wechatInfo);
        if (needToken) {
            AccessTokenResp accessTokenResp = accessTokenService.accessToken();
            if (wechatInfo == null) {
                wechatInfo = new WechatInfo();
            }
            if (accessTokenResp != null && StringUtils.isNotBlank(accessTokenResp.getAccess_token())) {
                wechatInfo.setAccessToken(accessTokenResp.getAccess_token());
                wechatInfo.setTokenPeriod(accessTokenResp.getExpires_in());
                wechatInfo.setStatus(1);
                wechatInfo.setTokenTime(new Date());
                wechatInfo.setUpdateTime(new Date());
                wechatInfoDao.saveAndFlush(wechatInfo);
            }
        }
        boolean needTicket = isNeedTicket(wechatInfo);
        if (needTicket) {
            TicketResp ticketResp = ticketService.ticket(wechatInfo.getAccessToken());
            if (ticketResp == null || ticketResp.getErrcode() != 0) {
                return;
            }
            if (StringUtils.isNotBlank(ticketResp.getTicket())) {
                wechatInfo.setJsapiTicket(ticketResp.getTicket());
                wechatInfo.setTicketPeriod(ticketResp.getExpires_in());
                wechatInfo.setStatus(1);
                wechatInfo.setTicketTime(new Date());
                wechatInfo.setUpdateTime(new Date());
                wechatInfoDao.saveAndFlush(wechatInfo);
            }
        }
        logger.debug("wechatInfo==={}", JsonUtils.toJson(wechatInfo));
    }

    private boolean isNeedTicket(WechatInfo wechatInfo) {
        if (wechatInfo == null || StringUtils.isBlank(wechatInfo.getJsapiTicket())) {
            return true;
        }
        Date startTime = wechatInfo.getTicketTime();
        if (startTime == null) {
            return true;
        }
        Integer interval = wechatInfo.getTicketPeriod();
        if (interval == null || interval <= 0) {
            interval = PERIOD;
        }
        interval = interval - PRE_REFRESH;
        Date expireTime = DateUtils.addSeconds(startTime, interval);
        return expireTime.before(new Date());
    }

    private boolean isNeedToken(WechatInfo wechatInfo) {
        if (wechatInfo == null || StringUtils.isBlank(wechatInfo.getAccessToken())) {
            return true;
        }
        Date startTime = wechatInfo.getTokenTime();
        if (startTime == null) {
            return true;
        }
        Integer interval = wechatInfo.getTokenPeriod();
        if (interval == null || interval <= 0) {
            interval = PERIOD;
        }
        interval = interval - PRE_REFRESH;
        Date expireTime = DateUtils.addSeconds(startTime, interval);
        return expireTime.before(new Date());
    }

}
